// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'dart:async';

import 'package:file/file.dart';
import 'package:flutter_tools/src/base/file_system.dart';
import 'package:flutter_tools/src/base/io.dart';
import 'package:flutter_tools/src/globals.dart' as globals;
import 'package:process/process.dart';

import '../src/common.dart';
import 'test_data/gen_l10n_project.dart';
import 'test_driver.dart';
import 'test_utils.dart';

// Verify that the code generated by gen_l10n executes correctly.
// It can fail if gen_l10n produces a lib/l10n/app_localizations.dart that:
// - Does not analyze cleanly.
// - Can't be processed by the intl_translation:generate_from_arb tool.
// The generate_from_arb step can take close to a minute on a lightly
// loaded workstation, so the test could time out on a heavily loaded bot.
void main() {
  Directory tempDir;
  final GenL10nProject _project = GenL10nProject();
  FlutterRunTestDriver _flutter;

  setUp(() async {
    tempDir = createResolvedTempDirectorySync('gen_l10n_test.');
    await _project.setUpIn(tempDir);
    _flutter = FlutterRunTestDriver(tempDir);
  });

  tearDown(() async {
    await _flutter.stop();
    tryToDelete(tempDir);
  });

  void runCommand(List<String> command) {
    final ProcessResult result = const LocalProcessManager().runSync(
      command,
      workingDirectory: tempDir.path,
      environment: <String, String>{ 'FLUTTER_ROOT': getFlutterRoot() },
    );
    if (result.exitCode != 0) {
      throw Exception('FAILED [${result.exitCode}]: ${command.join(' ')}\n${result.stderr}\n${result.stdout}');
    }
  }

  void setUpAndRunGenL10n({List<String> args}) {
    // Get the intl packages before running gen_l10n.
    final String flutterBin = globals.platform.isWindows ? 'flutter.bat' : 'flutter';
    final String flutterPath = globals.fs.path.join(getFlutterRoot(), 'bin', flutterBin);
    runCommand(<String>[flutterPath, 'pub', 'get']);

    // Generate lib/l10n/app_localizations.dart
    final String genL10nPath = globals.fs.path.join(getFlutterRoot(), 'dev', 'tools', 'localization', 'bin', 'gen_l10n.dart');
    final String dartBin = globals.platform.isWindows ? 'dart.exe' : 'dart';
    final String dartPath = globals.fs.path.join(getFlutterRoot(), 'bin', 'cache', 'dart-sdk', 'bin', dartBin);
    runCommand(<String>[dartPath, genL10nPath, args?.join(' ')]);
  }

  Future<StringBuffer> runApp() async {
    // Run the app defined in GenL10nProject.main and wait for it to
    // send '#l10n END' to its stdout.
    final Completer<void> l10nEnd = Completer<void>();
    final StringBuffer stdout = StringBuffer();
    final StreamSubscription<String> subscription = _flutter.stdout.listen((String line) {
      if (line.contains('#l10n')) {
        stdout.writeln(line.substring(line.indexOf('#l10n')));
      }
      if (line.contains('#l10n END')) {
        l10nEnd.complete();
      }
    });
    await _flutter.run();
    await l10nEnd.future;
    await subscription.cancel();
    return stdout;
  }

  void expectOutput(StringBuffer stdout) {
    expect(stdout.toString(),
      '#l10n 0 (--- supportedLocales tests ---)\n'
      '#l10n 1 (supportedLocales[0]: languageCode: en, countryCode: null, scriptCode: null)\n'
      '#l10n 2 (supportedLocales[1]: languageCode: en, countryCode: CA, scriptCode: null)\n'
      '#l10n 3 (supportedLocales[2]: languageCode: en, countryCode: GB, scriptCode: null)\n'
      '#l10n 4 (supportedLocales[3]: languageCode: es, countryCode: null, scriptCode: null)\n'
      '#l10n 5 (supportedLocales[4]: languageCode: es, countryCode: 419, scriptCode: null)\n'
      '#l10n 6 (supportedLocales[5]: languageCode: zh, countryCode: null, scriptCode: null)\n'
      '#l10n 7 (supportedLocales[6]: languageCode: zh, countryCode: null, scriptCode: Hans)\n'
      '#l10n 8 (supportedLocales[7]: languageCode: zh, countryCode: null, scriptCode: Hant)\n'
      '#l10n 9 (supportedLocales[8]: languageCode: zh, countryCode: TW, scriptCode: Hant)\n'
      '#l10n 10 (--- countryCode (en_CA) tests ---)\n'
      '#l10n 11 (CA Hello World)\n'
      '#l10n 12 (Hello CA fallback World)\n'
      '#l10n 13 (--- countryCode (en_GB) tests ---)\n'
      '#l10n 14 (GB Hello World)\n'
      '#l10n 15 (Hello GB fallback World)\n'
      '#l10n 16 (--- zh ---)\n'
      '#l10n 17 (你好世界)\n'
      '#l10n 18 (你好)\n'
      '#l10n 19 (你好世界)\n'
      '#l10n 20 (你好2个其他世界)\n'
      '#l10n 21 (Hello 世界)\n'
      '#l10n 22 (--- scriptCode: zh_Hans ---)\n'
      '#l10n 23 (简体你好世界)\n'
      '#l10n 24 (--- scriptCode - zh_Hant ---)\n'
      '#l10n 25 (繁體你好世界)\n'
      '#l10n 26 (--- scriptCode - zh_Hant_TW ---)\n'
      '#l10n 27 (台灣繁體你好世界)\n'
      '#l10n 28 (--- General formatting tests ---)\n'
      '#l10n 29 (Hello World)\n'
      '#l10n 30 (Hello _NEWLINE_ World)\n'
      '#l10n 31 (Hello World)\n'
      '#l10n 32 (Hello World)\n'
      '#l10n 33 (Hello World on Friday, January 1, 1960)\n'
      '#l10n 34 (Hello world argument on 1/1/1960 at 00:00)\n'
      '#l10n 35 (Hello World from 1960 to 2020)\n'
      '#l10n 36 (Hello for 123)\n'
      '#l10n 37 (Hello for price USD123.00)\n'
      '#l10n 38 (Hello)\n'
      '#l10n 39 (Hello World)\n'
      '#l10n 40 (Hello two worlds)\n'
      '#l10n 41 (Hello)\n'
      '#l10n 42 (Hello new World)\n'
      '#l10n 43 (Hello two new worlds)\n'
      '#l10n 44 (Hello on Friday, January 1, 1960)\n'
      '#l10n 45 (Hello World, on Friday, January 1, 1960)\n'
      '#l10n 46 (Hello two worlds, on Friday, January 1, 1960)\n'
      '#l10n 47 (Hello other 0 worlds, with a total of 100 citizens)\n'
      '#l10n 48 (Hello World of 101 citizens)\n'
      '#l10n 49 (Hello two worlds with 102 total citizens)\n'
      '#l10n 50 ([Hello] -World- #123#)\n'
      '#l10n 51 (\$!)\n'
      '#l10n 52 (One \$)\n'
      '#l10n 53 (Flutter\'s amazing!)\n'
      '#l10n 54 (Flutter\'s amazing, times 2!)\n'
      '#l10n 55 (Flutter is "amazing"!)\n'
      '#l10n 56 (Flutter is "amazing", times 2!)\n'
      '#l10n 57 (--- es ---)\n'
      '#l10n 58 (ES - Hello world)\n'
      '#l10n 59 (ES - Hello _NEWLINE_ World)\n'
      '#l10n 60 (ES - Hello Mundo)\n'
      '#l10n 61 (ES - Hola Mundo)\n'
      '#l10n 62 (ES - Hello World on viernes, 1 de enero de 1960)\n'
      '#l10n 63 (ES - Hello world argument on 1/1/1960 at 0:00)\n'
      '#l10n 64 (ES - Hello World from 1960 to 2020)\n'
      '#l10n 65 (ES - Hello for 123)\n'
      // Note that the space between "123.00" and "EUR" is a
      // non-breaking space (U+00A0) and not a regular space(U+0020)
      '#l10n 66 (ES - Hello for el precio 123,00 EUR)\n'
      '#l10n 67 (ES - Hello)\n'
      '#l10n 68 (ES - Hello World)\n'
      '#l10n 69 (ES - Hello two worlds)\n'
      '#l10n 70 (ES - Hello)\n'
      '#l10n 71 (ES - Hello nuevo World)\n'
      '#l10n 72 (ES - Hello two nuevo worlds)\n'
      '#l10n 73 (ES - Hello on viernes, 1 de enero de 1960)\n'
      '#l10n 74 (ES - Hello World, on viernes, 1 de enero de 1960)\n'
      '#l10n 75 (ES - Hello two worlds, on viernes, 1 de enero de 1960)\n'
      '#l10n 76 (ES - Hello other 0 worlds, with a total of 100 citizens)\n'
      '#l10n 77 (ES - Hello World of 101 citizens)\n'
      '#l10n 78 (ES - Hello two worlds with 102 total citizens)\n'
      '#l10n 79 (ES - [Hola] -Mundo- #123#)\n'
      '#l10n 80 (ES - \$!)\n'
      '#l10n 81 (ES - One \$)\n'
      '#l10n 82 (ES - Flutter\'s amazing!)\n'
      '#l10n 83 (ES - Flutter\'s amazing, times 2!)\n'
      '#l10n 84 (ES - Flutter is "amazing"!)\n'
      '#l10n 85 (ES - Flutter is "amazing", times 2!)\n'
      '#l10n 86 (--- es_419 ---)\n'
      '#l10n 87 (ES 419 - Hello World)\n'
      '#l10n 88 (ES 419 - Hello)\n'
      '#l10n 89 (ES 419 - Hello World)\n'
      '#l10n 90 (ES 419 - Hello two worlds)\n'
      '#l10n END\n'
    );
  }

  test('generated l10n classes produce expected localized strings', () async {
    setUpAndRunGenL10n();
    final StringBuffer stdout = await runApp();
    expectOutput(stdout);
  });

  test('generated l10n classes produce expected localized strings with deferred loading', () async {
    setUpAndRunGenL10n(args: <String>['--use-deferred-loading']);
    final StringBuffer stdout = await runApp();
    expectOutput(stdout);
  });
}
