// Copyright (c) 2014, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'dart:collection';

import 'package:_fe_analyzer_shared/src/base/errors.dart';
import 'package:_fe_analyzer_shared/src/scanner/errors.dart';
import 'package:analyzer/diagnostic/diagnostic.dart';
import 'package:analyzer/error/listener.dart';
import 'package:analyzer/src/dart/error/ffi_code.dart';
import 'package:analyzer/src/dart/scanner/scanner.dart' show ScannerErrorCode;
import 'package:analyzer/src/diagnostic/diagnostic.dart';
import 'package:analyzer/src/error/codes.dart';
import 'package:analyzer/src/generated/java_core.dart';
import 'package:analyzer/src/generated/parser.dart' show ParserErrorCode;
import 'package:analyzer/src/generated/source.dart';
import 'package:analyzer/src/manifest/manifest_warning_code.dart';

export 'package:_fe_analyzer_shared/src/base/errors.dart'
    show ErrorCode, ErrorSeverity, ErrorType;

const List<ErrorCode> errorCodeValues = [
  //
  // Manually generated. You can mostly reproduce this list by running the
  // following command from the root of the analyzer package:
  //
  // > cat
  //       lib/src/analysis_options/error/option_codes.dart
  //       lib/src/dart/error/ffi_code.dart
  //       lib/src/dart/error/hint_codes.dart
  //       lib/src/dart/error/lint_codes.dart
  //       lib/src/dart/error/todo_codes.dart
  //       lib/src/html/error/html_codes.dart
  //       lib/src/dart/error/syntactic_errors.dart
  //       lib/src/error/codes.dart
  //       ../front_end/lib/src/scanner/errors.dart |
  //     grep 'static const .*Code' |
  //     awk '{print $3"."$4","}' |
  //     sort > codes.txt
  //
  // There are a few error codes that are wrapped such that the name of the
  // error code in on the line following the pattern we're grepping for. Those
  // need to be filled in by hand.
  //
  AnalysisOptionsErrorCode.INCLUDED_FILE_PARSE_ERROR,
  AnalysisOptionsErrorCode.PARSE_ERROR,
  AnalysisOptionsHintCode.DEPRECATED_ANALYSIS_OPTIONS_FILE_NAME,
  AnalysisOptionsHintCode.PREVIEW_DART_2_SETTING_DEPRECATED,
  AnalysisOptionsHintCode.STRONG_MODE_SETTING_DEPRECATED,
  AnalysisOptionsHintCode.SUPER_MIXINS_SETTING_DEPRECATED,
  AnalysisOptionsWarningCode.ANALYSIS_OPTION_DEPRECATED,
  AnalysisOptionsWarningCode.INCLUDE_FILE_NOT_FOUND,
  AnalysisOptionsWarningCode.INCLUDED_FILE_WARNING,
  AnalysisOptionsWarningCode.INVALID_OPTION,
  AnalysisOptionsWarningCode.INVALID_SECTION_FORMAT,
  AnalysisOptionsWarningCode.SPEC_MODE_REMOVED,
  AnalysisOptionsWarningCode.UNRECOGNIZED_ERROR_CODE,
  AnalysisOptionsWarningCode.UNSUPPORTED_OPTION_WITH_LEGAL_VALUE,
  AnalysisOptionsWarningCode.UNSUPPORTED_OPTION_WITH_LEGAL_VALUES,
  AnalysisOptionsWarningCode.UNSUPPORTED_OPTION_WITHOUT_VALUES,
  AnalysisOptionsWarningCode.UNSUPPORTED_VALUE,
  CompileTimeErrorCode.ABSTRACT_SUPER_MEMBER_REFERENCE,
  CompileTimeErrorCode.ACCESS_PRIVATE_ENUM_FIELD,
  CompileTimeErrorCode.AMBIGUOUS_EXPORT,
  CompileTimeErrorCode.AMBIGUOUS_EXTENSION_MEMBER_ACCESS,
  CompileTimeErrorCode.AMBIGUOUS_IMPORT,
  CompileTimeErrorCode.AMBIGUOUS_SET_OR_MAP_LITERAL_BOTH,
  CompileTimeErrorCode.AMBIGUOUS_SET_OR_MAP_LITERAL_EITHER,
  CompileTimeErrorCode.ANNOTATION_WITH_NON_CLASS,
  CompileTimeErrorCode.ARGUMENT_TYPE_NOT_ASSIGNABLE,
  CompileTimeErrorCode.ASSERT_IN_REDIRECTING_CONSTRUCTOR,
  CompileTimeErrorCode.ASSIGNMENT_TO_CONST,
  CompileTimeErrorCode.ASSIGNMENT_TO_FINAL,
  CompileTimeErrorCode.ASSIGNMENT_TO_FINAL_LOCAL,
  CompileTimeErrorCode.ASSIGNMENT_TO_FINAL_NO_SETTER,
  CompileTimeErrorCode.ASSIGNMENT_TO_FUNCTION,
  CompileTimeErrorCode.ASSIGNMENT_TO_METHOD,
  CompileTimeErrorCode.ASSIGNMENT_TO_TYPE,
  CompileTimeErrorCode.ASYNC_FOR_IN_WRONG_CONTEXT,
  CompileTimeErrorCode.AWAIT_IN_LATE_LOCAL_VARIABLE_INITIALIZER,
  CompileTimeErrorCode.AWAIT_IN_WRONG_CONTEXT,
  CompileTimeErrorCode.BODY_MIGHT_COMPLETE_NORMALLY,
  CompileTimeErrorCode.BREAK_LABEL_ON_SWITCH_MEMBER,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_EXTENSION_NAME,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_PREFIX_NAME,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_TYPE,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_TYPE_NAME,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_TYPE_PARAMETER_NAME,
  CompileTimeErrorCode.BUILT_IN_IDENTIFIER_AS_TYPEDEF_NAME,
  CompileTimeErrorCode.CASE_BLOCK_NOT_TERMINATED,
  CompileTimeErrorCode.CASE_EXPRESSION_TYPE_IMPLEMENTS_EQUALS,
  CompileTimeErrorCode.CASE_EXPRESSION_TYPE_IS_NOT_SWITCH_EXPRESSION_SUBTYPE,
  CompileTimeErrorCode.CAST_TO_NON_TYPE,
  CompileTimeErrorCode.CONCRETE_CLASS_WITH_ABSTRACT_MEMBER,
  CompileTimeErrorCode.CONFLICTING_CONSTRUCTOR_AND_STATIC_FIELD,
  CompileTimeErrorCode.CONFLICTING_CONSTRUCTOR_AND_STATIC_METHOD,
  CompileTimeErrorCode.CONFLICTING_FIELD_AND_METHOD,
  CompileTimeErrorCode.CONFLICTING_GENERIC_INTERFACES,
  CompileTimeErrorCode.CONFLICTING_METHOD_AND_FIELD,
  CompileTimeErrorCode.CONFLICTING_STATIC_AND_INSTANCE,
  CompileTimeErrorCode.CONFLICTING_TYPE_VARIABLE_AND_CLASS,
  CompileTimeErrorCode.CONFLICTING_TYPE_VARIABLE_AND_EXTENSION,
  CompileTimeErrorCode.CONFLICTING_TYPE_VARIABLE_AND_MEMBER_CLASS,
  CompileTimeErrorCode.CONFLICTING_TYPE_VARIABLE_AND_MEMBER_EXTENSION,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_FIELD_TYPE_MISMATCH,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_PARAM_TYPE_MISMATCH,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_THROWS_EXCEPTION,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_WITH_FIELD_INITIALIZED_BY_NON_CONST,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_WITH_MIXIN_WITH_FIELD,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_WITH_MIXIN_WITH_FIELDS,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_WITH_NON_CONST_SUPER,
  CompileTimeErrorCode.CONST_CONSTRUCTOR_WITH_NON_FINAL_FIELD,
  CompileTimeErrorCode.CONST_DEFERRED_CLASS,
  CompileTimeErrorCode.CONST_EVAL_THROWS_EXCEPTION,
  CompileTimeErrorCode.CONST_EVAL_THROWS_IDBZE,
  CompileTimeErrorCode.CONST_EVAL_TYPE_BOOL,
  CompileTimeErrorCode.CONST_EVAL_TYPE_BOOL_INT,
  CompileTimeErrorCode.CONST_EVAL_TYPE_BOOL_NUM_STRING,
  CompileTimeErrorCode.CONST_EVAL_TYPE_INT,
  CompileTimeErrorCode.CONST_EVAL_TYPE_NUM,
  CompileTimeErrorCode.CONST_EVAL_TYPE_TYPE,
  CompileTimeErrorCode.CONST_FIELD_INITIALIZER_NOT_ASSIGNABLE,
  CompileTimeErrorCode.CONST_FORMAL_PARAMETER,
  CompileTimeErrorCode.CONST_INITIALIZED_WITH_NON_CONSTANT_VALUE,
  CompileTimeErrorCode
      .CONST_INITIALIZED_WITH_NON_CONSTANT_VALUE_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.CONST_INSTANCE_FIELD,
  CompileTimeErrorCode.CONST_MAP_KEY_EXPRESSION_TYPE_IMPLEMENTS_EQUALS,
  CompileTimeErrorCode.CONST_NOT_INITIALIZED,
  CompileTimeErrorCode.CONST_SET_ELEMENT_TYPE_IMPLEMENTS_EQUALS,
  CompileTimeErrorCode.CONST_SPREAD_EXPECTED_LIST_OR_SET,
  CompileTimeErrorCode.CONST_SPREAD_EXPECTED_MAP,
  CompileTimeErrorCode.CONST_WITH_NON_CONST,
  CompileTimeErrorCode.CONST_WITH_NON_CONSTANT_ARGUMENT,
  CompileTimeErrorCode.CONST_WITH_NON_TYPE,
  CompileTimeErrorCode.CONST_WITH_TYPE_PARAMETERS,
  CompileTimeErrorCode.CONST_WITH_UNDEFINED_CONSTRUCTOR,
  CompileTimeErrorCode.CONST_WITH_UNDEFINED_CONSTRUCTOR_DEFAULT,
  CompileTimeErrorCode.CONTINUE_LABEL_ON_SWITCH,
  CompileTimeErrorCode.COULD_NOT_INFER,
  CompileTimeErrorCode.DEFAULT_LIST_CONSTRUCTOR,
  CompileTimeErrorCode.DEFAULT_VALUE_IN_FUNCTION_TYPED_PARAMETER,
  CompileTimeErrorCode.DEFAULT_VALUE_IN_REDIRECTING_FACTORY_CONSTRUCTOR,
  CompileTimeErrorCode.DEFAULT_VALUE_ON_REQUIRED_PARAMETER,
  CompileTimeErrorCode.DEFERRED_IMPORT_OF_EXTENSION,
  CompileTimeErrorCode.DEFINITELY_UNASSIGNED_LATE_LOCAL_VARIABLE,
  CompileTimeErrorCode.DUPLICATE_CONSTRUCTOR_DEFAULT,
  CompileTimeErrorCode.DUPLICATE_CONSTRUCTOR_NAME,
  CompileTimeErrorCode.DUPLICATE_DEFINITION,
  CompileTimeErrorCode.DUPLICATE_NAMED_ARGUMENT,
  CompileTimeErrorCode.DUPLICATE_PART,
  CompileTimeErrorCode.ENUM_CONSTANT_SAME_NAME_AS_ENCLOSING,
  CompileTimeErrorCode.EQUAL_ELEMENTS_IN_CONST_SET,
  CompileTimeErrorCode.EQUAL_KEYS_IN_CONST_MAP,
  CompileTimeErrorCode.EXPECTED_ONE_LIST_TYPE_ARGUMENTS,
  CompileTimeErrorCode.EXPECTED_ONE_SET_TYPE_ARGUMENTS,
  CompileTimeErrorCode.EXPECTED_TWO_MAP_TYPE_ARGUMENTS,
  CompileTimeErrorCode.EXPORT_INTERNAL_LIBRARY,
  CompileTimeErrorCode.EXPORT_LEGACY_SYMBOL,
  CompileTimeErrorCode.EXPORT_OF_NON_LIBRARY,
  CompileTimeErrorCode.EXPRESSION_IN_MAP,
  CompileTimeErrorCode.EXTENDS_DEFERRED_CLASS,
  CompileTimeErrorCode.EXTENDS_DISALLOWED_CLASS,
  CompileTimeErrorCode.EXTENDS_NON_CLASS,
  CompileTimeErrorCode.EXTENSION_AS_EXPRESSION,
  CompileTimeErrorCode.EXTENSION_CONFLICTING_STATIC_AND_INSTANCE,
  CompileTimeErrorCode.EXTENSION_DECLARES_MEMBER_OF_OBJECT,
  CompileTimeErrorCode.EXTENSION_OVERRIDE_ACCESS_TO_STATIC_MEMBER,
  CompileTimeErrorCode.EXTENSION_OVERRIDE_ARGUMENT_NOT_ASSIGNABLE,
  CompileTimeErrorCode.EXTENSION_OVERRIDE_WITH_CASCADE,
  CompileTimeErrorCode.EXTENSION_OVERRIDE_WITHOUT_ACCESS,
  CompileTimeErrorCode.EXTRA_POSITIONAL_ARGUMENTS,
  CompileTimeErrorCode.EXTRA_POSITIONAL_ARGUMENTS_COULD_BE_NAMED,
  CompileTimeErrorCode.FIELD_INITIALIZED_BY_MULTIPLE_INITIALIZERS,
  CompileTimeErrorCode.FIELD_INITIALIZED_IN_INITIALIZER_AND_DECLARATION,
  CompileTimeErrorCode.FIELD_INITIALIZED_IN_PARAMETER_AND_INITIALIZER,
  CompileTimeErrorCode.FIELD_INITIALIZER_FACTORY_CONSTRUCTOR,
  CompileTimeErrorCode.FIELD_INITIALIZER_NOT_ASSIGNABLE,
  CompileTimeErrorCode.FIELD_INITIALIZER_OUTSIDE_CONSTRUCTOR,
  CompileTimeErrorCode.FIELD_INITIALIZER_REDIRECTING_CONSTRUCTOR,
  CompileTimeErrorCode.FIELD_INITIALIZING_FORMAL_NOT_ASSIGNABLE,
  CompileTimeErrorCode.FINAL_INITIALIZED_IN_DECLARATION_AND_CONSTRUCTOR,
  CompileTimeErrorCode.FINAL_INITIALIZED_MULTIPLE_TIMES,
  CompileTimeErrorCode.FINAL_NOT_INITIALIZED,
  CompileTimeErrorCode.FINAL_NOT_INITIALIZED_CONSTRUCTOR_1,
  CompileTimeErrorCode.FINAL_NOT_INITIALIZED_CONSTRUCTOR_2,
  CompileTimeErrorCode.FINAL_NOT_INITIALIZED_CONSTRUCTOR_3_PLUS,
  CompileTimeErrorCode.FOR_IN_OF_INVALID_ELEMENT_TYPE,
  CompileTimeErrorCode.FOR_IN_OF_INVALID_TYPE,
  CompileTimeErrorCode.FOR_IN_WITH_CONST_VARIABLE,
  CompileTimeErrorCode.GENERIC_FUNCTION_TYPE_CANNOT_BE_BOUND,
  CompileTimeErrorCode.GENERIC_FUNCTION_TYPE_CANNOT_BE_TYPE_ARGUMENT,
  CompileTimeErrorCode.GETTER_NOT_ASSIGNABLE_SETTER_TYPES,
  CompileTimeErrorCode.GETTER_NOT_SUBTYPE_SETTER_TYPES,
  CompileTimeErrorCode.IF_ELEMENT_CONDITION_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.ILLEGAL_ASYNC_GENERATOR_RETURN_TYPE,
  CompileTimeErrorCode.ILLEGAL_ASYNC_RETURN_TYPE,
  CompileTimeErrorCode.ILLEGAL_SYNC_GENERATOR_RETURN_TYPE,
  CompileTimeErrorCode.IMPLEMENTS_DEFERRED_CLASS,
  CompileTimeErrorCode.IMPLEMENTS_DISALLOWED_CLASS,
  CompileTimeErrorCode.IMPLEMENTS_NON_CLASS,
  CompileTimeErrorCode.IMPLEMENTS_REPEATED,
  CompileTimeErrorCode.IMPLEMENTS_SUPER_CLASS,
  CompileTimeErrorCode.IMPLICIT_THIS_REFERENCE_IN_INITIALIZER,
  CompileTimeErrorCode.IMPORT_INTERNAL_LIBRARY,
  CompileTimeErrorCode.IMPORT_OF_NON_LIBRARY,
  CompileTimeErrorCode.INCONSISTENT_CASE_EXPRESSION_TYPES,
  CompileTimeErrorCode.INCONSISTENT_INHERITANCE,
  CompileTimeErrorCode.INCONSISTENT_INHERITANCE_GETTER_AND_METHOD,
  CompileTimeErrorCode.INCONSISTENT_LANGUAGE_VERSION_OVERRIDE,
  CompileTimeErrorCode.INITIALIZER_FOR_NON_EXISTENT_FIELD,
  CompileTimeErrorCode.INITIALIZER_FOR_STATIC_FIELD,
  CompileTimeErrorCode.INITIALIZING_FORMAL_FOR_NON_EXISTENT_FIELD,
  CompileTimeErrorCode.INITIALIZING_FORMAL_FOR_STATIC_FIELD,
  CompileTimeErrorCode.INSTANCE_ACCESS_TO_STATIC_MEMBER,
  CompileTimeErrorCode.INSTANCE_MEMBER_ACCESS_FROM_FACTORY,
  CompileTimeErrorCode.INSTANCE_MEMBER_ACCESS_FROM_STATIC,
  CompileTimeErrorCode.INSTANTIATE_ABSTRACT_CLASS,
  CompileTimeErrorCode.INSTANTIATE_ENUM,
  CompileTimeErrorCode.INTEGER_LITERAL_IMPRECISE_AS_DOUBLE,
  CompileTimeErrorCode.INTEGER_LITERAL_OUT_OF_RANGE,
  CompileTimeErrorCode.INVALID_ANNOTATION,
  CompileTimeErrorCode.INVALID_ANNOTATION_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.INVALID_ANNOTATION_GETTER,
  CompileTimeErrorCode.INVALID_ASSIGNMENT,
  CompileTimeErrorCode.INVALID_CAST_FUNCTION,
  CompileTimeErrorCode.INVALID_CAST_FUNCTION_EXPR,
  CompileTimeErrorCode.INVALID_CAST_LITERAL,
  CompileTimeErrorCode.INVALID_CAST_LITERAL_LIST,
  CompileTimeErrorCode.INVALID_CAST_LITERAL_MAP,
  CompileTimeErrorCode.INVALID_CAST_LITERAL_SET,
  CompileTimeErrorCode.INVALID_CAST_METHOD,
  CompileTimeErrorCode.INVALID_CAST_NEW_EXPR,
  CompileTimeErrorCode.INVALID_CONSTANT,
  CompileTimeErrorCode.INVALID_CONSTRUCTOR_NAME,
  CompileTimeErrorCode.INVALID_EXTENSION_ARGUMENT_COUNT,
  CompileTimeErrorCode.INVALID_FACTORY_NAME_NOT_A_CLASS,
  CompileTimeErrorCode.INVALID_INLINE_FUNCTION_TYPE,
  CompileTimeErrorCode.INVALID_MODIFIER_ON_CONSTRUCTOR,
  CompileTimeErrorCode.INVALID_MODIFIER_ON_SETTER,
  CompileTimeErrorCode.INVALID_OVERRIDE,
  CompileTimeErrorCode.INVALID_REFERENCE_TO_THIS,
  CompileTimeErrorCode.INVALID_SUPER_INVOCATION,
  CompileTimeErrorCode.INVALID_TYPE_ARGUMENT_IN_CONST_LIST,
  CompileTimeErrorCode.INVALID_TYPE_ARGUMENT_IN_CONST_MAP,
  CompileTimeErrorCode.INVALID_TYPE_ARGUMENT_IN_CONST_SET,
  CompileTimeErrorCode.INVALID_URI,
  CompileTimeErrorCode.INVALID_USE_OF_COVARIANT,
  CompileTimeErrorCode.INVALID_USE_OF_NULL_VALUE,
  CompileTimeErrorCode.INVOCATION_OF_EXTENSION_WITHOUT_CALL,
  CompileTimeErrorCode.INVOCATION_OF_NON_FUNCTION,
  CompileTimeErrorCode.INVOCATION_OF_NON_FUNCTION_EXPRESSION,
  CompileTimeErrorCode.LABEL_IN_OUTER_SCOPE,
  CompileTimeErrorCode.LABEL_UNDEFINED,
  CompileTimeErrorCode.LATE_FINAL_FIELD_WITH_CONST_CONSTRUCTOR,
  CompileTimeErrorCode.LATE_FINAL_LOCAL_ALREADY_ASSIGNED,
  CompileTimeErrorCode.LIST_ELEMENT_TYPE_NOT_ASSIGNABLE,
  CompileTimeErrorCode.MAP_ENTRY_NOT_IN_MAP,
  CompileTimeErrorCode.MAP_KEY_TYPE_NOT_ASSIGNABLE,
  CompileTimeErrorCode.MAP_VALUE_TYPE_NOT_ASSIGNABLE,
  CompileTimeErrorCode.MISSING_CONST_IN_LIST_LITERAL,
  CompileTimeErrorCode.MISSING_CONST_IN_MAP_LITERAL,
  CompileTimeErrorCode.MISSING_CONST_IN_SET_LITERAL,
  CompileTimeErrorCode.MISSING_DART_LIBRARY,
  CompileTimeErrorCode.MISSING_DEFAULT_VALUE_FOR_PARAMETER,
  CompileTimeErrorCode.MISSING_REQUIRED_ARGUMENT,
  CompileTimeErrorCode.MIXIN_APPLICATION_CONCRETE_SUPER_INVOKED_MEMBER_TYPE,
  CompileTimeErrorCode.MIXIN_APPLICATION_NO_CONCRETE_SUPER_INVOKED_MEMBER,
  CompileTimeErrorCode.MIXIN_APPLICATION_NOT_IMPLEMENTED_INTERFACE,
  CompileTimeErrorCode.MIXIN_CLASS_DECLARES_CONSTRUCTOR,
  CompileTimeErrorCode.MIXIN_DECLARES_CONSTRUCTOR,
  CompileTimeErrorCode.MIXIN_DEFERRED_CLASS,
  CompileTimeErrorCode.MIXIN_INFERENCE_INCONSISTENT_MATCHING_CLASSES,
  CompileTimeErrorCode.MIXIN_INFERENCE_NO_MATCHING_CLASS,
  CompileTimeErrorCode.MIXIN_INFERENCE_NO_POSSIBLE_SUBSTITUTION,
  CompileTimeErrorCode.MIXIN_INHERITS_FROM_NOT_OBJECT,
  CompileTimeErrorCode.MIXIN_INSTANTIATE,
  CompileTimeErrorCode.MIXIN_OF_DISALLOWED_CLASS,
  CompileTimeErrorCode.MIXIN_OF_NON_CLASS,
  CompileTimeErrorCode.MIXIN_SUPER_CLASS_CONSTRAINT_DEFERRED_CLASS,
  CompileTimeErrorCode.MIXIN_SUPER_CLASS_CONSTRAINT_DISALLOWED_CLASS,
  CompileTimeErrorCode.MIXIN_SUPER_CLASS_CONSTRAINT_NON_INTERFACE,
  CompileTimeErrorCode.MIXIN_WITH_NON_CLASS_SUPERCLASS,
  CompileTimeErrorCode.MIXINS_SUPER_CLASS,
  CompileTimeErrorCode.MULTIPLE_REDIRECTING_CONSTRUCTOR_INVOCATIONS,
  CompileTimeErrorCode.MULTIPLE_SUPER_INITIALIZERS,
  CompileTimeErrorCode.NEW_WITH_NON_TYPE,
  CompileTimeErrorCode.NEW_WITH_UNDEFINED_CONSTRUCTOR,
  CompileTimeErrorCode.NEW_WITH_UNDEFINED_CONSTRUCTOR_DEFAULT,
  CompileTimeErrorCode.NO_ANNOTATION_CONSTRUCTOR_ARGUMENTS,
  CompileTimeErrorCode.NO_COMBINED_SUPER_SIGNATURE,
  CompileTimeErrorCode.NO_DEFAULT_SUPER_CONSTRUCTOR_EXPLICIT,
  CompileTimeErrorCode.NO_DEFAULT_SUPER_CONSTRUCTOR_IMPLICIT,
  CompileTimeErrorCode.NO_GENERATIVE_CONSTRUCTORS_IN_SUPERCLASS,
  CompileTimeErrorCode.NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER_FIVE_PLUS,
  CompileTimeErrorCode.NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER_FOUR,
  CompileTimeErrorCode.NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER_ONE,
  CompileTimeErrorCode.NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER_THREE,
  CompileTimeErrorCode.NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER_TWO,
  CompileTimeErrorCode.NON_BOOL_CONDITION,
  CompileTimeErrorCode.NON_BOOL_EXPRESSION,
  CompileTimeErrorCode.NON_BOOL_NEGATION_EXPRESSION,
  CompileTimeErrorCode.NON_BOOL_OPERAND,
  CompileTimeErrorCode.NON_CONST_MAP_AS_EXPRESSION_STATEMENT,
  CompileTimeErrorCode.NON_CONSTANT_ANNOTATION_CONSTRUCTOR,
  CompileTimeErrorCode.NON_CONSTANT_CASE_EXPRESSION,
  CompileTimeErrorCode.NON_CONSTANT_CASE_EXPRESSION_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.NON_CONSTANT_DEFAULT_VALUE,
  CompileTimeErrorCode.NON_CONSTANT_DEFAULT_VALUE_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.NON_CONSTANT_LIST_ELEMENT,
  CompileTimeErrorCode.NON_CONSTANT_LIST_ELEMENT_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.NON_CONSTANT_MAP_ELEMENT,
  CompileTimeErrorCode.NON_CONSTANT_MAP_KEY,
  CompileTimeErrorCode.NON_CONSTANT_MAP_KEY_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.NON_CONSTANT_MAP_VALUE,
  CompileTimeErrorCode.NON_CONSTANT_MAP_VALUE_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.NON_CONSTANT_SET_ELEMENT,
  CompileTimeErrorCode.NON_GENERATIVE_CONSTRUCTOR,
  CompileTimeErrorCode.NON_GENERATIVE_IMPLICIT_CONSTRUCTOR,
  CompileTimeErrorCode.NON_SYNC_FACTORY,
  CompileTimeErrorCode.NON_TYPE_AS_TYPE_ARGUMENT,
  CompileTimeErrorCode.NON_TYPE_IN_CATCH_CLAUSE,
  CompileTimeErrorCode.NON_VOID_RETURN_FOR_OPERATOR,
  CompileTimeErrorCode.NON_VOID_RETURN_FOR_SETTER,
  CompileTimeErrorCode.NOT_A_TYPE,
  CompileTimeErrorCode.NOT_ASSIGNED_POTENTIALLY_NON_NULLABLE_LOCAL_VARIABLE,
  CompileTimeErrorCode.NOT_ENOUGH_POSITIONAL_ARGUMENTS,
  CompileTimeErrorCode.NOT_INITIALIZED_NON_NULLABLE_INSTANCE_FIELD,
  CompileTimeErrorCode.NOT_INITIALIZED_NON_NULLABLE_INSTANCE_FIELD_CONSTRUCTOR,
  CompileTimeErrorCode.NOT_INITIALIZED_NON_NULLABLE_VARIABLE,
  CompileTimeErrorCode.NOT_INSTANTIATED_BOUND,
  CompileTimeErrorCode.NOT_ITERABLE_SPREAD,
  CompileTimeErrorCode.NOT_MAP_SPREAD,
  CompileTimeErrorCode.NOT_NULL_AWARE_NULL_SPREAD,
  CompileTimeErrorCode.NULLABLE_TYPE_IN_EXTENDS_CLAUSE,
  CompileTimeErrorCode.NULLABLE_TYPE_IN_IMPLEMENTS_CLAUSE,
  CompileTimeErrorCode.NULLABLE_TYPE_IN_ON_CLAUSE,
  CompileTimeErrorCode.NULLABLE_TYPE_IN_WITH_CLAUSE,
  CompileTimeErrorCode.OBJECT_CANNOT_EXTEND_ANOTHER_CLASS,
  CompileTimeErrorCode.ON_REPEATED,
  CompileTimeErrorCode.OPTIONAL_PARAMETER_IN_OPERATOR,
  CompileTimeErrorCode.PART_OF_DIFFERENT_LIBRARY,
  CompileTimeErrorCode.PART_OF_NON_PART,
  CompileTimeErrorCode.PART_OF_UNNAMED_LIBRARY,
  CompileTimeErrorCode.PREFIX_COLLIDES_WITH_TOP_LEVEL_MEMBER,
  CompileTimeErrorCode.PREFIX_IDENTIFIER_NOT_FOLLOWED_BY_DOT,
  CompileTimeErrorCode.PREFIX_SHADOWED_BY_LOCAL_DECLARATION,
  CompileTimeErrorCode.PRIVATE_COLLISION_IN_MIXIN_APPLICATION,
  CompileTimeErrorCode.PRIVATE_OPTIONAL_PARAMETER,
  CompileTimeErrorCode.PRIVATE_SETTER,
  CompileTimeErrorCode.RECURSIVE_COMPILE_TIME_CONSTANT,
  CompileTimeErrorCode.RECURSIVE_CONSTRUCTOR_REDIRECT,
  CompileTimeErrorCode.RECURSIVE_FACTORY_REDIRECT,
  CompileTimeErrorCode.RECURSIVE_INTERFACE_INHERITANCE,
  CompileTimeErrorCode.RECURSIVE_INTERFACE_INHERITANCE_EXTENDS,
  CompileTimeErrorCode.RECURSIVE_INTERFACE_INHERITANCE_IMPLEMENTS,
  CompileTimeErrorCode.RECURSIVE_INTERFACE_INHERITANCE_ON,
  CompileTimeErrorCode.RECURSIVE_INTERFACE_INHERITANCE_WITH,
  CompileTimeErrorCode.REDIRECT_GENERATIVE_TO_MISSING_CONSTRUCTOR,
  CompileTimeErrorCode.REDIRECT_GENERATIVE_TO_NON_GENERATIVE_CONSTRUCTOR,
  CompileTimeErrorCode.REDIRECT_TO_ABSTRACT_CLASS_CONSTRUCTOR,
  CompileTimeErrorCode.REDIRECT_TO_INVALID_FUNCTION_TYPE,
  CompileTimeErrorCode.REDIRECT_TO_INVALID_RETURN_TYPE,
  CompileTimeErrorCode.REDIRECT_TO_MISSING_CONSTRUCTOR,
  CompileTimeErrorCode.REDIRECT_TO_NON_CLASS,
  CompileTimeErrorCode.REDIRECT_TO_NON_CONST_CONSTRUCTOR,
  CompileTimeErrorCode.REFERENCED_BEFORE_DECLARATION,
  CompileTimeErrorCode.RETHROW_OUTSIDE_CATCH,
  CompileTimeErrorCode.RETURN_IN_GENERATIVE_CONSTRUCTOR,
  CompileTimeErrorCode.RETURN_IN_GENERATOR,
  CompileTimeErrorCode.RETURN_OF_INVALID_TYPE_FROM_CLOSURE,
  CompileTimeErrorCode.RETURN_OF_INVALID_TYPE_FROM_CONSTRUCTOR,
  CompileTimeErrorCode.RETURN_OF_INVALID_TYPE_FROM_FUNCTION,
  CompileTimeErrorCode.RETURN_OF_INVALID_TYPE_FROM_METHOD,
  CompileTimeErrorCode.RETURN_WITHOUT_VALUE,
  CompileTimeErrorCode.SET_ELEMENT_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.SET_ELEMENT_TYPE_NOT_ASSIGNABLE,
  CompileTimeErrorCode.SHARED_DEFERRED_PREFIX,
  CompileTimeErrorCode.SPREAD_EXPRESSION_FROM_DEFERRED_LIBRARY,
  CompileTimeErrorCode.STATIC_ACCESS_TO_INSTANCE_MEMBER,
  CompileTimeErrorCode.SUPER_IN_EXTENSION,
  CompileTimeErrorCode.SUPER_IN_INVALID_CONTEXT,
  CompileTimeErrorCode.SUPER_IN_REDIRECTING_CONSTRUCTOR,
  CompileTimeErrorCode.SUPER_INITIALIZER_IN_OBJECT,
  CompileTimeErrorCode.SWITCH_CASE_COMPLETES_NORMALLY,
  CompileTimeErrorCode.SWITCH_EXPRESSION_NOT_ASSIGNABLE,
  CompileTimeErrorCode.THROW_OF_INVALID_TYPE,
  CompileTimeErrorCode.TOP_LEVEL_CYCLE,
  CompileTimeErrorCode.TYPE_ALIAS_CANNOT_REFERENCE_ITSELF,
  CompileTimeErrorCode.TYPE_ANNOTATION_DEFERRED_CLASS,
  CompileTimeErrorCode.TYPE_ARGUMENT_NOT_MATCHING_BOUNDS,
  CompileTimeErrorCode.TYPE_PARAMETER_REFERENCED_BY_STATIC,
  CompileTimeErrorCode.TYPE_PARAMETER_SUPERTYPE_OF_ITS_BOUND,
  CompileTimeErrorCode.TYPE_TEST_WITH_NON_TYPE,
  CompileTimeErrorCode.TYPE_TEST_WITH_UNDEFINED_NAME,
  CompileTimeErrorCode.UNCHECKED_USE_OF_NULLABLE_VALUE,
  CompileTimeErrorCode.UNDEFINED_ANNOTATION,
  CompileTimeErrorCode.UNDEFINED_CLASS,
  CompileTimeErrorCode.UNDEFINED_CLASS_BOOLEAN,
  CompileTimeErrorCode.UNDEFINED_CONSTRUCTOR_IN_INITIALIZER,
  CompileTimeErrorCode.UNDEFINED_CONSTRUCTOR_IN_INITIALIZER_DEFAULT,
  CompileTimeErrorCode.UNDEFINED_ENUM_CONSTANT,
  CompileTimeErrorCode.UNDEFINED_EXTENSION_GETTER,
  CompileTimeErrorCode.UNDEFINED_EXTENSION_METHOD,
  CompileTimeErrorCode.UNDEFINED_EXTENSION_OPERATOR,
  CompileTimeErrorCode.UNDEFINED_EXTENSION_SETTER,
  CompileTimeErrorCode.UNDEFINED_FUNCTION,
  CompileTimeErrorCode.UNDEFINED_GETTER,
  CompileTimeErrorCode.UNDEFINED_IDENTIFIER,
  CompileTimeErrorCode.UNDEFINED_IDENTIFIER_AWAIT,
  CompileTimeErrorCode.UNDEFINED_METHOD,
  CompileTimeErrorCode.UNDEFINED_NAMED_PARAMETER,
  CompileTimeErrorCode.UNDEFINED_OPERATOR,
  CompileTimeErrorCode.UNDEFINED_PREFIXED_NAME,
  CompileTimeErrorCode.UNDEFINED_SETTER,
  CompileTimeErrorCode.UNDEFINED_SUPER_GETTER,
  CompileTimeErrorCode.UNDEFINED_SUPER_METHOD,
  CompileTimeErrorCode.UNDEFINED_SUPER_OPERATOR,
  CompileTimeErrorCode.UNDEFINED_SUPER_SETTER,
  CompileTimeErrorCode.UNQUALIFIED_REFERENCE_TO_NON_LOCAL_STATIC_MEMBER,
  CompileTimeErrorCode.UNQUALIFIED_REFERENCE_TO_STATIC_MEMBER_OF_EXTENDED_TYPE,
  CompileTimeErrorCode.URI_DOES_NOT_EXIST,
  CompileTimeErrorCode.URI_HAS_NOT_BEEN_GENERATED,
  CompileTimeErrorCode.URI_WITH_INTERPOLATION,
  CompileTimeErrorCode.USE_OF_VOID_RESULT,
  CompileTimeErrorCode.VARIABLE_TYPE_MISMATCH,
  CompileTimeErrorCode.WRONG_EXPLICIT_TYPE_PARAMETER_VARIANCE_IN_SUPERINTERFACE,
  CompileTimeErrorCode.WRONG_NUMBER_OF_PARAMETERS_FOR_OPERATOR,
  CompileTimeErrorCode.WRONG_NUMBER_OF_PARAMETERS_FOR_OPERATOR_MINUS,
  CompileTimeErrorCode.WRONG_NUMBER_OF_PARAMETERS_FOR_SETTER,
  CompileTimeErrorCode.WRONG_NUMBER_OF_TYPE_ARGUMENTS,
  CompileTimeErrorCode.WRONG_NUMBER_OF_TYPE_ARGUMENTS_CONSTRUCTOR,
  CompileTimeErrorCode.WRONG_NUMBER_OF_TYPE_ARGUMENTS_METHOD,
  CompileTimeErrorCode.WRONG_TYPE_PARAMETER_VARIANCE_IN_SUPERINTERFACE,
  CompileTimeErrorCode.WRONG_TYPE_PARAMETER_VARIANCE_POSITION,
  CompileTimeErrorCode.YIELD_EACH_IN_NON_GENERATOR,
  CompileTimeErrorCode.YIELD_IN_NON_GENERATOR,
  CompileTimeErrorCode.YIELD_OF_INVALID_TYPE,
  FfiCode.ANNOTATION_ON_POINTER_FIELD,
  FfiCode.EXTRA_ANNOTATION_ON_STRUCT_FIELD,
  FfiCode.FIELD_IN_STRUCT_WITH_INITIALIZER,
  FfiCode.FIELD_INITIALIZER_IN_STRUCT,
  FfiCode.GENERIC_STRUCT_SUBCLASS,
  FfiCode.INVALID_EXCEPTION_VALUE,
  FfiCode.INVALID_FIELD_TYPE_IN_STRUCT,
  FfiCode.MISMATCHED_ANNOTATION_ON_STRUCT_FIELD,
  FfiCode.MISSING_ANNOTATION_ON_STRUCT_FIELD,
  FfiCode.MISSING_EXCEPTION_VALUE,
  FfiCode.MISSING_FIELD_TYPE_IN_STRUCT,
  FfiCode.MUST_BE_A_NATIVE_FUNCTION_TYPE,
  FfiCode.MUST_BE_A_SUBTYPE,
  FfiCode.NON_CONSTANT_TYPE_ARGUMENT,
  FfiCode.NON_NATIVE_FUNCTION_TYPE_ARGUMENT_TO_POINTER,
  FfiCode.SUBTYPE_OF_FFI_CLASS_IN_EXTENDS,
  FfiCode.SUBTYPE_OF_FFI_CLASS_IN_IMPLEMENTS,
  FfiCode.SUBTYPE_OF_FFI_CLASS_IN_WITH,
  FfiCode.SUBTYPE_OF_STRUCT_CLASS_IN_EXTENDS,
  FfiCode.SUBTYPE_OF_STRUCT_CLASS_IN_IMPLEMENTS,
  FfiCode.SUBTYPE_OF_STRUCT_CLASS_IN_WITH,
  HintCode.CAN_BE_NULL_AFTER_NULL_AWARE,
  HintCode.DEAD_CODE,
  HintCode.DEAD_CODE_CATCH_FOLLOWING_CATCH,
  HintCode.DEAD_CODE_ON_CATCH_SUBTYPE,
  HintCode.DEPRECATED_EXTENDS_FUNCTION,
  HintCode.DEPRECATED_FUNCTION_CLASS_DECLARATION,
  HintCode.DEPRECATED_MEMBER_USE,
  HintCode.DEPRECATED_MEMBER_USE_FROM_SAME_PACKAGE,
  HintCode.DEPRECATED_MEMBER_USE_FROM_SAME_PACKAGE_WITH_MESSAGE,
  HintCode.DEPRECATED_MEMBER_USE_WITH_MESSAGE,
  HintCode.DEPRECATED_MIXIN_FUNCTION,
  HintCode.DIVISION_OPTIMIZATION,
  HintCode.DUPLICATE_HIDDEN_NAME,
  HintCode.DUPLICATE_IMPORT,
  HintCode.DUPLICATE_SHOWN_NAME,
  HintCode.EQUAL_ELEMENTS_IN_SET,
  HintCode.EQUAL_KEYS_IN_MAP,
  HintCode.FILE_IMPORT_INSIDE_LIB_REFERENCES_FILE_OUTSIDE,
  HintCode.FILE_IMPORT_OUTSIDE_LIB_REFERENCES_FILE_INSIDE,
  HintCode.IMPORT_DEFERRED_LIBRARY_WITH_LOAD_FUNCTION,
  HintCode.INFERENCE_FAILURE_ON_COLLECTION_LITERAL,
  HintCode.INFERENCE_FAILURE_ON_FUNCTION_RETURN_TYPE,
  HintCode.INFERENCE_FAILURE_ON_INSTANCE_CREATION,
  HintCode.INFERENCE_FAILURE_ON_UNINITIALIZED_VARIABLE,
  HintCode.INFERENCE_FAILURE_ON_UNTYPED_PARAMETER,
  HintCode.INVALID_FACTORY_ANNOTATION,
  HintCode.INVALID_FACTORY_METHOD_DECL,
  HintCode.INVALID_FACTORY_METHOD_IMPL,
  HintCode.INVALID_IMMUTABLE_ANNOTATION,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_AT_SIGN,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_EQUALS,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_GREATER,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_LOCATION,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_LOWER_CASE,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_NUMBER,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_PREFIX,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_TRAILING_CHARACTERS,
  HintCode.INVALID_LANGUAGE_VERSION_OVERRIDE_TWO_SLASHES,
  HintCode.INVALID_LITERAL_ANNOTATION,
  HintCode.INVALID_NON_VIRTUAL_ANNOTATION,
  HintCode.INVALID_OVERRIDE_OF_NON_VIRTUAL_MEMBER,
  HintCode.INVALID_REQUIRED_NAMED_PARAM,
  HintCode.INVALID_REQUIRED_OPTIONAL_POSITIONAL_PARAM,
  HintCode.INVALID_REQUIRED_POSITIONAL_PARAM,
  HintCode.INVALID_SEALED_ANNOTATION,
  HintCode.INVALID_USE_OF_PROTECTED_MEMBER,
  HintCode.INVALID_USE_OF_VISIBLE_FOR_TEMPLATE_MEMBER,
  HintCode.INVALID_USE_OF_VISIBLE_FOR_TESTING_MEMBER,
  HintCode.INVALID_VISIBILITY_ANNOTATION,
  HintCode.IS_DOUBLE,
  HintCode.IS_INT,
  HintCode.IS_NOT_DOUBLE,
  HintCode.IS_NOT_INT,
  HintCode.MISSING_JS_LIB_ANNOTATION,
  HintCode.MISSING_REQUIRED_PARAM,
  HintCode.MISSING_REQUIRED_PARAM_WITH_DETAILS,
  HintCode.MISSING_RETURN,
  HintCode.MIXIN_ON_SEALED_CLASS,
  HintCode.MUST_BE_IMMUTABLE,
  HintCode.MUST_CALL_SUPER,
  HintCode.NON_CONST_CALL_TO_LITERAL_CONSTRUCTOR,
  HintCode.NON_CONST_CALL_TO_LITERAL_CONSTRUCTOR_USING_NEW,
  HintCode.NULL_AWARE_BEFORE_OPERATOR,
  HintCode.NULL_AWARE_IN_CONDITION,
  HintCode.NULL_AWARE_IN_LOGICAL_OPERATOR,
  HintCode.NULLABLE_TYPE_IN_CATCH_CLAUSE,
  HintCode.OVERRIDE_EQUALS_BUT_NOT_HASH_CODE,
  HintCode.OVERRIDE_ON_NON_OVERRIDING_FIELD,
  HintCode.OVERRIDE_ON_NON_OVERRIDING_GETTER,
  HintCode.OVERRIDE_ON_NON_OVERRIDING_METHOD,
  HintCode.OVERRIDE_ON_NON_OVERRIDING_SETTER,
  HintCode.PACKAGE_IMPORT_CONTAINS_DOT_DOT,
  HintCode.RECEIVER_OF_TYPE_NEVER,
  HintCode.SDK_VERSION_AS_EXPRESSION_IN_CONST_CONTEXT,
  HintCode.SDK_VERSION_ASYNC_EXPORTED_FROM_CORE,
  HintCode.SDK_VERSION_BOOL_OPERATOR_IN_CONST_CONTEXT,
  HintCode.SDK_VERSION_EQ_EQ_OPERATOR_IN_CONST_CONTEXT,
  HintCode.SDK_VERSION_EXTENSION_METHODS,
  HintCode.SDK_VERSION_GT_GT_GT_OPERATOR,
  HintCode.SDK_VERSION_IS_EXPRESSION_IN_CONST_CONTEXT,
  HintCode.SDK_VERSION_NEVER,
  HintCode.SDK_VERSION_SET_LITERAL,
  HintCode.SDK_VERSION_UI_AS_CODE,
  HintCode.SDK_VERSION_UI_AS_CODE_IN_CONST_CONTEXT,
  HintCode.STRICT_RAW_TYPE,
  HintCode.SUBTYPE_OF_SEALED_CLASS,
  HintCode.TYPE_CHECK_IS_NOT_NULL,
  HintCode.TYPE_CHECK_IS_NULL,
  HintCode.UNDEFINED_HIDDEN_NAME,
  HintCode.UNDEFINED_SHOWN_NAME,
  HintCode.UNNECESSARY_CAST,
  HintCode.UNNECESSARY_NO_SUCH_METHOD,
  HintCode.UNNECESSARY_NULL_COMPARISON_FALSE,
  HintCode.UNNECESSARY_NULL_COMPARISON_TRUE,
  HintCode.UNNECESSARY_TYPE_CHECK_FALSE,
  HintCode.UNNECESSARY_TYPE_CHECK_TRUE,
  HintCode.UNUSED_CATCH_CLAUSE,
  HintCode.UNUSED_CATCH_STACK,
  HintCode.UNUSED_ELEMENT,
  HintCode.UNUSED_ELEMENT_PARAMETER,
  HintCode.UNUSED_FIELD,
  HintCode.UNUSED_IMPORT,
  HintCode.UNUSED_LABEL,
  HintCode.UNUSED_LOCAL_VARIABLE,
  HintCode.UNUSED_SHOWN_NAME,
  ManifestWarningCode.CAMERA_PERMISSIONS_INCOMPATIBLE,
  ManifestWarningCode.NO_TOUCHSCREEN_FEATURE,
  ManifestWarningCode.NON_RESIZABLE_ACTIVITY,
  ManifestWarningCode.PERMISSION_IMPLIES_UNSUPPORTED_HARDWARE,
  ManifestWarningCode.SETTING_ORIENTATION_ON_ACTIVITY,
  ManifestWarningCode.UNSUPPORTED_CHROME_OS_FEATURE,
  ManifestWarningCode.UNSUPPORTED_CHROME_OS_HARDWARE,
  ParserErrorCode.ABSTRACT_CLASS_MEMBER,
  ParserErrorCode.ABSTRACT_ENUM,
  ParserErrorCode.ABSTRACT_STATIC_METHOD,
  ParserErrorCode.ABSTRACT_TOP_LEVEL_FUNCTION,
  ParserErrorCode.ABSTRACT_TOP_LEVEL_VARIABLE,
  ParserErrorCode.ABSTRACT_TYPEDEF,
  ParserErrorCode.ANNOTATION_WITH_TYPE_ARGUMENTS,
  ParserErrorCode.ASYNC_KEYWORD_USED_AS_IDENTIFIER,
  ParserErrorCode.BREAK_OUTSIDE_OF_LOOP,
  ParserErrorCode.CATCH_SYNTAX,
  ParserErrorCode.CATCH_SYNTAX_EXTRA_PARAMETERS,
  ParserErrorCode.CLASS_IN_CLASS,
  ParserErrorCode.COLON_IN_PLACE_OF_IN,
  ParserErrorCode.CONFLICTING_MODIFIERS,
  ParserErrorCode.CONST_AFTER_FACTORY,
  ParserErrorCode.CONST_AND_COVARIANT,
  ParserErrorCode.CONST_AND_FINAL,
  ParserErrorCode.CONST_AND_VAR,
  ParserErrorCode.CONST_CLASS,
  ParserErrorCode.CONST_CONSTRUCTOR_WITH_BODY,
  ParserErrorCode.CONST_ENUM,
  ParserErrorCode.CONST_FACTORY,
  ParserErrorCode.CONST_METHOD,
  ParserErrorCode.CONST_TYPEDEF,
  ParserErrorCode.CONSTRUCTOR_WITH_RETURN_TYPE,
  ParserErrorCode.CONTINUE_OUTSIDE_OF_LOOP,
  ParserErrorCode.CONTINUE_WITHOUT_LABEL_IN_CASE,
  ParserErrorCode.COVARIANT_AFTER_FINAL,
  ParserErrorCode.COVARIANT_AFTER_VAR,
  ParserErrorCode.COVARIANT_AND_STATIC,
  ParserErrorCode.COVARIANT_CONSTRUCTOR,
  ParserErrorCode.COVARIANT_MEMBER,
  ParserErrorCode.COVARIANT_TOP_LEVEL_DECLARATION,
  ParserErrorCode.DEFAULT_VALUE_IN_FUNCTION_TYPE,
  ParserErrorCode.DEFERRED_AFTER_PREFIX,
  ParserErrorCode.DIRECTIVE_AFTER_DECLARATION,
  ParserErrorCode.DUPLICATE_DEFERRED,
  ParserErrorCode.DUPLICATE_LABEL_IN_SWITCH_STATEMENT,
  ParserErrorCode.DUPLICATE_PREFIX,
  ParserErrorCode.DUPLICATED_MODIFIER,
  ParserErrorCode.EMPTY_ENUM_BODY,
  ParserErrorCode.ENUM_IN_CLASS,
  ParserErrorCode.EQUALITY_CANNOT_BE_EQUALITY_OPERAND,
  ParserErrorCode.EXPECTED_BODY,
  ParserErrorCode.EXPECTED_CASE_OR_DEFAULT,
  ParserErrorCode.EXPECTED_CLASS_MEMBER,
  ParserErrorCode.EXPECTED_ELSE_OR_COMMA,
  ParserErrorCode.EXPECTED_EXECUTABLE,
  ParserErrorCode.EXPECTED_INSTEAD,
  ParserErrorCode.EXPECTED_LIST_OR_MAP_LITERAL,
  ParserErrorCode.EXPECTED_STRING_LITERAL,
  ParserErrorCode.EXPECTED_TOKEN,
  ParserErrorCode.EXPECTED_TYPE_NAME,
  ParserErrorCode.EXPERIMENT_NOT_ENABLED,
  ParserErrorCode.EXPORT_DIRECTIVE_AFTER_PART_DIRECTIVE,
  ParserErrorCode.EXTENSION_DECLARES_ABSTRACT_MEMBER,
  ParserErrorCode.EXTENSION_DECLARES_CONSTRUCTOR,
  ParserErrorCode.EXTENSION_DECLARES_INSTANCE_FIELD,
  ParserErrorCode.EXTERNAL_AFTER_CONST,
  ParserErrorCode.EXTERNAL_AFTER_FACTORY,
  ParserErrorCode.EXTERNAL_AFTER_STATIC,
  ParserErrorCode.EXTERNAL_CLASS,
  ParserErrorCode.EXTERNAL_CONSTRUCTOR_WITH_BODY,
  ParserErrorCode.EXTERNAL_CONSTRUCTOR_WITH_INITIALIZER,
  ParserErrorCode.EXTERNAL_ENUM,
  ParserErrorCode.EXTERNAL_FACTORY_REDIRECTION,
  ParserErrorCode.EXTERNAL_FACTORY_WITH_BODY,
  ParserErrorCode.EXTERNAL_FIELD,
  ParserErrorCode.EXTERNAL_GETTER_WITH_BODY,
  ParserErrorCode.EXTERNAL_METHOD_WITH_BODY,
  ParserErrorCode.EXTERNAL_OPERATOR_WITH_BODY,
  ParserErrorCode.EXTERNAL_SETTER_WITH_BODY,
  ParserErrorCode.EXTERNAL_TYPEDEF,
  ParserErrorCode.EXTRANEOUS_MODIFIER,
  ParserErrorCode.FACTORY_TOP_LEVEL_DECLARATION,
  ParserErrorCode.FACTORY_WITH_INITIALIZERS,
  ParserErrorCode.FACTORY_WITHOUT_BODY,
  ParserErrorCode.FIELD_INITIALIZED_OUTSIDE_DECLARING_CLASS,
  ParserErrorCode.FIELD_INITIALIZER_OUTSIDE_CONSTRUCTOR,
  ParserErrorCode.FINAL_AND_COVARIANT,
  ParserErrorCode.FINAL_AND_COVARIANT_LATE_WITH_INITIALIZER,
  ParserErrorCode.FINAL_AND_VAR,
  ParserErrorCode.FINAL_CLASS,
  ParserErrorCode.FINAL_CONSTRUCTOR,
  ParserErrorCode.FINAL_ENUM,
  ParserErrorCode.FINAL_METHOD,
  ParserErrorCode.FINAL_TYPEDEF,
  ParserErrorCode.FUNCTION_TYPED_PARAMETER_VAR,
  ParserErrorCode.GETTER_CONSTRUCTOR,
  ParserErrorCode.GETTER_IN_FUNCTION,
  ParserErrorCode.GETTER_WITH_PARAMETERS,
  ParserErrorCode.ILLEGAL_ASSIGNMENT_TO_NON_ASSIGNABLE,
  ParserErrorCode.IMPLEMENTS_BEFORE_EXTENDS,
  ParserErrorCode.IMPLEMENTS_BEFORE_ON,
  ParserErrorCode.IMPLEMENTS_BEFORE_WITH,
  ParserErrorCode.IMPORT_DIRECTIVE_AFTER_PART_DIRECTIVE,
  ParserErrorCode.INITIALIZED_VARIABLE_IN_FOR_EACH,
  ParserErrorCode.INVALID_AWAIT_IN_FOR,
  ParserErrorCode.INVALID_CODE_POINT,
  ParserErrorCode.INVALID_COMMENT_REFERENCE,
  ParserErrorCode.INVALID_CONSTRUCTOR_NAME,
  ParserErrorCode.INVALID_GENERIC_FUNCTION_TYPE,
  ParserErrorCode.INVALID_HEX_ESCAPE,
  ParserErrorCode.INVALID_INITIALIZER,
  ParserErrorCode.INVALID_LITERAL_IN_CONFIGURATION,
  ParserErrorCode.INVALID_OPERATOR,
  ParserErrorCode.INVALID_OPERATOR_FOR_SUPER,
  ParserErrorCode.INVALID_OPERATOR_QUESTIONMARK_PERIOD_FOR_SUPER,
  ParserErrorCode.INVALID_STAR_AFTER_ASYNC,
  ParserErrorCode.INVALID_SUPER_IN_INITIALIZER,
  ParserErrorCode.INVALID_SYNC,
  ParserErrorCode.INVALID_THIS_IN_INITIALIZER,
  ParserErrorCode.INVALID_UNICODE_ESCAPE,
  ParserErrorCode.INVALID_USE_OF_COVARIANT_IN_EXTENSION,
  ParserErrorCode.LIBRARY_DIRECTIVE_NOT_FIRST,
  ParserErrorCode.LOCAL_FUNCTION_DECLARATION_MODIFIER,
  ParserErrorCode.MEMBER_WITH_CLASS_NAME,
  ParserErrorCode.MISSING_ASSIGNABLE_SELECTOR,
  ParserErrorCode.MISSING_ASSIGNMENT_IN_INITIALIZER,
  ParserErrorCode.MISSING_CATCH_OR_FINALLY,
  ParserErrorCode.MISSING_CLASS_BODY,
  ParserErrorCode.MISSING_CLOSING_PARENTHESIS,
  ParserErrorCode.MISSING_CONST_FINAL_VAR_OR_TYPE,
  ParserErrorCode.MISSING_ENUM_BODY,
  ParserErrorCode.MISSING_EXPRESSION_IN_INITIALIZER,
  ParserErrorCode.MISSING_EXPRESSION_IN_THROW,
  ParserErrorCode.MISSING_FUNCTION_BODY,
  ParserErrorCode.MISSING_FUNCTION_KEYWORD,
  ParserErrorCode.MISSING_FUNCTION_PARAMETERS,
  ParserErrorCode.MISSING_GET,
  ParserErrorCode.MISSING_IDENTIFIER,
  ParserErrorCode.MISSING_INITIALIZER,
  ParserErrorCode.MISSING_KEYWORD_OPERATOR,
  ParserErrorCode.MISSING_METHOD_PARAMETERS,
  ParserErrorCode.MISSING_NAME_FOR_NAMED_PARAMETER,
  ParserErrorCode.MISSING_NAME_IN_LIBRARY_DIRECTIVE,
  ParserErrorCode.MISSING_NAME_IN_PART_OF_DIRECTIVE,
  ParserErrorCode.MISSING_PREFIX_IN_DEFERRED_IMPORT,
  ParserErrorCode.MISSING_STAR_AFTER_SYNC,
  ParserErrorCode.MISSING_STATEMENT,
  ParserErrorCode.MISSING_TERMINATOR_FOR_PARAMETER_GROUP,
  ParserErrorCode.MISSING_TYPEDEF_PARAMETERS,
  ParserErrorCode.MISSING_VARIABLE_IN_FOR_EACH,
  ParserErrorCode.MIXED_PARAMETER_GROUPS,
  ParserErrorCode.MIXIN_DECLARES_CONSTRUCTOR,
  ParserErrorCode.MODIFIER_OUT_OF_ORDER,
  ParserErrorCode.MULTIPLE_EXTENDS_CLAUSES,
  ParserErrorCode.MULTIPLE_IMPLEMENTS_CLAUSES,
  ParserErrorCode.MULTIPLE_LIBRARY_DIRECTIVES,
  ParserErrorCode.MULTIPLE_NAMED_PARAMETER_GROUPS,
  ParserErrorCode.MULTIPLE_ON_CLAUSES,
  ParserErrorCode.MULTIPLE_PART_OF_DIRECTIVES,
  ParserErrorCode.MULTIPLE_POSITIONAL_PARAMETER_GROUPS,
  ParserErrorCode.MULTIPLE_VARIABLES_IN_FOR_EACH,
  ParserErrorCode.MULTIPLE_VARIANCE_MODIFIERS,
  ParserErrorCode.MULTIPLE_WITH_CLAUSES,
  ParserErrorCode.NAMED_FUNCTION_EXPRESSION,
  ParserErrorCode.NAMED_FUNCTION_TYPE,
  ParserErrorCode.NAMED_PARAMETER_OUTSIDE_GROUP,
  ParserErrorCode.NATIVE_CLAUSE_IN_NON_SDK_CODE,
  ParserErrorCode.NATIVE_CLAUSE_SHOULD_BE_ANNOTATION,
  ParserErrorCode.NATIVE_FUNCTION_BODY_IN_NON_SDK_CODE,
  ParserErrorCode.NON_CONSTRUCTOR_FACTORY,
  ParserErrorCode.NON_IDENTIFIER_LIBRARY_NAME,
  ParserErrorCode.NON_PART_OF_DIRECTIVE_IN_PART,
  ParserErrorCode.NON_STRING_LITERAL_AS_URI,
  ParserErrorCode.NON_USER_DEFINABLE_OPERATOR,
  ParserErrorCode.NORMAL_BEFORE_OPTIONAL_PARAMETERS,
  ParserErrorCode.NULL_AWARE_CASCADE_OUT_OF_ORDER,
  ParserErrorCode.POSITIONAL_AFTER_NAMED_ARGUMENT,
  ParserErrorCode.POSITIONAL_PARAMETER_OUTSIDE_GROUP,
  ParserErrorCode.PREFIX_AFTER_COMBINATOR,
  ParserErrorCode.REDIRECTING_CONSTRUCTOR_WITH_BODY,
  ParserErrorCode.REDIRECTION_IN_NON_FACTORY_CONSTRUCTOR,
  ParserErrorCode.SETTER_CONSTRUCTOR,
  ParserErrorCode.SETTER_IN_FUNCTION,
  ParserErrorCode.STACK_OVERFLOW,
  ParserErrorCode.STATIC_AFTER_CONST,
  ParserErrorCode.STATIC_AFTER_FINAL,
  ParserErrorCode.STATIC_AFTER_VAR,
  ParserErrorCode.STATIC_CONSTRUCTOR,
  ParserErrorCode.STATIC_GETTER_WITHOUT_BODY,
  ParserErrorCode.STATIC_OPERATOR,
  ParserErrorCode.STATIC_SETTER_WITHOUT_BODY,
  ParserErrorCode.STATIC_TOP_LEVEL_DECLARATION,
  ParserErrorCode.SWITCH_HAS_CASE_AFTER_DEFAULT_CASE,
  ParserErrorCode.SWITCH_HAS_MULTIPLE_DEFAULT_CASES,
  ParserErrorCode.TOP_LEVEL_OPERATOR,
  ParserErrorCode.TYPE_ARGUMENTS_ON_TYPE_VARIABLE,
  ParserErrorCode.TYPE_BEFORE_FACTORY,
  ParserErrorCode.TYPE_PARAMETER_ON_CONSTRUCTOR,
  ParserErrorCode.TYPE_PARAMETER_ON_OPERATOR,
  ParserErrorCode.TYPEDEF_IN_CLASS,
  ParserErrorCode.UNEXPECTED_TERMINATOR_FOR_PARAMETER_GROUP,
  ParserErrorCode.UNEXPECTED_TOKEN,
  ParserErrorCode.VAR_AND_TYPE,
  ParserErrorCode.VAR_AS_TYPE_NAME,
  ParserErrorCode.VAR_CLASS,
  ParserErrorCode.VAR_ENUM,
  ParserErrorCode.VAR_RETURN_TYPE,
  ParserErrorCode.VAR_TYPEDEF,
  ParserErrorCode.VOID_WITH_TYPE_ARGUMENTS,
  ParserErrorCode.WITH_BEFORE_EXTENDS,
  ParserErrorCode.WRONG_SEPARATOR_FOR_POSITIONAL_PARAMETER,
  ParserErrorCode.WRONG_TERMINATOR_FOR_PARAMETER_GROUP,
  ScannerErrorCode.EXPECTED_TOKEN,
  ScannerErrorCode.ILLEGAL_CHARACTER,
  ScannerErrorCode.MISSING_DIGIT,
  ScannerErrorCode.MISSING_HEX_DIGIT,
  ScannerErrorCode.MISSING_IDENTIFIER,
  ScannerErrorCode.MISSING_QUOTE,
  ScannerErrorCode.UNABLE_GET_CONTENT,
  ScannerErrorCode.UNEXPECTED_DOLLAR_IN_STRING,
  ScannerErrorCode.UNSUPPORTED_OPERATOR,
  ScannerErrorCode.UNTERMINATED_MULTI_LINE_COMMENT,
  ScannerErrorCode.UNTERMINATED_STRING_LITERAL,
  StaticWarningCode.DEAD_NULL_AWARE_EXPRESSION,
  StaticWarningCode.INVALID_NULL_AWARE_OPERATOR,
  StaticWarningCode.INVALID_NULL_AWARE_OPERATOR_AFTER_SHORT_CIRCUIT,
  StaticWarningCode.INVALID_OVERRIDE_DIFFERENT_DEFAULT_VALUES_NAMED,
  StaticWarningCode.INVALID_OVERRIDE_DIFFERENT_DEFAULT_VALUES_POSITIONAL,
  StaticWarningCode.MISSING_ENUM_CONSTANT_IN_SWITCH,
  StaticWarningCode.UNNECESSARY_NON_NULL_ASSERTION,
  StrongModeCode.IMPLICIT_DYNAMIC_FIELD,
  StrongModeCode.IMPLICIT_DYNAMIC_FUNCTION,
  StrongModeCode.IMPLICIT_DYNAMIC_INVOKE,
  StrongModeCode.IMPLICIT_DYNAMIC_LIST_LITERAL,
  StrongModeCode.IMPLICIT_DYNAMIC_MAP_LITERAL,
  StrongModeCode.IMPLICIT_DYNAMIC_METHOD,
  StrongModeCode.IMPLICIT_DYNAMIC_PARAMETER,
  StrongModeCode.IMPLICIT_DYNAMIC_RETURN,
  StrongModeCode.IMPLICIT_DYNAMIC_TYPE,
  StrongModeCode.IMPLICIT_DYNAMIC_VARIABLE,
  StrongModeCode.TOP_LEVEL_FUNCTION_LITERAL_BLOCK,
  StrongModeCode.TOP_LEVEL_IDENTIFIER_NO_TYPE,
  StrongModeCode.TOP_LEVEL_INSTANCE_GETTER,
  StrongModeCode.TOP_LEVEL_INSTANCE_METHOD,
  TodoCode.TODO,
];

/// The lazy initialized map from [ErrorCode.uniqueName] to the [ErrorCode]
/// instance.
HashMap<String, ErrorCode> _uniqueNameToCodeMap;

/// Return the [ErrorCode] with the given [uniqueName], or `null` if not
/// found.
ErrorCode errorCodeByUniqueName(String uniqueName) {
  if (_uniqueNameToCodeMap == null) {
    _uniqueNameToCodeMap = HashMap<String, ErrorCode>();
    for (ErrorCode errorCode in errorCodeValues) {
      _uniqueNameToCodeMap[errorCode.uniqueName] = errorCode;
    }
  }
  return _uniqueNameToCodeMap[uniqueName];
}

/// An error discovered during the analysis of some Dart code.
///
/// See [AnalysisErrorListener].
class AnalysisError implements Diagnostic {
  /// An empty array of errors used when no errors are expected.
  static const List<AnalysisError> NO_ERRORS = <AnalysisError>[];

  /// A [Comparator] that sorts by the name of the file that the [AnalysisError]
  /// was found.
  static Comparator<AnalysisError> FILE_COMPARATOR =
      (AnalysisError o1, AnalysisError o2) =>
          o1.source.shortName.compareTo(o2.source.shortName);

  /// A [Comparator] that sorts error codes first by their severity (errors
  /// first, warnings second), and then by the error code type.
  static Comparator<AnalysisError> ERROR_CODE_COMPARATOR =
      (AnalysisError o1, AnalysisError o2) {
    ErrorCode errorCode1 = o1.errorCode;
    ErrorCode errorCode2 = o2.errorCode;
    ErrorSeverity errorSeverity1 = errorCode1.errorSeverity;
    ErrorSeverity errorSeverity2 = errorCode2.errorSeverity;
    if (errorSeverity1 == errorSeverity2) {
      ErrorType errorType1 = errorCode1.type;
      ErrorType errorType2 = errorCode2.type;
      return errorType1.compareTo(errorType2);
    } else {
      return errorSeverity2.compareTo(errorSeverity1);
    }
  };

  /// The error code associated with the error.
  final ErrorCode errorCode;

  /// The message describing the problem.
  DiagnosticMessage _problemMessage;

  /// The context messages associated with the problem. This list will be empty
  /// if there are no context messages.
  List<DiagnosticMessage> _contextMessages;

  /// The correction to be displayed for this error, or `null` if there is no
  /// correction information for this error.
  String _correction;

  /// The source in which the error occurred, or `null` if unknown.
  final Source source;

  /// Initialize a newly created analysis error. The error is associated with
  /// the given [source] and is located at the given [offset] with the given
  /// [length]. The error will have the given [errorCode] and the list of
  /// [arguments] will be used to complete the message and correction. If any
  /// [contextMessages] are provided, they will be recorded with the error.
  AnalysisError(this.source, int offset, int length, this.errorCode,
      [List<Object> arguments,
      List<DiagnosticMessage> contextMessages = const []]) {
    String message = formatList(errorCode.message, arguments);
    String correctionTemplate = errorCode.correction;
    if (correctionTemplate != null) {
      _correction = formatList(correctionTemplate, arguments);
    }
    _problemMessage = DiagnosticMessageImpl(
        filePath: source?.fullName,
        length: length,
        message: message,
        offset: offset);
    _contextMessages = contextMessages;
  }

  /// Initialize a newly created analysis error with given values.
  AnalysisError.forValues(this.source, int offset, int length, this.errorCode,
      String message, this._correction,
      {List<DiagnosticMessage> contextMessages = const []}) {
    _problemMessage = DiagnosticMessageImpl(
        filePath: source?.fullName,
        length: length,
        message: message,
        offset: offset);
    _contextMessages = contextMessages;
  }

  @override
  List<DiagnosticMessage> get contextMessages => _contextMessages;

  /// Return the template used to create the correction to be displayed for this
  /// error, or `null` if there is no correction information for this error. The
  /// correction should indicate how the user can fix the error.
  String get correction => _correction;

  @override
  String get correctionMessage => _correction;

  @override
  int get hashCode {
    int hashCode = offset;
    hashCode ^= (message != null) ? message.hashCode : 0;
    hashCode ^= (source != null) ? source.hashCode : 0;
    return hashCode;
  }

  /// The number of characters from the offset to the end of the source which
  /// encompasses the compilation error.
  int get length => _problemMessage.length;

  /// Return the message to be displayed for this error. The message should
  /// indicate what is wrong and why it is wrong.
  String get message => _problemMessage.message;

  /// The character offset from the beginning of the source (zero based) where
  /// the error occurred.
  int get offset => _problemMessage.offset;

  @override
  DiagnosticMessage get problemMessage => _problemMessage;

  @override
  Severity get severity {
    switch (errorCode.errorSeverity) {
      case ErrorSeverity.ERROR:
        return Severity.error;
      case ErrorSeverity.WARNING:
        return Severity.warning;
      case ErrorSeverity.INFO:
        return Severity.info;
      default:
        throw StateError('Invalid error severity: ${errorCode.errorSeverity}');
    }
  }

  @override
  bool operator ==(Object other) {
    if (identical(other, this)) {
      return true;
    }
    // prepare other AnalysisError
    if (other is AnalysisError) {
      // Quick checks.
      if (!identical(errorCode, other.errorCode)) {
        return false;
      }
      if (offset != other.offset || length != other.length) {
        return false;
      }
      // Deep checks.
      if (message != other.message) {
        return false;
      }
      if (source != other.source) {
        return false;
      }
      // OK
      return true;
    }
    return false;
  }

  @override
  String toString() {
    StringBuffer buffer = StringBuffer();
    buffer.write((source != null) ? source.fullName : "<unknown source>");
    buffer.write("(");
    buffer.write(offset);
    buffer.write("..");
    buffer.write(offset + length - 1);
    buffer.write("): ");
    //buffer.write("(" + lineNumber + ":" + columnNumber + "): ");
    buffer.write(message);
    return buffer.toString();
  }

  /// Merge all of the errors in the lists in the given list of [errorLists]
  /// into a single list of errors.
  static List<AnalysisError> mergeLists(List<List<AnalysisError>> errorLists) {
    Set<AnalysisError> errors = HashSet<AnalysisError>();
    for (List<AnalysisError> errorList in errorLists) {
      errors.addAll(errorList);
    }
    return errors.toList();
  }
}
