// Copyright (c) 2020, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// Parse the WinMD file and interpret its metadata

// Sources of inspiration:
// https://stackoverflow.com/questions/54375771/how-to-read-a-winmd-winrt-metadata-file
// https://docs.microsoft.com/en-us/windows/win32/api/rometadataresolution/nf-rometadataresolution-rogetmetadatafile

import 'dart:ffi';
import 'dart:io';

import 'package:ffi/ffi.dart';
import 'package:win32/win32.dart';

String toHex(int value32) =>
    '0x${value32.toUnsigned(32).toRadixString(16).padLeft(8, '0')}';

class WindowsRuntimeType {
  int token;
  String typeName;
  int flags;
  int baseTypeToken;

  WindowsRuntimeType(this.token, this.typeName, this.flags, this.baseTypeToken);
}

/// Takes a typename (e.g. `Windows.Globalization.Calendar`) and returns the
/// metadata file that contains the type.
File metadataFileContainingType(String typeName) {
  final hstrTypeName = convertToHString(typeName);

  final hstrMetaDataFilePath = calloc<IntPtr>();
  final spMetaDataImport = calloc<Pointer>();
  final typeDef = calloc<Uint32>();

  File path;

  // RoGetMetaDataFile can only be used for Windows Runtime classes in an app
  // that is not a Windows Store app.
  final hr = RoGetMetaDataFile(hstrTypeName.value, nullptr,
      hstrMetaDataFilePath, spMetaDataImport, typeDef);
  if (SUCCEEDED(hr)) {
    path = File(convertFromHString(hstrMetaDataFilePath));
  } else {
    throw WindowsException(hr);
  }

  WindowsDeleteString(hstrTypeName.value);
  WindowsDeleteString(hstrMetaDataFilePath.value);

  calloc.free(hstrTypeName);
  calloc.free(hstrMetaDataFilePath);

  return path;
}

List<WindowsRuntimeType> metadataTypesInFile(File file) {
  final types = <WindowsRuntimeType>[];

  final pDispenser = calloc<COMObject>();
  var hr = MetaDataGetDispenser(
      convertToCLSID(CLSID_CorMetaDataDispenser).cast(),
      convertToIID(IID_IMetaDataDispenser).cast(),
      pDispenser.cast());

  if (FAILED(hr)) {
    throw WindowsException(hr);
  }

  final dispenser = IMetaDataDispenser(pDispenser.cast());
  final szFile = TEXT(file.path);
  final pReader = calloc<IntPtr>();

  hr = dispenser.OpenScope(szFile, CorOpenFlags.ofRead,
      convertToIID(IID_IMetaDataImport2).cast(), pReader);
  if (FAILED(hr)) {
    throw WindowsException(hr);
  }

  final reader = IMetaDataImport2(pReader.cast());

  final phEnum = calloc<IntPtr>();
  final rgTypeDefs = calloc<Uint32>();
  final pcTypeDefs = calloc<Uint32>();

  hr = reader.EnumTypeDefs(phEnum, rgTypeDefs, 1, pcTypeDefs);
  while (hr == S_OK) {
    final token = rgTypeDefs.value;

    types.add(ProcessToken(reader, token));
    hr = reader.EnumTypeDefs(phEnum, rgTypeDefs, 1, pcTypeDefs);
  }
  reader.CloseEnum(phEnum.address);

  return types;
}

WindowsRuntimeType ProcessToken(IMetaDataImport reader, int token) {
  WindowsRuntimeType type;

  final nRead = calloc<Uint32>();
  final tdFlags = calloc<Uint32>();
  final baseClassToken = calloc<Uint32>();
  final typeName = calloc<Uint16>(256).cast<Utf16>();

  final hr = reader.GetTypeDefProps(
      token, typeName, 256, nRead, tdFlags, baseClassToken);

  if (SUCCEEDED(hr)) {
    type = WindowsRuntimeType(
        token, typeName.toDartString(), tdFlags.value, baseClassToken.value);

    calloc.free(nRead);
    calloc.free(tdFlags);
    calloc.free(baseClassToken);
    calloc.free(typeName);

    return type;
  } else {
    throw WindowsException(hr);
  }
}

/// Example usage:
/// ```
///   dart winmd.dart Windows.Storage.Pickers.FileOpenPicker
/// ```
void main(List<String> args) {
  final type = args.length == 1 ? args.first : 'Windows.Globalization.Calendar';

  final winmdFile = metadataFileContainingType(type);

  print('Type $type can be found in ${winmdFile.path}.');

  print('\nLooking for other types in the same file...\n');

  final types = metadataTypesInFile(winmdFile);
  print('Found ${types.length} types:');
  for (final type in types) {
    print(
        '[${toHex(type.token)}] ${type.typeName} (baseType: ${toHex(type.baseTypeToken)})');
  }
}
