---
title: Diagnostic messages
description: Details for diagnostics produced by the Dart analyzer.
---
{%- comment %}
WARNING: Do NOT EDIT this file directly. It is autogenerated by the script in
`pkg/analyzer/tool/diagnostics/generate.dart` in the sdk repository.
Update instructions: https://github.com/dart-lang/site-www/issues/1949
{% endcomment -%}

This page lists diagnostic messages produced by the Dart analyzer,
with details about what those messages mean and how you can fix your code.
For more information about the analyzer, see
[Customizing static analysis](/guides/language/analysis-options).

## Glossary

This page uses the following terms:

* [constant context][]
* [definite assignment][]
* [override inference][]
* [potentially non-nullable][]

[constant context]: #constant-context
[definite assignment]: #definite-assignment
[override inference]: #override-inference
[potentially non-nullable]: #potentially-non-nullable

### Constant context

A _constant context_ is a region of code in which it isn't necessary to include
the `const` keyword because it's implied by the fact that everything in that
region is required to be a constant. The following locations are constant
contexts:

* Everything inside a list, map or set literal that's prefixed by the
  `const` keyword. Example:

  ```dart
  var l = const [/*constant context*/];
  ```

* The arguments inside an invocation of a constant constructor. Example:

  ```dart
  var p = const Point(/*constant context*/);
  ```

* The initializer for a variable that's prefixed by the `const` keyword.
  Example:

  ```dart
  const v = /*constant context*/;
  ```

* Annotations

* The expression in a `case` clause. Example:

  ```dart
  void f(int e) {
    switch (e) {
      case /*constant context*/:
        break;
    }
  }
  ```

### Definite assignment

Definite assignment analysis is the process of determining, for each local
variable at each point in the code, which of the following is true:
- The variable has definitely been assigned a value (_definitely assigned_).
- The variable has definitely not been assigned a value (_definitely
  unassigned_).
- The variable might or might not have been assigned a value, depending on the
  execution path taken to arrive at that point.

Definite assignment analysis helps find problems in code, such as places where a
variable that might not have been assigned a value is being referenced, or
places where a variable that can only be assigned a value one time is being
assigned after it might already have been assigned a value.

For example, in the following code the variable `s` is definitely unassigned
when it’s passed as an argument to `print`:

```dart
void f() {
  String s;
  print(s);
}
```

But in the following code, the variable `s` is definitely assigned:

```dart
void f(String name) {
  String s = 'Hello $name!';
  print(s);
}
```

Definite assignment analysis can even tell whether a variable is definitely
assigned (or unassigned) when there are multiple possible execution paths. In
the following code the `print` function is called if execution goes through
either the true or the false branch of the `if` statement, but because `s` is
assigned no matter which branch is taken, it’s definitely assigned before it’s
passed to `print`:

```dart
void f(String name, bool casual) {
  String s;
  if (casual) {
    s = 'Hi $name!';
  } else {
    s = 'Hello $name!';
  }
  print(s);
}
```

In flow analysis, the end of the `if` statement is referred to as a _join_—a
place where two or more execution paths merge back together. Where there's a
join, the analysis says that a variable is definitely assigned if it’s
definitely assigned along all of the paths that are merging, and definitely
unassigned if it’s definitely unassigned along all of the paths.

Sometimes a variable is assigned a value on one path but not on another, in
which case the variable might or might not have been assigned a value. In the
following example, the true branch of the `if` statement might or might not be
executed, so the variable might or might be assigned a value:

```dart
void f(String name, bool casual) {
  String s;
  if (casual) {
    s = 'Hi $name!';
  }
  print(s);
}
```

The same is true if there is a false branch that doesn’t assign a value to `s`.

The analysis of loops is a little more complicated, but it follows the same
basic reasoning. For example, the condition in a `while` loop is always
executed, but the body might or might not be. So just like an `if` statement,
there's a join at the end of the `while` statement between the path in which the
condition is `true` and the path in which the condition is `false`.

For additional details, see the
[specification of definite assignment][definiteAssignmentSpec].

[definiteAssignmentSpec](https://github.com/dart-lang/language/blob/master/resources/type-system/flow-analysis.md)

### Override inference

Override inference is the process by which any missing types in a method
declaration are inferred based on the corresponding types from the method or
methods that it overrides.

If a candidate method (the method that's missing type information) overrides a
single inherited method, then the corresponding types from the overridden method
are inferred. For example, consider the following code:

```dart
class A {
  int m(String s) => 0;
}

class B extends A {
  @override
  m(s) => 1;
}
```

The declaration of `m` in `B` is a candidate because it's missing both the
return type and the parameter type. Because it overrides a single method (the
method `m` in `A`), the types from the overridden method will be used to infer
the missing types and it will be as if the method in `B` had been declared as
`int m(String s) => 1;`.

If a candidate method overrides multiple methods, and the function type one of
those overridden methods, M<sub>s</sub>, is a supertype of the function types of
all of the other overridden methods, then M<sub>s</sub> is used to infer the
missing types. For example, consider the following code:

```dart
class A {
  int m(num n) => 0;
}

class B {
  num m(int i) => 0;
}

class C implements A, B {
  @override
  m(n) => 1;
}
```

The declaration of `m` in `C` is a candidate for override inference because it's
missing both the return type and the parameter type. It overrides both `m` in
`A` and `m` in `B`, so we need to choose one of them from which the missing
types can be inferred. But because the function type of `m` in `A`
(`int Function(num)`) is a supertype of the function type of `m` in `B`
(`num Function(int)`), the function in `A` is used to infer the missing types.
The result is the same as declaring the method in `C` as `int m(num n) => 1;`.

It is an error if none of the overridden methods has a function type that is a
supertype of all the other overridden methods.

### Potentially non-nullable

A type is _potentially non-nullable_ if it's either explicitly non-nullable or
if it's a type parameter.

A type is explicitly non-nullable if it is a type name that isn't followed by a
question mark. Note that there are a few types that are always nullable, such as
`Null` and `dynamic`, and that `FutureOr` is only non-nullable if it isn't
followed by a question mark _and_ the type argument is non-nullable (such as
`FutureOr<String>`).

Type parameters are potentially non-nullable because the actual runtime type
(the type specified as a type argument) might be non-nullable. For example,
given a declaration of `class C<T> {}`, the type `C` could be used with a
non-nullable type argument as in `C<int>`.

## Diagnostics

The analyzer produces the following diagnostics for code that
doesn't conform to the language specification or
that might work in unexpected ways.

### abstract_field_initializer

_Abstract fields can't have initializers._

#### Description

The analyzer produces this diagnostic when a field that has the `abstract`
modifier also has an initializer.

#### Example

The following code produces this diagnostic because `f` is marked as
`abstract` and has an initializer:

{% prettify dart tag=pre+code %}
abstract class C {
  abstract int [!f!] = 0;
}
{% endprettify %}

The following code produces this diagnostic because `f` is marked as
`abstract` and there's an initializer in the constructor:

{% prettify dart tag=pre+code %}
abstract class C {
  abstract int f;

  C() : [!f!] = 0;
}
{% endprettify %}

#### Common fixes

If the field must be abstract, then remove the initializer:

{% prettify dart tag=pre+code %}
abstract class C {
  abstract int f;
}
{% endprettify %}

If the field isn't required to be abstract, then remove the keyword:

{% prettify dart tag=pre+code %}
abstract class C {
  int f = 0;
}
{% endprettify %}

### abstract_super_member_reference

_The {0} '{1}' is always abstract in the supertype._

#### Description

The analyzer produces this diagnostic when an inherited member is
referenced using `super`, but there is no concrete implementation of the
member in the superclass chain. Abstract members can't be invoked.

#### Examples

The following code produces this diagnostic because `B` doesn't inherit a
concrete implementation of `a`:

{% prettify dart tag=pre+code %}
abstract class A {
  int get a;
}
class B extends A {
  int get a => super.[!a!];
}
{% endprettify %}

#### Common fixes

Remove the invocation of the abstract member, possibly replacing it with an
invocation of a concrete member.

### ambiguous_export

_The name '{0}' is defined in the libraries '{1}' and '{2}'._

#### Description

The analyzer produces this diagnostic when two or more export directives
cause the same name to be exported from multiple libraries.

#### Example

Given a file named `a.dart` containing

{% prettify dart tag=pre+code %}
class C {}
{% endprettify %}

And a file named `b.dart` containing

{% prettify dart tag=pre+code %}
class C {}
{% endprettify %}

The following code produces this diagnostic because the name `C` is being
exported from both `a.dart` and `b.dart`:

{% prettify dart tag=pre+code %}
export 'a.dart';
export [!'b.dart'!];
{% endprettify %}

#### Common fixes

If none of the names in one of the libraries needs to be exported, then
remove the unnecessary export directives:

{% prettify dart tag=pre+code %}
export 'a.dart';
{% endprettify %}

If all of the export directives are needed, then hide the name in all
except one of the directives:

{% prettify dart tag=pre+code %}
export 'a.dart';
export 'b.dart' hide C;
{% endprettify %}

### ambiguous_extension_member_access

_A member named '{0}' is defined in extensions '{1}' and '{2}' and neither is
more specific._

#### Description

When code refers to a member of an object (for example, `o.m()` or `o.m` or
`o[i]`) where the static type of `o` doesn't declare the member (`m` or
`[]`, for example), then the analyzer tries to find the member in an
extension. For example, if the member is `m`, then the analyzer looks for
extensions that declare a member named `m` and have an extended type that
the static type of `o` can be assigned to. When there's more than one such
extension in scope, the extension whose extended type is most specific is
selected.

The analyzer produces this diagnostic when none of the extensions has an
extended type that's more specific than the extended types of all of the
other extensions, making the reference to the member ambiguous.

#### Examples

The following code produces this diagnostic because there's no way to
choose between the member in `E1` and the member in `E2`:

{% prettify dart tag=pre+code %}
extension E1 on String {
  int get charCount => 1;
}

extension E2 on String {
  int get charCount => 2;
}

void f(String s) {
  print(s.[!charCount!]);
}
{% endprettify %}

#### Common fixes

If you don't need both extensions, then you can delete or hide one of them.

If you need both, then explicitly select the one you want to use by using
an extension override:

{% prettify dart tag=pre+code %}
extension E1 on String {
  int get charCount => length;
}

extension E2 on String {
  int get charCount => length;
}

void f(String s) {
  print(E2(s).charCount);
}
{% endprettify %}

### ambiguous_import

_The name '{0}' is defined in the libraries {1}._

#### Description

The analyzer produces this diagnostic when a name is referenced that is
declared in two or more imported libraries.

#### Examples

Given a library (`a.dart`) that defines a class (`C` in this example):

{% prettify dart tag=pre+code %}
class A {}
class C {}
{% endprettify %}

And a library (`b.dart`) that defines a different class with the same name:

{% prettify dart tag=pre+code %}
class B {}
class C {}
{% endprettify %}

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
import 'a.dart';
import 'b.dart';

void f([!C!] c1, [!C!] c2) {}
{% endprettify %}

#### Common fixes

If any of the libraries aren't needed, then remove the import directives
for them:

{% prettify dart tag=pre+code %}
import 'a.dart';

void f(C c1, C c2) {}
{% endprettify %}

If the name is still defined by more than one library, then add a `hide`
clause to the import directives for all except one library:

{% prettify dart tag=pre+code %}
import 'a.dart' hide C;
import 'b.dart';

void f(C c1, C c2) {}
{% endprettify %}

If you must be able to reference more than one of these types, then add a
prefix to each of the import directives, and qualify the references with
the appropriate prefix:

{% prettify dart tag=pre+code %}
import 'a.dart' as a;
import 'b.dart' as b;

void f(a.C c1, b.C c2) {}
{% endprettify %}

### ambiguous_set_or_map_literal_both

_This literal contains both 'Map' and 'Iterable' spreads, which makes it
impossible to determine whether the literal is a map or a set._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments and all of the
elements are spread elements (which are allowed in both kinds of literals),
then the analyzer uses the types of the expressions that are being spread.
If all of the expressions have the type `Iterable`, then it's a set
literal; if they all have the type `Map`, then it's a map literal.

The analyzer produces this diagnostic when some of the expressions being
spread have the type `Iterable` and others have the type `Map`, making it
impossible for the analyzer to determine whether you are writing a map
literal or a set literal.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    [!{...a, ...b, ...c}!];
{% endprettify %}

The list `b` can only be spread into a set, and the maps `a` and `c` can
only be spread into a map, and the literal can't be both.

#### Common fixes

There are two common ways to fix this problem. The first is to remove all
of the spread elements of one kind or another, so that the elements are
consistent. In this case, that likely means removing the list and deciding
what to do about the now unused parameter:

{% prettify dart tag=pre+code %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, ...c};
{% endprettify %}

The second fix is to change the elements of one kind into elements that are
consistent with the other elements. For example, you can add the elements
of the list as keys that map to themselves:

{% prettify dart tag=pre+code %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, for (String s in b) s: s, ...c};
{% endprettify %}

### ambiguous_set_or_map_literal_either

_This literal must be either a map or a set, but the elements don't have enough
information for type inference to work._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments and all of the
elements are spread elements (which are allowed in both kinds of literals)
then the analyzer uses the types of the expressions that are being spread.
If all of the expressions have the type `Iterable`, then it's a set
literal; if they all have the type `Map`, then it's a map literal.

This diagnostic is produced when none of the expressions being spread have
a type that allows the analyzer to decide whether you were writing a map
literal or a set literal.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
union(a, b) => [!{...a, ...b}!];
{% endprettify %}

The problem occurs because there are no type arguments, and there is no
information about the type of either `a` or `b`.

#### Common fixes

There are three common ways to fix this problem. The first is to add type
arguments to the literal. For example, if the literal is intended to be a
map literal, you might write something like this:

{% prettify dart tag=pre+code %}
union(a, b) => <String, String>{...a, ...b};
{% endprettify %}

The second fix is to add type information so that the expressions have
either the type `Iterable` or the type `Map`. You can add an explicit cast
or, in this case, add types to the declarations of the two parameters:

{% prettify dart tag=pre+code %}
union(List<int> a, List<int> b) => {...a, ...b};
{% endprettify %}

The third fix is to add context information. In this case, that means
adding a return type to the function:

{% prettify dart tag=pre+code %}
Set<String> union(a, b) => {...a, ...b};
{% endprettify %}

In other cases, you might add a type somewhere else. For example, say the
original code looks like this:

{% prettify dart tag=pre+code %}
union(a, b) {
  var x = [!{...a, ...b}!];
  return x;
}
{% endprettify %}

You might add a type annotation on `x`, like this:

{% prettify dart tag=pre+code %}
union(a, b) {
  Map<String, String> x = {...a, ...b};
  return x;
}
{% endprettify %}

### argument_type_not_assignable

_The argument type '{0}' can't be assigned to the parameter type '{1}'._

#### Description

The analyzer produces this diagnostic when the static type of an argument
can't be assigned to the static type of the corresponding parameter.

#### Examples

The following code produces this diagnostic because a `num` can't be
assigned to a `String`:

{% prettify dart tag=pre+code %}
String f(String x) => x;
String g(num y) => f([!y!]);
{% endprettify %}

#### Common fixes

If possible, rewrite the code so that the static type is assignable. In the
example above you might be able to change the type of the parameter `y`:

{% prettify dart tag=pre+code %}
String f(String x) => x;
String g(String y) => f(y);
{% endprettify %}

If that fix isn't possible, then add code to handle the case where the
argument value isn't the required type. One approach is to coerce other
types to the required type:

{% prettify dart tag=pre+code %}
String f(String x) => x;
String g(num y) => f(y.toString());
{% endprettify %}

Another approach is to add explicit type tests and fallback code:

{% prettify dart tag=pre+code %}
String f(String x) => x;
String g(num y) => f(y is String ? y : '');
{% endprettify %}

If you believe that the runtime type of the argument will always be the
same as the static type of the parameter, and you're willing to risk having
an exception thrown at runtime if you're wrong, then add an explicit cast:

{% prettify dart tag=pre+code %}
String f(String x) => x;
String g(num y) => f(y as String);
{% endprettify %}

### assert_in_redirecting_constructor

_A redirecting constructor can't have an 'assert' initializer._

#### Description

The analyzer produces this diagnostic when a redirecting constructor (a
constructor that redirects to another constructor in the same class) has an
assert in the initializer list.

#### Example

The following code produces this diagnostic because the unnamed constructor
is a redirecting constructor and also has an assert in the initializer
list:

{% prettify dart tag=pre+code %}
class C {
  C(int x) : [!assert(x > 0)!], this.name();
  C.name() {}
}
{% endprettify %}

#### Common fixes

If the assert isn't needed, then remove it:

{% prettify dart tag=pre+code %}
class C {
  C(int x) : this.name();
  C.name() {}
}
{% endprettify %}

If the assert is needed, then convert the constructor into a factory
constructor:

{% prettify dart tag=pre+code %}
class C {
  factory C(int x) {
    assert(x > 0);
    return C.name();
  }
  C.name() {}
}
{% endprettify %}

### assignment_to_const

_Constant variables can't be assigned a value._

#### Description

The analyzer produces this diagnostic when it finds an assignment to a
top-level variable, a static field, or a local variable that has the
`const` modifier. The value of a compile-time constant can't be changed at
runtime.

#### Example

The following code produces this diagnostic because `c` is being assigned a
value even though it has the `const` modifier:

{% prettify dart tag=pre+code %}
const c = 0;

void f() {
  [!c!] = 1;
  print(c);
}
{% endprettify %}

#### Common fixes

If the variable must be assignable, then remove the `const` modifier:

{% prettify dart tag=pre+code %}
var c = 0;

void f() {
  c = 1;
  print(c);
}
{% endprettify %}

If the constant shouldn't be changed, then either remove the assignment or
use a local variable in place of references to the constant:

{% prettify dart tag=pre+code %}
const c = 0;

void f() {
  var v = 1;
  print(v);
}
{% endprettify %}

### assignment_to_final

_'{0}' can't be used as a setter because it's final._

#### Description

The analyzer produces this diagnostic when it finds an invocation of a
setter, but there's no setter because the field with the same name was
declared to be `final` or `const`.

#### Examples

The following code produces this diagnostic because `v` is final:

{% prettify dart tag=pre+code %}
class C {
  final v = 0;
}

f(C c) {
  c.[!v!] = 1;
}
{% endprettify %}

#### Common fixes

If you need to be able to set the value of the field, then remove the
modifier `final` from the field:

{% prettify dart tag=pre+code %}
class C {
  int v = 0;
}

f(C c) {
  c.v = 1;
}
{% endprettify %}

### assignment_to_final_local

_The final variable '{0}' can only be set once._

#### Description

The analyzer produces this diagnostic when a local variable that was
declared to be final is assigned after it was initialized.

#### Examples

The following code produces this diagnostic because `x` is final, so it
can't have a value assigned to it after it was initialized:

{% prettify dart tag=pre+code %}
void f() {
  final x = 0;
  [!x!] = 3;
  print(x);
}
{% endprettify %}

#### Common fixes

Remove the keyword `final`, and replace it with `var` if there's no type
annotation:

{% prettify dart tag=pre+code %}
void f() {
  var x = 0;
  x = 3;
  print(x);
}
{% endprettify %}

### assignment_to_final_no_setter

_There isn’t a setter named '{0}' in class '{1}'._

#### Description

The analyzer produces this diagnostic when a reference to a setter is
found; there is no setter defined for the type; but there is a getter
defined with the same name.

#### Examples

The following code produces this diagnostic because there is no setter
named `x` in `C`, but there is a getter named `x`:

{% prettify dart tag=pre+code %}
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.[!x!] = 1;
}
{% endprettify %}

#### Common fixes

If you want to invoke an existing setter, then correct the name:

{% prettify dart tag=pre+code %}
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.y = 1;
}
{% endprettify %}

If you want to invoke the setter but it just doesn't exist yet, then
declare it:

{% prettify dart tag=pre+code %}
class C {
  int get x => 0;
  set x(int p) {}
  set y(int p) {}
}

void f(C c) {
  c.x = 1;
}
{% endprettify %}

### assignment_to_function

_Functions can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the name of a function appears
on the left-hand side of an assignment expression.

#### Example

The following code produces this diagnostic because the assignment to the
function `f` is invalid:

{% prettify dart tag=pre+code %}
void f() {}

void g() {
  [!f!] = () {};
}
{% endprettify %}

#### Common fixes

If the right-hand side should be assigned to something else, such as a
local variable, then change the left-hand side:

{% prettify dart tag=pre+code %}
void f() {}

void g() {
  var x = () {};
  print(x);
}
{% endprettify %}

If the intent is to change the implementation of the function, then define
a function-valued variable instead of a function:

{% prettify dart tag=pre+code %}
void Function() f = () {};

void g() {
  f = () {};
}
{% endprettify %}

### assignment_to_method

_Methods can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the target of an assignment is a
method.

#### Examples

The following code produces this diagnostic because `f` can't be assigned a
value because it's a method:

{% prettify dart tag=pre+code %}
class C {
  void f() {}

  void g() {
    [!f!] = null;
  }
}
{% endprettify %}

#### Common fixes

Rewrite the code so that there isn't an assignment to a method.

### assignment_to_type

_Types can't be assigned a value._

#### Description

The analyzer produces this diagnostic when the name of a type name appears
on the left-hand side of an assignment expression.

#### Example

The following code produces this diagnostic because the assignment to the
class `C` is invalid:

{% prettify dart tag=pre+code %}
class C {}

void f() {
  [!C!] = null;
}
{% endprettify %}

#### Common fixes

If the right-hand side should be assigned to something else, such as a
local variable, then change the left-hand side:

{% prettify dart tag=pre+code %}
void f() {}

void g() {
  var c = null;
  print(c);
}
{% endprettify %}

### async_for_in_wrong_context

_The async for-in loop can only be used in an async function._

#### Description

The analyzer produces this diagnostic when an async for-in loop is found in
a function or method whose body isn't marked as being either `async` or
`async*`.

#### Example

The following code produces this diagnostic because the body of `f` isn't
marked as being either `async` or `async*`, but `f` contains an async
for-in loop:

{% prettify dart tag=pre+code %}
void f(list) {
  await for (var e [!in!] list) {
    print(e);
  }
}
{% endprettify %}

#### Common fixes

If the function should return a `Future`, then mark the body with `async`:

{% prettify dart tag=pre+code %}
Future<void> f(list) async {
  await for (var e in list) {
    print(e);
  }
}
{% endprettify %}

If the function should return a `Stream` of values, then mark the body with
`async*`:

{% prettify dart tag=pre+code %}
Stream<void> f(list) async* {
  await for (var e in list) {
    print(e);
  }
}
{% endprettify %}

If the function should be synchronous, then remove the `await` before the
loop:

{% prettify dart tag=pre+code %}
void f(list) {
  for (var e in list) {
    print(e);
  }
}
{% endprettify %}

### await_in_late_local_variable_initializer

_The 'await' expression can't be used in a 'late' local variable's initializer._

#### Description

The analyzer produces this diagnostic when a local variable that has the
`late` modifier uses an `await` expression in the initializer.

#### Example

The following code produces this diagnostic because an `await` expression
is used in the initializer for `v`, a local variable that is marked `late`:

{% prettify dart tag=pre+code %}
Future<int> f() async {
  late var v = [!await!] 42;
  return v;
}
{% endprettify %}

#### Common fixes

If the initializer can be rewritten to not use `await`, then rewrite it:

{% prettify dart tag=pre+code %}
Future<int> f() async {
  late var v = 42;
  return v;
}
{% endprettify %}

If the initializer can't be rewritten, then remove the `late` modifier:

{% prettify dart tag=pre+code %}
Future<int> f() async {
  var v = await 42;
  return v;
}
{% endprettify %}

### body_might_complete_normally

_The body might complete normally, causing 'null' to be returned, but the return
type is a potentially non-nullable type._

#### Description

The analyzer produces this diagnostic when a method or function has a
return type that's [potentially non-nullable][] but would implicitly return
`null` if control reached the end of the function.

#### Example

The following code produces this diagnostic because the method `m` has an
implicit return of `null` inserted at the end of the method, but the method
is declared to not return `null`:

{% prettify dart tag=pre+code %}
class C {
  int [!m!](int t) {
    print(t);
  }
}
{% endprettify %}

The following code produces this diagnostic because the method `m` has an
implicit return of `null` inserted at the end of the method, but because
the class `C` can be instantiated with a non-nullable type argument, the
method is effectively declared to not return `null`:

{% prettify dart tag=pre+code %}
class C<T> {
  T [!m!](T t) {
    print(t);
  }
}
{% endprettify %}

#### Common fixes

If there's a reasonable value that can be returned, then add a `return`
statement at the end of the method:

{% prettify dart tag=pre+code %}
class C<T> {
  T m(T t) {
    print(t);
    return t;
  }
}
{% endprettify %}

If the method won't reach the implicit return, then add a `throw` at the
end of the method:

{% prettify dart tag=pre+code %}
class C<T> {
  T m(T t) {
    print(t);
    throw '';
  }
}
{% endprettify %}

If the method intentionally returns `null` at the end, then change the
return type so that it's valid to return `null`:

{% prettify dart tag=pre+code %}
class C<T> {
  T? m(T t) {
    print(t);
  }
}
{% endprettify %}

### break_label_on_switch_member

_A break label resolves to the 'case' or 'default' statement._

#### Description

The analyzer produces this diagnostic when a break in a case clause inside
a switch statement has a label that is associated with another case clause.

#### Example

The following code produces this diagnostic because the label `l` is
associated with the case clause for `0`:

{% prettify dart tag=pre+code %}
void f(int i) {
  switch (i) {
    l: case 0:
      break;
    case 1:
      break [!l!];
  }
}
{% endprettify %}

#### Common fixes

If the intent is to transfer control to the statement after the switch,
then remove the label from the break statement:

{% prettify dart tag=pre+code %}
void f(int i) {
  switch (i) {
    case 0:
      break;
    case 1:
      break;
  }
}
{% endprettify %}

If the intent is to transfer control to a different case block, then use
`continue` rather than `break`:

{% prettify dart tag=pre+code %}
void f(int i) {
  switch (i) {
    l: case 0:
      break;
    case 1:
      continue l;
  }
}
{% endprettify %}

### built_in_identifier_as_extension_name

_The built-in identifier '{0}' can't be used as an extension name._

#### Description

The analyzer produces this diagnostic when the name of an extension is a
built-in identifier. Built-in identifiers can’t be used as extension names.

#### Examples

The following code produces this diagnostic because `mixin` is a built-in
identifier:

{% prettify dart tag=pre+code %}
extension [!mixin!] on int {}
{% endprettify %}

#### Common fixes

Choose a different name for the extension.

### built_in_identifier_as_type

_The built-in identifier '{0}' can't be used as a type._

#### Description

The analyzer produces this diagnostic when a built-in identifier is used
where a type name is expected.

#### Examples

The following code produces this diagnostic because `import` can't be used
as a type because it's a built-in identifier:

{% prettify dart tag=pre+code %}
[!import!]<int> x;
{% endprettify %}

#### Common fixes

Replace the built-in identifier with the name of a valid type:

{% prettify dart tag=pre+code %}
List<int> x;
{% endprettify %}

### case_block_not_terminated

_The last statement of the 'case' should be 'break', 'continue', 'rethrow',
'return', or 'throw'._

#### Description

The analyzer produces this diagnostic when the last statement in a `case`
block isn't one of the required terminators: `break`, `continue`,
`rethrow`, `return`, or `throw`.

#### Examples

The following code produces this diagnostic because the `case` block ends
with an assignment:

{% prettify dart tag=pre+code %}
void f(int x) {
  switch (x) {
    [!case!] 0:
      x += 2;
    default:
      x += 1;
  }
}
{% endprettify %}

#### Common fixes

Add one of the required terminators:

{% prettify dart tag=pre+code %}
void f(int x) {
  switch (x) {
    case 0:
      x += 2;
      break;
    default:
      x += 1;
  }
}
{% endprettify %}

### case_expression_type_implements_equals

_The switch case expression type '{0}' can't override the '==' operator._

#### Description

The analyzer produces this diagnostic when the type of the expression
following the keyword `case` has an implementation of the `==` operator
other than the one in `Object`.

#### Example

The following code produces this diagnostic because the expression
following the keyword `case` (`C(0)`) has the type `C`, and the class `C`
overrides the `==` operator:

{% prettify dart tag=pre+code %}
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  switch (c) {
    case [!C(0)!]:
      break;
  }
}
{% endprettify %}

#### Common fixes

If there isn't a strong reason not to do so, then rewrite the code to use
an if-else structure:

{% prettify dart tag=pre+code %}
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  if (c == C(0)) {
    // ...
  }
}
{% endprettify %}

If you can't rewrite the switch statement and the implementation of `==`
isn't necessary, then remove it:

{% prettify dart tag=pre+code %}
class C {
  final int value;

  const C(this.value);
}

void f(C c) {
  switch (c) {
    case C(0):
      break;
  }
}
{% endprettify %}

If you can't rewrite the switch statement and you can't remove the
definition of `==`, then find some other value that can be used to control
the switch:

{% prettify dart tag=pre+code %}
class C {
  final int value;

  const C(this.value);

  bool operator ==(Object other) {
    return false;
  }
}

void f(C c) {
  switch (c.value) {
    case 0:
      break;
  }
}
{% endprettify %}

### case_expression_type_is_not_switch_expression_subtype

_The switch case expression type '{0}' must be a subtype of the switch
expression type '{1}'._

#### Description

The analyzer produces this diagnostic when the expression following `case`
in a `switch` statement has a static type that isn't a subtype of the
static type of the expression following `switch`.

#### Example

The following code produces this diagnostic because `1` is an `int`, which
isn't a subtype of `String` (the type of `s`):

{% prettify dart tag=pre+code %}
void f(String s) {
  switch (s) {
    case [!1!]:
      break;
  }
}
{% endprettify %}

#### Common fixes

If the value of the `case` expression is wrong, then change the `case`
expression so that it has the required type:

{% prettify dart tag=pre+code %}
void f(String s) {
  switch (s) {
    case '1':
      break;
  }
}
{% endprettify %}

If the value of the `case` expression is correct, then change the `switch`
expression to have the required type:

{% prettify dart tag=pre+code %}
void f(int s) {
  switch (s) {
    case 1:
      break;
  }
}
{% endprettify %}

### cast_to_non_type

_The name '{0}' isn't a type, so it can't be used in an 'as' expression._

#### Description

The analyzer produces this diagnostic when the name following the `as` in a
cast expression is defined to be something other than a type.

#### Examples

The following code produces this diagnostic because `x` is a variable, not
a type:

{% prettify dart tag=pre+code %}
num x = 0;
int y = x as [!x!];
{% endprettify %}

#### Common fixes

Replace the name with the name of a type:

{% prettify dart tag=pre+code %}
num x = 0;
int y = x as int;
{% endprettify %}

### concrete_class_with_abstract_member

_'{0}' must have a method body because '{1}' isn't abstract._

#### Description

The analyzer produces this diagnostic when a member of a concrete class is
found that doesn't have a concrete implementation. Concrete classes aren't
allowed to contain abstract members.

#### Examples

The following code produces this diagnostic because `m` is an abstract
method but `C` isn't an abstract class:

{% prettify dart tag=pre+code %}
class C {
  [!void m();!]
}
{% endprettify %}

#### Common fixes

If it's valid to create instances of the class, provide an implementation
for the member:

{% prettify dart tag=pre+code %}
class C {
  void m() {}
}
{% endprettify %}

If it isn't valid to create instances of the class, mark the class as being
abstract:

{% prettify dart tag=pre+code %}
abstract class C {
  void m();
}
{% endprettify %}

### const_constructor_param_type_mismatch

_A value of type '{0}' can't be assigned to a parameter of type '{1}' in a const
constructor._

#### Description

The analyzer produces this diagnostic when the runtime type of a constant
value can't be assigned to the static type of a constant constructor's
parameter.

#### Example

The following code produces this diagnostic because the runtime type of `i`
is `int`, which can't be assigned to the static type of `s`:

{% prettify dart tag=pre+code %}
class C {
  final String s;

  const C(this.s);
}

const dynamic i = 0;

void f() {
  const C([!i!]);
}
{% endprettify %}

#### Common fixes

Pass a value of the correct type to the constructor:

{% prettify dart tag=pre+code %}
class C {
  final String s;

  const C(this.s);
}

const dynamic i = 0;

void f() {
  const C('$i');
}
{% endprettify %}

### const_constructor_with_field_initialized_by_non_const

_Can't define the 'const' constructor because the field '{0}' is initialized
with a non-constant value._

#### Description

The analyzer produces this diagnostic when a constructor has the keyword
`const`, but a field in the class is initialized to a non-constant value.

#### Example

The following code produces this diagnostic because the field `s` is
initialized to a non-constant value:

{% prettify dart tag=pre+code %}
class C {
  final String s = 3.toString();
  [!const!] C();
}
{% endprettify %}

#### Common fixes

If the field can be initialized to a constant value, then change the
initializer to a constant expression:

{% prettify dart tag=pre+code %}
class C {
  final String s = '3';
  const C();
}
{% endprettify %}

If the field can't be initialized to a constant value, then remove the
keyword `const` from the constructor:

{% prettify dart tag=pre+code %}
class C {
  final String s = 3.toString();
  C();
}
{% endprettify %}

### const_constructor_with_non_final_field

_Can't define a const constructor for a class with non-final fields._

#### Description

The analyzer produces this diagnostic when a constructor is marked as a
const constructor, but the constructor is defined in a class that has at
least one non-final instance field (either directly or by inheritance).

#### Examples

The following code produces this diagnostic because the field `x` isn't
final:

{% prettify dart tag=pre+code %}
class C {
  int x;

  const [!C!](this.x);
}
{% endprettify %}

#### Common fixes

If it's possible to mark all of the fields as final, then do so:

{% prettify dart tag=pre+code %}
class C {
  final int x;

  const C(this.x);
}
{% endprettify %}

If it isn't possible to mark all of the fields as final, then remove the
keyword `const` from the constructor:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C(this.x);
}
{% endprettify %}

### const_initialized_with_non_constant_value

_Const variables must be initialized with a constant value._

#### Description

The analyzer produces this diagnostic when a value that isn't statically
known to be a constant is assigned to a variable that's declared to be a
`const` variable.

#### Examples

The following code produces this diagnostic because `x` isn't declared to
be `const`:

{% prettify dart tag=pre+code %}
var x = 0;
const y = [!x!];
{% endprettify %}

#### Common fixes

If the value being assigned can be declared to be `const`, then change the
declaration:

{% prettify dart tag=pre+code %}
const x = 0;
const y = x;
{% endprettify %}

If the value can't be declared to be `const`, then remove the `const`
modifier from the variable, possibly using `final` in its place:

{% prettify dart tag=pre+code %}
var x = 0;
final y = x;
{% endprettify %}

### const_instance_field

_Only static fields can be declared as const._

#### Description

The analyzer produces this diagnostic when an instance field is marked as
being const.

#### Examples

The following code produces this diagnostic because `f` is an instance
field:

{% prettify dart tag=pre+code %}
class C {
  [!const!] int f = 3;
}
{% endprettify %}

#### Common fixes

If the field needs to be an instance field, then remove the keyword
`const`, or replace it with `final`:

{% prettify dart tag=pre+code %}
class C {
  final int f = 3;
}
{% endprettify %}

If the field really should be a const field, then make it a static field:

{% prettify dart tag=pre+code %}
class C {
  static const int f = 3;
}
{% endprettify %}

### const_not_initialized

_The constant '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a variable that is declared to
be a constant doesn't have an initializer.

#### Examples

The following code produces this diagnostic because `c` isn't initialized:

{% prettify dart tag=pre+code %}
const [!c!];
{% endprettify %}

#### Common fixes

Add an initializer:

{% prettify dart tag=pre+code %}
const c = 'c';
{% endprettify %}

### const_spread_expected_list_or_set

_A list or a set is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant list or set evaluates to something other than a list
or a set.

#### Examples

The following code produces this diagnostic because the value of `list1` is
`null`, which is neither a list nor a set:

{% prettify dart tag=pre+code %}
const List<int> list1 = null;
const List<int> list2 = [...[!list1!]];
{% endprettify %}

#### Common fixes

Change the expression to something that evaluates to either a constant list
or a constant set:

{% prettify dart tag=pre+code %}
const List<int> list1 = [];
const List<int> list2 = [...list1];
{% endprettify %}

### const_spread_expected_map

_A map is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant map evaluates to something other than a map.

#### Examples

The following code produces this diagnostic because the value of `map1` is
`null`, which isn't a map:

{% prettify dart tag=pre+code %}
const Map<String, int> map1 = null;
const Map<String, int> map2 = {...[!map1!]};
{% endprettify %}

#### Common fixes

Change the expression to something that evaluates to a constant map:

{% prettify dart tag=pre+code %}
const Map<String, int> map1 = {};
const Map<String, int> map2 = {...map1};
{% endprettify %}

### const_with_non_const

_The constructor being called isn't a const constructor._

#### Description

The analyzer produces this diagnostic when the keyword `const` is used to
invoke a constructor that isn't marked with `const`.

#### Examples

The following code produces this diagnostic because the constructor in `A`
isn't a const constructor:

{% prettify dart tag=pre+code %}
class A {
  A();
}

A f() => [!const!] A();
{% endprettify %}

#### Common fixes

If it's desirable and possible to make the class a constant class (by
making all of the fields of the class, including inherited fields, final),
then add the keyword `const` to the constructor:

{% prettify dart tag=pre+code %}
class A {
  const A();
}

A f() => const A();
{% endprettify %}

Otherwise, remove the keyword `const`:

{% prettify dart tag=pre+code %}
class A {
  A();
}

A f() => A();
{% endprettify %}

### const_with_non_constant_argument

_Arguments of a constant creation must be constant expressions._

#### Description

The analyzer produces this diagnostic when a const constructor is invoked
with an argument that isn't a constant expression.

#### Examples

The following code produces this diagnostic because `i` isn't a constant:

{% prettify dart tag=pre+code %}
class C {
  final int i;
  const C(this.i);
}
C f(int i) => const C([!i!]);
{% endprettify %}

#### Common fixes

Either make all of the arguments constant expressions, or remove the
`const` keyword to use the non-constant form of the constructor:

{% prettify dart tag=pre+code %}
class C {
  final int i;
  const C(this.i);
}
C f(int i) => C(i);
{% endprettify %}

### const_with_type_parameters

_A constant creation can't use a type parameter as a type argument._

#### Description

The analyzer produces this diagnostic when a type parameter is used as a
type argument in a `const` invocation of a constructor. This isn't allowed
because the value of the type parameter (the actual type that will be used
at runtime) can't be known at compile time.

#### Example

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant:

{% prettify dart tag=pre+code %}
class C<T> {
  const C();
}

C<T> newC<T>() => const C<[!T!]>();
{% endprettify %}

#### Common fixes

If the type that will be used for the type parameter can be known at
compile time, then remove the use of the type parameter:

{% prettify dart tag=pre+code %}
class C<T> {
  const C();
}

C<int> newC() => const C<int>();
{% endprettify %}

If the type that will be used for the type parameter can't be known until
runtime, then remove the keyword `const`:

{% prettify dart tag=pre+code %}
class C<T> {
  const C();
}

C<T> newC<T>() => C<T>();
{% endprettify %}

### creation_with_non_type

_The name '{0}' isn't a class._

#### Description

The analyzer produces this diagnostic when an instance creation using
either `new` or `const` specifies a name that isn't defined as a class.

#### Example

The following code produces this diagnostic because `f` is a function
rather than a class:

{% prettify dart tag=pre+code %}
int f() => 0;

void g() {
  new [!f!]();
}
{% endprettify %}

#### Common fixes

If a class should be created, then replace the invalid name with the name
of a valid class:

{% prettify dart tag=pre+code %}
int f() => 0;

void g() {
  new Object();
}
{% endprettify %}

If the name is the name of a function and you want that function to be
invoked, then remove the `new` or `const` keyword:

{% prettify dart tag=pre+code %}
int f() => 0;

void g() {
  f();
}
{% endprettify %}

### dead_code

_Dead code._

#### Description

The analyzer produces this diagnostic when code is found that won't be
executed because execution will never reach the code.

#### Examples

The following code produces this diagnostic because the invocation of
`print` occurs after the function has returned:

{% prettify dart tag=pre+code %}
void f() {
  return;
  [!print('here');!]
}
{% endprettify %}

#### Common fixes

If the code isn't needed, then remove it:

{% prettify dart tag=pre+code %}
void f() {
  return;
}
{% endprettify %}

If the code needs to be executed, then either move the code to a place
where it will be executed:

{% prettify dart tag=pre+code %}
void f() {
  print('here');
  return;
}
{% endprettify %}

Or, rewrite the code before it, so that it can be reached:

{% prettify dart tag=pre+code %}
void f({bool skipPrinting = true}) {
  if (skipPrinting) {
    return;
  }
  print('here');
}
{% endprettify %}

### dead_code_catch_following_catch

_Dead code: Catch clauses after a 'catch (e)' or an 'on Object catch (e)' are
never reached._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found that
can't be executed because it’s after a `catch` clause of the form
`catch (e)` or `on Object catch (e)`. The first `catch` clause that matches
the thrown object is selected, and both of those forms will match any
object, so no `catch` clauses that follow them will be selected.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } catch (e) {
  } [!on String {
  }!]
}
{% endprettify %}

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } on String {
  } catch (e) {
  }
}
{% endprettify %}

If the clause doesn't need to be selectable, then remove it:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } catch (e) {
  }
}
{% endprettify %}

### dead_code_on_catch_subtype

_Dead code: This on-catch block won’t be executed because '{0}' is a subtype of
'{1}' and hence will have been caught already._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found that
can't be executed because it is after a `catch` clause that catches either
the same type or a supertype of the clause's type. The first `catch` clause
that matches the thrown object is selected, and the earlier clause always
matches anything matchable by the highlighted clause, so the highlighted
clause will never be selected.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } on num {
  } [!on int {
  }!]
}
{% endprettify %}

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } on int {
  } on num {
  }
}
{% endprettify %}

If the clause doesn't need to be selectable, then remove it:

{% prettify dart tag=pre+code %}
void f() {
  try {
  } on num {
  }
}
{% endprettify %}

### dead_null_aware_expression

_The left operand can't be null, so the right operand is never executed._

#### Description

The analyzer produces this diagnostic in two cases.

The first is when the left operand of an `??` operator can't be `null`.
The right operand is only evaluated if the left operand has the value
`null`, and because the left operand can't be `null`, the right operand is
never evaluated.

The second is when the left-hand side of an assignment using the `??=`
operator can't be `null`. The right-hand side is only evaluated if the
left-hand side has the value `null`, and because the left-hand side can't
be `null`, the right-hand side is never evaluated.

#### Example

The following code produces this diagnostic because `x` can't be `null`:

{% prettify dart tag=pre+code %}
int f(int x) {
  return x ?? [!0!];
}
{% endprettify %}

The following code produces this diagnostic because `f` can't be `null`:

{% prettify dart tag=pre+code %}
class C {
  int f = -1;

  void m(int x) {
    f ??= [!x!];
  }
}
{% endprettify %}

#### Common fixes

If the diagnostic is reported for an `??` operator, then remove the `??`
operator and the right operand:

{% prettify dart tag=pre+code %}
int f(int x) {
  return x;
}
{% endprettify %}

If the diagnostic is reported for an assignment, and the assignment isn't
needed, then remove the assignment:

{% prettify dart tag=pre+code %}
class C {
  int f = -1;

  void m(int x) {
  }
}
{% endprettify %}

If the assignment is needed, but should be based on a different condition,
then rewrite the code to use `=` and the different condition:

{% prettify dart tag=pre+code %}
class C {
  int f = -1;

  void m(int x) {
    if (f < 0) {
      f = x;
    }
  }
}
{% endprettify %}

### default_list_constructor

_The default 'List' constructor isn't available when null safety is enabled._

#### Description

The analyzer produces this diagnostic when it finds a use of the default
constructor for the class `List` in code that has opted in to null safety.

#### Example

Assuming the following code is opted in to null safety, it produces this
diagnostic because it uses the default `List` constructor:

{% prettify dart tag=pre+code %}
var l = [!List<int>!]();
{% endprettify %}

#### Common fixes

If no initial size is provided, then convert the code to use a list
literal:

{% prettify dart tag=pre+code %}
var l = <int>[];
{% endprettify %}

If an initial size needs to be provided and there is a single reasonable
initial value for the elements, then use `List.filled`:

{% prettify dart tag=pre+code %}
var l = List.filled(3, 0);
{% endprettify %}

If an initial size needs to be provided but each element needs to be
computed, then use `List.generate`:

{% prettify dart tag=pre+code %}
var l = List.generate(3, (i) => i);
{% endprettify %}

### default_value_in_function_type

_Parameters in a function type can't have default values._

#### Description

The analyzer produces this diagnostic when a function type associated with
a parameter includes optional parameters that have a default value. This
isn't allowed because the default values of parameters aren't part of the
function's type, and therefore including them doesn't provide any value.

#### Example

The following code produces this diagnostic because the parameter `p` has a
default value even though it's part of the type of the parameter `g`:

{% prettify dart tag=pre+code %}
void f(void Function([int p [!=!] 0]) g) {
}
{% endprettify %}

#### Common fixes

Remove the default value from the function-type's parameter:

{% prettify dart tag=pre+code %}
void f(void Function([int p]) g) {
}
{% endprettify %}

### default_value_in_redirecting_factory_constructor

_Default values aren't allowed in factory constructors that redirect to another
constructor._

#### Description

The analyzer produces this diagnostic when a factory constructor that
redirects to another constructor specifies a default value for an optional
parameter.

#### Example

The following code produces this diagnostic because the factory constructor
in `A` has a default value for the optional parameter `x`:

{% prettify dart tag=pre+code %}
class A {
  factory A([int [!x!] = 0]) = B;
}

class B implements A {
  B([int x = 1]) {}
}
{% endprettify %}

#### Common fixes

Remove the default value from the factory constructor:

{% prettify dart tag=pre+code %}
class A {
  factory A([int x]) = B;
}

class B implements A {
  B([int x = 1]) {}
}
{% endprettify %}

Note that this fix might change the value used when the optional parameter
is omitted. If that happens, and if that change is a problem, then consider
making the optional parameter a required parameter in the factory method:

{% prettify dart tag=pre+code %}
class A {
 factory A(int x) = B;
}

class B implements A {
  B([int x = 1]) {}
}
{% endprettify %}

### definitely_unassigned_late_local_variable

_The late local variable '{0}' is definitely unassigned at this point._

#### Description

The analyzer produces this diagnostic when [definite assignment][] analysis
shows that a local variable that's marked as `late` is read before being
assigned.

#### Example

The following code produces this diagnostic because `x` wasn't assigned a
value before being read:

{% prettify dart tag=pre+code %}
void f(bool b) {
  late int x;
  print([!x!]);
}
{% endprettify %}

#### Common fixes

Assign a value to the variable before reading from it:

{% prettify dart tag=pre+code %}
void f(bool b) {
  late int x;
  x = b ? 1 : 0;
  print(x);
}
{% endprettify %}

### deprecated_member_use

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}._

#### Description

The analyzer produces this diagnostic when a deprecated library or class
member is used in a different package.

#### Examples

If the method `m` in the class `C` is annotated with `@deprecated`, then
the following code produces this diagnostic:

{% prettify dart tag=pre+code %}
void f(C c) {
  c.[!m!]();
}
{% endprettify %}

#### Common fixes

The documentation for declarations that are annotated with `@deprecated`
should indicate what code to use in place of the deprecated code.

### deprecated_member_use_from_same_package

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}._

#### Description

The analyzer produces this diagnostic when a deprecated library member or
class member is used in the same package in which it's declared.

#### Examples

The following code produces this diagnostic because `x` is deprecated:

{% prettify dart tag=pre+code %}
@deprecated
var x = 0;
var y = [!x!];
{% endprettify %}

#### Common fixes

The fix depends on what's been deprecated and what the replacement is. The
documentation for deprecated declarations should indicate what code to use
in place of the deprecated code.

### duplicate_constructor

_The constructor with name '{0}' is already defined._

_The default constructor is already defined._

#### Description

The analyzer produces this diagnostic when a class declares more than one
unnamed constructor or when it declares more than one constructor with the
same name.

#### Examples

The following code produces this diagnostic because there are two
declarations for the unnamed constructor:

{% prettify dart tag=pre+code %}
class C {
  C();

  [!C!]();
}
{% endprettify %}

The following code produces this diagnostic because there are two
declarations for the constructor named `m`:

{% prettify dart tag=pre+code %}
class C {
  C.m();

  [!C.m!]();
}
{% endprettify %}

#### Common fixes

If there are multiple unnamed constructors and all of the constructors are
needed, then give all of them, or all except one of them, a name:

{% prettify dart tag=pre+code %}
class C {
  C();

  C.n();
}
{% endprettify %}

If there are multiple unnamed constructors and all except one of them are
unneeded, then remove the constructors that aren't needed:

{% prettify dart tag=pre+code %}
class C {
  C();
}
{% endprettify %}

If there are multiple named constructors and all of the constructors are
needed, then rename all except one of them:

{% prettify dart tag=pre+code %}
class C {
  C.m();

  C.n();
}
{% endprettify %}

If there are multiple named constructors and all except one of them are
unneeded, then remove the constructorsthat aren't needed:

{% prettify dart tag=pre+code %}
class C {
  C.m();
}
{% endprettify %}

### duplicate_definition

_The name '{0}' is already defined._

#### Description

The analyzer produces this diagnostic when a name is declared, and there is
a previous declaration with the same name in the same scope.

#### Examples

The following code produces this diagnostic because the name `x` is
declared twice:

{% prettify dart tag=pre+code %}
int x = 0;
int [!x!] = 1;
{% endprettify %}

#### Common fixes

Choose a different name for one of the declarations.

{% prettify dart tag=pre+code %}
int x = 0;
int y = 1;
{% endprettify %}

### duplicate_hidden_name

_Duplicate hidden name._

#### Description

The analyzer produces this diagnostic when a name occurs multiple times in
a `hide` clause. Repeating the name is unnecessary.

#### Example

The following code produces this diagnostic because the name `min` is
hidden more than once:

{% prettify dart tag=pre+code %}
import 'dart:math' hide min, [!min!];

var x = pi;
{% endprettify %}

#### Common fixes

If the name was mistyped in one or more places, then correct the mistyped
names:

{% prettify dart tag=pre+code %}
import 'dart:math' hide max, min;

var x = pi;
{% endprettify %}

If the name wasn't mistyped, then remove the unnecessary name from the
list:

{% prettify dart tag=pre+code %}
import 'dart:math' hide min;

var x = pi;
{% endprettify %}

### duplicate_ignore

_The diagnostic '{0}' doesn't need to be ignored here because it's already being
ignored._

#### Description

The analyzer produces this diagnostic when a diagnostic name appears in an
`ignore` comment, but the diagnostic is already being ignored, either
because it's already included in the same `ignore` comment or because it
appears in an `ignore-in-file` comment.

#### Example

The following code produces this diagnostic because the diagnostic named
`unused_local_variable` is already being ignored for the whole file so it
doesn't need to be ignored on a specific line:

{% prettify dart tag=pre+code %}
// ignore_for_file: unused_local_variable
void f() {
  // ignore: [!unused_local_variable!]
  var x = 0;
}
{% endprettify %}

The following code produces this diagnostic because the diagnostic named
`unused_local_variable` is being ignored twice on the same line:

{% prettify dart tag=pre+code %}
void f() {
  // ignore: unused_local_variable, [!unused_local_variable!]
  var x = 0;
}
{% endprettify %}

#### Common fixes

Remove the ignore comment, or remove the unnecessary diagnostic name if the
ignore comment is ignoring more than one diagnostic:

{% prettify dart tag=pre+code %}
// ignore_for_file: unused_local_variable
void f() {
  var x = 0;
}
{% endprettify %}

### duplicate_import

_Duplicate import._

#### Description

The analyzer produces this diagnostic when an import directive is found
that is the same as an import before it in the file. The second import
doesn’t add value and should be removed.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';
import [!'package:meta/meta.dart'!];

@sealed class C {}
{% endprettify %}

#### Common fixes

Remove the unnecessary import:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

@sealed class C {}
{% endprettify %}

### duplicate_named_argument

_The argument for the named parameter '{0}' was already specified._

#### Description

The analyzer produces this diagnostic when an invocation has two or more
named arguments that have the same name.

#### Examples

The following code produces this diagnostic because there are two arguments
with the name `a`:

{% prettify dart tag=pre+code %}
void f(C c) {
  c.m(a: 0, [!a!]: 1);
}

class C {
  void m({int a, int b}) {}
}
{% endprettify %}

#### Common fixes

If one of the arguments should have a different name, then change the name:

{% prettify dart tag=pre+code %}
void f(C c) {
  c.m(a: 0, b: 1);
}

class C {
  void m({int a, int b}) {}
}
{% endprettify %}

If one of the arguments is wrong, then remove it:

{% prettify dart tag=pre+code %}
void f(C c) {
  c.m(a: 1);
}

class C {
  void m({int a, int b}) {}
}
{% endprettify %}

### duplicate_part

_The library already contains a part with the URI '{0}'._

#### Description

The analyzer produces this diagnostic when a single file is referenced in
multiple part directives.

#### Example

Given a file named `part.dart` containing

{% prettify dart tag=pre+code %}
part of lib;
{% endprettify %}

The following code produces this diagnostic because the file `part.dart` is
included multiple times:

{% prettify dart tag=pre+code %}
library lib;

part 'part.dart';
part [!'part.dart'!];
{% endprettify %}

#### Common fixes

Remove all except the first of the duplicated part directives:

{% prettify dart tag=pre+code %}
library lib;

part 'part.dart';
{% endprettify %}

### duplicate_shown_name

_Duplicate shown name._

#### Description

The analyzer produces this diagnostic when a name occurs multiple times in
a `show` clause. Repeating the name is unnecessary.

#### Example

The following code produces this diagnostic because the name `min` is shown
more than once:

{% prettify dart tag=pre+code %}
import 'dart:math' show min, [!min!];

var x = min(2, min(0, 1));
{% endprettify %}

#### Common fixes

If the name was mistyped in one or more places, then correct the mistyped
names:

{% prettify dart tag=pre+code %}
import 'dart:math' show max, min;

var x = max(2, min(0, 1));
{% endprettify %}

If the name wasn't mistyped, then remove the unnecessary name from the
list:

{% prettify dart tag=pre+code %}
import 'dart:math' show min;

var x = min(2, min(0, 1));
{% endprettify %}

### equal_elements_in_const_set

_Two elements in a constant set literal can't be equal._

#### Description

The analyzer produces this diagnostic when two elements in a constant set
literal have the same value. The set can only contain each value once,
which means that one of the values is unnecessary.

#### Examples

The following code produces this diagnostic because the string `'a'` is
specified twice:

{% prettify dart tag=pre+code %}
const Set<String> set = {'a', [!'a'!]};
{% endprettify %}

#### Common fixes

Remove one of the duplicate values:

{% prettify dart tag=pre+code %}
const Set<String> set = {'a'};
{% endprettify %}

Note that literal sets preserve the order of their elements, so the choice
of which element to remove might affect the order in which elements are
returned by an iterator.

### equal_elements_in_set

_Two elements in a set literal shouldn't be equal._

#### Description

The analyzer produces this diagnostic when an element in a non-constant set
is the same as a previous element in the same set. If two elements are the
same, then the second value is ignored, which makes having both elements
pointless and likely signals a bug.

#### Example

The following code produces this diagnostic because the element `1` appears
twice:

{% prettify dart tag=pre+code %}
const a = 1;
const b = 1;
var s = <int>{a, [!b!]};
{% endprettify %}

#### Common fixes

If both elements should be included in the set, then change one of the
elements:

{% prettify dart tag=pre+code %}
const a = 1;
const b = 2;
var s = <int>{a, b};
{% endprettify %}

If only one of the elements is needed, then remove the one that isn't
needed:

{% prettify dart tag=pre+code %}
const a = 1;
var s = <int>{a};
{% endprettify %}

Note that literal sets preserve the order of their elements, so the choice
of which element to remove might affect the order in which elements are
returned by an iterator.

### equal_keys_in_const_map

_Two keys in a constant map literal can't be equal._

#### Description

The analyzer produces this diagnostic when a key in a constant map is the
same as a previous key in the same map. If two keys are the same, then the
second value would overwrite the first value, which makes having both pairs
pointless.

#### Examples

The following code produces this diagnostic because the key `1` is used
twice:

{% prettify dart tag=pre+code %}
const map = <int, String>{1: 'a', 2: 'b', [!1!]: 'c', 4: 'd'};
{% endprettify %}

#### Common fixes

If both entries should be included in the map, then change one of the keys
to be different:

{% prettify dart tag=pre+code %}
const map = <int, String>{1: 'a', 2: 'b', 3: 'c', 4: 'd'};
{% endprettify %}

If only one of the entries is needed, then remove the one that isn't
needed:

{% prettify dart tag=pre+code %}
const map = <int, String>{1: 'a', 2: 'b', 4: 'd'};
{% endprettify %}

Note that literal maps preserve the order of their entries, so the choice
of which entry to remove might affect the order in which keys and values
are returned by an iterator.

### equal_keys_in_map

_Two keys in a map literal shouldn't be equal._

#### Description

The analyzer produces this diagnostic when a key in a non-constant map is
the same as a previous key in the same map. If two keys are the same, then
the second value overwrites the first value, which makes having both pairs
pointless and likely signals a bug.

#### Example

The following code produces this diagnostic because the keys `a` and `b`
have the same value:

{% prettify dart tag=pre+code %}
const a = 1;
const b = 1;
var m = <int, String>{a: 'a', [!b!]: 'b'};
{% endprettify %}

#### Common fixes

If both entries should be included in the map, then change one of the keys:

{% prettify dart tag=pre+code %}
const a = 1;
const b = 2;
var m = <int, String>{a: 'a', b: 'b'};
{% endprettify %}

If only one of the entries is needed, then remove the one that isn't
needed:

{% prettify dart tag=pre+code %}
const a = 1;
var m = <int, String>{a: 'a'};
{% endprettify %}

Note that literal maps preserve the order of their entries, so the choice
of which entry to remove might affect the order in which the keys and
values are returned by an iterator.

### expected_one_list_type_arguments

_List literals require one type argument or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a list literal has more than one
type argument.

#### Example

The following code produces this diagnostic because the list literal has
two type arguments when it can have at most one:

{% prettify dart tag=pre+code %}
var l = [!<int, int>!][];
{% endprettify %}

#### Common fixes

Remove all except one of the type arguments:

{% prettify dart tag=pre+code %}
var l = <int>[];
{% endprettify %}

### expected_one_set_type_arguments

_Set literals require one type argument or none, but {0} were found._

#### Description

The analyzer produces this diagnostic when a set literal has more than one
type argument.

#### Example

The following code produces this diagnostic because the set literal has
three type arguments when it can have at most one:

{% prettify dart tag=pre+code %}
var s = [!<int, String, int>!]{0, 'a', 1};
{% endprettify %}

#### Common fixes

Remove all except one of the type arguments:

{% prettify dart tag=pre+code %}
var s = <int>{0, 1};
{% endprettify %}

### expected_two_map_type_arguments

_Map literals require two type arguments or none, but {0} found._

#### Description

The analyzer produces this diagnostic when a map literal has either one or
more than two type arguments.

#### Example

The following code produces this diagnostic because the map literal has
three type arguments when it can have either two or zero:

{% prettify dart tag=pre+code %}
var m = [!<int, String, int>!]{};
{% endprettify %}

#### Common fixes

Remove all except two of the type arguments:

{% prettify dart tag=pre+code %}
var m = <int, String>{};
{% endprettify %}

### export_legacy_symbol

_The symbol '{0}' is defined in a legacy library, and can't be re-exported from
a library with null safety enabled._

#### Description

The analyzer produces this diagnostic when a library that was opted in to
null safety exports another library, and the exported library is opted out
of null safety.

#### Example

Given a library that is opted out of null safety:

{% prettify dart tag=pre+code %}
// @dart = 2.8
String s;
{% endprettify %}

The following code produces this diagnostic because it's exporting symbols
from an opted-out library:

{% prettify dart tag=pre+code %}
export [!'optedOut.dart'!];

class C {}
{% endprettify %}

#### Common fixes

If you're able to do so, migrate the exported library so that it doesn't
need to opt out:

{% prettify dart tag=pre+code %}
String? s;
{% endprettify %}

If you can't migrate the library, then remove the export:

{% prettify dart tag=pre+code %}
class C {}
{% endprettify %}

If the exported library (the one that is opted out) itself exports an
opted-in library, then it's valid for your library to indirectly export the
symbols from the opted-in library. You can do so by adding a hide
combinator to the export directive in your library that hides all of the
names declared in the opted-out library.

### export_of_non_library

_The exported library '{0}' can't have a part-of directive._

#### Description

The analyzer produces this diagnostic when an export directive references a
part rather than a library.

#### Example

Given a file named `part.dart` containing

{% prettify dart tag=pre+code %}
part of lib;
{% endprettify %}

The following code produces this diagnostic because the file `part.dart` is
a part, and only libraries can be exported:

{% prettify dart tag=pre+code %}
library lib;

export [!'part.dart'!];
{% endprettify %}

#### Common fixes

Either remove the export directive, or change the URI to be the URI of the
library containing the part.

### expression_in_map

_Expressions can't be used in a map literal._

#### Description

The analyzer produces this diagnostic when the analyzer finds an
expression, rather than a map entry, in what appears to be a map literal.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
var map = <String, int>{'a': 0, 'b': 1, [!'c'!]};
{% endprettify %}

#### Common fixes

If the expression is intended to compute either a key or a value in an
entry, fix the issue by replacing the expression with the key or the value.
For example:

{% prettify dart tag=pre+code %}
var map = <String, int>{'a': 0, 'b': 1, 'c': 2};
{% endprettify %}

### extends_non_class

_Classes can only extend other classes._

#### Description

The analyzer produces this diagnostic when an `extends` clause contains a
name that is declared to be something other than a class.

#### Examples

The following code produces this diagnostic because `f` is declared to be a
function:

{% prettify dart tag=pre+code %}
void f() {}

class C extends [!f!] {}
{% endprettify %}

#### Common fixes

If you want the class to extend a class other than `Object`, then replace
the name in the `extends` clause with the name of that class:

{% prettify dart tag=pre+code %}
void f() {}

class C extends B {}

class B {}
{% endprettify %}

If you want the class to extend `Object`, then remove the `extends` clause:

{% prettify dart tag=pre+code %}
void f() {}

class C {}
{% endprettify %}

### extension_as_expression

_Extension '{0}' can't be used as an expression._

#### Description

The analyzer produces this diagnostic when the name of an extension is used
in an expression other than in an extension override or to qualify an
access to a static member of the extension. Because classes define a type,
the name of a class can be used to refer to the instance of `Type`
representing the type of the class. Extensions, on the other hand, don't
define a type and can't be used as a type literal.

#### Examples

The following code produces this diagnostic because `E` is an extension:

{% prettify dart tag=pre+code %}
extension E on int {
  static String m() => '';
}

var x = [!E!];
{% endprettify %}

#### Common fixes

Replace the name of the extension with a name that can be referenced, such
as a static member defined on the extension:

{% prettify dart tag=pre+code %}
extension E on int {
  static String m() => '';
}

var x = E.m();
{% endprettify %}

### extension_conflicting_static_and_instance

_Extension '{0}' can't define static member '{1}' and an instance member with
the same name._

#### Description

The analyzer produces this diagnostic when an extension declaration
contains both an instance member and a static member that have the same
name. The instance member and the static member can't have the same name
because it's unclear which member is being referenced by an unqualified use
of the name within the body of the extension.

#### Examples

The following code produces this diagnostic because the name `a` is being
used for two different members:

{% prettify dart tag=pre+code %}
extension E on Object {
  int get a => 0;
  static int [!a!]() => 0;
}
{% endprettify %}

#### Common fixes

Rename or remove one of the members:

{% prettify dart tag=pre+code %}
extension E on Object {
  int get a => 0;
  static int b() => 0;
}
{% endprettify %}

### extension_declares_abstract_member

_Extensions can't declare abstract members._

#### Description

The analyzer produces this diagnostic when an abstract declaration is
declared in an extension. Extensions can declare only concrete members.

#### Examples

The following code produces this diagnostic because the method `a` doesn't
have a body:

{% prettify dart tag=pre+code %}
extension E on String {
  int [!a!]();
}
{% endprettify %}

#### Common fixes

Either provide an implementation for the member or remove it.

### extension_declares_constructor

_Extensions can't declare constructors._

#### Description

The analyzer produces this diagnostic when a constructor declaration is
found in an extension. It isn't valid to define a constructor because
extensions aren't classes, and it isn't possible to create an instance of
an extension.

#### Examples

The following code produces this diagnostic because there is a constructor
declaration in `E`:

{% prettify dart tag=pre+code %}
extension E on String {
  [!E!]() : super();
}
{% endprettify %}

#### Common fixes

Remove the constructor or replace it with a static method.

### extension_declares_instance_field

_Extensions can't declare instance fields_

#### Description

The analyzer produces this diagnostic when an instance field declaration is
found in an extension. It isn't valid to define an instance field because
extensions can only add behavior, not state.

#### Examples

The following code produces this diagnostic because `s` is an instance
field:

{% prettify dart tag=pre+code %}
extension E on String {
  String [!s!];
}
{% endprettify %}

#### Common fixes

Remove the field, make it a static field, or convert it to be a getter,
setter, or method.

### extension_declares_member_of_object

_Extensions can't declare members with the same name as a member declared by
'Object'._

#### Description

The analyzer produces this diagnostic when an extension declaration
declares a member with the same name as a member declared in the class
`Object`. Such a member can never be used because the member in `Object` is
always found first.

#### Examples

The following code produces this diagnostic because `toString` is defined
by `Object`:

{% prettify dart tag=pre+code %}
extension E on String {
  String [!toString!]() => this;
}
{% endprettify %}

#### Common fixes

Remove the member or rename it so that the name doesn't conflict with the
member in `Object`:

{% prettify dart tag=pre+code %}
extension E on String {
  String displayString() => this;
}
{% endprettify %}

### extension_override_access_to_static_member

_An extension override can't be used to access a static member from an
extension._

#### Description

The analyzer produces this diagnostic when an extension override is the
receiver of the invocation of a static member. Similar to static members in
classes, the static members of an extension should be accessed using the
name of the extension, not an extension override.

#### Examples

The following code produces this diagnostic because `m` is static:

{% prettify dart tag=pre+code %}
extension E on String {
  static void m() {}
}

void f() {
  E('').[!m!]();
}
{% endprettify %}

#### Common fixes

Replace the extension override with the name of the extension:

{% prettify dart tag=pre+code %}
extension E on String {
  static void m() {}
}

void f() {
  E.m();
}
{% endprettify %}

### extension_override_argument_not_assignable

_The type of the argument to the extension override '{0}' isn't assignable to
the extended type '{1}'._

#### Description

The analyzer produces this diagnostic when the argument to an extension
override isn't assignable to the type being extended by the extension.

#### Examples

The following code produces this diagnostic because `3` isn't a `String`:

{% prettify dart tag=pre+code %}
extension E on String {
  void method() {}
}

void f() {
  E([!3!]).method();
}
{% endprettify %}

#### Common fixes

If you're using the correct extension, then update the argument to have the
correct type:

{% prettify dart tag=pre+code %}
extension E on String {
  void method() {}
}

void f() {
  E(3.toString()).method();
}
{% endprettify %}

If there's a different extension that's valid for the type of the argument,
then either replace the name of the extension or unwrap the argument so
that the correct extension is found.

### extension_override_without_access

_An extension override can only be used to access instance members._

#### Description

The analyzer produces this diagnostic when an extension override is found
that isn't being used to access one of the members of the extension. The
extension override syntax doesn't have any runtime semantics; it only
controls which member is selected at compile time.

#### Examples

The following code produces this diagnostic because `E(i)` isn't an
expression:

{% prettify dart tag=pre+code %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print([!E(i)!]);
}
{% endprettify %}

#### Common fixes

If you want to invoke one of the members of the extension, then add the
invocation:

{% prettify dart tag=pre+code %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(E(i).a);
}
{% endprettify %}

If you don't want to invoke a member, then unwrap the argument:

{% prettify dart tag=pre+code %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(i);
}
{% endprettify %}

### extension_override_with_cascade

_Extension overrides have no value so they can't be used as the receiver of a
cascade expression._

#### Description

The analyzer produces this diagnostic when an extension override is used as
the receiver of a cascade expression. The value of a cascade expression
`e..m` is the value of the receiver `e`, but extension overrides aren't
expressions and don't have a value.

#### Examples

The following code produces this diagnostic because `E(3)` isn't an
expression:

{% prettify dart tag=pre+code %}
extension E on int {
  void m() {}
}
f() {
  [!E!](3)..m();
}
{% endprettify %}

#### Common fixes

Use `.` rather than `..`:

{% prettify dart tag=pre+code %}
extension E on int {
  void m() {}
}
f() {
  E(3).m();
}
{% endprettify %}

If there are multiple cascaded accesses, you'll need to duplicate the
extension override for each one.

### extra_positional_arguments

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows.

#### Examples

The following code produces this diagnostic because `f` defines 2
parameters but is invoked with 3 arguments:

{% prettify dart tag=pre+code %}
void f(int a, int b) {}
void g() {
  f(1, 2, [!3!]);
}
{% endprettify %}

#### Common fixes

Remove the arguments that don't correspond to parameters:

{% prettify dart tag=pre+code %}
void f(int a, int b) {}
void g() {
  f(1, 2);
}
{% endprettify %}

### extra_positional_arguments_could_be_named

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows, but the
method or function defines named parameters.

#### Examples

The following code produces this diagnostic because `f` defines 2
positional parameters but has a named parameter that could be used for the
third argument:

{% prettify dart tag=pre+code %}
void f(int a, int b, {int c}) {}
void g() {
  f(1, 2, [!3!]);
}
{% endprettify %}

#### Common fixes

If some of the arguments should be values for named parameters, then add
the names before the arguments:

{% prettify dart tag=pre+code %}
void f(int a, int b, {int c}) {}
void g() {
  f(1, 2, c: 3);
}
{% endprettify %}

Otherwise, remove the arguments that don't correspond to positional
parameters:

{% prettify dart tag=pre+code %}
void f(int a, int b, {int c}) {}
void g() {
  f(1, 2);
}
{% endprettify %}

### field_initialized_by_multiple_initializers

_The field '{0}' can't be initialized twice in the same constructor._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor initializes a field more than once. There is no value to allow
both initializers because only the last value is preserved.

#### Example

The following code produces this diagnostic because the field `f` is being
initialized twice:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C() : f = 0, [!f!] = 1;
}
{% endprettify %}

#### Common fixes

Remove one of the initializers:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C() : f = 0;
}
{% endprettify %}

### field_initialized_in_initializer_and_declaration

_Fields can't be initialized in the constructor if they are final and were
already initialized at their declaration._

#### Description

The analyzer produces this diagnostic when a final field is initialized in
both the declaration of the field and in an initializer in a constructor.
Final fields can only be assigned once, so it can't be initialized in both
places.

#### Example

The following code produces this diagnostic because `f` is :

{% prettify dart tag=pre+code %}
class C {
  final int f = 0;
  C() : [!f!] = 1;
}
{% endprettify %}

#### Common fixes

If the initialization doesn't depend on any values passed to the
constructor, and if all of the constructors need to initialize the field to
the same value, then remove the initializer from the constructor:

{% prettify dart tag=pre+code %}
class C {
  final int f = 0;
  C();
}
{% endprettify %}

If the initialization depends on a value passed to the constructor, or if
different constructors need to initialize the field differently, then
remove the initializer in the field's declaration:

{% prettify dart tag=pre+code %}
class C {
  final int f;
  C() : f = 1;
}
{% endprettify %}

### field_initialized_in_parameter_and_initializer

_Fields can't be initialized in both the parameter list and the initializers._

#### Description

The analyzer produces this diagnostic when a field is initialized in both
the parameter list and in the initializer list of a constructor.

#### Example

The following code produces this diagnostic because the field `f` is
initialized both by a field formal parameter and in the initializer list:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f) : [!f!] = 0;
}
{% endprettify %}

#### Common fixes

If the field should be initialized by the parameter, then remove the
initialization in the initializer list:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);
}
{% endprettify %}

If the field should be initialized in the initializer list and the
parameter isn't needed, then remove the parameter:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C() : f = 0;
}
{% endprettify %}

If the field should be initialized in the initializer list and the
parameter is needed, then make it a normal parameter:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(int g) : f = g * 2;
}
{% endprettify %}

### field_initializer_factory_constructor

_Initializing formal parameters can't be used in factory constructors._

#### Description

The analyzer produces this diagnostic when a factory constructor has a
field formal parameter. Factory constructors can't assign values to fields
because no instance is created; hence, there is no field to assign.

#### Example

The following code produces this diagnostic because the factory constructor
uses a field formal parameter:

{% prettify dart tag=pre+code %}
class C {
  int? f;

  factory C([!this.f!]) => throw 0;
}
{% endprettify %}

#### Common fixes

Replace the field formal parameter with a normal parameter:

{% prettify dart tag=pre+code %}
class C {
  int? f;

  factory C(int f) => throw 0;
}
{% endprettify %}

### field_initializer_not_assignable

_The initializer type '{0}' can't be assigned to the field type '{1}' in a const
constructor._

_The initializer type '{0}' can't be assigned to the field type '{1}'._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor initializes a field to a value that isn't assignable to the
field.

#### Example

The following code produces this diagnostic because `0` has the type `int`,
and an `int` can't be assigned to a field of type `String`:

{% prettify dart tag=pre+code %}
class C {
  String s;

  C() : s = [!0!];
}
{% endprettify %}

#### Common fixes

If the type of the field is correct, then change the value assigned to it
so that the value has a valid type:

{% prettify dart tag=pre+code %}
class C {
  String s;

  C() : s = '0';
}
{% endprettify %}

If the type of the value is correct, then change the type of the field to
allow the assignment:

{% prettify dart tag=pre+code %}
class C {
  int s;

  C() : s = 0;
}
{% endprettify %}

### field_initializer_redirecting_constructor

_The redirecting constructor can't have a field initializer._

#### Description

The analyzer produces this diagnostic when a redirecting constructor
initializes a field in the object. This isn't allowed because the instance
that has the field hasn't been created at the point at which it should be
initialized.

#### Example

The following code produces this diagnostic because the constructor
`C.zero`, which redirects to the constructor `C`, has a field formal
parameter that initializes the field `f`:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);

  C.zero([!this.f!]) : this(f);
}
{% endprettify %}

The following code produces this diagnostic because the constructor
`C.zero`, which redirects to the constructor `C`, has an initializer that
initializes the field `f`:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);

  C.zero() : [!f = 0!], this(1);
}
{% endprettify %}

#### Common fixes

If the initialization is done by a field formal parameter, then use a
normal parameter:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);

  C.zero(int f) : this(f);
}
{% endprettify %}

If the initialization is done in an initializer, then remove the
initializer:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);

  C.zero() : this(0);
}
{% endprettify %}

### field_initializing_formal_not_assignable

_The parameter type '{0}' is incompatible with the field type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of a field formal
parameter isn't assignable to the type of the field being initialized.

#### Example

The following code produces this diagnostic because the field formal
parameter has the type `String`, but the type of the field is `int`. The
parameter must have a type that is a subtype of the field's type.

{% prettify dart tag=pre+code %}
class C {
  int f;

  C([!String this.f!]);
}
{% endprettify %}

#### Common fixes

If the type of the field is incorrect, then change the type of the field to
match the type of the parameter, and consider removing the type from the
parameter:

{% prettify dart tag=pre+code %}
class C {
  String f;

  C(this.f);
}
{% endprettify %}

If the type of the parameter is incorrect, then remove the type of the
parameter:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(this.f);
}
{% endprettify %}

If the types of both the field and the parameter are correct, then use an
initializer rather than a field formal parameter to convert the parameter
value into a value of the correct type:

{% prettify dart tag=pre+code %}
class C {
  int f;

  C(String s) : f = int.parse(s);
}
{% endprettify %}

### final_initialized_in_declaration_and_constructor

_'{0}' is final and was given a value when it was declared, so it can't be set
to a new value._

#### Description

The analyzer produces this diagnostic when a final field is initialized
twice: once where it's declared and once by a constructor's parameter.

#### Example

The following code produces this diagnostic because the field `f` is
initialized twice:

{% prettify dart tag=pre+code %}
class C {
  final int f = 0;

  C(this.[!f!]);
}
{% endprettify %}

#### Common fixes

If the field should have the same value for all instances, then remove the
initialization in the parameter list:

{% prettify dart tag=pre+code %}
class C {
  final int f = 0;

  C();
}
{% endprettify %}

If the field can have different values in different instances, then remove
the initialization in the declaration:

{% prettify dart tag=pre+code %}
class C {
  final int f;

  C(this.f);
}
{% endprettify %}

### final_not_initialized

_The final variable '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a final field or variable isn't
initialized.

#### Examples

The following code produces this diagnostic because `x` doesn't have an
initializer:

{% prettify dart tag=pre+code %}
final [!x!];
{% endprettify %}

#### Common fixes

For variables and static fields, you can add an initializer:

{% prettify dart tag=pre+code %}
final x = 0;
{% endprettify %}

For instance fields, you can add an initializer as shown in the previous
example, or you can initialize the field in every constructor. You can
initialize the field by using a field formal parameter:

{% prettify dart tag=pre+code %}
class C {
  final int x;
  C(this.x);
}
{% endprettify %}

You can also initialize the field by using an initializer in the
constructor:

{% prettify dart tag=pre+code %}
class C {
  final int x;
  C(int y) : x = y * 2;
}
{% endprettify %}

### final_not_initialized_constructor

_All final variables must be initialized, but '{0}' and '{1}' aren't._

_All final variables must be initialized, but '{0}' isn't._

_All final variables must be initialized, but '{0}', '{1}', and {2} others
aren't._

#### Description

The analyzer produces this diagnostic when a class defines one or more
final instance fields without initializers and has at least one constructor
that doesn't initialize those fields. All final instance fields must be
initialized when the instance is created, either by the field's initializer
or by the constructor.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
class C {
  final String value;

  [!C!]();
}
{% endprettify %}

#### Common fixes

If the value should be passed in to the constructor directly, then use a
field formal parameter to initialize the field `value`:

{% prettify dart tag=pre+code %}
class C {
  final String value;

  C(this.value);
}
{% endprettify %}

If the value should be computed indirectly from a value provided by the
caller, then add a parameter and include an initializer:

{% prettify dart tag=pre+code %}
class C {
  final String value;

  C(Object o) : value = o.toString();
}
{% endprettify %}

If the value of the field doesn't depend on values that can be passed to
the constructor, then add an initializer for the field as part of the field
declaration:

{% prettify dart tag=pre+code %}
class C {
  final String value = '';

  C();
}
{% endprettify %}

If the value of the field doesn't depend on values that can be passed to
the constructor but different constructors need to initialize it to
different values, then add an initializer for the field in the initializer
list:

{% prettify dart tag=pre+code %}
class C {
  final String value;

  C() : value = '';

  C.named() : value = 'c';
}
{% endprettify %}

However, if the value is the same for all instances, then consider using a
static field instead of an instance field:

{% prettify dart tag=pre+code %}
class C {
  static const String value = '';

  C();
}
{% endprettify %}

### for_in_of_invalid_type

_The type '{0}' used in the 'for' loop must implement {1}._

#### Description

The analyzer produces this diagnostic when the expression following `in` in
a for-in loop has a type that isn't a subclass of `Iterable`.

#### Examples

The following code produces this diagnostic because `m` is a `Map`, and
`Map` isn't a subclass of `Iterable`:

{% prettify dart tag=pre+code %}
void f(Map<String, String> m) {
  for (String s in [!m!]) {
    print(s);
  }
}
{% endprettify %}

#### Common fixes

Replace the expression with one that produces an iterable value:

{% prettify dart tag=pre+code %}
void f(Map<String, String> m) {
  for (String s in m.values) {
    print(s);
  }
}
{% endprettify %}

### getter_not_subtype_setter_types

_The return type of getter '{0}' is '{1}' which isn't a subtype of the type
'{2}' of its setter '{3}'._

#### Description

The analyzer produces this diagnostic when the return type of a getter
isn't a subtype of the type of the parameter of a setter with the same
name.

The subtype relationship is a requirement whether the getter and setter are
in the same class or whether one of them is in a superclass of the other.

#### Example

The following code produces this diagnostic because the return type of the
getter `x` is `num`, the parameter type of the setter `x` is `int`, and
`num` isn't a subtype of `int`:

{% prettify dart tag=pre+code %}
class C {
  num get [!x!] => 0;

  set x(int y) {}
}
{% endprettify %}

#### Common fixes

If the type of the getter is correct, then change the type of the setter:

{% prettify dart tag=pre+code %}
class C {
  num get x => 0;

  set x(num y) {}
}
{% endprettify %}

If the type of the setter is correct, then change the type of the getter:

{% prettify dart tag=pre+code %}
class C {
  int get x => 0;

  set x(int y) {}
}
{% endprettify %}

### illegal_async_generator_return_type

_Functions marked 'async*' must have a return type that is a supertype of
'Stream<T>' for some type 'T'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`async*` modifier even though the return type of the function isn't either
`Stream` or a supertype of `Stream`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the 'async*' modifier even though the return type `int`
isn't a supertype of `Stream`:

{% prettify dart tag=pre+code %}
[!int!] f() async* {}
{% endprettify %}

#### Common fixes

If the function should be asynchronous, then change the return type to be
either `Stream` or a supertype of `Stream`:

{% prettify dart tag=pre+code %}
Stream<int> f() async* {}
{% endprettify %}

If the function should be synchronous, then remove the `async*` modifier:

{% prettify dart tag=pre+code %}
int f() => 0;
{% endprettify %}

### illegal_async_return_type

_Functions marked 'async' must have a return type assignable to 'Future'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`async` modifier even though the return type of the function isn't
assignable to `Future`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the `async` modifier even though the return type isn't
assignable to `Future`:

{% prettify dart tag=pre+code %}
[!int!] f() async {
  return 0;
}
{% endprettify %}

#### Common fixes

If the function should be asynchronous, then change the return type to be
assignable to `Future`:

{% prettify dart tag=pre+code %}
Future<int> f() async {
  return 0;
}
{% endprettify %}

If the function should be synchronous, then remove the `async` modifier:

{% prettify dart tag=pre+code %}
int f() => 0;
{% endprettify %}

### illegal_sync_generator_return_type

_Functions marked 'sync*' must have a return type that is a supertype of
'Iterable<T>' for some type 'T'._

#### Description

The analyzer produces this diagnostic when the body of a function has the
`sync*` modifier even though the return type of the function isn't either
`Iterable` or a supertype of `Iterable`.

#### Example

The following code produces this diagnostic because the body of the
function `f` has the 'sync*' modifier even though the return type `int`
isn't a supertype of `Iterable`:

{% prettify dart tag=pre+code %}
[!int!] f() sync* {}
{% endprettify %}

#### Common fixes

If the function should return an iterable, then change the return type to
be either `Iterable` or a supertype of `Iterable`:

{% prettify dart tag=pre+code %}
Iterable<int> f() sync* {}
{% endprettify %}

If the function should return a single value, then remove the `sync*`
modifier:

{% prettify dart tag=pre+code %}
int f() => 0;
{% endprettify %}

### implements_non_class

_Classes and mixins can only implement other classes and mixins._

#### Description

The analyzer produces this diagnostic when a name used in the `implements`
clause of a class or mixin declaration is defined to be something other
than a class or mixin.

#### Examples

The following code produces this diagnostic because `x` is a variable
rather than a class or mixin:

{% prettify dart tag=pre+code %}
var x;
class C implements [!x!] {}
{% endprettify %}

#### Common fixes

If the name is the name of an existing class or mixin that's already being
imported, then add a prefix to the import so that the local definition of
the name doesn't shadow the imported name.

If the name is the name of an existing class or mixin that isn't being
imported, then add an import, with a prefix, for the library in which it’s
declared.

Otherwise, either replace the name in the `implements` clause with the name
of an existing class or mixin, or remove the name from the `implements`
clause.

### implements_repeated

_'{0}' can only be implemented once._

#### Description

The analyzer produces this diagnostic when a single class is specified more
than once in an `implements` clause.

#### Examples

The following code produces this diagnostic because `A` is in the list
twice:

{% prettify dart tag=pre+code %}
class A {}
class B implements A, [!A!] {}
{% endprettify %}

#### Common fixes

Remove all except one occurrence of the class name:

{% prettify dart tag=pre+code %}
class A {}
class B implements A {}
{% endprettify %}

### implements_super_class

_'{0}' can't be used in both the 'extends' and 'implements' clauses._

#### Description

The analyzer produces this diagnostic when one class is listed in both the
`extends` and `implements` clauses of another class.

#### Example

The following code produces this diagnostic because the class `A` is used
in both the `extends` and `implements` clauses for the class `B`:

{% prettify dart tag=pre+code %}
class A {}

class B extends A implements [!A!] {}
{% endprettify %}

#### Common fixes

If you want to inherit the implementation from the class, then remove the
class from the `implements` clause:

{% prettify dart tag=pre+code %}
class A {}

class B extends A {}
{% endprettify %}

If you don't want to inherit the implementation from the class, then remove
the `extends` clause:

{% prettify dart tag=pre+code %}
class A {}

class B implements A {}
{% endprettify %}

### implicit_this_reference_in_initializer

_The instance member '{0}' can't be accessed in an initializer._

#### Description

The analyzer produces this diagnostic when it finds a reference to an
instance member in a constructor's initializer list.

#### Examples

The following code produces this diagnostic because `defaultX` is an
instance member:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C() : x = [!defaultX!];

  int get defaultX => 0;
}
{% endprettify %}

#### Common fixes

If the member can be made static, then do so:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C() : x = defaultX;

  static int get defaultX => 0;
}
{% endprettify %}

If not, then replace the reference in the initializer with a different
expression that doesn't use an instance member:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C() : x = 0;

  int get defaultX => 0;
}
{% endprettify %}

### import_internal_library

_The library '{0}' is internal and can't be imported._

#### Description

The analyzer produces this diagnostic when it finds an import whose `dart:`
URI references an internal library.

#### Example

The following code produces this diagnostic because `_interceptors` is an
internal library:

{% prettify dart tag=pre+code %}
import [!'dart:_interceptors'!];
{% endprettify %}

#### Common fixes

Remove the import directive.

### inconsistent_inheritance

_Superinterfaces don't have a valid override for '{0}': {1}._

#### Description

The analyzer produces this diagnostic when a class inherits two or more
conflicting signatures for a member and doesn't provide an implementation
that satisfies all the inherited signatures.

#### Example

The following code produces this diagnostic because `C` is inheriting the
declaration of `m` from `A`, and that implementation isn't consistent with
the signature of `m` that's inherited from `B`:

{% prettify dart tag=pre+code %}
class A {
  void m({int a}) {}
}

class B {
  void m({int b}) {}
}

class [!C!] extends A implements B {
}
{% endprettify %}

#### Common fixes

Add an implementation of the method that satisfies all the inherited
signatures:

{% prettify dart tag=pre+code %}
class A {
  void m({int a}) {}
}

class B {
  void m({int b}) {}
}

class C extends A implements B {
  void m({int a, int b}) {}
}
{% endprettify %}

### initializer_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when a constructor initializes a
field that isn't declared in the class containing the constructor.
Constructors can't initialize fields that aren't declared and fields that
are inherited from superclasses.

#### Examples

The following code produces this diagnostic because the initializer is
initializing `x`, but `x` isn't a field in the class:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C() : [!x = 0!];
}
{% endprettify %}

#### Common fixes

If a different field should be initialized, then change the name to the
name of the field:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C() : y = 0;
}
{% endprettify %}

If the field must be declared, then add a declaration:

{% prettify dart tag=pre+code %}
class C {
  int x;
  int y;

  C() : x = 0;
}
{% endprettify %}

### initializing_formal_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when a field formal parameter is
found in a constructor in a class that doesn't declare the field being
initialized. Constructors can't initialize fields that aren't declared and
fields that are inherited from superclasses.

#### Examples

The following code produces this diagnostic because the field `x` isn't
defined:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C([!this.x!]);
}
{% endprettify %}

#### Common fixes

If the field name was wrong, then change it to the name of an existing
field:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C(this.y);
}
{% endprettify %}

If the field name is correct but hasn't yet been defined, then declare the
field:

{% prettify dart tag=pre+code %}
class C {
  int x;
  int y;

  C(this.x);
}
{% endprettify %}

If the parameter is needed but shouldn't initialize a field, then convert
it to a normal parameter and use it:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C(int x) : y = x * 2;
}
{% endprettify %}

If the parameter isn't needed, then remove it:

{% prettify dart tag=pre+code %}
class C {
  int y;

  C();
}
{% endprettify %}

### instance_access_to_static_member

_Static {1} '{0}' can't be accessed through an instance._

#### Description

The analyzer produces this diagnostic when an access operator is used to
access a static member through an instance of the class.

#### Examples

The following code produces this diagnostic because `zero` is a static
field, but it’s being accessed as if it were an instance field:

{% prettify dart tag=pre+code %}
void f(C c) {
  c.[!zero!];
}

class C {
  static int zero = 0;
}
{% endprettify %}

#### Common fixes

Use the class to access the static member:

{% prettify dart tag=pre+code %}
void f(C c) {
  C.zero;
}

class C {
  static int zero = 0;
}
{% endprettify %}

### instance_member_access_from_factory

_Instance members can't be accessed from a factory constructor._

#### Description

The analyzer produces this diagnostic when a factory constructor contains
an unqualified reference to an instance member. In a generative
constructor, the instance of the class is created and initialized before
the body of the constructor is executed, so the instance can be bound to
`this` and accessed just like it would be in an instance method. But, in a
factory constructor, the instance isn't created before executing the body,
so `this` can't be used to reference it.

#### Examples

The following code produces this diagnostic because `x` isn't in scope in
the factory constructor:

{% prettify dart tag=pre+code %}
class C {
  int x;
  factory C() {
    return C._([!x!]);
  }
  C._(this.x);
}
{% endprettify %}

#### Common fixes

Rewrite the code so that it doesn't reference the instance member:

{% prettify dart tag=pre+code %}
class C {
  int x;
  factory C() {
    return C._(0);
  }
  C._(this.x);
}
{% endprettify %}

### instance_member_access_from_static

_Instance members can't be accessed from a static method._

#### Description

The analyzer produces this diagnostic when a static method contains an
unqualified reference to an instance member.

#### Examples

The following code produces this diagnostic because the instance field `x`
is being referenced in a static method:

{% prettify dart tag=pre+code %}
class C {
  int x;

  static int m() {
    return [!x!];
  }
}
{% endprettify %}

#### Common fixes

If the method must reference the instance member, then it can't be static,
so remove the keyword:

{% prettify dart tag=pre+code %}
class C {
  int x;

  int m() {
    return x;
  }
}
{% endprettify %}

If the method can't be made an instance method, then add a parameter so
that an instance of the class can be passed in:

{% prettify dart tag=pre+code %}
class C {
  int x;

  static int m(C c) {
    return c.x;
  }
}
{% endprettify %}

### instantiate_abstract_class

_Abstract classes can't be instantiated._

#### Description

The analyzer produces this diagnostic when it finds a constructor
invocation and the constructor is declared in an abstract class. Even
though you can't create an instance of an abstract class, abstract classes
can declare constructors that can be invoked by subclasses.

#### Examples

The following code produces this diagnostic because `C` is an abstract
class:

{% prettify dart tag=pre+code %}
abstract class C {}

var c = new [!C!]();
{% endprettify %}

#### Common fixes

If there's a concrete subclass of the abstract class that can be used, then
create an instance of the concrete subclass.

### integer_literal_out_of_range

_The integer literal {0} can't be represented in 64 bits._

#### Description

The analyzer produces this diagnostic when an integer literal has a value
that is too large (positive) or too small (negative) to be represented in a
64-bit word.

#### Example

The following code produces this diagnostic because the value can't be
represented in 64 bits:

{% prettify dart tag=pre+code %}
var x = [!9223372036854775810!];
{% endprettify %}

#### Common fixes

If you need to represent the current value, then wrap it in an instance of
the class `BigInt`:

{% prettify dart tag=pre+code %}
var x = BigInt.parse('9223372036854775810');
{% endprettify %}

### invalid_annotation

_Annotation must be either a const variable reference or const constructor
invocation._

#### Description

The analyzer produces this diagnostic when an annotation is found that is
using something that is neither a variable marked as `const` or the
invocation of a `const` constructor.

Getters can't be used as annotations.

#### Example

The following code produces this diagnostic because the variable `v` isn't
a `const` variable:

{% prettify dart tag=pre+code %}
var v = 0;

[!@v!]
void f() {
}
{% endprettify %}

The following code produces this diagnostic because `f` isn't a variable:

{% prettify dart tag=pre+code %}
[!@f!]
void f() {
}
{% endprettify %}

The following code produces this diagnostic because `f` isn't a
constructor:

{% prettify dart tag=pre+code %}
[!@f()!]
void f() {
}
{% endprettify %}

The following code produces this diagnostic because `g` is a getter:

{% prettify dart tag=pre+code %}
[!@g!]
int get g => 0;
{% endprettify %}

#### Common fixes

If the annotation is referencing a variable that isn't a `const`
constructor, add the keyword `const` to the variable's declaration:

{% prettify dart tag=pre+code %}
const v = 0;

@v
void f() {
}
{% endprettify %}

If the annotation isn't referencing a variable, then remove it:

{% prettify dart tag=pre+code %}
int v = 0;

void f() {
}
{% endprettify %}

### invalid_assignment

_A value of type '{0}' can't be assigned to a variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the static type of an expression
that is assigned to a variable isn't assignable to the type of the
variable.

#### Examples

The following code produces this diagnostic because the type of the
initializer (`int`) isn't assignable to the type of the variable
(`String`):

{% prettify dart tag=pre+code %}
int i = 0;
String s = [!i!];
{% endprettify %}

#### Common fixes

If the value being assigned is always assignable at runtime, even though
the static types don't reflect that, then add an explicit cast.

Otherwise, change the value being assigned so that it has the expected
type. In the previous example, this might look like:

{% prettify dart tag=pre+code %}
int i = 0;
String s = i.toString();
{% endprettify %}

If you can’t change the value, then change the type of the variable to be
compatible with the type of the value being assigned:

{% prettify dart tag=pre+code %}
int i = 0;
int s = i;
{% endprettify %}

### invalid_extension_argument_count

_Extension overrides must have exactly one argument: the value of 'this' in the
extension method._

#### Description

The analyzer produces this diagnostic when an extension override doesn't
have exactly one argument. The argument is the expression used to compute
the value of `this` within the extension method, so there must be one
argument.

#### Examples

The following code produces this diagnostic because there are no arguments:

{% prettify dart tag=pre+code %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!()!].join('b');
}
{% endprettify %}

And, the following code produces this diagnostic because there's more than
one argument:

{% prettify dart tag=pre+code %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!('a', 'b')!].join('c');
}
{% endprettify %}

#### Common fixes

Provide one argument for the extension override:

{% prettify dart tag=pre+code %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E('a').join('b');
}
{% endprettify %}

### invalid_factory_name_not_a_class

_The name of a factory constructor must be the same as the name of the
immediately enclosing class._

#### Description

The analyzer produces this diagnostic when the name of a factory
constructor isn't the same as the name of the surrounding class.

#### Examples

The following code produces this diagnostic because the name of the factory
constructor (`A`) isn't the same as the surrounding class (`C`):

{% prettify dart tag=pre+code %}
class A {}

class C {
  factory [!A!]() => throw 0;
}
{% endprettify %}

#### Common fixes

If the factory returns an instance of the surrounding class, then rename
the factory:

{% prettify dart tag=pre+code %}
class A {}

class C {
  factory C() => throw 0;
}
{% endprettify %}

If the factory returns an instance of a different class, then move the
factory to that class:

{% prettify dart tag=pre+code %}
class A {
  factory A() => throw 0;
}

class C {}
{% endprettify %}

If the factory returns an instance of a different class, but you can't
modify that class or don't want to move the factory, then convert it to be
a static method:

{% prettify dart tag=pre+code %}
class A {}

class C {
  static A a() => throw 0;
}
{% endprettify %}

### invalid_inline_function_type

_Inline function types can't be used for parameters in a generic function type._

#### Description

The analyzer produces this diagnostic when a generic function type has a
function-valued parameter that is written using the older inline function
type syntax.

#### Example

The following code produces this diagnostic because the parameter `f`, in
the generic function type used to define `F`, uses the inline function
type syntax:

{% prettify dart tag=pre+code %}
typedef F = int Function(int f[!(!]String s));
{% endprettify %}

#### Common fixes

Use the generic function syntax for the parameter's type:

{% prettify dart tag=pre+code %}
typedef F = int Function(int Function(String));
{% endprettify %}

### invalid_literal_annotation

_Only const constructors can have the `@literal` annotation._

#### Description

The analyzer produces this diagnostic when the `@literal` annotation is
applied to anything other than a const constructor.

#### Examples

The following code produces this diagnostic because the constructor isn't
a `const` constructor:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class C {
  [!@literal!]
  C();
}
{% endprettify %}

The following code produces this diagnostic because `x` isn't a
constructor:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

[!@literal!]
var x;
{% endprettify %}

#### Common fixes

If the annotation is on a constructor and the constructor should always be
invoked with `const`, when possible, then mark the constructor with the
`const` keyword:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}
{% endprettify %}

If the constructor can't be marked as `const`, then remove the annotation.

If the annotation is on anything other than a constructor, then remove the
annotation:

{% prettify dart tag=pre+code %}
var x;
{% endprettify %}

### invalid_null_aware_operator

_The receiver can't be null because of short-circuiting, so the null-aware
operator '{0}' can't be used._

_The receiver can't be null, so the null-aware operator '{0}' is unnecessary._

#### Description

The analyzer produces this diagnostic when a null-aware operator (`?.`,
`?..`, `?[`, `?..[`, or `...?`) is used on a target that's known to be
non-nullable.

#### Example

The following code produces this diagnostic because `s` can't be `null`:

{% prettify dart tag=pre+code %}
int? getLength(String s) {
  return s[!?.!]length;
}
{% endprettify %}

The following code produces this diagnostic because `a` can't be `null`:

{% prettify dart tag=pre+code %}
var a = [];
var b = [[!...?!]a];
{% endprettify %}

The following code produces this diagnostic because `s?.length` can't
return `null`:

{% prettify dart tag=pre+code %}
void f(String? s) {
  s?.length[!?.!]isEven;
}
{% endprettify %}

The reason `s?.length` can't return `null` is because the null-aware
operator following `s` short-circuits the evaluation of both `length` and
`isEven` if `s` is `null`. In other words, if `s` is `null`, then neither
`length` nor `isEven` will be invoked, and if `s` is non-`null`, then
`length` can't return a `null` value. Either way, `isEven` can't be invoked
on a `null` value, so the null-aware operator is not necessary. See
[Understanding null safety](/null-safety/understanding-null-safety#smarter-null-aware-methods)
for more details.

The following code produces this diagnostic because `s` can't be `null`.

{% prettify dart tag=pre+code %}
void f(Object? o) {
  var s = o as String;
  s[!?.!]length;
}
{% endprettify %}

The reason `s` can't be null, despite the fact that `o` can be `null`, is
because of the cast to `String`, which is a non-nullable type. If `o` ever
has the value `null`, the cast will fail and the invocation of `length`
will not happen.

#### Common fixes

Replace the null-aware operator with a non-null-aware equivalent; for
example, change `?.` to  `.`:

{% prettify dart tag=pre+code %}
int getLength(String s) {
  return s.length;
}
{% endprettify %}

(Note that the return type was also changed to be non-nullable, which might
not be appropriate in some cases.)

### invalid_override

_'{1}.{0}' ('{2}') isn't a valid override of '{3}.{0}' ('{4}')._

#### Description

The analyzer produces this diagnostic when a member of a class is found
that overrides a member from a supertype and the override isn't valid. An
override is valid if all of these are true:
* It allows all of the arguments allowed by the overridden member.
* It doesn't require any arguments that aren't required by the overridden
  member.
* The type of every parameter of the overridden member is assignable to the
  corresponding parameter of the override.
* The return type of the override is assignable to the return type of the
  overridden member.

#### Examples

The following code produces this diagnostic because the type of the
parameter `s` (`String`) isn't assignable to the type of the parameter `i`
(`int`):

{% prettify dart tag=pre+code %}
class A {
  void m(int i) {}
}

class B extends A {
  void [!m!](String s) {}
}
{% endprettify %}

#### Common fixes

If the invalid method is intended to override the method from the
superclass, then change it to conform:

{% prettify dart tag=pre+code %}
class A {
  void m(int i) {}
}

class B extends A {
  void m(int i) {}
}
{% endprettify %}

If it isn't intended to override the method from the superclass, then
rename it:

{% prettify dart tag=pre+code %}
class A {
  void m(int i) {}
}

class B extends A {
  void m2(String s) {}
}
{% endprettify %}

### invalid_reference_to_this

_Invalid reference to 'this' expression._

#### Description

The analyzer produces this diagnostic when `this` is used outside of an
instance method or a generative constructor. The reserved word `this` is
only defined in the context of an instance method or a generative
constructor.

#### Examples

The following code produces this diagnostic because `v` is a top-level
variable:

{% prettify dart tag=pre+code %}
C f() => [!this!];

class C {}
{% endprettify %}

#### Common fixes

Use a variable of the appropriate type in place of `this`, declaring it if
necessary:

{% prettify dart tag=pre+code %}
C f(C c) => c;

class C {}
{% endprettify %}

### invalid_super_invocation

_The superclass call must be last in an initializer list: '{0}'._

#### Description

The analyzer produces this diagnostic when the initializer list of a
constructor contains an invocation of a constructor in the superclass, but
the invocation isn't the last item in the initializer list.

#### Example

The following code produces this diagnostic because the invocation of the
superclass' constructor isn't the last item in the initializer list:

{% prettify dart tag=pre+code %}
class A {
  A(int x);
}

class B extends A {
  B(int x) : [!super!](x), assert(x >= 0);
}
{% endprettify %}

#### Common fixes

Move the invocation of the superclass' constructor to the end of the
initializer list:

{% prettify dart tag=pre+code %}
class A {
  A(int x);
}

class B extends A {
  B(int x) : assert(x >= 0), super(x);
}
{% endprettify %}

### invalid_type_argument_in_const_literal

_Constant list literals can't include a type parameter as a type argument, such
as '{0}'._

_Constant map literals can't include a type parameter as a type argument, such
as '{0}'._

_Constant set literals can't include a type parameter as a type argument, such
as '{0}'._

#### Description

The analyzer produces this diagnostic when a type parameter is used as a
type argument in a list, map, or set literal that is prefixed by `const`.
This isn't allowed because the value of the type parameter (the actual type
that will be used at runtime) can't be known at compile time.

#### Example

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant list:

{% prettify dart tag=pre+code %}
List<T> newList<T>() => const <[!T!]>[];
{% endprettify %}

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant map:

{% prettify dart tag=pre+code %}
Map<String, T> newSet<T>() => const <String, [!T!]>{};
{% endprettify %}

The following code produces this diagnostic because the type parameter `T`
is being used as a type argument when creating a constant set:

{% prettify dart tag=pre+code %}
Set<T> newSet<T>() => const <[!T!]>{};
{% endprettify %}

#### Common fixes

If the type that will be used for the type parameter can be known at
compile time, then remove the type parameter:

{% prettify dart tag=pre+code %}
List<int> newList() => const <int>[];
{% endprettify %}

If the type that will be used for the type parameter can't be known until
runtime, then remove the keyword `const`:

{% prettify dart tag=pre+code %}
List<T> newList<T>() => <T>[];
{% endprettify %}

### invalid_uri

_Invalid URI syntax: '{0}'._

#### Description

The analyzer produces this diagnostic when a URI in a directive doesn't
conform to the syntax of a valid URI.

#### Examples

The following code produces this diagnostic because `'#'` isn't a valid
URI:

{% prettify dart tag=pre+code %}
import [!'#'!];
{% endprettify %}

#### Common fixes

Replace the invalid URI with a valid URI.

### invalid_use_of_covariant_in_extension

_Can't have modifier '#lexeme' in an extension._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the keyword `covariant` in the declaration of a parameter.
Extensions aren't classes and don't have subclasses, so the keyword serves
no purpose.

#### Examples

The following code produces this diagnostic because `i` is marked as being
covariant:

{% prettify dart tag=pre+code %}
extension E on String {
  void a([!covariant!] int i) {}
}
{% endprettify %}

#### Common fixes

Remove the `covariant` keyword:

{% prettify dart tag=pre+code %}
extension E on String {
  void a(int i) {}
}
{% endprettify %}

### invalid_use_of_null_value

_An expression whose value is always 'null' can't be dereferenced._

#### Description

The analyzer produces this diagnostic when an expression whose value will
always be `null` is dererenced.

#### Example

The following code produces this diagnostic because `x` will always be
`null`:

{% prettify dart tag=pre+code %}
int f(Null x) {
  return [!x!].length;
}
{% endprettify %}

#### Common fixes

If the value is allowed to be something other than `null`, then change the
type of the expression:

{% prettify dart tag=pre+code %}
int f(String? x) {
  return x!.length;
}
{% endprettify %}

### invalid_visibility_annotation

_The member '{0}' is annotated with '{1}', but this annotation is only
meaningful on declarations of public members._

#### Description

The analyzer produces this diagnostic when either the `@visibleForTemplate`
or `@visibleForTesting` annotation is applied to a non-public declaration.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

[!@visibleForTesting!]
void _someFunction() {}

void f() => _someFunction();
{% endprettify %}

#### Common fixes

If the declaration doesn't need to be used by test code, then remove the
annotation:

{% prettify dart tag=pre+code %}
void _someFunction() {}

void f() => _someFunction();
{% endprettify %}

If it does, then make it public:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

@visibleForTesting
void someFunction() {}

void f() => someFunction();
{% endprettify %}

### invocation_of_extension_without_call

_The extension '{0}' doesn't define a 'call' method so the override can't be
used in an invocation._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a function but the extension doesn't declare a `call` method.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't define a `call` method:

{% prettify dart tag=pre+code %}
extension E on String {}

void f() {
  [!E('')!]();
}
{% endprettify %}

#### Common fixes

If the extension is intended to define a `call` method, then declare it:

{% prettify dart tag=pre+code %}
extension E on String {
  int call() => 0;
}

void f() {
  E('')();
}
{% endprettify %}

If the extended type defines a `call` method, then remove the extension
override.

If the `call` method isn't defined, then rewrite the code so that it
doesn't invoke the `call` method.

### invocation_of_non_function

_'{0}' isn't a function._

#### Description

The analyzer produces this diagnostic when it finds a function invocation,
but the name of the function being invoked is defined to be something other
than a function.

#### Examples

The following code produces this diagnostic because `Binary` is the name of
a function type, not a function:

{% prettify dart tag=pre+code %}
typedef Binary = int Function(int, int);

int f() {
  return [!Binary!](1, 2);
}
{% endprettify %}

#### Common fixes

Replace the name with the name of a function.

### invocation_of_non_function_expression

_The expression doesn't evaluate to a function, so it can't be invoked._

#### Description

The analyzer produces this diagnostic when a function invocation is found,
but the name being referenced isn't the name of a function, or when the
expression computing the function doesn't compute a function.

#### Examples

The following code produces this diagnostic because `x` isn't a function:

{% prettify dart tag=pre+code %}
int x = 0;

int f() => x;

var y = [!x!]();
{% endprettify %}

The following code produces this diagnostic because `f()` doesn't return a
function:

{% prettify dart tag=pre+code %}
int x = 0;

int f() => x;

var y = [!f()!]();
{% endprettify %}

#### Common fixes

If you need to invoke a function, then replace the code before the argument
list with the name of a function or with an expression that computes a
function:

{% prettify dart tag=pre+code %}
int x = 0;

int f() => x;

var y = f();
{% endprettify %}

### label_in_outer_scope

_Can't reference label '{0}' declared in an outer method._

#### Description

The analyzer produces this diagnostic when a `break` or `continue`
statement references a label that is declared in a method or function
containing the function in which the `break` or `continue` statement
appears. The `break` and `continue` statements can't be used to transfer
control outside the function that contains them.

#### Example

The following code produces this diagnostic because the label `loop` is
declared outside the local function `g`:

{% prettify dart tag=pre+code %}
void f() {
  loop:
  while (true) {
    void g() {
      break [!loop!];
    }

    g();
  }
}
{% endprettify %}

#### Common fixes

Try rewriting the code so that it isn't necessary to transfer control
outside the local function, possibly by inlining the local function:

{% prettify dart tag=pre+code %}
void f() {
  loop:
  while (true) {
    break loop;
  }
}
{% endprettify %}

If that isn't possible, then try rewriting the local function so that a
value returned by the function can be used to determine whether control is
transferred:

{% prettify dart tag=pre+code %}
void f() {
  loop:
  while (true) {
    bool g() {
      return true;
    }

    if (g()) {
      break loop;
    }
  }
}
{% endprettify %}

### label_undefined

_Can't reference an undefined label '{0}'._

#### Description

The analyzer produces this diagnostic when it finds a reference to a label
that isn't defined in the scope of the `break` or `continue` statement that
is referencing it.

#### Example

The following code produces this diagnostic because the label `loop` isn't
defined anywhere:

{% prettify dart tag=pre+code %}
void f() {
  for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      break [!loop!];
    }
  }
}
{% endprettify %}

#### Common fixes

If the label should be on the innermost enclosing `do`, `for`, `switch`, or
`while` statement, then remove the label:

{% prettify dart tag=pre+code %}
void f() {
  for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      break;
    }
  }
}
{% endprettify %}

If the label should be on some other statement, then add the label:

{% prettify dart tag=pre+code %}
void f() {
  loop: for (int i = 0; i < 10; i++) {
    for (int j = 0; j < 10; j++) {
      break loop;
    }
  }
}
{% endprettify %}

### late_final_field_with_const_constructor

_Can't have a late final field in a class with a const constructor._

#### Description

The analyzer produces this diagnostic when a class that has at least one
`const` constructor also has a field marked both `late` and `final`.

#### Example

The following code produces this diagnostic because the class `A` has a
`const` constructor and the `final` field `f` is marked as `late`:

{% prettify dart tag=pre+code %}
class A {
  [!late!] final int f;

  const A();
}
{% endprettify %}

#### Common fixes

If the field doesn't need to be marked `late`, then remove the `late`
modifier from the field:

{% prettify dart tag=pre+code %}
class A {
  final int f = 0;

  const A();
}
{% endprettify %}

If the field must be marked `late`, then remove the `const` modifier from
the constructors:

{% prettify dart tag=pre+code %}
class A {
  late final int f;

  A();
}
{% endprettify %}

### late_final_local_already_assigned

_The late final local variable is already assigned._

#### Description

The analyzer produces this diagnostic when the analyzer can prove that a
local variable marked as both `late` and `final` was already assigned a
value at the point where another assignment occurs.

Because `final` variables can only be assigned once, subsequent assignments
are guaranteed to fail, so they're flagged.

#### Example

The following code produces this diagnostic because the `final` variable
`v` is assigned a value in two places:

{% prettify dart tag=pre+code %}
int f() {
  late final int v;
  v = 0;
  [!v!] += 1;
  return v;
}
{% endprettify %}

#### Common fixes

If you need to be able to reassign the variable, then remove the `final`
keyword:

{% prettify dart tag=pre+code %}
int f() {
  late int v;
  v = 0;
  v += 1;
  return v;
}
{% endprettify %}

If you don't need to reassign the variable, then remove all except the
first of the assignments:

{% prettify dart tag=pre+code %}
int f() {
  late final int v;
  v = 0;
  return v;
}
{% endprettify %}

### list_element_type_not_assignable

_The element type '{0}' can't be assigned to the list type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of an element in a list
literal isn't assignable to the element type of the list.

#### Examples

The following code produces this diagnostic because `2.5` is a double, and
the list can hold only integers:

{% prettify dart tag=pre+code %}
List<int> x = [1, [!2.5!], 3];
{% endprettify %}

#### Common fixes

If you intended to add a different object to the list, then replace the
element with an expression that computes the intended object:

{% prettify dart tag=pre+code %}
List<int> x = [1, 2, 3];
{% endprettify %}

If the object shouldn't be in the list, then remove the element:

{% prettify dart tag=pre+code %}
List<int> x = [1, 3];
{% endprettify %}

If the object being computed is correct, then widen the element type of the
list to allow all of the different types of objects it needs to contain:

{% prettify dart tag=pre+code %}
List<num> x = [1, 2.5, 3];
{% endprettify %}

### main_first_positional_parameter_type

_The type of the first positional parameter of the 'main' function must be a
supertype of 'List<String>'._

#### Description

The analyzer produces this diagnostic when the first positional parameter
of a function named `main` isn't a supertype of `List<String>`.

#### Example

The following code produces this diagnostic because `List<int>` isn't a
supertype of `List<String>`:

{% prettify dart tag=pre+code %}
void main([!List<int>!] args) {}
{% endprettify %}

#### Common fixes

If the function is an entry point, then change the type of the first
positional parameter to be a supertype of `List<String>`:

{% prettify dart tag=pre+code %}
void main(List<String> args) {}
{% endprettify %}

If the function isn't an entry point, then change the name of the function:

{% prettify dart tag=pre+code %}
void f(List<int> args) {}
{% endprettify %}

### main_has_required_named_parameters

_The function 'main' can't have any required named parameters._

#### Description

The analyzer produces this diagnostic when a function named `main` has one
or more required named parameters.

#### Example

The following code produces this diagnostic because the function named
`main` has a required named parameter (`x`):

{% prettify dart tag=pre+code %}
void [!main!]({required int x}) {}
{% endprettify %}

#### Common fixes

If the function is an entry point, then remove the `required` keyword:

{% prettify dart tag=pre+code %}
void main({int? x}) {}
{% endprettify %}

If the function isn't an entry point, then change the name of the function:

{% prettify dart tag=pre+code %}
void f({required int x}) {}
{% endprettify %}

### main_has_too_many_required_positional_parameters

_The function 'main' can't have more than two required positional parameters._

#### Description

The analyzer produces this diagnostic when a function named `main` has more
than two required positional parameters.

#### Example

The following code produces this diagnostic because the function `main` has
three required positional parameters:

{% prettify dart tag=pre+code %}
void [!main!](List<String> args, int x, int y) {}
{% endprettify %}

#### Common fixes

If the function is an entry point and the extra parameters aren't used,
then remove them:

{% prettify dart tag=pre+code %}
void main(List<String> args, int x) {}
{% endprettify %}

If the function is an entry point, but the extra parameters used are for
when the function isn't being used as an entry point, then make the extra
parameters optional:

{% prettify dart tag=pre+code %}
void main(List<String> args, int x, [int y = 0]) {}
{% endprettify %}

If the function isn't an entry point, then change the name of the function:

{% prettify dart tag=pre+code %}
void f(List<String> args, int x, int y) {}
{% endprettify %}

### main_is_not_function

_The declaration named 'main' must be a function._

#### Description

The analyzer produces this diagnostic when a library contains a declaration
of the name `main` that isn't the declaration of a top-level function.

#### Example

The following code produces this diagnostic because the name `main` is
being used to declare a top-level variable:

{% prettify dart tag=pre+code %}
var [!main!] = 3;
{% endprettify %}

#### Common fixes

Use a different name for the declaration:

{% prettify dart tag=pre+code %}
var mainIndex = 3;
{% endprettify %}

### map_entry_not_in_map

_Map entries can only be used in a map literal._

#### Description

The analyzer produces this diagnostic when a map entry (a key/value pair)
is found in a set literal.

#### Examples

The following code produces this diagnostic because the literal has a map
entry even though it's a set literal:

{% prettify dart tag=pre+code %}
const collection = <String>{[!'a' : 'b'!]};
{% endprettify %}

#### Common fixes

If you intended for the collection to be a map, then change the code so
that it is a map. In the previous example, you could do this by adding
another type argument:

{% prettify dart tag=pre+code %}
const collection = <String, String>{'a' : 'b'};
{% endprettify %}

In other cases, you might need to change the explicit type from `Set` to
`Map`.

If you intended for the collection to be a set, then remove the map entry,
possibly by replacing the colon with a comma if both values should be
included in the set:

{% prettify dart tag=pre+code %}
const collection = <String>{'a', 'b'};
{% endprettify %}

### map_key_type_not_assignable

_The element type '{0}' can't be assigned to the map key type '{1}'._

#### Description

The analyzer produces this diagnostic when a key of a key-value pair in a
map literal has a type that isn't assignable to the key type of the map.

#### Examples

The following code produces this diagnostic because `2` is an `int`, but
the keys of the map are required to be `String`s:

{% prettify dart tag=pre+code %}
var m = <String, String>{[!2!] : 'a'};
{% endprettify %}

#### Common fixes

If the type of the map is correct, then change the key to have the correct
type:

{% prettify dart tag=pre+code %}
var m = <String, String>{'2' : 'a'};
{% endprettify %}

If the type of the key is correct, then change the key type of the map:

{% prettify dart tag=pre+code %}
var m = <int, String>{2 : 'a'};
{% endprettify %}

### map_value_type_not_assignable

_The element type '{0}' can't be assigned to the map value type '{1}'._

#### Description

The analyzer produces this diagnostic when a value of a key-value pair in a
map literal has a type that isn't assignable to the the value type of the
map.

#### Examples

The following code produces this diagnostic because `2` is an `int`, but/
the values of the map are required to be `String`s:

{% prettify dart tag=pre+code %}
var m = <String, String>{'a' : [!2!]};
{% endprettify %}

#### Common fixes

If the type of the map is correct, then change the value to have the
correct type:

{% prettify dart tag=pre+code %}
var m = <String, String>{'a' : '2'};
{% endprettify %}

If the type of the value is correct, then change the value type of the map:

{% prettify dart tag=pre+code %}
var m = <String, int>{'a' : 2};
{% endprettify %}

### missing_default_value_for_parameter

_The parameter '{0}' can't have a value of 'null' because of its type, but the
implicit default value is 'null'._

#### Description

The analyzer produces this diagnostic when an optional parameter, whether
positional or named, has a [potentially non-nullable][] type and doesn't
specify a default value. Optional parameters that have no explicit default
value have an implicit default value of `null`. If the type of the
parameter doesn't allow the parameter to have a value of `null`, then the
implicit default value isn't valid.

#### Example

The following code produces this diagnostic because `x` can't be `null`,
and no non-`null` default value is specified:

{% prettify dart tag=pre+code %}
void f([int [!x!]]) {}
{% endprettify %}

As does this:

{% prettify dart tag=pre+code %}
void g({int [!x!]}) {}
{% endprettify %}

#### Common fixes

If you want to use `null` to indicate that no value was provided, then you
need to make the type nullable:

{% prettify dart tag=pre+code %}
void f([int? x]) {}
void g({int? x}) {}
{% endprettify %}

If the parameter can't be null, then either provide a default value:

{% prettify dart tag=pre+code %}
void f([int x = 1]) {}
void g({int x = 2}) {}
{% endprettify %}

or make the parameter a required parameter:

{% prettify dart tag=pre+code %}
void f(int x) {}
void g({required int x}) {}
{% endprettify %}

### missing_enum_constant_in_switch

_Missing case clause for '{0}'._

#### Description

The analyzer produces this diagnostic when a `switch` statement for an enum
doesn't include an option for one of the values in the enumeration.

Note that `null` is always a possible value for an enum and therefore also
must be handled.

#### Examples

The following code produces this diagnostic because the enum constant `e2`
isn't handled:

{% prettify dart tag=pre+code %}
enum E { e1, e2 }

void f(E e) {
  [!switch (e)!] {
    case E.e1:
      break;
  }
}
{% endprettify %}

#### Common fixes

If there's special handling for the missing values, then add a `case`
clause for each of the missing values:

{% prettify dart tag=pre+code %}
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    case E.e2:
      break;
  }
}
{% endprettify %}

If the missing values should be handled the same way, then add a `default`
clause:

{% prettify dart tag=pre+code %}
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    default:
      break;
  }
}
{% endprettify %}

### missing_required_argument

_The named parameter '{0}' is required, but there's no corresponding argument._

#### Description

The analyzer produces this diagnostic when an invocation of a function is
missing a required named parameter.

#### Example

The following code produces this diagnostic because the invocation of `f`
doesn't include a value for the required named parameter `end`:

{% prettify dart tag=pre+code %}
void f(int start, {required int end}) {}
void g() {
  [!f!](3);
}
{% endprettify %}

#### Common fixes

Add a named argument corresponding to the missing required parameter:

{% prettify dart tag=pre+code %}
void f(int start, {required int end}) {}
void g() {
  f(3, end: 5);
}
{% endprettify %}

### missing_required_param

_The parameter '{0}' is required._

_The parameter '{0}' is required. {1}._

#### Description

The analyzer produces this diagnostic when a method or function with a
named parameter that is annotated as being required is invoked without
providing a value for the parameter.

#### Examples

The following code produces this diagnostic because the named parameter `x`
is required:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

void f({@required int x}) {}

void g() {
  [!f!]();
}
{% endprettify %}

#### Common fixes

Provide the required value:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

void f({@required int x}) {}

void g() {
  f(x: 2);
}
{% endprettify %}

### missing_return

_This function has a return type of '{0}', but doesn't end with a return
statement._

#### Description

Any function or method that doesn't end with either an explicit return or a
throw implicitly returns `null`. This is rarely the desired behavior. The
analyzer produces this diagnostic when it finds an implicit return.

#### Examples

The following code produces this diagnostic because `f` doesn't end with a
return:

{% prettify dart tag=pre+code %}
int [!f!](int x) {
  if (x < 0) {
    return 0;
  }
}
{% endprettify %}

#### Common fixes

Add a `return` statement that makes the return value explicit, even if
`null` is the appropriate value.

### mixin_of_non_class

_Classes can only mix in mixins and classes._

#### Description

The analyzer produces this diagnostic when a name in a `with` clause is
defined to be something other than a mixin or a class.

#### Examples

The following code produces this diagnostic because `F` is defined to be a
function type:

{% prettify dart tag=pre+code %}
typedef F = int Function(String);

class C with [!F!] {}
{% endprettify %}

#### Common fixes

Remove the invalid name from the list, possibly replacing it with the name
of the intended mixin or class:

{% prettify dart tag=pre+code %}
typedef F = int Function(String);

class C {}
{% endprettify %}

### mixin_on_sealed_class

_The class '{0}' shouldn't be used as a mixin constraint because it is sealed,
and any class mixing in this mixin must have '{0}' as a superclass._

#### Description

The analyzer produces this diagnostic when the superclass constraint of a
mixin is a class from a different package that was marked as `@sealed`.
Classes that are sealed can't be extended, implemented, mixed in, or used
as a superclass constraint.

#### Examples

If the package `p` defines a sealed class:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

@sealed
class C {}
{% endprettify %}

Then, the following code, when in a package other than `p`, produces this
diagnostic:

{% prettify dart tag=pre+code %}
import 'package:p/p.dart';

[!mixin M on C {}!]
{% endprettify %}

#### Common fixes

If the classes that use the mixin don't need to be subclasses of the sealed
class, then consider adding a field and delegating to the wrapped instance
of the sealed class.

### mixin_super_class_constraint_non_interface

_Only classes and mixins can be used as superclass constraints._

#### Description

The analyzer produces this diagnostic when a type following the `on`
keyword in a mixin declaration is neither a class nor a mixin.

#### Examples

The following code produces this diagnostic because `F` is neither a class
nor a mixin:

{% prettify dart tag=pre+code %}
typedef F = void Function();

mixin M on [!F!] {}
{% endprettify %}

#### Common fixes

If the type was intended to be a class but was mistyped, then replace the
name.

Otherwise, remove the type from the `on` clause.

### must_be_immutable

_This class (or a class that this class inherits from) is marked as
'@immutable', but one or more of its instance fields aren't final: {0}_

#### Description

The analyzer produces this diagnostic when an immutable class defines one
or more instance fields that aren't final. A class is immutable if it's
marked as being immutable using the annotation `@immutable` or if it's a
subclass of an immutable class.

#### Examples

The following code produces this diagnostic because the field `x` isn't
final:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

@immutable
class [!C!] {
  int x;

  C(this.x);
}
{% endprettify %}

#### Common fixes

If instances of the class should be immutable, then add the keyword `final`
to all non-final field declarations:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

@immutable
class C {
  final int x;

  C(this.x);
}
{% endprettify %}

If the instances of the class should be mutable, then remove the
annotation, or choose a different superclass if the annotation is
inherited:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C(this.x);
}
{% endprettify %}

### must_call_super

_This method overrides a method annotated as '@mustCallSuper' in '{0}', but
doesn't invoke the overridden method._

#### Description

The analyzer produces this diagnostic when a method that overrides a method
that is annotated as `@mustCallSuper` doesn't invoke the overridden method
as required.

#### Examples

The following code produces this diagnostic because the method `m` in `B`
doesn't invoke the overridden method `m` in `A`:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class A {
  @mustCallSuper
  m() {}
}

class B extends A {
  @override
  [!m!]() {}
}
{% endprettify %}

#### Common fixes

Add an invocation of the overridden method in the overriding method:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class A {
  @mustCallSuper
  m() {}
}

class B extends A {
  @override
  m() {
    super.m();
  }
}
{% endprettify %}

### new_with_undefined_constructor_default

_The class '{0}' doesn't have a default constructor._

#### Description

The analyzer produces this diagnostic when an unnamed constructor is
invoked on a class that defines named constructors but the class doesn’t
have an unnamed constructor.

#### Examples

The following code produces this diagnostic because `A` doesn't define an
unnamed constructor:

{% prettify dart tag=pre+code %}
class A {
  A.a();
}

A f() => [!A!]();
{% endprettify %}

#### Common fixes

If one of the named constructors does what you need, then use it:

{% prettify dart tag=pre+code %}
class A {
  A.a();
}

A f() => A.a();
{% endprettify %}

If none of the named constructors does what you need, and you're able to
add an unnamed constructor, then add the constructor:

{% prettify dart tag=pre+code %}
class A {
  A();
  A.a();
}

A f() => A();
{% endprettify %}

### non_abstract_class_inherits_abstract_member

_Missing concrete implementation of '{0}'._

_Missing concrete implementations of '{0}' and '{1}'._

_Missing concrete implementations of '{0}', '{1}', '{2}', '{3}', and {4} more._

_Missing concrete implementations of '{0}', '{1}', '{2}', and '{3}'._

_Missing concrete implementations of '{0}', '{1}', and '{2}'._

#### Description

The analyzer produces this diagnostic when a concrete class inherits one or
more abstract members, and doesn't provide or inherit an implementation for
at least one of those abstract members.

#### Examples

The following code produces this diagnostic because the class `B` doesn't
have a concrete implementation of `m`:

{% prettify dart tag=pre+code %}
abstract class A {
  void m();
}

class [!B!] extends A {}
{% endprettify %}

#### Common fixes

If the subclass can provide a concrete implementation for some or all of
the abstract inherited members, then add the concrete implementations:

{% prettify dart tag=pre+code %}
abstract class A {
  void m();
}

class B extends A {
  void m() {}
}
{% endprettify %}

If there is a mixin that provides an implementation of the inherited
methods, then apply the mixin to the subclass:

{% prettify dart tag=pre+code %}
abstract class A {
  void m();
}

class B extends A with M {}

mixin M {
  void m() {}
}
{% endprettify %}

If the subclass can't provide a concrete implementation for all of the
abstract inherited members, then mark the subclass as being abstract:

{% prettify dart tag=pre+code %}
abstract class A {
  void m();
}

abstract class B extends A {}
{% endprettify %}

### non_bool_condition

_Conditions must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when a condition, such as an `if` or
`while` loop, doesn't have the static type `bool`.

#### Examples

The following code produces this diagnostic because `x` has the static type
`int`:

{% prettify dart tag=pre+code %}
void f(int x) {
  if ([!x!]) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

Change the condition so that it produces a Boolean value:

{% prettify dart tag=pre+code %}
void f(int x) {
  if (x == 0) {
    // ...
  }
}
{% endprettify %}

### non_bool_expression

_The expression in an assert must be of type 'bool'._

#### Description

The analyzer produces this diagnostic when the first expression in an
assert has a type other than `bool`.

#### Examples

The following code produces this diagnostic because the type of `p` is
`int`, but a `bool` is required:

{% prettify dart tag=pre+code %}
void f(int p) {
  assert([!p!]);
}
{% endprettify %}

#### Common fixes

Change the expression so that it has the type `bool`:

{% prettify dart tag=pre+code %}
void f(int p) {
  assert(p > 0);
}
{% endprettify %}

### non_bool_negation_expression

_A negation operand must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when the operand of the unary
negation operator (`!`) doesn't have the type `bool`.

#### Examples

The following code produces this diagnostic because `x` is an `int` when it
must be a `bool`:

{% prettify dart tag=pre+code %}
int x = 0;
bool y = ![!x!];
{% endprettify %}

#### Common fixes

Replace the operand with an expression that has the type `bool`:

{% prettify dart tag=pre+code %}
int x = 0;
bool y = !(x > 0);
{% endprettify %}

### non_bool_operand

_The operands of the operator '{0}' must be assignable to 'bool'._

#### Description

The analyzer produces this diagnostic when one of the operands of either
the `&&` or `||` operator doesn't have the type `bool`.

#### Examples

The following code produces this diagnostic because `a` isn't a Boolean
value:

{% prettify dart tag=pre+code %}
int a = 3;
bool b = [!a!] || a > 1;
{% endprettify %}

#### Common fixes

Change the operand to a Boolean value:

{% prettify dart tag=pre+code %}
int a = 3;
bool b = a == 0 || a > 1;
{% endprettify %}

### non_constant_annotation_constructor

_Annotation creation can only call a const constructor._

#### Description

The analyzer produces this diagnostic when an annotation is the invocation
of an existing constructor even though the invoked constructor isn't a
const constructor.

#### Example

The following code produces this diagnostic because the constructor for `C`
isn't a const constructor:

{% prettify dart tag=pre+code %}
[!@C()!]
void f() {
}

class C {
  C();
}
{% endprettify %}

#### Common fixes

If it's valid for the class to have a const constructor, then create a
const constructor that can be used for the annotation:

{% prettify dart tag=pre+code %}
@C()
void f() {
}

class C {
  const C();
}
{% endprettify %}

If it isn't valid for the class to have a const constructor, then either
remove the annotation or use a different class for the annotation.

### non_constant_case_expression

_Case expressions must be constant._

#### Description

The analyzer produces this diagnostic when the expression in a `case`
clause isn't a constant expression.

#### Examples

The following code produces this diagnostic because `j` isn't a constant:

{% prettify dart tag=pre+code %}
void f(int i, int j) {
  switch (i) {
    case [!j!]:
      // ...
      break;
  }
}
{% endprettify %}

#### Common fixes

Either make the expression a constant expression, or rewrite the `switch`
statement as a sequence of `if` statements:

{% prettify dart tag=pre+code %}
void f(int i, int j) {
  if (i == j) {
    // ...
  }
}
{% endprettify %}

### non_constant_default_value

_The default value of an optional parameter must be constant._

#### Description

The analyzer produces this diagnostic when an optional parameter, either
named or positional, has a default value that isn't a compile-time
constant.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
var defaultValue = 3;

void f([int value = [!defaultValue!]]) {}
{% endprettify %}

#### Common fixes

If the default value can be converted to be a constant, then convert it:

{% prettify dart tag=pre+code %}
const defaultValue = 3;

void f([int value = defaultValue]) {}
{% endprettify %}

If the default value needs to change over time, then apply the default
value inside the function:

{% prettify dart tag=pre+code %}
var defaultValue = 3;

void f([int value]) {
  value ??= defaultValue;
}
{% endprettify %}

### non_constant_list_element

_The values in a const list literal must be constants._

#### Description

The analyzer produces this diagnostic when an element in a constant list
literal isn't a constant value. The list literal can be constant either
explicitly (because it's prefixed by the `const` keyword) or implicitly
(because it appears in a [constant context][]).

#### Examples

The following code produces this diagnostic because `x` isn't a constant,
even though it appears in an implicitly constant list literal:

{% prettify dart tag=pre+code %}
var x = 2;
var y = const <int>[0, 1, [!x!]];
{% endprettify %}

#### Common fixes

If the list needs to be a constant list, then convert the element to be a
constant. In the example above, you might add the `const` keyword to the
declaration of `x`:

{% prettify dart tag=pre+code %}
const x = 2;
var y = const <int>[0, 1, x];
{% endprettify %}

If the expression can't be made a constant, then the list can't be a
constant either, so you must change the code so that the list isn't a
constant. In the example above this means removing the `const` keyword
before the list literal:

{% prettify dart tag=pre+code %}
var x = 2;
var y = <int>[0, 1, x];
{% endprettify %}

### non_constant_map_element

_The elements in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when an `if` element or a spread
element in a constant map isn't a constant element.

#### Examples

The following code produces this diagnostic because it's attempting to
spread a non-constant map:

{% prettify dart tag=pre+code %}
var notConst = <int, int>{};
var map = const <int, int>{...[!notConst!]};
{% endprettify %}

Similarly, the following code produces this diagnostic because the
condition in the `if` element isn't a constant expression:

{% prettify dart tag=pre+code %}
bool notConst = true;
var map = const <int, int>{if ([!notConst!]) 1 : 2};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the elements constants.
In the spread example, you might do that by making the collection being
spread a constant:

{% prettify dart tag=pre+code %}
const notConst = <int, int>{};
var map = const <int, int>{...notConst};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart tag=pre+code %}
bool notConst = true;
var map = <int, int>{if (notConst) 1 : 2};
{% endprettify %}

### non_constant_map_key

_The keys in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a key in a constant map literal
isn't a constant value.

#### Examples

The following code produces this diagnostic beause `a` isn't a constant:

{% prettify dart tag=pre+code %}
var a = 'a';
var m = const {[!a!]: 0};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

{% prettify dart tag=pre+code %}
const a = 'a';
var m = const {a: 0};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart tag=pre+code %}
var a = 'a';
var m = {a: 0};
{% endprettify %}

### non_constant_map_value

_The values in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a value in a constant map
literal isn't a constant value.

#### Examples

The following code produces this diagnostic because `a` isn't a constant:

{% prettify dart tag=pre+code %}
var a = 'a';
var m = const {0: [!a!]};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

{% prettify dart tag=pre+code %}
const a = 'a';
var m = const {0: a};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart tag=pre+code %}
var a = 'a';
var m = {0: a};
{% endprettify %}

### non_constant_set_element

_The values in a const set literal must be constants._

#### Description

The analyzer produces this diagnostic when a constant set literal contains
an element that isn't a compile-time constant.

#### Examples

The following code produces this diagnostic because `i` isn't a constant:

{% prettify dart tag=pre+code %}
var i = 0;

var s = const {[!i!]};
{% endprettify %}

#### Common fixes

If the element can be changed to be a constant, then change it:

{% prettify dart tag=pre+code %}
const i = 0;

var s = const {i};
{% endprettify %}

If the element can't be a constant, then remove the keyword `const`:

{% prettify dart tag=pre+code %}
var i = 0;

var s = {i};
{% endprettify %}

### non_const_call_to_literal_constructor

_This instance creation must be 'const', because the {0} constructor is marked
as '@literal'._

#### Description

The analyzer produces this diagnostic when a constructor that has the
`@literal` annotation is invoked without using the `const` keyword, but all
of the arguments to the constructor are constants. The annotation indicates
that the constructor should be used to create a constant value whenever
possible.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

C f() => [!C()!];
{% endprettify %}

#### Common fixes

Add the keyword `const` before the constructor invocation:

{% prettify dart tag=pre+code %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

void f() => const C();
{% endprettify %}

### non_sync_factory

_Factory bodies can't use 'async', 'async*', or 'sync*'._

#### Description

The analyzer produces this diagnostic when the body of a factory
constructor is marked with `async`, `async*`, or `sync*`. All constructors,
including factory constructors, are required to return an instance of the
class in which they're declared, not a `Future`, `Stream`, or `Iterator`.

#### Example

The following code produces this diagnostic because the body of the factory
constructor is marked with `async`:

{% prettify dart tag=pre+code %}
class C {
  factory C() [!async!] {
    return C._();
  }
  C._();
}
{% endprettify %}

#### Common fixes

If the member must be declared as a factory constructor, then remove the
keyword appearing before the body:

{% prettify dart tag=pre+code %}
class C {
  factory C() {
    return C._();
  }
  C._();
}
{% endprettify %}

If the member must return something other than an instance of the enclosing
class, then make the member a static method:

{% prettify dart tag=pre+code %}
class C {
  static Future<C> m() async {
    return C._();
  }
  C._();
}
{% endprettify %}

### non_type_as_type_argument

_The name '{0}' isn't a type so it can't be used as a type argument._

#### Description

The analyzer produces this diagnostic when an identifier that isn't a type
is used as a type argument.

#### Examples

The following code produces this diagnostic because `x` is a variable, not
a type:

{% prettify dart tag=pre+code %}
var x = 0;
List<[!x!]> xList = [];
{% endprettify %}

#### Common fixes

Change the type argument to be a type:

{% prettify dart tag=pre+code %}
var x = 0;
List<int> xList = [];
{% endprettify %}

### non_type_in_catch_clause

_The name '{0}' isn't a type and can't be used in an on-catch clause._

#### Description

The analyzer produces this diagnostic when the identifier following the
`on` in a `catch` clause is defined to be something other than a type.

#### Examples

The following code produces this diagnostic because `f` is a function, not
a type:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } on [!f!] {
    // ...
  }
}
{% endprettify %}

#### Common fixes

Change the name to the type of object that should be caught:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } on FormatException {
    // ...
  }
}
{% endprettify %}

### not_assigned_potentially_non_nullable_local_variable

_The non-nullable local variable '{0}' must be assigned before it can be used._

#### Description

The analyzer produces this diagnostic when a local variable is referenced
and has all these characteristics:
- Has a type that's [potentially non-nullable][].
- Doesn't have an initializer.
- Isn't marked as `late`.
- The analyzer can't prove that the local variable will be assigned before
  the reference based on the specification of [definite assignment.][]

#### Example

The following code produces this diagnostic because `x` can't have a value
of `null`, but is referenced before a value was assigned to it:

{% prettify dart tag=pre+code %}
String f() {
  int x;
  return [!x!].toString();
}
{% endprettify %}

The following code produces this diagnostic because the assignment to `x`
might not be executed, so it might have a value of `null`:

{% prettify dart tag=pre+code %}
int g(bool b) {
  int x;
  if (b) {
    x = 1;
  }
  return [!x!] * 2;
}
{% endprettify %}

The following code produces this diagnostic because the analyzer can't
prove, based on definite assignment analysis, that `x` won't be referenced
without having a value assigned to it:

{% prettify dart tag=pre+code %}
int h(bool b) {
  int x;
  if (b) {
    x = 1;
  }
  if (b) {
    return [!x!] * 2;
  }
  return 0;
}
{% endprettify %}

#### Common fixes

If `null` is a valid value, then make the variable nullable:

{% prettify dart tag=pre+code %}
String f() {
  int? x;
  return x!.toString();
}
{% endprettify %}

If `null` isn’t a valid value, and there's a reasonable default value, then
add an initializer:

{% prettify dart tag=pre+code %}
int g(bool b) {
  int x = 2;
  if (b) {
    x = 1;
  }
  return x * 2;
}
{% endprettify %}

Otherwise, ensure that a value was assigned on every possible code path
before the value is accessed:

{% prettify dart tag=pre+code %}
int g(bool b) {
  int x;
  if (b) {
    x = 1;
  } else {
    x = 2;
  }
  return x * 2;
}
{% endprettify %}

You can also mark the variable as `late`, which removes the diagnostic, but
if the variable isn't assigned a value before it's accessed, then it
results in an exception being thrown at runtime. This approach should only
be used if you're sure that the variable will always be assigned, even
though the analyzer can't prove it based on definite assignment analysis.

{% prettify dart tag=pre+code %}
int h(bool b) {
  late int x;
  if (b) {
    x = 1;
  }
  if (b) {
    return x * 2;
  }
  return 0;
}
{% endprettify %}

### not_a_type

_{0} isn't a type._

#### Description

The analyzer produces this diagnostic when a name is used as a type but
declared to be something other than a type.

#### Examples

The following code produces this diagnostic because `f` is a function:

{% prettify dart tag=pre+code %}
f() {}
g([!f!] v) {}
{% endprettify %}

#### Common fixes

Replace the name with the name of a type.

### not_enough_positional_arguments

_{0} positional argument(s) expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has fewer positional arguments than the number of required positional
parameters.

#### Examples

The following code produces this diagnostic because `f` declares two
required parameters, but only one argument is provided:

{% prettify dart tag=pre+code %}
void f(int a, int b) {}
void g() {
  f[!(0)!];
}
{% endprettify %}

#### Common fixes

Add arguments corresponding to the remaining parameters:

{% prettify dart tag=pre+code %}
void f(int a, int b) {}
void g() {
  f(0, 1);
}
{% endprettify %}

### not_initialized_non_nullable_instance_field

_Non-nullable instance field '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a field is declared and has all
these characteristics:
- Has a type that's [potentially non-nullable][]
- Doesn't have an initializer
- Isn't marked as `late`

#### Example

The following code produces this diagnostic because `x` is implicitly
initialized to `null` when it isn't allowed to be `null`:

{% prettify dart tag=pre+code %}
class C {
  int [!x!];
}
{% endprettify %}

Similarly, the following code produces this diagnostic because `x` is
implicitly initialized to `null`, when it isn't allowed to be `null`, by
one of the constructors, even though it's initialized by other
constructors:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C(this.x);

  [!C!].n();
}
{% endprettify %}

#### Common fixes

If there's a reasonable default value for the field that’s the same for all
instances, then add an initializer expression:

{% prettify dart tag=pre+code %}
class C {
  int x = 0;
}
{% endprettify %}

If the value of the field should be provided when an instance is created,
then add a constructor that sets the value of the field or update an
existing constructor:

{% prettify dart tag=pre+code %}
class C {
  int x;

  C(this.x);
}
{% endprettify %}

You can also mark the field as `late`, which removes the diagnostic, but if
the field isn't assigned a value before it's accessed, then it results in
an exception being thrown at runtime. This approach should only be used if
you're sure that the field will always be assigned before it's referenced.

{% prettify dart tag=pre+code %}
class C {
  late int x;
}
{% endprettify %}

### not_initialized_non_nullable_variable

_The non-nullable variable '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a static field or top-level
variable has a type that's non-nullable and doesn't have an initializer.
Fields and variables that don't have an initializer are normally
initialized to `null`, but the type of the field or variable doesn't allow
it to be set to `null`, so an explicit initializer must be provided.

#### Example

The following code produces this diagnostic because the field `f` can't be
initialized to `null`:

{% prettify dart tag=pre+code %}
class C {
  static int [!f!];
}
{% endprettify %}

Similarly, the following code produces this diagnostic because the
top-level variable `v` can't be initialized to `null`:

{% prettify dart tag=pre+code %}
int [!v!];
{% endprettify %}

#### Common fixes

If the field or variable can't be initialized to `null`, then add an
initializer that sets it to a non-null value:

{% prettify dart tag=pre+code %}
class C {
  static int f = 0;
}
{% endprettify %}

If the field or variable should be initialized to `null`, then change the
type to be nullable:

{% prettify dart tag=pre+code %}
int? v;
{% endprettify %}

If the field or variable can't be initialized in the declaration but will
always be initialized before it's referenced, then mark it as being `late`:

{% prettify dart tag=pre+code %}
class C {
  static late int f;
}
{% endprettify %}

### not_iterable_spread

_Spread elements in list or set literals must implement 'Iterable'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in either a list literal or a
set literal doesn't implement the type `Iterable`.

#### Examples

The following code produces this diagnostic:

{% prettify dart tag=pre+code %}
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...[!m!]};
{% endprettify %}

#### Common fixes

The most common fix is to replace the expression with one that produces an
iterable object:

{% prettify dart tag=pre+code %}
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...m.keys};
{% endprettify %}

### not_map_spread

_Spread elements in map literals must implement 'Map'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in a map literal doesn't
implement the type `Map`.

#### Examples

The following code produces this diagnostic because `l` isn't a `Map`:

{% prettify dart tag=pre+code %}
var l =  <String>['a', 'b'];
var m = <int, String>{...[!l!]};
{% endprettify %}

#### Common fixes

The most common fix is to replace the expression with one that produces a
map:

{% prettify dart tag=pre+code %}
var l =  <String>['a', 'b'];
var m = <int, String>{...l.asMap()};
{% endprettify %}

### no_annotation_constructor_arguments

_Annotation creation must have arguments._

#### Description

The analyzer produces this diagnostic when an annotation consists of a
single identifier, but that identifier is the name of a class rather than a
variable. To create an instance of the class, the identifier must be
followed by an argument list.

#### Examples

The following code produces this diagnostic because `C` is a class, and a
class can't be used as an annotation without invoking a `const` constructor
from the class:

{% prettify dart tag=pre+code %}
class C {
  const C();
}

[!@C!]
var x;
{% endprettify %}

#### Common fixes

Add the missing argument list:

{% prettify dart tag=pre+code %}
class C {
  const C();
}

@C()
var x;
{% endprettify %}

### no_combined_super_signature

_Can't infer missing types in '{0}' from overridden methods: {1}._

#### Description

The analyzer produces this diagnostic when there is a method declaration
for which one or more types needs to be inferred, and those types can't be
inferred because none of the overridden methods has a function type that is
a supertype of all the other overridden methods, as specified by
[override inference][].

#### Example

The following code produces this diagnostic because the method `m` declared
in the class `C` is missing both the return type and the type of the
parameter `a`, and neither of the missing types can be inferred for it:

{% prettify dart tag=pre+code %}
abstract class A {
  A m(String a);
}

abstract class B {
  B m(int a);
}

abstract class C implements A, B {
  [!m!](a);
}
{% endprettify %}

In this example, override inference can't be performed because the
overridden methods are incompatible in these ways:
- Neither parameter type (`String` and `int`) is a supertype of the other.
- Neither return type is a subtype of the other.

#### Common fixes

If possible, add types to the method in the subclass that are consistent
with the types from all the overridden methods:

{% prettify dart tag=pre+code %}
abstract class A {
  A m(String a);
}

abstract class B {
  B m(int a);
}

abstract class C implements A, B {
  C m(Object a);
}
{% endprettify %}

### nullable_type_in_catch_clause

_A potentially nullable type can't be used in an 'on' clause because it isn't
valid to throw a nullable expression._

#### Description

The analyzer produces this diagnostic when the type following `on` in a
`catch` clause is a nullable type. It isn't valid to specify a nullable
type because it isn't possible to catch `null` (because it's a runtime
error to throw `null`).

#### Example

The following code produces this diagnostic because the exception type is
specified to allow `null` when `null` can't be thrown:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } on [!FormatException?!] {
  }
}
{% endprettify %}

#### Common fixes

Remove the question mark from the type:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } on FormatException {
  }
}
{% endprettify %}

### nullable_type_in_extends_clause

_A class can't extend a nullable type._

#### Description

The analyzer produces this diagnostic when a class declaration uses an
`extends` clause to specify a superclass, and the superclass is followed by
a `?`.

It isn't valid to specify a nullable superclass because doing so would have
no meaning; it wouldn't change either the interface or implementation being
inherited by the class containing the `extends` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the superclass, such as `class A extends B<C?> {}`.

#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in an `extends` clause:

{% prettify dart tag=pre+code %}
class A {}
class B extends [!A?!] {}
{% endprettify %}

#### Common fixes

Remove the question mark from the type:

{% prettify dart tag=pre+code %}
class A {}
class B extends A {}
{% endprettify %}

### nullable_type_in_implements_clause

_A class or mixin can't implement a nullable type._

#### Description

The analyzer produces this diagnostic when a class or mixin declaration has
an `implements` clause, and an interface is followed by a `?`.

It isn't valid to specify a nullable interface because doing so would have
no meaning; it wouldn't change the interface being inherited by the class
containing the `implements` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the interface, such as `class A implements B<C?> {}`.


#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in an `implements` clause:

{% prettify dart tag=pre+code %}
class A {}
class B implements [!A?!] {}
{% endprettify %}

#### Common fixes

Remove the question mark from the type:

{% prettify dart tag=pre+code %}
class A {}
class B implements A {}
{% endprettify %}

### nullable_type_in_on_clause

_A mixin can't have a nullable type as a superclass constraint._

#### Description

The analyzer produces this diagnostic when a mixin declaration uses an `on`
clause to specify a superclass constraint, and the class that's specified
is followed by a `?`.

It isn't valid to specify a nullable superclass constraint because doing so
would have no meaning; it wouldn't change the interface being depended on
by the mixin containing the `on` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the superclass constraint, such as `mixin A on B<C?> {}`.


#### Example

The following code produces this diagnostic because `A?` is a nullable type
and nullable types can't be used in an `on` clause:

{% prettify dart tag=pre+code %}
class C {}
mixin M on [!C?!] {}
{% endprettify %}

#### Common fixes

Remove the question mark from the type:

{% prettify dart tag=pre+code %}
class C {}
mixin M on C {}
{% endprettify %}

### nullable_type_in_with_clause

_A class or mixin can't mix in a nullable type._

#### Description

The analyzer produces this diagnostic when a class or mixin declaration has
a `with` clause, and a mixin is followed by a `?`.

It isn't valid to specify a nullable mixin because doing so would have no
meaning; it wouldn't change either the interface or implementation being
inherited by the class containing the `with` clause.

Note, however, that it _is_ valid to use a nullable type as a type argument
to the mixin, such as `class A with B<C?> {}`.

#### Example

The following code produces this diagnostic because `A?` is a nullable
type, and nullable types can't be used in a `with` clause:

{% prettify dart tag=pre+code %}
mixin M {}
class C with [!M?!] {}
{% endprettify %}

#### Common fixes

Remove the question mark from the type:

{% prettify dart tag=pre+code %}
mixin M {}
class C with M {}
{% endprettify %}

### on_repeated

_The type '{0}' can be included in the superclass constraints only once._

#### Description

The analyzer produces this diagnostic when the same type is listed in the
superclass constraints of a mixin multiple times.

#### Example

The following code produces this diagnostic because `A` is included twice
in the superclass constraints for `M`:

{% prettify dart tag=pre+code %}
mixin M on A, [!A!] {
}

class A {}
class B {}
{% endprettify %}

#### Common fixes

If a different type should be included in the superclass constraints, then
replace one of the occurrences with the other type:

{% prettify dart tag=pre+code %}
mixin M on A, B {
}

class A {}
class B {}
{% endprettify %}

If no other type was intended, then remove the repeated type name:

{% prettify dart tag=pre+code %}
mixin M on A {
}

class A {}
class B {}
{% endprettify %}

### override_on_non_overriding_member

_The field doesn't override an inherited getter or setter._

_The getter doesn't override an inherited getter._

_The method doesn't override an inherited method._

_The setter doesn't override an inherited setter._

#### Description

The analyzer produces this diagnostic when a class member is annotated with
the `@override` annotation, but the member isn’t declared in any of the
supertypes of the class.

#### Examples

The following code produces this diagnostic because `m` isn't declared in
any of the supertypes of `C`:

{% prettify dart tag=pre+code %}
class C {
  @override
  String [!m!]() => '';
}
{% endprettify %}

#### Common fixes

If the member is intended to override a member with a different name, then
update the member to have the same name:

{% prettify dart tag=pre+code %}
class C {
  @override
  String toString() => '';
}
{% endprettify %}

If the member is intended to override a member that was removed from the
superclass, then consider removing the member from the subclass.

If the member can't be removed, then remove the annotation.

### part_of_different_library

_Expected this library to be part of '{0}', not '{1}'._

#### Description

The analyzer produces this diagnostic when a library attempts to include a
file as a part of itself when the other file is a part of a different
library.

#### Example

Given a file named `part.dart` containing

{% prettify dart tag=pre+code %}
part of 'library.dart';
{% endprettify %}

The following code, in any file other than `library.dart`, produces this
diagnostic because it attempts to include `part.dart` as a part of itself
when `part.dart` is a part of a different library:

{% prettify dart tag=pre+code %}
part [!'package:a/part.dart'!];
{% endprettify %}

#### Common fixes

If the library should be using a different file as a part, then change the
URI in the part directive to be the URI of the other file.

If the part file should be a part of this library, then update the URI (or
library name) in the part-of directive to be the URI (or name) of the
correct library.

### part_of_non_part

_The included part '{0}' must have a part-of directive._

#### Description

The analyzer produces this diagnostic when a part directive is found and
the referenced file doesn't have a part-of directive.

#### Examples

Given a file (`a.dart`) containing:

{% prettify dart tag=pre+code %}
class A {}
{% endprettify %}

The following code produces this diagnostic because `a.dart` doesn't
contain a part-of directive:

{% prettify dart tag=pre+code %}
part [!'a.dart'!];
{% endprettify %}

#### Common fixes

If the referenced file is intended to be a part of another library, then
add a part-of directive to the file:

{% prettify dart tag=pre+code %}
part of 'test.dart';

class A {}
{% endprettify %}

If the referenced file is intended to be a library, then replace the part
directive with an import directive:

{% prettify dart tag=pre+code %}
import 'a.dart';
{% endprettify %}

### prefix_collides_with_top_level_member

_The name '{0}' is already used as an import prefix and can't be used to name a
top-level element._

#### Description

The analyzer produces this diagnostic when a name is used as both an import
prefix and the name of a top-level declaration in the same library.

#### Example

The following code produces this diagnostic because `f` is used as both an
import prefix and the name of a function:

{% prettify dart tag=pre+code %}
import 'dart:math' as f;

int [!f!]() => f.min(0, 1);
{% endprettify %}

#### Common fixes

If you want to use the name for the import prefix, then rename the
top-level declaration:

{% prettify dart tag=pre+code %}
import 'dart:math' as f;

int g() => f.min(0, 1);
{% endprettify %}

If you want to use the name for the top-level declaration, then rename the
import prefix:

{% prettify dart tag=pre+code %}
import 'dart:math' as math;

int f() => math.min(0, 1);
{% endprettify %}

### prefix_identifier_not_followed_by_dot

_The name '{0}' refers to an import prefix, so it must be followed by '.'._

#### Description

The analyzer produces this diagnostic when an import prefix is used by
itself, without accessing any of the names declared in the libraries
associated with the prefix. Prefixes aren't variables, and therefore can't
be used as a value.

#### Example

The following code produces this diagnostic because the prefix `math` is
being used as if it were a variable:

{% prettify dart tag=pre+code %}
import 'dart:math' as math;

void f() {
  print([!math!]);
}
{% endprettify %}

#### Common fixes

If the code is incomplete, then reference something in one of the libraries
associated with the prefix:

{% prettify dart tag=pre+code %}
import 'dart:math' as math;

void f() {
  print(math.pi);
}
{% endprettify %}

If the name is wrong, then correct the name.

### private_optional_parameter

_Named parameters can't start with an underscore._

#### Description

The analyzer produces this diagnostic when the name of a named parameter
starts with an underscore.

#### Example

The following code produces this diagnostic because the named parameter
`_x` starts with an underscore:

{% prettify dart tag=pre+code %}
class C {
  void m({int [!_x!] = 0}) {}
}
{% endprettify %}

#### Common fixes

Rename the parameter so that it doesn't start with an underscore:

{% prettify dart tag=pre+code %}
class C {
  void m({int x = 0}) {}
}
{% endprettify %}

### recursive_compile_time_constant

_The compile-time constant expression depends on itself._

#### Description

The analyzer produces this diagnostic when the value of a compile-time
constant is defined in terms of itself, either directly or indirectly,
creating an infinite loop.

#### Example

The following code produces this diagnostic twice because both of the
constants are defined in terms of the other:

{% prettify dart tag=pre+code %}
const [!secondsPerHour!] = minutesPerHour * 60;
const [!minutesPerHour!] = secondsPerHour / 60;
{% endprettify %}

#### Common fixes

Break the cycle by finding an alternative way of defining at least one of
the constants:

{% prettify dart tag=pre+code %}
const secondsPerHour = minutesPerHour * 60;
const minutesPerHour = 60;
{% endprettify %}

### recursive_constructor_redirect

_Constructors can't redirect to themselves either directly or indirectly._

#### Description

The analyzer produces this diagnostic when a constructor redirects to
itself, either directly or indirectly, creating an infinite loop.

#### Example

The following code produces this diagnostic because the generative
constructors `C.a` and `C.b` each redirect to the other:

{% prettify dart tag=pre+code %}
class C {
  C.a() : [!this.b()!];
  C.b() : [!this.a()!];
}
{% endprettify %}

The following code produces this diagnostic because the factory
constructors `A` and `B` each redirect to the other:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = [!B!];
}
class B implements A {
  factory B() = [!A!];
  B.named();
}
{% endprettify %}

#### Common fixes

In the case of generative constructors, break the cycle by finding defining
at least one of the constructors to not redirect to another constructor:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b();
  C.b();
}
{% endprettify %}

In the case of factory constructors, break the cycle by defining at least
one of the factory constructors to do one of the following:

- Redirect to a generative constructor:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = B;
}
class B implements A {
  factory B() = B.named;
  B.named();
}
{% endprettify %}

- Not redirect to another constructor:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = B;
}
class B implements A {
  factory B() {
    return B.named();
  }

  B.named();
}
{% endprettify %}

- Not be a factory constructor:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = B;
}
class B implements A {
  B();
  B.named();
}
{% endprettify %}

### redirect_generative_to_missing_constructor

_The constructor '{0}' couldn't be found in '{1}'._

#### Description

The analyzer produces this diagnostic when a generative constructor
redirects to a constructor that isn't defined.

#### Example

The following code produces this diagnostic because the constructor `C.a`
redirects to the constructor `C.b`, but `C.b` isn't defined:

{% prettify dart tag=pre+code %}
class C {
  C.a() : [!this.b()!];
}
{% endprettify %}

#### Common fixes

If the missing constructor must be called, then define it:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b();
  C.b();
}
{% endprettify %}

If the missing constructor doesn't need to be called, then remove the
redirect:

{% prettify dart tag=pre+code %}
class C {
  C.a();
}
{% endprettify %}

### redirect_generative_to_non_generative_constructor

_Generative constructors can't redirect to a factory constructor._

#### Description

The analyzer produces this diagnostic when a generative constructor
redirects to a factory constructor.

#### Example

The following code produces this diagnostic because the generative
constructor `C.a` redirects to the factory constructor `C.b`:

{% prettify dart tag=pre+code %}
class C {
  C.a() : [!this.b()!];
  factory C.b() => C.a();
}
{% endprettify %}

#### Common fixes

If the generative constructor doesn't need to redirect to another
constructor, then remove the redirect.

{% prettify dart tag=pre+code %}
class C {
  C.a();
  factory C.b() => C.a();
}
{% endprettify %}

If the generative constructor must redirect to another constructor, then
make the other constructor be a generative (non-factory) constructor:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b();
  C.b();
}
{% endprettify %}

### redirect_to_invalid_function_type

_The redirected constructor '{0}' has incompatible parameters with '{1}'._

#### Description

The analyzer produces this diagnostic when a factory constructor attempts
to redirect to another constructor, but the two have incompatible
parameters. The parameters are compatible if all of the parameters of the
redirecting constructor can be passed to the other constructor and if the
other constructor doesn't require any parameters that aren't declared by
the redirecting constructor.

#### Examples

The following code produces this diagnostic because the constructor for `A`
doesn't declare a parameter that the constructor for `B` requires:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = [!B!];
}

class B implements A {
  B(int x);
  B.zero();
}
{% endprettify %}

The following code produces this diagnostic because the constructor for `A`
declares a named parameter (`y`) that the constructor for `B` doesn't
allow:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A(int x, {int y}) = [!B!];
}

class B implements A {
  B(int x);
}
{% endprettify %}

#### Common fixes

If there's a different constructor that is compatible with the redirecting
constructor, then redirect to that constructor:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A() = B.zero;
}

class B implements A {
  B(int x);
  B.zero();
}
{% endprettify %}

Otherwise, update the redirecting constructor to be compatible:

{% prettify dart tag=pre+code %}
abstract class A {
  factory A(int x) = B;
}

class B implements A {
  B(int x);
}
{% endprettify %}

### redirect_to_invalid_return_type

_The return type '{0}' of the redirected constructor isn't a subtype of '{1}'._

#### Description

The analyzer produces this diagnostic when a factory constructor redirects
to a constructor whose return type isn't a subtype of the type that the
factory constructor is declared to produce.

#### Examples

The following code produces this diagnostic because `A` isn't a subclass
of `C`, which means that the value returned by the constructor `A()`
couldn't be returned from the constructor `C()`:

{% prettify dart tag=pre+code %}
class A {}

class B implements C {}

class C {
  factory C() = [!A!];
}
{% endprettify %}

#### Common fixes

If the factory constructor is redirecting to a constructor in the wrong
class, then update the factory constructor to redirect to the correct
constructor:

{% prettify dart tag=pre+code %}
class A {}

class B implements C {}

class C {
  factory C() = B;
}
{% endprettify %}

If the class defining the constructor being redirected to is the class that
should be returned, then make it a subtype of the factory's return type:

{% prettify dart tag=pre+code %}
class A implements C {}

class B implements C {}

class C {
  factory C() = A;
}
{% endprettify %}

### redirect_to_non_class

_The name '{0}' isn't a type and can't be used in a redirected constructor._

#### Description

One way to implement a factory constructor is to redirect to another
constructor by referencing the name of the constructor. The analyzer
produces this diagnostic when the redirect is to something other than a
constructor.

#### Examples

The following code produces this diagnostic because `f` is a function:

{% prettify dart tag=pre+code %}
C f() => throw 0;

class C {
  factory C() = [!f!];
}
{% endprettify %}

#### Common fixes

If the constructor isn't defined, then either define it or replace it with
a constructor that is defined.

If the constructor is defined but the class that defines it isn't visible,
then you probably need to add an import.

If you're trying to return the value returned by a function, then rewrite
the constructor to return the value from the constructor's body:

{% prettify dart tag=pre+code %}
C f() => throw 0;

class C {
  factory C() => f();
}
{% endprettify %}

### redirect_to_non_const_constructor

_A constant redirecting constructor can't redirect to a non-constant
constructor._

#### Description

The analyzer produces this diagnostic when a constructor marked as `const`
redirects to a constructor that isn't marked as `const`.

#### Example

The following code produces this diagnostic because the constructor `C.a`
is marked as `const` but redirects to the constructor `C.b`, which isn't:

{% prettify dart tag=pre+code %}
class C {
  const C.a() : this.[!b!]();
  C.b();
}
{% endprettify %}

#### Common fixes

If the non-constant constructor can be marked as `const`, then mark it as
`const`:

{% prettify dart tag=pre+code %}
class C {
  const C.a() : this.b();
  const C.b();
}
{% endprettify %}

If the non-constant constructor can't be marked as `const`, then either
remove the redirect or remove `const` from the redirecting constructor:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b();
  C.b();
}
{% endprettify %}

### referenced_before_declaration

_Local variable '{0}' can't be referenced before it is declared._

#### Description

The analyzer produces this diagnostic when a variable is referenced before
it’s declared. In Dart, variables are visible everywhere in the block in
which they are declared, but can only be referenced after they are
declared.

The analyzer also produces a context message that indicates where the
declaration is located.

#### Examples

The following code produces this diagnostic because `i` is used before it
is declared:

{% prettify dart tag=pre+code %}
void f() {
  print([!i!]);
  int i = 5;
}
{% endprettify %}

#### Common fixes

If you intended to reference the local variable, move the declaration
before the first reference:

{% prettify dart tag=pre+code %}
void f() {
  int i = 5;
  print(i);
}
{% endprettify %}

If you intended to reference a name from an outer scope, such as a
parameter, instance field or top-level variable, then rename the local
declaration so that it doesn't hide the outer variable.

{% prettify dart tag=pre+code %}
void f(int i) {
  print(i);
  int x = 5;
  print(x);
}
{% endprettify %}

### rethrow_outside_catch

_A rethrow must be inside of a catch clause._

#### Description

The analyzer produces this diagnostic when a `rethrow` statement is outside
a `catch` clause. The `rethrow` statement is used to throw a caught
exception again, but there's no caught exception outside of a `catch`
clause.

#### Example

The following code produces this diagnostic because the`rethrow` statement
is outside of a `catch` clause:

{% prettify dart tag=pre+code %}
void f() {
  [!rethrow!];
}
{% endprettify %}

#### Common fixes

If you're trying to rethrow an exception, then wrap the `rethrow` statement
in a `catch` clause:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } catch (exception) {
    rethrow;
  }
}
{% endprettify %}

If you're trying to throw a new exception, then replace the `rethrow`
statement with a `throw` expression:

{% prettify dart tag=pre+code %}
void f() {
  throw UnsupportedError('Not yet implemented');
}
{% endprettify %}

### return_in_generative_constructor

_Constructors can't return values._

#### Description

The analyzer produces this diagnostic when a generative constructor
contains a `return` statement that specifies a value to be returned.
Generative constructors always return the object that was created, and
therefore can't return a different object.

#### Example

The following code produces this diagnostic because the `return` statement
has an expression:

{% prettify dart tag=pre+code %}
class C {
  C() {
    return [!this!];
  }
}
{% endprettify %}

#### Common fixes

If the constructor should create a new instance, then remove either the
`return` statement or the expression:

{% prettify dart tag=pre+code %}
class C {
  C();
}
{% endprettify %}

If the constructor shouldn't create a new instance, then convert it to be a
factory constructor:

{% prettify dart tag=pre+code %}
class C {
  factory C() {
    return _instance;
  }

  static C _instance = C._();

  C._();
}
{% endprettify %}

### return_of_invalid_type

_A value of type '{0}' can't be returned from the constructor '{2}' because it
has a return type of '{1}'._

_A value of type '{0}' can't be returned from the function '{2}' because it has
a return type of '{1}'._

_A value of type '{0}' can't be returned from the method '{2}' because it has a
return type of '{1}'._

#### Description

The analyzer produces this diagnostic when a method or function returns a
value whose type isn't assignable to the declared return type.

#### Examples

The following code produces this diagnostic because `f` has a return type
of `String` but is returning an `int`:

{% prettify dart tag=pre+code %}
String f() => [!3!];
{% endprettify %}

#### Common fixes

If the return type is correct, then replace the value being returned with a
value of the correct type, possibly by converting the existing value:

{% prettify dart tag=pre+code %}
String f() => 3.toString();
{% endprettify %}

If the value is correct, then change the return type to match:

{% prettify dart tag=pre+code %}
int f() => 3;
{% endprettify %}

### return_of_invalid_type_from_closure

_The return type '{0}' isn't a '{1}', as required by the closure's context._

#### Description

The analyzer produces this diagnostic when the static type of a returned
expression isn't assignable to the return type that the closure is required
to have.

#### Examples

The following code produces this diagnostic because `f` is defined to be a
function that returns a `String`, but the closure assigned to it returns an
`int`:

{% prettify dart tag=pre+code %}
String Function(String) f = (s) => [!3!];
{% endprettify %}

#### Common fixes

If the return type is correct, then replace the returned value with a value
of the correct type, possibly by converting the existing value:

{% prettify dart tag=pre+code %}
String Function(String) f = (s) => 3.toString();
{% endprettify %}

### return_without_value

_The return value is missing after 'return'._

#### Description

The analyzer produces this diagnostic when it finds a `return` statement
without an expression in a function that declares a return type.

#### Examples

The following code produces this diagnostic because the function `f` is
expected to return an `int`, but no value is being returned:

{% prettify dart tag=pre+code %}
int f() {
  [!return!];
}
{% endprettify %}

#### Common fixes

Add an expression that computes the value to be returned:

{% prettify dart tag=pre+code %}
int f() {
  return 0;
}
{% endprettify %}

### sdk_version_async_exported_from_core

_The class '{0}' wasn't exported from 'dart:core' until version 2.1, but this
code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when either the class `Future` or
`Stream` is referenced in a library that doesn't import `dart:async` in
code that has an SDK constraint whose lower bound is less than 2.1.0. In
earlier versions, these classes weren't defined in `dart:core`, so the
import was necessary.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.1.0:

```yaml
environment:
  sdk: '>=2.0.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart tag=pre+code %}
void f([!Future!] f) {}
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the classes to be referenced:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

If you need to support older versions of the SDK, then import the
`dart:async` library.

{% prettify dart tag=pre+code %}
import 'dart:async';

void f(Future f) {}
{% endprettify %}

### sdk_version_as_expression_in_const_context

_The use of an as expression in a constant expression wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an `as` expression inside a
[constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.3.2. Using an `as` expression in a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

{% prettify dart tag=pre+code %}
const num n = 3;
const int i = [!n as int!];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use an `as` expression, or change the code so that the `as`
expression isn't in a [constant context][]:

{% prettify dart tag=pre+code %}
num x = 3;
int y = x as int;
{% endprettify %}

### sdk_version_bool_operator_in_const_context

_The use of the operator '{0}' for 'bool' operands in a constant context wasn't
supported until version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when any use of the `&`, `|`, or `^`
operators on the class `bool` inside a [constant context][] is found in
code that has an SDK constraint whose lower bound is less than 2.3.2. Using
these operators in a [constant context][] wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart tag=pre+code %}
const bool a = true;
const bool b = false;
const bool c = a [!&!] b;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operators to be used:

```yaml
environment:
 sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use these operators, or change the code so that the expression
isn't in a [constant context][]:

{% prettify dart tag=pre+code %}
const bool a = true;
const bool b = false;
bool c = a & b;
{% endprettify %}

### sdk_version_eq_eq_operator_in_const_context

_Using the operator '==' for non-primitive types wasn't supported until version
2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when the operator `==` is used on a
non-primitive type inside a [constant context][] is found in code that has
an SDK constraint whose lower bound is less than 2.3.2. Using this operator
in a [constant context][] wasn't supported in earlier versions, so this
code won't be able to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart tag=pre+code %}
class C {}
const C a = null;
const C b = null;
const bool same = a [!==!] b;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operator to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the `==` operator, or change the code so that the
expression isn't in a [constant context][]:

{% prettify dart tag=pre+code %}
class C {}
const C a = null;
const C b = null;
bool same = a == b;
{% endprettify %}

### sdk_version_extension_methods

_Extension methods weren't supported until version 2.6.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an extension declaration or an
extension override is found in code that has an SDK constraint whose lower
bound is less than 2.6.0. Using extensions wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.6.0:

```yaml
environment:
 sdk: '>=2.4.0 <2.7.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

{% prettify dart tag=pre+code %}
[!extension!] E on String {
  void sayHello() {
    print('Hello $this');
  }
}
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.6.0 <2.7.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of extensions. The most common way to do this is to rewrite
the members of the extension as top-level functions (or methods) that take
the value that would have been bound to `this` as a parameter:

{% prettify dart tag=pre+code %}
void sayHello(String s) {
  print('Hello $s');
}
{% endprettify %}

### sdk_version_is_expression_in_const_context

_The use of an is expression in a constant context wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an `is` expression inside a
[constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.3.2. Using an `is` expression in a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

{% prettify dart tag=pre+code %}
const x = 4;
const y = [!x is int!] ? 0 : 1;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the `is` operator, or, if that isn't possible, change the
code so that the `is` expression isn't in a
[constant context][]:

{% prettify dart tag=pre+code %}
const x = 4;
var y = x is int ? 0 : 1;
{% endprettify %}

### sdk_version_never

_The type 'Never' wasn't supported until version 2.X.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a reference to the class `Never`
is found in code that has an SDK constraint whose lower bound is less than
2.12.0. This class wasn't defined in earlier versions, so this code won't
be able to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.12.0:

```yaml
environment:
  sdk: '>=2.5.0 <2.6.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart tag=pre+code %}
[!Never!] n;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the type to be used:

```yaml
environment:
  sdk: '>=2.12.0 <2.13.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not reference this class:

{% prettify dart tag=pre+code %}
dynamic x;
{% endprettify %}

### sdk_version_set_literal

_Set literals weren't supported until version 2.2, but this code is required to
be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a set literal is found in code
that has an SDK constraint whose lower bound is less than 2.2.0. Set
literals weren't supported in earlier versions, so this code won't be able
to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.2.0:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart tag=pre+code %}
var s = [!<int>{}!];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

If you do need to support older versions of the SDK, then replace the set
literal with code that creates the set without the use of a literal:

{% prettify dart tag=pre+code %}
var s = new Set<int>();
{% endprettify %}

### sdk_version_ui_as_code

_The for, if, and spread elements weren't supported until version 2.3.0, but
this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a for, if, or spread element is
found in code that has an SDK constraint whose lower bound is less than
2.3.0. Using a for, if, or spread element wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.0:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

{% prettify dart tag=pre+code %}
var digits = [[!for (int i = 0; i < 10; i++) i!]];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.3.0 <2.4.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

{% prettify dart tag=pre+code %}
var digits = _initializeDigits();

List<int> _initializeDigits() {
  var digits = <int>[];
  for (int i = 0; i < 10; i++) {
    digits.add(i);
  }
  return digits;
}
{% endprettify %}

### sdk_version_ui_as_code_in_const_context

_The if and spread elements weren't supported in constant expressions until
version 2.5.0, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an if or spread element inside
a [constant context][] is found in code that has an SDK constraint whose
lower bound is less than 2.5.0. Using an if or spread element inside a
[constant context][] wasn't supported in earlier versions, so this code
won't be able to run against earlier versions of the SDK.

#### Examples

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.5.0:

```yaml
environment:
  sdk: '>=2.4.0 <2.6.0'
```

In the package that has that pubspec, code like the following produces
this diagnostic:

{% prettify dart tag=pre+code %}
const a = [1, 2];
const b = [[!...a!]];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.5.0 <2.6.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

{% prettify dart tag=pre+code %}
const a = [1, 2];
const b = [1, 2];
{% endprettify %}

If that isn't possible, change the code so that the element isn't in a
[constant context][]:

{% prettify dart tag=pre+code %}
const a = [1, 2];
var b = [...a];
{% endprettify %}

### shared_deferred_prefix

_The prefix of a deferred import can't be used in other import directives._

#### Description

The analyzer produces this diagnostic when a prefix in a deferred import is
also used as a prefix in other imports (whether deferred or not). The
prefix in a deferred import can't be shared with other imports because the
prefix is used to load the imported library.

#### Example

The following code produces this diagnostic because the prefix `x` is used
as the prefix for a deferred import and is also used for one other import:

{% prettify dart tag=pre+code %}
import 'dart:math' [!deferred!] as x;
import 'dart:convert' as x;

var y = x.json.encode(x.min(0, 1));
{% endprettify %}

#### Common fixes

If you can use a different name for the deferred import, then do so:

{% prettify dart tag=pre+code %}
import 'dart:math' deferred as math;
import 'dart:convert' as x;

var y = x.json.encode(math.min(0, 1));
{% endprettify %}

If you can use a different name for the other imports, then do so:

{% prettify dart tag=pre+code %}
import 'dart:math' deferred as x;
import 'dart:convert' as convert;

var y = convert.json.encode(x.min(0, 1));
{% endprettify %}

### static_access_to_instance_member

_Instance member '{0}' can't be accessed using static access._

#### Description

The analyzer produces this diagnostic when a class name is used to access
an instance field. Instance fields don't exist on a class; they exist only
on an instance of the class.

#### Examples

The following code produces this diagnostic because `x` is an instance
field:

{% prettify dart tag=pre+code %}
class C {
  static int a;

  int b;
}

int f() => C.[!b!];
{% endprettify %}

#### Common fixes

If you intend to access a static field, then change the name of the field
to an existing static field:

{% prettify dart tag=pre+code %}
class C {
  static int a;

  int b;
}

int f() => C.a;
{% endprettify %}

If you intend to access the instance field, then use an instance of the
class to access the field:

{% prettify dart tag=pre+code %}
class C {
  static int a;

  int b;
}

int f(C c) => c.b;
{% endprettify %}

### subtype_of_disallowed_type

_''{0}' can't be used as a superclass constraint._

_Classes and mixins can't implement '{0}'._

_Classes can't extend '{0}'._

_Classes can't mixin '{0}'._

#### Description

The analyzer produces this diagnostic when one of the restricted classes is
used in either an `extends`, `implements`, `with`, or `on` clause. The
classes `bool`, `double`, `FutureOr`, `int`, `Null`, `num`, and `String`
are all restricted in this way, to allow for more efficient
implementations.

#### Example

The following code produces this diagnostic because `String` is used in an
`extends` clause:

{% prettify dart tag=pre+code %}
class A extends [!String!] {}
{% endprettify %}

The following code produces this diagnostic because `String` is used in an
`implements` clause:

{% prettify dart tag=pre+code %}
class B implements [!String!] {}
{% endprettify %}

The following code produces this diagnostic because `String` is used in a
`with` clause:

{% prettify dart tag=pre+code %}
class C with [!String!] {}
{% endprettify %}

The following code produces this diagnostic because `String` is used in an
`on` clause:

{% prettify dart tag=pre+code %}
mixin M on [!String!] {}
{% endprettify %}

#### Common fixes

If a different type should be specified, then replace the type:

{% prettify dart tag=pre+code %}
class A extends Object {}
{% endprettify %}

If there isn't a different type that would be appropriate, then remove the
type, and possibly the whole clause:

{% prettify dart tag=pre+code %}
class B {}
{% endprettify %}

### super_in_extension

_The 'super' keyword can't be used in an extension because an extension doesn't
have a superclass._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the `super` keyword . Extensions aren't classes and don't
have superclasses, so the `super` keyword serves no purpose.

#### Examples

The following code produces this diagnostic because `super` can't be used
in an extension:

{% prettify dart tag=pre+code %}
extension E on Object {
  String get displayString => [!super!].toString();
}
{% endprettify %}

#### Common fixes

Remove the `super` keyword :

{% prettify dart tag=pre+code %}
extension E on Object {
  String get displayString => toString();
}
{% endprettify %}

### super_in_invalid_context

_Invalid context for 'super' invocation._

#### Description

The analyzer produces this diagnostic when the keyword `super` is used
outside of a instance method.

#### Examples

The following code produces this diagnostic because `super` is used in a
top-level function:

{% prettify dart tag=pre+code %}
void f() {
  [!super!].f();
}
{% endprettify %}

#### Common fixes

Rewrite the code to not use `super`.

### super_in_redirecting_constructor

_The redirecting constructor can't have a 'super' initializer._

#### Description

The analyzer produces this diagnostic when a constructor that redirects to
another constructor also attempts to invoke a constructor from the
superclass. The superclass constructor will be invoked when the constructor
that the redirecting constructor is redirected to is invoked.

#### Example

The following code produces this diagnostic because the constructor `C.a`
both redirects to `C.b` and invokes a constructor from the superclass:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b(), [!super()!];
  C.b();
}
{% endprettify %}

#### Common fixes

Remove the invocation of the `super` constructor:

{% prettify dart tag=pre+code %}
class C {
  C.a() : this.b();
  C.b();
}
{% endprettify %}

### switch_expression_not_assignable

_Type '{0}' of the switch expression isn't assignable to the type '{1}' of case
expressions._

#### Description

The analyzer produces this diagnostic when the type of the expression in a
`switch` statement isn't assignable to the type of the expressions in the
`case` clauses.

#### Example

The following code produces this diagnostic because the type of `s`
(`String`) isn't assignable to the type of `0` (`int`):

{% prettify dart tag=pre+code %}
void f(String s) {
  switch ([!s!]) {
    case 0:
      break;
  }
}
{% endprettify %}

#### Common fixes

If the type of the `case` expressions is correct, then change the
expression in the `switch` statement to have the correct type:

{% prettify dart tag=pre+code %}
void f(String s) {
  switch (int.parse(s)) {
    case 0:
      break;
  }
}
{% endprettify %}

If the type of the `switch` expression is correct, then change the `case`
expressions to have the correct type:

{% prettify dart tag=pre+code %}
void f(String s) {
  switch (s) {
    case '0':
      break;
  }
}
{% endprettify %}

### throw_of_invalid_type

_The type '{0}' of the thrown expression must be assignable to 'Object'._

#### Description

The analyzer produces this diagnostic when the type of the expression in a
throw expression isn't assignable to `Object`. It isn't valid to throw
`null`, so it isn't valid to use an expression that might evaluate to
`null`.

#### Example

The following code produces this diagnostic because `s` might be `null`:

{% prettify dart tag=pre+code %}
void f(String? s) {
  throw [!s!];
}
{% endprettify %}

#### Common fixes

Add an explicit null check to the expression:

{% prettify dart tag=pre+code %}
void f(String? s) {
  throw s!;
}
{% endprettify %}

### type_argument_not_matching_bounds

_'{0}' doesn't conform to the bound '{2}' of the type parameter '{1}'._

#### Description

The analyzer produces this diagnostic when a type argument isn't the same
as or a subclass of the bounds of the corresponding type parameter.

#### Examples

The following code produces this diagnostic because `String` isn't a
subclass of `num`:

{% prettify dart tag=pre+code %}
class A<E extends num> {}

var a = A<[!String!]>();
{% endprettify %}

#### Common fixes

Change the type argument to be a subclass of the bounds:

{% prettify dart tag=pre+code %}
class A<E extends num> {}

var a = A<int>();
{% endprettify %}

### type_parameter_referenced_by_static

_Static members can't reference type parameters of the class._

#### Description

The analyzer produces this diagnostic when a static member references a
type parameter that is declared for the class. Type parameters only have
meaning for instances of the class.

#### Example

The following code produces this diagnostic because the static method
`hasType` has a reference to the type parameter `T`:

{% prettify dart tag=pre+code %}
class C<T> {
  static bool hasType(Object o) => o is [!T!];
}
{% endprettify %}

#### Common fixes

If the member can be an instance member, then remove the keyword `static`:

{% prettify dart tag=pre+code %}
class C<T> {
  bool hasType(Object o) => o is T;
}
{% endprettify %}

If the member must be a static member, then make the member be generic:

{% prettify dart tag=pre+code %}
class C<T> {
  static bool hasType<S>(Object o) => o is S;
}
{% endprettify %}

Note, however, that there isn’t a relationship between `T` and `S`, so this
second option changes the semantics from what was likely to be intended.

### type_test_with_undefined_name

_The name '{0}' isn't defined, so it can't be used in an 'is' expression._

#### Description

The analyzer produces this diagnostic when the name following the `is` in a
type test expression isn't defined.

#### Examples

The following code produces this diagnostic because the name `Srting` isn't
defined:

{% prettify dart tag=pre+code %}
void f(Object o) {
  if (o is [!Srting!]) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

Replace the name with the name of a type:

{% prettify dart tag=pre+code %}
void f(Object o) {
  if (o is String) {
    // ...
  }
}
{% endprettify %}

### unchecked_use_of_nullable_value

_A nullable expression can't be used as a condition._

_A nullable expression can't be used as an iterator in a for-in loop._

_A nullable expression can't be used in a spread._

_A nullable expression can't be used in a yield-each statement._

_An expression whose value can be 'null' must be null-checked before it can be
dereferenced._

_The function can't be unconditionally invoked because it can be 'null'._

_The method '{0}' can't be unconditionally invoked because the receiver can be
'null'._

_The operator '{0}' can't be unconditionally invoked because the receiver can be
'null'._

_The property '{0}' can't be unconditionally accessed because the receiver can
be 'null'._

#### Description

The analyzer produces this diagnostic when an expression whose type is
[potentially non-nullable][] is dereferenced without first verifying that
the value isn't `null`.

#### Example

The following code produces this diagnostic because `s` can be `null` at
the point where it's referenced:

{% prettify dart tag=pre+code %}
void f(String? s) {
  if ([!s!].length > 3) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

If the value really can be `null`, then add a test to ensure that members
are only accessed when the value isn't `null`:

{% prettify dart tag=pre+code %}
void f(String? s) {
  if (s != null && s.length > 3) {
    // ...
  }
}
{% endprettify %}

If the expression is a variable and the value should never be `null`, then
change the type of the variable to be non-nullable:

{% prettify dart tag=pre+code %}
void f(String s) {
  if (s.length > 3) {
    // ...
  }
}
{% endprettify %}

If you believe that the value of the expression should never be `null`, but
you can't change the type of the variable, and you're willing to risk
having an exception thrown at runtime if you're wrong, then you can assert
that the value isn't null:

{% prettify dart tag=pre+code %}
void f(String? s) {
  if (s!.length > 3) {
    // ...
  }
}
{% endprettify %}

### undefined_annotation

_Undefined name '{0}' used as an annotation._

#### Description

The analyzer produces this diagnostic when a name that isn't defined is
used as an annotation.

#### Examples

The following code produces this diagnostic because the name `undefined`
isn't defined:

{% prettify dart tag=pre+code %}
[!@undefined!]
void f() {}
{% endprettify %}

#### Common fixes

If the name is correct, but it isn’t declared yet, then declare the name as
a constant value:

{% prettify dart tag=pre+code %}
const undefined = 'undefined';

@undefined
void f() {}
{% endprettify %}

If the name is wrong, replace the name with the name of a valid constant:

{% prettify dart tag=pre+code %}
@deprecated
void f() {}
{% endprettify %}

Otherwise, remove the annotation.

### undefined_class

_Undefined class '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a class but either isn't defined or isn't visible
in the scope in which it's being referenced.

#### Examples

The following code produces this diagnostic because `Piont` isn't defined:

{% prettify dart tag=pre+code %}
class Point {}

void f([!Piont!] p) {}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a class that is defined. The example above can be corrected by
fixing the spelling of the class:

{% prettify dart tag=pre+code %}
class Point {}

void f(Point p) {}
{% endprettify %}

If the class is defined but isn't visible, then you probably need to add an
import.

### undefined_constructor_in_initializer

_The class '{0}' doesn't have a constructor named '{1}'._

_The class '{0}' doesn't have an unnamed constructor._

#### Description

The analyzer produces this diagnostic when a superclass constructor is
invoked in the initializer list of a constructor, but the superclass
doesn't define the constructor being invoked.

#### Examples

The following code produces this diagnostic because `A` doesn't have an
unnamed constructor:

{% prettify dart tag=pre+code %}
class A {
  A.n();
}
class B extends A {
  B() : [!super()!];
}
{% endprettify %}

The following code produces this diagnostic because `A` doesn't have a
constructor named `m`:

{% prettify dart tag=pre+code %}
class A {
  A.n();
}
class B extends A {
  B() : [!super.m()!];
}
{% endprettify %}

#### Common fixes

If the superclass defines a constructor that should be invoked, then change
the constructor being invoked:

{% prettify dart tag=pre+code %}
class A {
  A.n();
}
class B extends A {
  B() : super.n();
}
{% endprettify %}

If the superclass doesn't define an appropriate constructor, then define
the constructor being invoked:

{% prettify dart tag=pre+code %}
class A {
  A.m();
  A.n();
}
class B extends A {
  B() : super.m();
}
{% endprettify %}

### undefined_enum_constant

_There's no constant named '{0}' in '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of an enum constant, and the name either isn't
defined or isn't visible in the scope in which it's being referenced.

#### Examples

The following code produces this diagnostic because `E` doesn't define a
constant named `c`:

{% prettify dart tag=pre+code %}
enum E {a, b}

var e = E.[!c!];
{% endprettify %}

#### Common fixes

If the constant should be defined, then add it to the declaration of the
enum:

{% prettify dart tag=pre+code %}
enum E {a, b, c}

var e = E.c;
{% endprettify %}

If the constant shouldn't be defined, then change the name to the name of
an existing constant:

{% prettify dart tag=pre+code %}
enum E {a, b}

var e = E.b;
{% endprettify %}

### undefined_extension_getter

_The getter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a getter, but the getter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static getter is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance getter named `b`:

{% prettify dart tag=pre+code %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').[!b!];
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static getter named `a`:

{% prettify dart tag=pre+code %}
extension E on String {}

var x = E.[!a!];
{% endprettify %}

#### Common fixes

If the name of the getter is incorrect, then change it to the name of an
existing getter:

{% prettify dart tag=pre+code %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').a;
}
{% endprettify %}

If the name of the getter is correct but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart tag=pre+code %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  F('c').b;
}
{% endprettify %}

If the name of the getter and extension are both correct, but the getter
isn't defined, then define the getter:

{% prettify dart tag=pre+code %}
extension E on String {
  String get a => 'a';
  String get b => 'z';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').b;
}
{% endprettify %}

### undefined_extension_method

_The method '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a method, but the method isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static method is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance method named `b`:

{% prettify dart tag=pre+code %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').[!b!]();
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static method named `a`:

{% prettify dart tag=pre+code %}
extension E on String {}

var x = E.[!a!]();
{% endprettify %}

#### Common fixes

If the name of the method is incorrect, then change it to the name of an
existing method:

{% prettify dart tag=pre+code %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').a();
}
{% endprettify %}

If the name of the method is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart tag=pre+code %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  F('c').b();
}
{% endprettify %}

If the name of the method and extension are both correct, but the method
isn't defined, then define the method:

{% prettify dart tag=pre+code %}
extension E on String {
  String a() => 'a';
  String b() => 'z';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').b();
}
{% endprettify %}

### undefined_extension_setter

_The setter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a setter, but the setter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static setter is
referenced but isn't defined by the specified extension.

#### Examples

The following code produces this diagnostic because the extension `E`
doesn't declare an instance setter named `b`:

{% prettify dart tag=pre+code %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').[!b!] = 'd';
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static setter named `a`:

{% prettify dart tag=pre+code %}
extension E on String {}

void f() {
  E.[!a!] = 3;
}
{% endprettify %}

#### Common fixes

If the name of the setter is incorrect, then change it to the name of an
existing setter:

{% prettify dart tag=pre+code %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').a = 'd';
}
{% endprettify %}

If the name of the setter is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart tag=pre+code %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  F('c').b = 'd';
}
{% endprettify %}

If the name of the setter and extension are both correct, but the setter
isn't defined, then define the setter:

{% prettify dart tag=pre+code %}
extension E on String {
  set a(String v) {}
  set b(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').b = 'd';
}
{% endprettify %}

### undefined_function

_The function '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a function but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Examples

The following code produces this diagnostic because the name `emty` isn't
defined:

{% prettify dart tag=pre+code %}
List<int> empty() => [];

void main() {
  print([!emty!]());
}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a function that is defined. The example above can be corrected
by fixing the spelling of the function:

{% prettify dart tag=pre+code %}
List<int> empty() => [];

void main() {
  print(empty());
}
{% endprettify %}

If the function is defined but isn't visible, then you probably need to add
an import or re-arrange your code to make the function visible.

### undefined_getter

_The getter '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a getter but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Examples

The following code produces this diagnostic because `String` has no member
named `len`:

{% prettify dart tag=pre+code %}
int f(String s) => s.[!len!];
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a getter that is defined. The example above can be corrected by
fixing the spelling of the getter:

{% prettify dart tag=pre+code %}
int f(String s) => s.length;
{% endprettify %}

### undefined_hidden_name

_The library '{0}' doesn't export a member with the hidden name '{1}'._

#### Description

The analyzer produces this diagnostic when a hide combinator includes a
name that isn't defined by the library being imported.

#### Examples

The following code produces this diagnostic because `dart:math` doesn't
define the name `String`:

{% prettify dart tag=pre+code %}
import 'dart:math' hide [!String!], max;

var x = min(0, 1);
{% endprettify %}

#### Common fixes

If a different name should be hidden, then correct the name. Otherwise,
remove the name from the list:

{% prettify dart tag=pre+code %}
import 'dart:math' hide max;

var x = min(0, 1);
{% endprettify %}

### undefined_identifier

_Undefined name '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
either isn't defined or isn't visible in the scope in which it's being
referenced.

#### Examples

The following code produces this diagnostic because the name `rihgt` isn't
defined:

{% prettify dart tag=pre+code %}
int min(int left, int right) => left <= [!rihgt!] ? left : right;
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
an identifier that is defined. The example above can be corrected by
fixing the spelling of the variable:

{% prettify dart tag=pre+code %}
int min(int left, int right) => left <= right ? left : right;
{% endprettify %}

If the identifier is defined but isn't visible, then you probably need to
add an import or re-arrange your code to make the identifier visible.

### undefined_identifier_await

_Undefined name 'await' in function body not marked with 'async'._

#### Description

The analyzer produces this diagnostic when the name `await` is used in a
method or function body without being declared, and the body isn't marked
with the `async` keyword. The name `await` only introduces an await
expression in an asynchronous function.

#### Example

The following code produces this diagnostic because the name `await` is
used in the body of `f` even though the body of `f` isn't marked with the
`async` keyword:

{% prettify dart tag=pre+code %}
void f(p) { [!await!] p; }
{% endprettify %}

#### Common fixes

Add the keyword `async` to the function body:

{% prettify dart tag=pre+code %}
void f(p) async { await p; }
{% endprettify %}

### undefined_method

_The method '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a method but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Examples

The following code produces this diagnostic because the identifier
`removeMiddle` isn't defined:

{% prettify dart tag=pre+code %}
int f(List<int> l) => l.[!removeMiddle!]();
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a method that is defined. The example above can be corrected by
fixing the spelling of the method:

{% prettify dart tag=pre+code %}
int f(List<int> l) => l.removeLast();
{% endprettify %}

### undefined_named_parameter

_The named parameter '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has a named argument, but the method or function being invoked doesn't
define a parameter with the same name.

#### Examples

The following code produces this diagnostic because `m` doesn't declare a
named parameter named `a`:

{% prettify dart tag=pre+code %}
class C {
  m({int b}) {}
}

void f(C c) {
  c.m([!a!]: 1);
}
{% endprettify %}

#### Common fixes

If the argument name is mistyped, then replace it with the correct name.
The example above can be fixed by changing `a` to `b`:

{% prettify dart tag=pre+code %}
class C {
  m({int b}) {}
}

void f(C c) {
  c.m(b: 1);
}
{% endprettify %}

If a subclass adds a parameter with the name in question, then cast the
receiver to the subclass:

{% prettify dart tag=pre+code %}
class C {
  m({int b}) {}
}

class D extends C {
  m({int a, int b}) {}
}

void f(C c) {
  (c as D).m(a: 1);
}
{% endprettify %}

If the parameter should be added to the function, then add it:

{% prettify dart tag=pre+code %}
class C {
  m({int a, int b}) {}
}

void f(C c) {
  c.m(a: 1);
}
{% endprettify %}

### undefined_operator

_The operator '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when a user-definable operator is
invoked on an object for which the operator isn't defined.

#### Examples

The following code produces this diagnostic because the class `C` doesn't
define the operator `+`:

{% prettify dart tag=pre+code %}
class C {}

C f(C c) => c [!+!] 2;
{% endprettify %}

#### Common fixes

If the operator should be defined for the class, then define it:

{% prettify dart tag=pre+code %}
class C {
  C operator +(int i) => this;
}

C f(C c) => c + 2;
{% endprettify %}

### undefined_prefixed_name

_The name '{0}' is being referenced through the prefix '{1}', but it isn't
defined in any of the libraries imported using that prefix._

#### Description

The analyzer produces this diagnostic when a prefixed identifier is found
where the prefix is valid, but the identifier isn't declared in any of the
libraries imported using that prefix.

#### Examples

The following code produces this diagnostic because `dart:core` doesn't
define anything named `a`:

{% prettify dart tag=pre+code %}
import 'dart:core' as p;

void f() {
  p.[!a!];
}
{% endprettify %}

#### Common fixes

If the library in which the name is declared isn't imported yet, add an
import for the library.

If the name is wrong, then change it to one of the names that's declared in
the imported libraries.

### undefined_setter

_The setter '{0}' isn't defined for the type '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a setter but either isn't defined or isn't
visible in the scope in which the identifier is being referenced.

#### Examples

The following code produces this diagnostic because there isn't a setter
named `z`:

{% prettify dart tag=pre+code %}
class C {
  int x = 0;
  void m(int y) {
    this.[!z!] = y;
  }
}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a setter that is defined. The example above can be corrected by
fixing the spelling of the setter:

{% prettify dart tag=pre+code %}
class C {
  int x = 0;
  void m(int y) {
    this.x = y;
  }
}
{% endprettify %}

### undefined_shown_name

_The library '{0}' doesn't export a member with the shown name '{1}'._

#### Description

The analyzer produces this diagnostic when a show combinator includes a
name that isn't defined by the library being imported.

#### Examples

The following code produces this diagnostic because `dart:math` doesn't
define the name `String`:

{% prettify dart tag=pre+code %}
import 'dart:math' show min, [!String!];

var x = min(0, 1);
{% endprettify %}

#### Common fixes

If a different name should be shown, then correct the name. Otherwise,
remove the name from the list:

{% prettify dart tag=pre+code %}
import 'dart:math' show min;

var x = min(0, 1);
{% endprettify %}

### undefined_super_member

_The getter '{0}' isn't defined in a superclass of '{1}'._

_The method '{0}' isn't defined in a superclass of '{1}'._

_The operator '{0}' isn't defined in a superclass of '{1}'._

_The setter '{0}' isn't defined in a superclass of '{1}'._

#### Description

The analyzer produces this diagnostic when an inherited member (method,
getter, setter, or operator) is referenced using `super`, but there’s no
member with that name in the superclass chain.

#### Examples

The following code produces this diagnostic because `Object` doesn't define
a method named `n`:

{% prettify dart tag=pre+code %}
class C {
  void m() {
    super.[!n!]();
  }
}
{% endprettify %}

The following code produces this diagnostic because `Object` doesn't define
a getter named `g`:

{% prettify dart tag=pre+code %}
class C {
  void m() {
    super.[!g!];
  }
}
{% endprettify %}

#### Common fixes

If the inherited member you intend to invoke has a different name, then
make the name of the invoked member match the inherited member.

If the member you intend to invoke is defined in the same class, then
remove the `super.`.

If the member isn’t defined, then either add the member to one of the
superclasses or remove the invocation.

### unnecessary_cast

_Unnecessary cast._

#### Description

The analyzer produces this diagnostic when the value being cast is already
known to be of the type that it's being cast to.

#### Examples

The following code produces this diagnostic because `n` is already known to
be an `int` as a result of the `is` test:

{% prettify dart tag=pre+code %}
void f(num n) {
  if (n is int) {
    ([!n as int!]).isEven;
  }
}
{% endprettify %}

#### Common fixes

Remove the unnecessary cast:

{% prettify dart tag=pre+code %}
void f(num n) {
  if (n is int) {
    n.isEven;
  }
}
{% endprettify %}

### unnecessary_non_null_assertion

_The '!' will have no effect because the receiver can't be null._

#### Description

The analyzer produces this diagnostic when the operand of the `!` operator
can't be `null`.

#### Example

The following code produces this diagnostic because `x` can't be `null`:

{% prettify dart tag=pre+code %}
int f(int x) {
  return x[!!!];
}
{% endprettify %}

#### Common fixes

Remove the null check operator (`!`):

{% prettify dart tag=pre+code %}
int f(int x) {
  return x;
}
{% endprettify %}

### unnecessary_null_comparison

_The operand can't be null, so the condition is always false._

_The operand can't be null, so the condition is always true._

#### Description

The analyzer produces this diagnostic when it finds an equality comparison
(either `==` or `!=`) with one operand of `null` and the other operand
can't be `null`. Such comparisons are always either `true` or `false`, so
they serve no purpose.

#### Example

The following code produces this diagnostic because `x` can never be
`null`, so the comparison always evaluates to `true`:

{% prettify dart tag=pre+code %}
void f(int x) {
  if (x [!!= null!]) {
    print(x);
  }
}
{% endprettify %}

The following code produces this diagnostic because `x` can never be
`null`, so the comparison always evaluates to `false`:

{% prettify dart tag=pre+code %}
void f(int x) {
  if (x [!== null!]) {
    throw ArgumentError("x can't be null");
  }
}
{% endprettify %}

#### Common fixes

If the other operand should be able to be `null`, then change the type of
the operand:

{% prettify dart tag=pre+code %}
void f(int? x) {
  if (x != null) {
    print(x);
  }
}
{% endprettify %}

If the other operand really can't be `null`, then remove the condition:

{% prettify dart tag=pre+code %}
void f(int x) {
  print(x);
}
{% endprettify %}

### unnecessary_type_check

_Unnecessary type check; the result is always 'false'._

_Unnecessary type check; the result is always 'true'._

#### Description

The analyzer produces this diagnostic when the value of a type check (using
either `is` or `is!`) is known at compile time.

#### Example

The following code produces this diagnostic because the test `a is Object?`
is always `true`:

{% prettify dart tag=pre+code %}
bool f<T>(T a) => [!a is Object?!];
{% endprettify %}

#### Common fixes

If the type check doesn't check what you intended to check, then change the
test:

{% prettify dart tag=pre+code %}
bool f<T>(T a) => a is Object;
{% endprettify %}

If the type check does check what you intended to check, then replace the
type check with its known value or completely remove it:

{% prettify dart tag=pre+code %}
bool f<T>(T a) => true;
{% endprettify %}

### unqualified_reference_to_non_local_static_member

_Static members from supertypes must be qualified by the name of the defining
type._

#### Description

The analyzer produces this diagnostic when code in one class references a
static member in a superclass without prefixing the member's name with the
name of the superclass. Static members can only be referenced without a
prefix in the class in which they're declared.

#### Example

The following code produces this diagnostic because the static field `x` is
referenced in the getter `g` without prefixing it with the name of the
defining class:

{% prettify dart tag=pre+code %}
class A {
  static int x = 3;
}

class B extends A {
  int get g => [!x!];
}
{% endprettify %}

#### Common fixes

Prefix the name of the static member with the name of the declaring class:

{% prettify dart tag=pre+code %}
class A {
  static int x = 3;
}

class B extends A {
  int get g => A.x;
}
{% endprettify %}

### unqualified_reference_to_static_member_of_extended_type

_Static members from the extended type or one of its superclasses must be
qualified by the name of the defining type._

#### Description

The analyzer produces this diagnostic when an undefined name is found, and
the name is the same as a static member of the extended type or one of its
superclasses.

#### Examples

The following code produces this diagnostic because `m` is a static member
of the extended type `C`:

{% prettify dart tag=pre+code %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    [!m!]();
  }
}
{% endprettify %}

#### Common fixes

If you're trying to reference a static member that's declared outside the
extension, then add the name of the class or extension before the reference
to the member:

{% prettify dart tag=pre+code %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    C.m();
  }
}
{% endprettify %}

If you're referencing a member that isn't declared yet, add a declaration:

{% prettify dart tag=pre+code %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    m();
  }

  void m() {}
}
{% endprettify %}

### unused_catch_clause

_The exception variable '{0}' isn't used, so the 'catch' clause can be removed._

#### Description

The analyzer produces this diagnostic when a `catch` clause is found, and
neither the exception parameter nor the optional stack trace parameter are
used in the `catch` block.

#### Examples

The following code produces this diagnostic because `e` isn't referenced:

{% prettify dart tag=pre+code %}
void f() {
  try {
    int.parse(';');
  } on FormatException catch ([!e!]) {
    // ignored
  }
}
{% endprettify %}

#### Common fixes

Remove the unused `catch` clause:

{% prettify dart tag=pre+code %}
void f() {
  try {
    int.parse(';');
  } on FormatException {
    // ignored
  }
}
{% endprettify %}

### unused_catch_stack

_The stack trace variable '{0}' isn't used and can be removed._

#### Description

The analyzer produces this diagnostic when the stack trace parameter in a
`catch` clause isn't referenced within the body of the `catch` block.

#### Examples

The following code produces this diagnostic because `stackTrace` isn't
referenced:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } catch (exception, [!stackTrace!]) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

If you need to reference the stack trace parameter, then add a reference to
it. Otherwise, remove it:

{% prettify dart tag=pre+code %}
void f() {
  try {
    // ...
  } catch (exception) {
    // ...
  }
}
{% endprettify %}

### unused_element

_A value for optional parameter '{0}' isn't ever given._

_The declaration '{0}' isn't referenced._

#### Description

The analyzer produces this diagnostic when a private declaration isn't
referenced in the library that contains the declaration. The following
kinds of declarations are analyzed:
- Private top-level declarations, such as classes, enums, mixins, typedefs,
  top-level variables, and top-level functions
- Private static and instance methods
- Optional parameters of private functions for which a value is never
  passed, even when the parameter doesn't have a private name

#### Examples

Assuming that no code in the library references `_C`, the following code
produces this diagnostic:

{% prettify dart tag=pre+code %}
class [!_C!] {}
{% endprettify %}

Assuming that no code in the library passes a value for `y` in any
invocation of `_m`, the following code produces this diagnostic:

{% prettify dart tag=pre+code %}
class C {
  void _m(int x, [int [!y!]]) {}

  void n() => _m(0);
}
{% endprettify %}

#### Common fixes

If the declaration isn't needed, then remove it:

{% prettify dart tag=pre+code %}
class C {
  void _m(int x) {}

  void n() => _m(0);
}
{% endprettify %}

If the declaration is intended to be used, then add the code to use it.

### unused_field

_The value of the field '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a private field is declared but
never read, even if it's written in one or more places.

#### Examples

The following code produces this diagnostic because `_x` isn't referenced
anywhere in the library:

{% prettify dart tag=pre+code %}
class Point {
  int [!_x!];
}
{% endprettify %}

#### Common fixes

If the field isn't needed, then remove it.

If the field was intended to be used, then add the missing code.

### unused_import

_Unused import: '{0}'._

#### Description

The analyzer produces this diagnostic when an import isn't needed because
none of the names that are imported are referenced within the importing
library.

#### Examples

The following code produces this diagnostic because nothing defined in
`dart:async` is referenced in the library:

{% prettify dart tag=pre+code %}
import [!'dart:async'!];

void main() {}
{% endprettify %}

#### Common fixes

If the import isn't needed, then remove it.

If some of the imported names are intended to be used, then add the missing
code.

### unused_label

_The label '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a label that isn't used is
found.

#### Examples

The following code produces this diagnostic because the label `loop` isn't
referenced anywhere in the method:

{% prettify dart tag=pre+code %}
void f(int limit) {
  [!loop:!] for (int i = 0; i < limit; i++) {
    print(i);
  }
}
{% endprettify %}

#### Common fixes

If the label isn't needed, then remove it:

{% prettify dart tag=pre+code %}
void f(int limit) {
  for (int i = 0; i < limit; i++) {
    print(i);
  }
}
{% endprettify %}

If the label is needed, then use it:

{% prettify dart tag=pre+code %}
void f(int limit) {
  loop: for (int i = 0; i < limit; i++) {
    print(i);
    break loop;
  }
}
{% endprettify %}

### unused_local_variable

_The value of the local variable '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a local variable is declared but
never read, even if it's written in one or more places.

#### Examples

The following code produces this diagnostic because the value of `count` is
never read:

{% prettify dart tag=pre+code %}
void main() {
  int [!count!] = 0;
}
{% endprettify %}

#### Common fixes

If the variable isn't needed, then remove it.

If the variable was intended to be used, then add the missing code.

### unused_shown_name

_The name {0} is shown, but isn’t used._

#### Description

The analyzer produces this diagnostic when a show combinator includes a
name that isn't used within the library. Because it isn't referenced, the
name can be removed.

#### Examples

The following code produces this diagnostic because the function `max`
isn't used:

{% prettify dart tag=pre+code %}
import 'dart:math' show min, [!max!];

var x = min(0, 1);
{% endprettify %}

#### Common fixes

Either use the name or remove it:

{% prettify dart tag=pre+code %}
import 'dart:math' show min;

var x = min(0, 1);
{% endprettify %}

### uri_does_not_exist

_Target of URI doesn't exist: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist.

#### Examples

If the file `lib.dart` doesn't exist, the following code produces this
diagnostic:

{% prettify dart tag=pre+code %}
import [!'lib.dart'!];
{% endprettify %}

#### Common fixes

If the URI was mistyped or invalid, then correct the URI.

If the URI is correct, then create the file.

### uri_has_not_been_generated

_Target of URI hasn't been generated: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist and
the name of the file ends with a pattern that's commonly produced by code
generators, such as one of the following:
- `.g.dart`
- `.pb.dart`
- `.pbenum.dart`
- `.pbserver.dart`
- `.pbjson.dart`
- `.template.dart`

#### Examples

If the file `lib.g.dart` doesn't exist, the following code produces this
diagnostic:

{% prettify dart tag=pre+code %}
import [!'lib.g.dart'!];
{% endprettify %}

#### Common fixes

If the file is a generated file, then run the generator that generates the
file.

If the file isn't a generated file, then check the spelling of the URI or
create the file.

### uri_with_interpolation

_URIs can't use string interpolation._

#### Description

The analyzer produces this diagnostic when the string literal in an
`import`, `export`, or `part` directive contains an interpolation. The
resolution of the URIs in directives must happen before the declarations
are compiled, so expressions can’t be  evaluated  while determining the
values of the URIs.

#### Example

The following code produces this diagnostic because the string in the
`import` directive contains an interpolation:

{% prettify dart tag=pre+code %}
import [!'dart:$m'!];

const m = 'math';
{% endprettify %}

#### Common fixes

Remove the interpolation from the URI:

{% prettify dart tag=pre+code %}
import 'dart:math';

var zero = min(0, 0);
{% endprettify %}

### use_of_void_result

_This expression has a type of 'void' so its value can't be used._

#### Description

The analyzer produces this diagnostic when it finds an expression whose
type is `void`, and the expression is used in a place where a value is
expected, such as before a member access or on the right-hand side of an
assignment.

#### Examples

The following code produces this diagnostic because `f` doesn't produce an
object on which `toString` can be invoked:

{% prettify dart tag=pre+code %}
void f() {}

void g() {
  [!f()!].toString();
}
{% endprettify %}

#### Common fixes

Either rewrite the code so that the expression has a value or rewrite the
code so that it doesn't depend on the value.

### variable_type_mismatch

_A value of type '{0}' can't be assigned to a const variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the evaluation of a constant
expression would result in a `CastException`.

#### Examples

The following code produces this diagnostic because the value of `x` is an
`int`, which can't be assigned to `y` because an `int` isn't a `String`:

{% prettify dart tag=pre+code %}
const Object x = 0;
const String y = [!x!];
{% endprettify %}

#### Common fixes

If the declaration of the constant is correct, then change the value being
assigned to be of the correct type:

{% prettify dart tag=pre+code %}
const Object x = 0;
const String y = '$x';
{% endprettify %}

If the assigned value is correct, then change the declaration to have the
correct type:

{% prettify dart tag=pre+code %}
const Object x = 0;
const int y = x;
{% endprettify %}

### wrong_number_of_parameters_for_operator

_Operator '-' should declare 0 or 1 parameter, but {0} found._

_Operator '{0}' should declare exactly {1} parameters, but {2} found._

#### Description

The analyzer produces this diagnostic when a declaration of an operator has
the wrong number of parameters.

#### Examples

The following code produces this diagnostic because the operator `+` must
have a single parameter corresponding to the right operand:

{% prettify dart tag=pre+code %}
class C {
  int operator [!+!](a, b) => 0;
}
{% endprettify %}

#### Common fixes

Add or remove parameters to match the required number:

{% prettify dart tag=pre+code %}
class C {
  int operator +(a) => 0;
}
{% endprettify %}

### wrong_number_of_parameters_for_setter

_Setters must declare exactly one required positional parameter._

#### Description

The analyzer produces this diagnostic when a setter is found that doesn't
declare exactly one required positional parameter.

#### Examples

The following code produces this diagnostic because the setter `s` declares
two required parameters:

{% prettify dart tag=pre+code %}
class C {
  set [!s!](int x, int y) {}
}
{% endprettify %}

The following code produces this diagnostic because the setter `s` declares
one optional parameter:

{% prettify dart tag=pre+code %}
class C {
  set [!s!]([int x]) {}
}
{% endprettify %}

#### Common fixes

Change the declaration so that there's exactly one required positional
parameter:

{% prettify dart tag=pre+code %}
class C {
  set s(int x) {}
}
{% endprettify %}

### wrong_number_of_type_arguments

_The type '{0}' is declared with {1} type parameters, but {2} type arguments
were given._

#### Description

The analyzer produces this diagnostic when a type that has type parameters
is used and type arguments are provided, but the number of type arguments
isn't the same as the number of type parameters.

The analyzer also produces this diagnostic when a constructor is invoked
and the number of type arguments doesn't match the number of type
parameters declared for the class.

#### Examples

The following code produces this diagnostic because `C` has one type
parameter but two type arguments are provided when it is used as a type
annotation:

{% prettify dart tag=pre+code %}
class C<E> {}

void f([!C<int, int>!] x) {}
{% endprettify %}

The following code produces this diagnostic because `C` declares one type
parameter, but two type arguments are provided when creating an instance:

{% prettify dart tag=pre+code %}
class C<E> {}

var c = [!C<int, int>!]();
{% endprettify %}

#### Common fixes

Add or remove type arguments, as necessary, to match the number of type
parameters defined for the type:

{% prettify dart tag=pre+code %}
class C<E> {}

void f(C<int> x) {}
{% endprettify %}

### wrong_number_of_type_arguments_method

_The method '{0}' is declared with {1} type parameters, but {2} type arguments
are given._

#### Description

The analyzer produces this diagnostic when a method or function is invoked
with a different number of type arguments than the number of type
parameters specified in its declaration. There must either be no type
arguments or the number of arguments must match the number of parameters.

#### Example

The following code produces this diagnostic because the invocation of the
method `m` has two type arguments, but the declaration of `m` only has one
type parameter:

{% prettify dart tag=pre+code %}
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m[!<int, int>!](2);
{% endprettify %}

#### Common fixes

If the type arguments are necessary, then make them match the number of
type parameters by either adding or removing type arguments:

{% prettify dart tag=pre+code %}
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m<int>(2);
{% endprettify %}

If the type arguments aren't necessary, then remove them:

{% prettify dart tag=pre+code %}
class C {
  int m<A>(A a) => 0;
}

int f(C c) => c.m(2);
{% endprettify %}

### yield_of_invalid_type

_The type '{0}' implied by the 'yield' expression must be assignable to '{1}'._

#### Description

The analyzer produces this diagnostic when the type of object produced by a
`yield` expression doesn't match the type of objects that are to be
returned from the `Iterable` or `Stream` types that are returned from a
generator (a function or method marked with either `sync*` or `async*`).

#### Example

The following code produces this diagnostic because the getter `zero` is
declared to return an `Iterable` that returns integers, but the `yield` is
returning a string from the iterable:

{% prettify dart tag=pre+code %}
Iterable<int> get zero sync* {
  yield [!'0'!];
}
{% endprettify %}

#### Common fixes

If the return type of the function is correct, then fix the expression
following the keyword `yield` to return the correct type:

{% prettify dart tag=pre+code %}
Iterable<int> get zero sync* {
  yield 0;
}
{% endprettify %}

If the expression following the `yield` is correct, then change the return
type of the function to allow it:

{% prettify dart tag=pre+code %}
Iterable<String> get zero sync* {
  yield '0';
}
{% endprettify %}

### undefined_super_method

See [undefined_super_member](#undefined_super_member).
