// Copyright (c) 2020, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// Generates a file to test that all the Win32 API typedefs can be loaded (i.e.
// that lookupFunction works for all the APIs generated.)

import 'dart:io';

import 'package:winmd/winmd.dart';

import '../manual_gen/win32api.dart';
import 'generate_win32.dart';
import 'projection/typeprinter.dart';

const structFileHeader = '''
// Copyright (c) 2020, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// Dart representations of common structs used in the Win32 API.

// THIS FILE IS GENERATED AUTOMATICALLY AND SHOULD NOT BE EDITED DIRECTLY.

// -----------------------------------------------------------------------------
// Linter exceptions
// -----------------------------------------------------------------------------
// ignore_for_file: camel_case_types
// ignore_for_file: camel_case_extensions
//
// Why? The linter defaults to throw a warning for types not named as camel
// case. We deliberately break this convention to match the Win32 underlying
// types.
//
//
// ignore_for_file: unused_field
//
// Why? The linter complains about unused fields (e.g. a class that contains
// underscore-prefixed members that are not used in the library. In this class,
// we use this feature to ensure that sizeOf<STRUCT_NAME> returns a size at
// least as large as the underlying native struct. See, for example,
// ENUMLOGFONTEX.
//
//
// ignore_for_file: unnecessary_getters_setters
//
// Why? In structs like VARIANT, we're using getters and setters to project the
// same underlying data property to various union types. The trivial overhead is
// outweighed by readability.
//
//
// ignore_for_file: unused_import
//
// Why? We speculatively include some imports that we might generate a
// requirement for.
// -----------------------------------------------------------------------------

import 'dart:ffi';
import 'dart:typed_data';

import 'package:ffi/ffi.dart';

import 'callbacks.dart';
import 'com/IDispatch.dart';
import 'com/IUnknown.dart';
import 'combase.dart';
import 'oleaut32.dart';
import 'structs.dart';
''';

int generateStructs(Win32API win32) {
  final scope = MetadataStore.getWin32Scope();

  var structsGenerated = 0;
  final writer = File('lib/src/structs.g.dart').openSync(mode: FileMode.write);

  writer.writeStringSync(structFileHeader);

  for (final struct in win32.structs.keys) {
    final win32struct = win32.structs[struct]!;
    final typedef = scope.findTypeDef(win32struct.namespace);
    if (typedef == null) {
      throw Exception('$struct missing');
    }

    writer.writeStringSync(wrapCommentText(win32struct.comment));

    writer.writeStringSync('\n///\n/// {@category Struct}\n');
    final packingAlignment = typedef.classLayout.packingAlignment;
    if (packingAlignment != null && packingAlignment > 0) {
      writer.writeStringSync('@Packed($packingAlignment)\n');
    }

    writer.writeStringSync(TypePrinter.printStruct(typedef, struct));
    structsGenerated++;
  }

  return structsGenerated;
}
