// Copyright (c) 2019, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:analyzer/dart/analysis/results.dart';
import 'package:analyzer/dart/ast/ast.dart';
import 'package:analyzer/dart/element/element.dart';
import 'package:analyzer/dart/element/type.dart';
import 'package:analyzer/src/dart/element/element.dart';
import 'package:analyzer/src/test_utilities/mock_sdk.dart';
import 'package:test/test.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import '../dart/resolution/context_collection_resolution.dart';
import 'element_text.dart';
import 'elements_base.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(ElementsKeepLinkingTest);
    defineReflectiveTests(ElementsFromBytesTest);
    // defineReflectiveTests(ApplyCheckElementTextReplacements);
  });
}

@reflectiveTest
class ApplyCheckElementTextReplacements {
  test_applyReplacements() {
    applyCheckElementTextReplacements();
  }
}

@reflectiveTest
class ElementsFromBytesTest extends ElementsTest {
  @override
  bool get keepLinkingLibraries => false;
}

@reflectiveTest
class ElementsKeepLinkingTest extends ElementsTest {
  @override
  bool get keepLinkingLibraries => true;
}

abstract class ElementsTest extends ElementsBaseTest {
  @override
  List<MockSdkLibrary> additionalMockSdkLibraries = [];

  test_annotationArgument_recordLiteral() async {
    var library = await buildLibrary('''
@A((2, a: 3))
class C {}
class A {
  const A(o);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @20
        metadata
          Annotation
            atSign: @ @0
            name: SimpleIdentifier
              token: A @1
              staticElement: self::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @2
              arguments
                RecordLiteral
                  leftParenthesis: ( @3
                  fields
                    IntegerLiteral
                      literal: 2 @4
                      staticType: int
                    NamedExpression
                      name: Label
                        label: SimpleIdentifier
                          token: a @7
                          staticElement: <null>
                          staticType: null
                        colon: : @8
                      expression: IntegerLiteral
                        literal: 3 @10
                        staticType: int
                  rightParenthesis: ) @11
                  staticType: (int, {int a})
              rightParenthesis: ) @12
            element: self::@class::A::@constructor::new
        constructors
          synthetic @-1
      class A @31
        constructors
          const @43
            parameters
              requiredPositional o @45
                type: dynamic
''');
  }

  test_annotationArgument_recordLiteral_withConst() async {
    var library = await buildLibrary('''
@A(const ('',))
class C {}
class A {
  const A(o);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        metadata
          Annotation
            atSign: @ @0
            name: SimpleIdentifier
              token: A @1
              staticElement: self::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @2
              arguments
                RecordLiteral
                  constKeyword: const @3
                  leftParenthesis: ( @9
                  fields
                    SimpleStringLiteral
                      literal: '' @10
                  rightParenthesis: ) @13
                  staticType: (String)
              rightParenthesis: ) @14
            element: self::@class::A::@constructor::new
        constructors
          synthetic @-1
      class A @33
        constructors
          const @45
            parameters
              requiredPositional o @47
                type: dynamic
''');
  }

  test_augmentation_augmentationImports_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
import augment 'b.dart';
class A {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'a.dart';
class B {}
''');
    final library = await buildLibrary(r'''
import augment 'a.dart';
class C {}
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      augmentationImports
        package:test/b.dart
          definingUnit
            classes
              class B @32
                constructors
                  synthetic @-1
      definingUnit
        classes
          class A @60
            constructors
              synthetic @-1
  definingUnit
    classes
      class C @31
        constructors
          synthetic @-1
''');
  }

  test_augmentation_class_constructor_superConstructor_generic_named() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class B extends A<int> {
  B() : super.named(0);
}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
class A<T> {
  A.named(T a);
}
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class B @35
            supertype: A<int>
            constructors
              @56
                superConstructor: ConstructorMember
                  base: self::@class::A::@constructor::named
                  substitution: {T: int}
  definingUnit
    classes
      class A @31
        typeParameters
          covariant T @33
            defaultType: dynamic
        constructors
          named @42
            periodOffset: 41
            nameEnd: 47
            parameters
              requiredPositional a @50
                type: T
''');
  }

  test_augmentation_class_constructor_superConstructor_notGeneric_named() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class B extends A {
  B() : super.named();
}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
class A {
  A.named();
}
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class B @35
            supertype: A
            constructors
              @51
                superConstructor: self::@class::A::@constructor::named
  definingUnit
    classes
      class A @31
        constructors
          named @39
            periodOffset: 38
            nameEnd: 44
''');
  }

  test_augmentation_class_constructor_superConstructor_notGeneric_unnamed_explicit() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class B extends A {
  B() : super();
}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
class A {}
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class B @35
            supertype: A
            constructors
              @51
                superConstructor: self::@class::A::@constructor::new
  definingUnit
    classes
      class A @31
        constructors
          synthetic @-1
''');
  }

  test_augmentation_class_notSimplyBounded_circularity_via_typedef() async {
    // C's type parameter T is not simply bounded because its bound, F, expands
    // to `dynamic F(C)`, which refers to C.
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class C<T extends F> {}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
typedef F(C value);
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          notSimplyBounded class C @35
            typeParameters
              covariant T @37
                bound: dynamic
                defaultType: dynamic
            constructors
              synthetic @-1
  definingUnit
    typeAliases
      functionTypeAliasBased notSimplyBounded F @33
        aliasedType: dynamic Function(C<dynamic>)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional value @37
              type: C<dynamic>
          returnType: dynamic
''');
  }

  test_augmentation_class_notSimplyBounded_self() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class C<T extends C> {}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          notSimplyBounded class C @35
            typeParameters
              covariant T @37
                bound: C<dynamic>
                defaultType: dynamic
            constructors
              synthetic @-1
  definingUnit
''');
  }

  test_augmentation_defaultValue_class_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  static const a = 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
void f({int x = A.a}) {}
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        functions
          f @51
            parameters
              optionalNamed default x @58
                type: int
                constantInitializer
                  PrefixedIdentifier
                    prefix: SimpleIdentifier
                      token: A @62
                      staticElement: package:test/a.dart::@class::A
                      staticType: null
                    period: . @63
                    identifier: SimpleIdentifier
                      token: a @64
                      staticElement: package:test/a.dart::@class::A::@getter::a
                      staticType: int
                    staticElement: package:test/a.dart::@class::A::@getter::a
                    staticType: int
            returnType: void
  definingUnit
''');
  }

  test_augmentation_defaultValue_prefix_class_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  static const a = 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart' as prefix;
void f({int x = prefix.A.a}) {}
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart as prefix @48
      definingUnit
        functions
          f @61
            parameters
              optionalNamed default x @68
                type: int
                constantInitializer
                  PropertyAccess
                    target: PrefixedIdentifier
                      prefix: SimpleIdentifier
                        token: prefix @72
                        staticElement: self::@augmentation::package:test/b.dart::@prefix::prefix
                        staticType: null
                      period: . @78
                      identifier: SimpleIdentifier
                        token: A @79
                        staticElement: package:test/a.dart::@class::A
                        staticType: null
                      staticElement: package:test/a.dart::@class::A
                      staticType: null
                    operator: . @80
                    propertyName: SimpleIdentifier
                      token: a @81
                      staticElement: package:test/a.dart::@class::A::@getter::a
                      staticType: int
                    staticType: int
            returnType: void
  definingUnit
''');
  }

  test_augmentation_documented() async {
    newFile('$testPackageLibPath/a.dart', r'''
/// My documentation.
library augment 'test.dart';
''');
    final library = await buildLibrary(r'''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      documentationComment: /// My documentation.
      definingUnit
  definingUnit
''');
  }

  test_augmentation_importScope_constant() async {
    newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
const b = a;
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        topLevelVariables
          static const b @52
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              SimpleIdentifier
                token: a @56
                staticElement: package:test/a.dart::@getter::a
                staticType: int
        accessors
          synthetic static get b @-1
            returnType: int
  definingUnit
''');
  }

  test_augmentation_importScope_constant_class_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  static const a = 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
const b = A.a;
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        topLevelVariables
          static const b @52
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: A @56
                  staticElement: package:test/a.dart::@class::A
                  staticType: null
                period: . @57
                identifier: SimpleIdentifier
                  token: a @58
                  staticElement: package:test/a.dart::@class::A::@getter::a
                  staticType: int
                staticElement: package:test/a.dart::@class::A::@getter::a
                staticType: int
        accessors
          synthetic static get b @-1
            returnType: int
  definingUnit
''');
  }

  test_augmentation_importScope_constant_instanceCreation() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  const A {};
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
const a = A();
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        topLevelVariables
          static const a @52
            type: A
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: A @56
                    element: package:test/a.dart::@class::A
                    type: A
                  staticElement: package:test/a.dart::@class::A::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @57
                  rightParenthesis: ) @58
                staticType: A
        accessors
          synthetic static get a @-1
            returnType: A
  definingUnit
''');
  }

  test_augmentation_importScope_constant_prefix_class_field() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  static const a = 0;
}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart' as prefix;
const b = prefix.A.a;
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart as prefix @48
      definingUnit
        topLevelVariables
          static const b @62
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              PropertyAccess
                target: PrefixedIdentifier
                  prefix: SimpleIdentifier
                    token: prefix @66
                    staticElement: self::@augmentation::package:test/b.dart::@prefix::prefix
                    staticType: null
                  period: . @72
                  identifier: SimpleIdentifier
                    token: A @73
                    staticElement: package:test/a.dart::@class::A
                    staticType: null
                  staticElement: package:test/a.dart::@class::A
                  staticType: null
                operator: . @74
                propertyName: SimpleIdentifier
                  token: a @75
                  staticElement: package:test/a.dart::@class::A::@getter::a
                  staticType: int
                staticType: int
        accessors
          synthetic static get b @-1
            returnType: int
  definingUnit
''');
  }

  test_augmentation_importScope_prefixed() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart' as prefix;
prefix.A f() {}
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart as prefix @48
      definingUnit
        functions
          f @65
            returnType: A
  definingUnit
''');
  }

  test_augmentation_importScope_topInference() async {
    newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
final b = a;
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
''');

    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        topLevelVariables
          static final b @52
            type: int
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic static get b @-1
            returnType: int
  definingUnit
''');
  }

  test_augmentation_importScope_types_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'a.dart';
A f() {}
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
A f() {}
''');

    // The augmentation imports `a.dart`, so can resolve `A`.
    // But the library does not import, so there `A` is unresolved.
    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      imports
        package:test/a.dart
      definingUnit
        functions
          f @48
            returnType: A
  definingUnit
    functions
      f @27
        returnType: InvalidType
''');
  }

  test_augmentation_importScope_types_library() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
A f() {}
''');

    final library = await buildLibrary(r'''
import augment 'b.dart';
import 'a.dart';
A f() {}
''');

    // The library imports `a.dart`, so can resolve `A`.
    // But the augmentation does not import, so there `A` is unresolved.
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  augmentationImports
    package:test/b.dart
      definingUnit
        functions
          f @31
            returnType: InvalidType
  definingUnit
    functions
      f @44
        returnType: A
''');
  }

  test_augmentation_libraryExports_library() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
export 'dart:async';
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
export 'dart:collection';
export 'dart:math';
''');
    final library = await buildLibrary(r'''
import 'dart:io';
import augment 'a.dart';
import augment 'b.dart';
''');
    checkElementText(library, r'''
library
  imports
    dart:io
  augmentationImports
    package:test/a.dart
      exports
        dart:async
      definingUnit
    package:test/b.dart
      exports
        dart:collection
        dart:math
      definingUnit
  definingUnit
''');
  }

  test_augmentation_libraryImports_library() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
import 'dart:async';
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
import 'dart:collection';
import 'dart:math';
''');
    final library = await buildLibrary(r'''
import 'dart:io';
import augment 'a.dart';
import augment 'b.dart';
''');
    checkElementText(library, r'''
library
  imports
    dart:io
  augmentationImports
    package:test/a.dart
      imports
        dart:async
      definingUnit
    package:test/b.dart
      imports
        dart:collection
        dart:math
      definingUnit
  definingUnit
''');
  }

  test_augmentation_topScope_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class A {}
A f() {}
''');

    final library = await buildLibrary(r'''
import augment 'a.dart';
A f() {}
''');

    // The augmentation declares `A`, and can it be used in the library.
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class A @35
            constructors
              synthetic @-1
        functions
          f @42
            returnType: A
  definingUnit
    functions
      f @27
        returnType: A
''');
  }

  test_augmentation_topScope_library() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
A f() {}
''');

    final library = await buildLibrary(r'''
import augment 'a.dart';
class A {}
A f() {}
''');

    // The library declares `A`, and can it be used in the augmentation.
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        functions
          f @31
            returnType: A
  definingUnit
    classes
      class A @31
        constructors
          synthetic @-1
    functions
      f @38
        returnType: A
''');
  }

  test_class_abstract() async {
    var library = await buildLibrary('abstract class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        constructors
          synthetic @-1
''');
  }

  test_class_base() async {
    var library = await buildLibrary('base class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      base class C @11
        constructors
          synthetic @-1
''');
  }

  test_class_constructor_const() async {
    var library = await buildLibrary('class C { const C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @16
''');
  }

  test_class_constructor_const_external() async {
    var library = await buildLibrary('class C { external const C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          external const @25
''');
  }

  test_class_constructor_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  C();
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @34
            documentationComment: /**\n   * Docs\n   */
''');
  }

  test_class_constructor_explicit_named() async {
    var library = await buildLibrary('class C { C.foo(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          foo @12
            periodOffset: 11
            nameEnd: 15
''');
  }

  test_class_constructor_explicit_type_params() async {
    var library = await buildLibrary('class C<T, U> { C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          @16
''');
  }

  test_class_constructor_explicit_unnamed() async {
    var library = await buildLibrary('class C { C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @10
''');
  }

  test_class_constructor_external() async {
    var library = await buildLibrary('class C { external C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          external @19
''');
  }

  test_class_constructor_factory() async {
    var library = await buildLibrary('class C { factory C() => throw 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          factory @18
''');
  }

  test_class_constructor_field_formal_dynamic_dynamic() async {
    var library =
        await buildLibrary('class C { dynamic x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @18
            type: dynamic
            shouldUseTypeForInitializerInference: true
        constructors
          @21
            parameters
              requiredPositional final this.x @36
                type: dynamic
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_dynamic_typed() async {
    var library = await buildLibrary('class C { dynamic x; C(int this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @18
            type: dynamic
            shouldUseTypeForInitializerInference: true
        constructors
          @21
            parameters
              requiredPositional final this.x @32
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_dynamic_untyped() async {
    var library = await buildLibrary('class C { dynamic x; C(this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @18
            type: dynamic
            shouldUseTypeForInitializerInference: true
        constructors
          @21
            parameters
              requiredPositional final this.x @28
                type: dynamic
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_noReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(this.x(double b));
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @16
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @21
            parameters
              requiredPositional final this.x @28
                type: dynamic Function(double)
                parameters
                  requiredPositional b @37
                    type: double
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(int this.x(double b));
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @16
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @21
            parameters
              requiredPositional final this.x @32
                type: int Function(double)
                parameters
                  requiredPositional b @41
                    type: double
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType_generic() async {
    var library = await buildLibrary(r'''
class C {
  Function() f;
  C(List<U> this.f<T, U>(T t));
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          f @23
            type: dynamic Function()
            shouldUseTypeForInitializerInference: true
        constructors
          @28
            parameters
              requiredPositional final this.f @43
                type: List<U> Function<T, U>(T)
                typeParameters
                  covariant T @45
                  covariant U @48
                parameters
                  requiredPositional t @53
                    type: T
                field: self::@class::C::@field::f
        accessors
          synthetic get f @-1
            returnType: dynamic Function()
          synthetic set f @-1
            parameters
              requiredPositional _f @-1
                type: dynamic Function()
            returnType: void
''');
  }

  test_class_constructor_field_formal_multiple_matching_fields() async {
    // This is a compile-time error but it should still analyze consistently.
    var library = await buildLibrary('class C { C(this.x); int x; String x; }',
        allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @25
            type: int
            shouldUseTypeForInitializerInference: true
          x @35
            type: String
            shouldUseTypeForInitializerInference: true
        constructors
          @10
            parameters
              requiredPositional final this.x @17
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
          synthetic get x @-1
            returnType: String
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: String
            returnType: void
''');
  }

  test_class_constructor_field_formal_no_matching_field() async {
    // This is a compile-time error but it should still analyze consistently.
    var library =
        await buildLibrary('class C { C(this.x); }', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @10
            parameters
              requiredPositional final this.x @17
                type: dynamic
                field: <null>
''');
  }

  test_class_constructor_field_formal_typed_dynamic() async {
    var library = await buildLibrary('class C { num x; C(dynamic this.x); }',
        allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: num
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              requiredPositional final this.x @32
                type: dynamic
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: num
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: num
            returnType: void
''');
  }

  test_class_constructor_field_formal_typed_typed() async {
    var library = await buildLibrary('class C { num x; C(int this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: num
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              requiredPositional final this.x @28
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: num
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: num
            returnType: void
''');
  }

  test_class_constructor_field_formal_typed_untyped() async {
    var library = await buildLibrary('class C { num x; C(this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: num
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              requiredPositional final this.x @24
                type: num
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: num
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: num
            returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_dynamic() async {
    var library = await buildLibrary('class C { var x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @17
            parameters
              requiredPositional final this.x @32
                type: dynamic
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_typed() async {
    var library = await buildLibrary('class C { var x; C(int this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @17
            parameters
              requiredPositional final this.x @28
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_field_formal_untyped_untyped() async {
    var library = await buildLibrary('class C { var x; C(this.x); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @17
            parameters
              requiredPositional final this.x @24
                type: dynamic
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_constructor_fieldFormal_named_noDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x}); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              optionalNamed default final this.x @25
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_constructor_fieldFormal_named_withDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x: 42}); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              optionalNamed default final this.x @25
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 42 @28
                    staticType: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_constructor_fieldFormal_optional_noDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x]); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              optionalPositional default final this.x @25
                type: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_constructor_fieldFormal_optional_withDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x = 42]); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          @17
            parameters
              optionalPositional default final this.x @25
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 42 @29
                    staticType: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_constructor_implicit_type_params() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_constructor_initializers_assertInvocation() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
            parameters
              requiredPositional x @24
                type: int
            constantInitializers
              AssertInitializer
                assertKeyword: assert @29
                leftParenthesis: ( @35
                condition: BinaryExpression
                  leftOperand: SimpleIdentifier
                    token: x @36
                    staticElement: self::@class::C::@constructor::new::@parameter::x
                    staticType: int
                  operator: >= @38
                  rightOperand: IntegerLiteral
                    literal: 42 @41
                    staticType: int
                  staticElement: dart:core::@class::num::@method::>=
                  staticInvokeType: bool Function(num)
                  staticType: bool
                rightParenthesis: ) @43
''');
  }

  test_class_constructor_initializers_assertInvocation_message() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42, 'foo');
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
            parameters
              requiredPositional x @24
                type: int
            constantInitializers
              AssertInitializer
                assertKeyword: assert @29
                leftParenthesis: ( @35
                condition: BinaryExpression
                  leftOperand: SimpleIdentifier
                    token: x @36
                    staticElement: self::@class::C::@constructor::new::@parameter::x
                    staticType: int
                  operator: >= @38
                  rightOperand: IntegerLiteral
                    literal: 42 @41
                    staticType: int
                  staticElement: dart:core::@class::num::@method::>=
                  staticInvokeType: bool Function(num)
                  staticType: bool
                comma: , @43
                message: SimpleStringLiteral
                  literal: 'foo' @45
                rightParenthesis: ) @50
''');
  }

  test_class_constructor_initializers_field() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = 42;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @35
                  staticElement: self::@class::C::@field::x
                  staticType: null
                equals: = @37
                expression: IntegerLiteral
                  literal: 42 @39
                  staticType: int
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_class_constructor_initializers_field_notConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = foo();
}
int foo() => 42;
''', allowErrors: true);
    // It is OK to keep non-constant initializers.
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @35
                  staticElement: self::@class::C::@field::x
                  staticType: null
                equals: = @37
                expression: MethodInvocation
                  methodName: SimpleIdentifier
                    token: foo @39
                    staticElement: self::@function::foo
                    staticType: int Function()
                  argumentList: ArgumentList
                    leftParenthesis: ( @42
                    rightParenthesis: ) @43
                  staticInvokeType: int Function()
                  staticType: int
        accessors
          synthetic get x @-1
            returnType: dynamic
    functions
      foo @52
        returnType: int
''');
  }

  test_class_constructor_initializers_field_optionalPositionalParameter() async {
    var library = await buildLibrary('''
class A {
  final int _f;
  const A([int f = 0]) : _f = f;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _f @22
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          const @34
            parameters
              optionalPositional default f @41
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 0 @45
                    staticType: int
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: _f @51
                  staticElement: self::@class::A::@field::_f
                  staticType: null
                equals: = @54
                expression: SimpleIdentifier
                  token: f @56
                  staticElement: self::@class::A::@constructor::new::@parameter::f
                  staticType: int
        accessors
          synthetic get _f @-1
            returnType: int
''');
  }

  test_class_constructor_initializers_field_recordLiteral() async {
    var library = await buildLibrary('''
class C {
  final Object x;
  const C(int a) : x = (0, a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @25
            type: Object
            shouldUseTypeForInitializerInference: true
        constructors
          const @36
            parameters
              requiredPositional a @42
                type: int
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @47
                  staticElement: self::@class::C::@field::x
                  staticType: null
                equals: = @49
                expression: RecordLiteral
                  leftParenthesis: ( @51
                  fields
                    IntegerLiteral
                      literal: 0 @52
                      staticType: int
                    SimpleIdentifier
                      token: a @55
                      staticElement: self::@class::C::@constructor::new::@parameter::a
                      staticType: int
                  rightParenthesis: ) @56
                  staticType: (int, int)
        accessors
          synthetic get x @-1
            returnType: Object
''');
  }

  test_class_constructor_initializers_field_withParameter() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C(int p) : x = 1 + p;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            parameters
              requiredPositional p @35
                type: int
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @40
                  staticElement: self::@class::C::@field::x
                  staticType: null
                equals: = @42
                expression: BinaryExpression
                  leftOperand: IntegerLiteral
                    literal: 1 @44
                    staticType: int
                  operator: + @46
                  rightOperand: SimpleIdentifier
                    token: p @48
                    staticElement: self::@class::C::@constructor::new::@parameter::p
                    staticType: int
                  staticElement: dart:core::@class::num::@method::+
                  staticInvokeType: num Function(num)
                  staticType: int
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_class_constructor_initializers_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  const B(dynamic x);
  const B.f()
   : this(A<Function()>());
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class B @34
        constructors
          const @46
            parameters
              requiredPositional x @56
                type: dynamic
          const f @70
            periodOffset: 69
            nameEnd: 71
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @79
                argumentList: ArgumentList
                  leftParenthesis: ( @83
                  arguments
                    InstanceCreationExpression
                      constructorName: ConstructorName
                        type: NamedType
                          name: A @84
                          typeArguments: TypeArgumentList
                            leftBracket: < @85
                            arguments
                              GenericFunctionType
                                functionKeyword: Function @86
                                parameters: FormalParameterList
                                  leftParenthesis: ( @94
                                  rightParenthesis: ) @95
                                declaredElement: GenericFunctionTypeElement
                                  parameters
                                  returnType: dynamic
                                  type: dynamic Function()
                                type: dynamic Function()
                            rightBracket: > @96
                          element: self::@class::A
                          type: A<dynamic Function()>
                        staticElement: ConstructorMember
                          base: self::@class::A::@constructor::new
                          substitution: {T: dynamic Function()}
                      argumentList: ArgumentList
                        leftParenthesis: ( @97
                        rightParenthesis: ) @98
                      staticType: A<dynamic Function()>
                  rightParenthesis: ) @99
                staticElement: self::@class::B::@constructor::new
            redirectedConstructor: self::@class::B::@constructor::new
''');
  }

  test_class_constructor_initializers_superInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
}
class B extends A {
  const B() : super(const []);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional values @33
                type: List<String>
      class B @50
        supertype: A
        constructors
          const @72
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @78
                argumentList: ArgumentList
                  leftParenthesis: ( @83
                  arguments
                    ListLiteral
                      constKeyword: const @84
                      leftBracket: [ @90
                      rightBracket: ] @91
                      staticType: List<String>
                  rightParenthesis: ) @92
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_initializers_superInvocation_named() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(int p);
}
class C extends A {
  const C() : super.aaa(42);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const aaa @20
            periodOffset: 19
            nameEnd: 23
            parameters
              requiredPositional p @28
                type: int
      class C @40
        supertype: A
        constructors
          const @62
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @68
                period: . @73
                constructorName: SimpleIdentifier
                  token: aaa @74
                  staticElement: self::@class::A::@constructor::aaa
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @77
                  arguments
                    IntegerLiteral
                      literal: 42 @78
                      staticType: int
                  rightParenthesis: ) @80
                staticElement: self::@class::A::@constructor::aaa
            superConstructor: self::@class::A::@constructor::aaa
''');
  }

  test_class_constructor_initializers_superInvocation_named_underscore() async {
    var library = await buildLibrary('''
class A {
  const A._();
}
class B extends A {
  const B() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const _ @20
            periodOffset: 19
            nameEnd: 21
      class B @33
        supertype: A
        constructors
          const @55
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @61
                period: . @66
                constructorName: SimpleIdentifier
                  token: _ @67
                  staticElement: self::@class::A::@constructor::_
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @68
                  rightParenthesis: ) @69
                staticElement: self::@class::A::@constructor::_
            superConstructor: self::@class::A::@constructor::_
''');
  }

  test_class_constructor_initializers_superInvocation_namedExpression() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(a, {int b});
}
class C extends A {
  const C() : super.aaa(1, b: 2);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const aaa @20
            periodOffset: 19
            nameEnd: 23
            parameters
              requiredPositional a @24
                type: dynamic
              optionalNamed default b @32
                type: int
      class C @45
        supertype: A
        constructors
          const @67
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @73
                period: . @78
                constructorName: SimpleIdentifier
                  token: aaa @79
                  staticElement: self::@class::A::@constructor::aaa
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @82
                  arguments
                    IntegerLiteral
                      literal: 1 @83
                      staticType: int
                    NamedExpression
                      name: Label
                        label: SimpleIdentifier
                          token: b @86
                          staticElement: self::@class::A::@constructor::aaa::@parameter::b
                          staticType: null
                        colon: : @87
                      expression: IntegerLiteral
                        literal: 2 @89
                        staticType: int
                  rightParenthesis: ) @90
                staticElement: self::@class::A::@constructor::aaa
            superConstructor: self::@class::A::@constructor::aaa
''');
  }

  test_class_constructor_initializers_superInvocation_unnamed() async {
    var library = await buildLibrary('''
class A {
  const A(int p);
}
class C extends A {
  const C.ccc() : super(42);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional p @24
                type: int
      class C @36
        supertype: A
        constructors
          const ccc @60
            periodOffset: 59
            nameEnd: 63
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @68
                argumentList: ArgumentList
                  leftParenthesis: ( @73
                  arguments
                    IntegerLiteral
                      literal: 42 @74
                      staticType: int
                  rightParenthesis: ) @76
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_initializers_thisInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
  const A.empty() : this(const []);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional values @33
                type: List<String>
          const empty @52
            periodOffset: 51
            nameEnd: 57
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @62
                argumentList: ArgumentList
                  leftParenthesis: ( @66
                  arguments
                    ListLiteral
                      constKeyword: const @67
                      leftBracket: [ @73
                      rightBracket: ] @74
                      staticType: List<String>
                  rightParenthesis: ) @75
                staticElement: self::@class::A::@constructor::new
            redirectedConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_initializers_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, 'bbb');
  const C.named(int a, String b);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @24
                period: . @28
                constructorName: SimpleIdentifier
                  token: named @29
                  staticElement: self::@class::C::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @34
                  arguments
                    IntegerLiteral
                      literal: 1 @35
                      staticType: int
                    SimpleStringLiteral
                      literal: 'bbb' @38
                  rightParenthesis: ) @43
                staticElement: self::@class::C::@constructor::named
            redirectedConstructor: self::@class::C::@constructor::named
          const named @56
            periodOffset: 55
            nameEnd: 61
            parameters
              requiredPositional a @66
                type: int
              requiredPositional b @76
                type: String
''');
  }

  test_class_constructor_initializers_thisInvocation_namedExpression() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, b: 2);
  const C.named(a, {int b});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @24
                period: . @28
                constructorName: SimpleIdentifier
                  token: named @29
                  staticElement: self::@class::C::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @34
                  arguments
                    IntegerLiteral
                      literal: 1 @35
                      staticType: int
                    NamedExpression
                      name: Label
                        label: SimpleIdentifier
                          token: b @38
                          staticElement: self::@class::C::@constructor::named::@parameter::b
                          staticType: null
                        colon: : @39
                      expression: IntegerLiteral
                        literal: 2 @41
                        staticType: int
                  rightParenthesis: ) @42
                staticElement: self::@class::C::@constructor::named
            redirectedConstructor: self::@class::C::@constructor::named
          const named @55
            periodOffset: 54
            nameEnd: 60
            parameters
              requiredPositional a @61
                type: dynamic
              optionalNamed default b @69
                type: int
''');
  }

  test_class_constructor_initializers_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C.named() : this(1, 'bbb');
  const C(int a, String b);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const named @20
            periodOffset: 19
            nameEnd: 25
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @30
                argumentList: ArgumentList
                  leftParenthesis: ( @34
                  arguments
                    IntegerLiteral
                      literal: 1 @35
                      staticType: int
                    SimpleStringLiteral
                      literal: 'bbb' @38
                  rightParenthesis: ) @43
                staticElement: self::@class::C::@constructor::new
            redirectedConstructor: self::@class::C::@constructor::new
          const @54
            parameters
              requiredPositional a @60
                type: int
              requiredPositional b @70
                type: String
''');
  }

  test_class_constructor_parameters_super_explicitType_function() async {
    var library = await buildLibrary('''
class A {
  A(Object? a);
}

class B extends A {
  B(int super.a<T extends num>(T d)?);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @22
                type: Object?
      class B @35
        supertype: A
        constructors
          @51
            parameters
              requiredPositional final super.a @63
                type: int Function<T extends num>(T)?
                typeParameters
                  covariant T @65
                    bound: num
                parameters
                  requiredPositional d @82
                    type: T
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_explicitType_interface() async {
    var library = await buildLibrary('''
class A {
  A(num a);
}

class B extends A {
  B(int super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: num
      class B @31
        supertype: A
        constructors
          @47
            parameters
              requiredPositional final super.a @59
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_explicitType_interface_nullable() async {
    var library = await buildLibrary('''
class A {
  A(num? a);
}

class B extends A {
  B(int? super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @19
                type: num?
      class B @32
        supertype: A
        constructors
          @48
            parameters
              requiredPositional final super.a @61
                type: int?
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_invalid_topFunction() async {
    var library = await buildLibrary('''
void f(super.a) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional final super.a @13
            type: dynamic
            superConstructorParameter: <null>
        returnType: void
''');
  }

  test_class_constructor_parameters_super_optionalNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({String o1, super.a, String o2, super.b}) : super();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredNamed default a @28
                type: int
              requiredNamed default b @47
                type: double
      class B @61
        supertype: A
        constructors
          @77
            parameters
              optionalNamed default o1 @87
                type: String
              optionalNamed default final super.a @97
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
              optionalNamed default o2 @107
                type: String
              optionalNamed default final super.b @117
                type: double
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::b
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved() async {
    var library = await buildLibrary('''
class A {
  A({required int a});
}

class B extends A {
  B({super.b});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredNamed default a @28
                type: int
      class B @42
        supertype: A
        constructors
          @58
            parameters
              optionalNamed default final super.b @67
                type: dynamic
                superConstructorParameter: <null>
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A(int a);
}

class B extends A {
  B({super.a});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: int
      class B @31
        supertype: A
        constructors
          @47
            parameters
              optionalNamed default final super.a @56
                type: dynamic
                superConstructorParameter: <null>
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_optionalPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B([String o1, super.a, String o2, super.b]) : super();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: int
              requiredPositional b @28
                type: double
      class B @41
        supertype: A
        constructors
          @57
            parameters
              optionalPositional default o1 @67
                type: String
              optionalPositional default final super.a @77
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
              optionalPositional default o2 @87
                type: String
              optionalPositional default final super.b @97
                type: double
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::b
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({
    required String o1,
    required super.a,
    required String o2,
    required super.b,
  }) : super();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredNamed default a @28
                type: int
              requiredNamed default b @47
                type: double
      class B @61
        supertype: A
        constructors
          @77
            parameters
              requiredNamed default o1 @101
                type: String
              requiredNamed default final super.a @124
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
              requiredNamed default o2 @147
                type: String
              requiredNamed default final super.b @170
                type: double
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::b
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B(String o1, super.a, String o2, super.b) : super();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: int
              requiredPositional b @28
                type: double
      class B @41
        supertype: A
        constructors
          @57
            parameters
              requiredPositional o1 @66
                type: String
              requiredPositional final super.a @76
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
              requiredPositional o2 @86
                type: String
              requiredPositional final super.b @96
                type: double
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::b
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder() async {
    // It is important that `B` is declared after `C`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
abstract class A {
  A(int a);
}

class C extends B {
  C(super.a);
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        constructors
          @21
            parameters
              requiredPositional a @27
                type: int
      class C @40
        supertype: B
        constructors
          @56
            parameters
              requiredPositional final super.a @64
                type: int
                superConstructorParameter: self::@class::B::@constructor::new::@parameter::a
            superConstructor: self::@class::B::@constructor::new
      class B @77
        supertype: A
        constructors
          @93
            parameters
              requiredPositional final super.a @101
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder_generic() async {
    // It is important that `C` is declared before `B`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
class A {
  A(int a);
}

class C extends B<String> {
  C(super.a);
}

class B<T> extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: int
      class C @31
        supertype: B<String>
        constructors
          @55
            parameters
              requiredPositional final super.a @63
                type: int
                superConstructorParameter: SuperFormalParameterMember
                  base: self::@class::B::@constructor::new::@parameter::a
                  substitution: {T: String}
            superConstructor: ConstructorMember
              base: self::@class::B::@constructor::new
              substitution: {T: String}
      class B @76
        typeParameters
          covariant T @78
            defaultType: dynamic
        supertype: A
        constructors
          @95
            parameters
              requiredPositional final super.a @103
                type: int
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::a
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved() async {
    var library = await buildLibrary('''
class A {}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @18
        supertype: A
        constructors
          @34
            parameters
              requiredPositional final super.a @42
                type: dynamic
                superConstructorParameter: <null>
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A({required int a})
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredNamed default a @28
                type: int
      class B @41
        supertype: A
        constructors
          @57
            parameters
              requiredPositional final super.a @65
                type: dynamic
                superConstructorParameter: <null>
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_params() async {
    var library = await buildLibrary('class C { C(x, int y); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @10
            parameters
              requiredPositional x @12
                type: dynamic
              requiredPositional y @19
                type: int
''');
  }

  test_class_constructor_redirected_factory_named() async {
    var library = await buildLibrary('''
class C {
  factory C() = D.named;
  C._();
}
class D extends C {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          factory @20
            redirectedConstructor: self::@class::D::@constructor::named
          _ @39
            periodOffset: 38
            nameEnd: 40
      class D @52
        supertype: C
        constructors
          named @70
            periodOffset: 69
            nameEnd: 75
            superConstructor: self::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          factory @26
            redirectedConstructor: ConstructorMember
              base: self::@class::D::@constructor::named
              substitution: {T: U, U: T}
          _ @51
            periodOffset: 50
            nameEnd: 52
      class D @64
        typeParameters
          covariant T @66
            defaultType: dynamic
          covariant U @69
            defaultType: dynamic
        supertype: C<U, T>
        constructors
          named @94
            periodOffset: 93
            nameEnd: 99
            superConstructor: ConstructorMember
              base: self::@class::C::@constructor::_
              substitution: {T: U, U: T}
''');
  }

  test_class_constructor_redirected_factory_named_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>.named;
  B._();
}
class C<T, U> extends A<U, T> {
  C.named() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @33
        typeParameters
          covariant T @35
            defaultType: dynamic
          covariant U @38
            defaultType: dynamic
        constructors
          factory @53
            redirectedConstructor: ConstructorMember
              base: self::@class::C::@constructor::named
              substitution: {T: U, U: T}
          _ @78
            periodOffset: 77
            nameEnd: 79
      class C @91
        typeParameters
          covariant T @93
            defaultType: dynamic
          covariant U @96
            defaultType: dynamic
        supertype: C<U, T>
          alias: self::@typeAlias::A
            typeArguments
              U
              T
        constructors
          named @121
            periodOffset: 120
            nameEnd: 126
    typeAliases
      A @8
        typeParameters
          covariant T @10
            defaultType: dynamic
          covariant U @13
            defaultType: dynamic
        aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_named_imported() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class C @25
        constructors
          factory @39
            redirectedConstructor: package:test/foo.dart::@class::D::@constructor::named
          _ @58
            periodOffset: 57
            nameEnd: 59
''');
  }

  test_class_constructor_redirected_factory_named_imported_generic() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class C @25
        typeParameters
          covariant T @27
            defaultType: dynamic
          covariant U @30
            defaultType: dynamic
        constructors
          factory @45
            redirectedConstructor: ConstructorMember
              base: package:test/foo.dart::@class::D::@constructor::named
              substitution: {T: U, U: T}
          _ @70
            periodOffset: 69
            nameEnd: 71
''');
  }

  test_class_constructor_redirected_factory_named_prefixed() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @32
        constructors
          factory @46
            redirectedConstructor: package:test/foo.dart::@class::D::@constructor::named
          _ @69
            periodOffset: 68
            nameEnd: 70
''');
  }

  test_class_constructor_redirected_factory_named_prefixed_generic() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @32
        typeParameters
          covariant T @34
            defaultType: dynamic
          covariant U @37
            defaultType: dynamic
        constructors
          factory @52
            redirectedConstructor: ConstructorMember
              base: package:test/foo.dart::@class::D::@constructor::named
              substitution: {T: U, U: T}
          _ @81
            periodOffset: 80
            nameEnd: 82
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_class() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D.named<E>;
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant E @8
            defaultType: dynamic
        constructors
          factory @23
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_constructor() async {
    var library = await buildLibrary('''
class D {}
class C<E> {
  factory C() = D.named<E>;
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class D @6
        constructors
          synthetic @-1
      class C @17
        typeParameters
          covariant E @19
            defaultType: dynamic
        constructors
          factory @34
''');
  }

  test_class_constructor_redirected_factory_unnamed() async {
    var library = await buildLibrary('''
class C {
  factory C() = D;
  C._();
}
class D extends C {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          factory @20
            redirectedConstructor: self::@class::D::@constructor::new
          _ @33
            periodOffset: 32
            nameEnd: 34
      class D @46
        supertype: C
        constructors
          @62
            superConstructor: self::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          factory @26
            redirectedConstructor: ConstructorMember
              base: self::@class::D::@constructor::new
              substitution: {T: U, U: T}
          _ @45
            periodOffset: 44
            nameEnd: 46
      class D @58
        typeParameters
          covariant T @60
            defaultType: dynamic
          covariant U @63
            defaultType: dynamic
        supertype: C<U, T>
        constructors
          @86
            superConstructor: ConstructorMember
              base: self::@class::C::@constructor::_
              substitution: {T: U, U: T}
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>;
  B_();
}
class C<T, U> extends B<U, T> {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @33
        typeParameters
          covariant T @35
            defaultType: dynamic
          covariant U @38
            defaultType: dynamic
        constructors
          factory @53
            redirectedConstructor: ConstructorMember
              base: self::@class::C::@constructor::new
              substitution: {T: U, U: T}
        methods
          abstract B_ @70
            returnType: dynamic
      class C @84
        typeParameters
          covariant T @86
            defaultType: dynamic
          covariant U @89
            defaultType: dynamic
        supertype: B<U, T>
        constructors
          @112
    typeAliases
      A @8
        typeParameters
          covariant T @10
            defaultType: dynamic
          covariant U @13
            defaultType: dynamic
        aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class C @25
        constructors
          factory @39
            redirectedConstructor: package:test/foo.dart::@class::D::@constructor::new
          _ @52
            periodOffset: 51
            nameEnd: 53
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_generic() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class C @25
        typeParameters
          covariant T @27
            defaultType: dynamic
          covariant U @30
            defaultType: dynamic
        constructors
          factory @45
            redirectedConstructor: ConstructorMember
              base: package:test/foo.dart::@class::D::@constructor::new
              substitution: {T: U, U: T}
          _ @64
            periodOffset: 63
            nameEnd: 65
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_viaTypeAlias() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = A;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class C @25
        constructors
          factory @39
            redirectedConstructor: package:test/foo.dart::@class::B::@constructor::new
          _ @52
            periodOffset: 51
            nameEnd: 53
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @32
        constructors
          factory @46
            redirectedConstructor: package:test/foo.dart::@class::D::@constructor::new
          _ @63
            periodOffset: 62
            nameEnd: 64
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_generic() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @32
        typeParameters
          covariant T @34
            defaultType: dynamic
          covariant U @37
            defaultType: dynamic
        constructors
          factory @52
            redirectedConstructor: ConstructorMember
              base: package:test/foo.dart::@class::D::@constructor::new
              substitution: {T: U, U: T}
          _ @75
            periodOffset: 74
            nameEnd: 76
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_viaTypeAlias() async {
    addSource('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.A;
  C._();
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @32
        constructors
          factory @46
            redirectedConstructor: package:test/foo.dart::@class::B::@constructor::new
          _ @63
            periodOffset: 62
            nameEnd: 64
''');
  }

  test_class_constructor_redirected_factory_unnamed_unresolved() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D<E>;
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant E @8
            defaultType: dynamic
        constructors
          factory @23
''');
  }

  test_class_constructor_redirected_factory_unnamed_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A = C;
class B {
  factory B() = A;
  B._();
}
class C extends B {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @21
        constructors
          factory @35
            redirectedConstructor: self::@class::C::@constructor::new
          _ @48
            periodOffset: 47
            nameEnd: 49
      class C @61
        supertype: B
        constructors
          @77
            superConstructor: self::@class::B::@constructor::_
    typeAliases
      A @8
        aliasedType: C
''');
  }

  test_class_constructor_redirected_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const named @20
            periodOffset: 19
            nameEnd: 25
          const @37
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @43
                period: . @47
                constructorName: SimpleIdentifier
                  token: named @48
                  staticElement: self::@class::C::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @53
                  rightParenthesis: ) @54
                staticElement: self::@class::C::@constructor::named
            redirectedConstructor: self::@class::C::@constructor::named
''');
  }

  test_class_constructor_redirected_thisInvocation_named_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const named @23
            periodOffset: 22
            nameEnd: 28
          const @40
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @46
                period: . @50
                constructorName: SimpleIdentifier
                  token: named @51
                  staticElement: self::@class::C::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @56
                  rightParenthesis: ) @57
                staticElement: self::@class::C::@constructor::named
            redirectedConstructor: self::@class::C::@constructor::named
''');
  }

  test_class_constructor_redirected_thisInvocation_named_notConst() async {
    var library = await buildLibrary('''
class C {
  C.named();
  C() : this.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          named @14
            periodOffset: 13
            nameEnd: 19
          @25
            redirectedConstructor: self::@class::C::@constructor::named
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
          const named @33
            periodOffset: 32
            nameEnd: 38
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @43
                argumentList: ArgumentList
                  leftParenthesis: ( @47
                  rightParenthesis: ) @48
                staticElement: self::@class::C::@constructor::new
            redirectedConstructor: self::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
          const named @36
            periodOffset: 35
            nameEnd: 41
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @46
                argumentList: ArgumentList
                  leftParenthesis: ( @50
                  rightParenthesis: ) @51
                staticElement: self::@class::C::@constructor::new
            redirectedConstructor: self::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_notConst() async {
    var library = await buildLibrary('''
class C {
  C();
  C.named() : this();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @12
          named @21
            periodOffset: 20
            nameEnd: 26
            redirectedConstructor: self::@class::C::@constructor::new
''');
  }

  test_class_constructor_superConstructor_generic_named() async {
    var library = await buildLibrary('''
class A<T> {
  A.named(T a);
}
class B extends A<int> {
  B() : super.named(0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          named @17
            periodOffset: 16
            nameEnd: 22
            parameters
              requiredPositional a @25
                type: T
      class B @37
        supertype: A<int>
        constructors
          @58
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::named
              substitution: {T: int}
''');
  }

  test_class_constructor_superConstructor_notGeneric_named() async {
    var library = await buildLibrary('''
class A {
  A.named();
}
class B extends A {
  B() : super.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          named @14
            periodOffset: 13
            nameEnd: 19
      class B @31
        supertype: A
        constructors
          @47
            superConstructor: self::@class::A::@constructor::named
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_explicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B() : super();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        supertype: A
        constructors
          @33
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        supertype: A
        constructors
          @33
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit2() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_constructor_unnamed_implicit() async {
    var library = await buildLibrary('class C {}');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
            displayName: C
''');
  }

  test_class_constructor_withCycles_const() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = const D();
}
class D {
  final x;
  const D() : x = const C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @35
                  staticElement: self::@class::C::@field::x
                  staticType: null
                equals: = @37
                expression: InstanceCreationExpression
                  keyword: const @39
                  constructorName: ConstructorName
                    type: NamedType
                      name: D @45
                      element: self::@class::D
                      type: D
                    staticElement: self::@class::D::@constructor::new
                  argumentList: ArgumentList
                    leftParenthesis: ( @46
                    rightParenthesis: ) @47
                  staticType: D
        accessors
          synthetic get x @-1
            returnType: dynamic
      class D @58
        fields
          final x @70
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @81
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @87
                  staticElement: self::@class::D::@field::x
                  staticType: null
                equals: = @89
                expression: InstanceCreationExpression
                  keyword: const @91
                  constructorName: ConstructorName
                    type: NamedType
                      name: C @97
                      element: self::@class::C
                      type: C
                    staticElement: self::@class::C::@constructor::new
                  argumentList: ArgumentList
                    leftParenthesis: ( @98
                    rightParenthesis: ) @99
                  staticType: C
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_class_constructor_withCycles_nonConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  C() : x = new D();
}
class D {
  final x;
  D() : x = new C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @23
        accessors
          synthetic get x @-1
            returnType: dynamic
      class D @50
        fields
          final x @62
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @67
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_class_constructors_named() async {
    var library = await buildLibrary('''
class C {
  C.foo();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          foo @14
            displayName: C.foo
            periodOffset: 13
            nameEnd: 17
''');
  }

  test_class_constructors_unnamed() async {
    var library = await buildLibrary('''
class C {
  C();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @12
            displayName: C
''');
  }

  test_class_constructors_unnamed_new() async {
    var library = await buildLibrary('''
class C {
  C.new();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @14
            displayName: C
            periodOffset: 13
            nameEnd: 17
''');
  }

  test_class_documented() async {
    var library = await buildLibrary('''
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        documentationComment: /**\n * Docs\n */
        constructors
          synthetic @-1
''');
  }

  test_class_documented_mix() async {
    var library = await buildLibrary('''
/**
 * aaa
 */
/**
 * bbb
 */
class A {}

/**
 * aaa
 */
/// bbb
/// ccc
class B {}

/// aaa
/// bbb
/**
 * ccc
 */
class C {}

/// aaa
/// bbb
/**
 * ccc
 */
/// ddd
class D {}

/**
 * aaa
 */
// bbb
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @36
        documentationComment: /**\n * bbb\n */
        constructors
          synthetic @-1
      class B @79
        documentationComment: /// bbb\n/// ccc
        constructors
          synthetic @-1
      class C @122
        documentationComment: /**\n * ccc\n */
        constructors
          synthetic @-1
      class D @173
        documentationComment: /// ddd
        constructors
          synthetic @-1
      class E @207
        documentationComment: /**\n * aaa\n */
        constructors
          synthetic @-1
''');
  }

  test_class_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// first
/// second
/// third
class C {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @37
        documentationComment: /// first\n/// second\n/// third
        constructors
          synthetic @-1
''');
  }

  test_class_documented_with_references() async {
    var library = await buildLibrary('''
/**
 * Docs referring to [D] and [E]
 */
class C {}

class D {}
class E {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @47
        documentationComment: /**\n * Docs referring to [D] and [E]\n */
        constructors
          synthetic @-1
      class D @59
        constructors
          synthetic @-1
      class E @70
        constructors
          synthetic @-1
''');
  }

  test_class_documented_with_windows_line_endings() async {
    var library = await buildLibrary('/**\r\n * Docs\r\n */\r\nclass C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @25
        documentationComment: /**\n * Docs\n */
        constructors
          synthetic @-1
''');
  }

  test_class_documented_withLeadingNotDocumentation() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @66
        documentationComment: /**\n * Docs\n */
        constructors
          synthetic @-1
''');
  }

  test_class_documented_withMetadata() async {
    var library = await buildLibrary('''
/// Comment 1
/// Comment 2
@Annotation()
class BeforeMeta {}

/// Comment 1
/// Comment 2
@Annotation.named()
class BeforeMetaNamed {}

@Annotation()
/// Comment 1
/// Comment 2
class AfterMeta {}

/// Comment 1
@Annotation()
/// Comment 2
class AroundMeta {}

/// Doc comment.
@Annotation()
// Not doc comment.
class DocBeforeMetaNotDocAfter {}

class Annotation {
  const Annotation();
  const Annotation.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class BeforeMeta @48
        documentationComment: /// Comment 1\n/// Comment 2
        metadata
          Annotation
            atSign: @ @28
            name: SimpleIdentifier
              token: Annotation @29
              staticElement: self::@class::Annotation
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @39
              rightParenthesis: ) @40
            element: self::@class::Annotation::@constructor::new
        constructors
          synthetic @-1
      class BeforeMetaNamed @117
        documentationComment: /// Comment 1\n/// Comment 2
        metadata
          Annotation
            atSign: @ @91
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: Annotation @92
                staticElement: self::@class::Annotation
                staticType: null
              period: . @102
              identifier: SimpleIdentifier
                token: named @103
                staticElement: self::@class::Annotation::@constructor::named
                staticType: null
              staticElement: self::@class::Annotation::@constructor::named
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @108
              rightParenthesis: ) @109
            element: self::@class::Annotation::@constructor::named
        constructors
          synthetic @-1
      class AfterMeta @185
        documentationComment: /// Comment 1\n/// Comment 2
        metadata
          Annotation
            atSign: @ @137
            name: SimpleIdentifier
              token: Annotation @138
              staticElement: self::@class::Annotation
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @148
              rightParenthesis: ) @149
            element: self::@class::Annotation::@constructor::new
        constructors
          synthetic @-1
      class AroundMeta @247
        documentationComment: /// Comment 2
        metadata
          Annotation
            atSign: @ @213
            name: SimpleIdentifier
              token: Annotation @214
              staticElement: self::@class::Annotation
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @224
              rightParenthesis: ) @225
            element: self::@class::Annotation::@constructor::new
        constructors
          synthetic @-1
      class DocBeforeMetaNotDocAfter @319
        documentationComment: /// Doc comment.
        metadata
          Annotation
            atSign: @ @279
            name: SimpleIdentifier
              token: Annotation @280
              staticElement: self::@class::Annotation
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @290
              rightParenthesis: ) @291
            element: self::@class::Annotation::@constructor::new
        constructors
          synthetic @-1
      class Annotation @354
        constructors
          const @375
          const named @408
            periodOffset: 407
            nameEnd: 413
''');
  }

  test_class_field_abstract() async {
    var library = await buildLibrary('''
abstract class C {
  abstract int i;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        fields
          abstract i @34
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic abstract get i @-1
            returnType: int
          synthetic abstract set i @-1
            parameters
              requiredPositional _i @-1
                type: int
            returnType: void
''');
  }

  test_class_field_const() async {
    var library = await buildLibrary('class C { static const int i = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const i @27
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 0 @31
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get i @-1
            returnType: int
''');
  }

  test_class_field_const_late() async {
    var library =
        await buildLibrary('class C { static late const int i = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static late const i @32
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 0 @36
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get i @-1
            returnType: int
''');
  }

  test_class_field_covariant() async {
    var library = await buildLibrary('''
class C {
  covariant int x;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          covariant x @26
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional covariant _x @-1
                type: int
            returnType: void
''');
  }

  test_class_field_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  var x;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @38
            documentationComment: /**\n   * Docs\n   */
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_field_duplicate_getter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  int get foo => 0;
}
''');
    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
            id: field_0
            getter: getter_0
            setter: setter_0
          synthetic foo @-1
            type: int
            id: field_1
            getter: getter_1
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
            id: getter_0
            variable: field_0
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
            id: setter_0
            variable: field_0
          get foo @35
            returnType: int
            id: getter_1
            variable: field_1
''');
  }

  test_class_field_duplicate_setter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  set foo(int _) {}
}
''');
    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
            id: field_0
            getter: getter_0
            setter: setter_0
          synthetic foo @-1
            type: int
            id: field_1
            setter: setter_1
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
            id: getter_0
            variable: field_0
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
            id: setter_0
            variable: field_0
          set foo @31
            parameters
              requiredPositional _ @39
                type: int
            returnType: void
            id: setter_1
            variable: field_1
''');
  }

  test_class_field_external() async {
    var library = await buildLibrary('''
abstract class C {
  external int i;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        fields
          external i @34
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get i @-1
            returnType: int
          synthetic set i @-1
            parameters
              requiredPositional _i @-1
                type: int
            returnType: void
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
  const C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 42 @22
                staticType: int
        constructors
          const @34
        accessors
          synthetic get x @-1
            returnType: int
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  final f = const A<int Function(double a)>();
  const B();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class B @34
        fields
          final f @46
            type: A<int Function(double)>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                keyword: const @50
                constructorName: ConstructorName
                  type: NamedType
                    name: A @56
                    typeArguments: TypeArgumentList
                      leftBracket: < @57
                      arguments
                        GenericFunctionType
                          returnType: NamedType
                            name: int @58
                            element: dart:core::@class::int
                            type: int
                          functionKeyword: Function @62
                          parameters: FormalParameterList
                            leftParenthesis: ( @70
                            parameter: SimpleFormalParameter
                              type: NamedType
                                name: double @71
                                element: dart:core::@class::double
                                type: double
                              name: a @78
                              declaredElement: a@78
                                type: double
                            rightParenthesis: ) @79
                          declaredElement: GenericFunctionTypeElement
                            parameters
                              a
                                kind: required positional
                                type: double
                            returnType: int
                            type: int Function(double)
                          type: int Function(double)
                      rightBracket: > @80
                    element: self::@class::A
                    type: A<int Function(double)>
                  staticElement: ConstructorMember
                    base: self::@class::A::@constructor::new
                    substitution: {T: int Function(double)}
                argumentList: ArgumentList
                  leftParenthesis: ( @81
                  rightParenthesis: ) @82
                staticType: A<int Function(double)>
        constructors
          const @93
        accessors
          synthetic get f @-1
            returnType: A<int Function(double)>
''');
  }

  test_class_field_final_hasInitializer_noConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
''');
  }

  test_class_field_final_withSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int foo;
  A(this.foo);
  set foo(int newValue) {}
}
''');
    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final foo @22
            type: int
            shouldUseTypeForInitializerInference: true
            id: field_0
            getter: getter_0
            setter: setter_0
        constructors
          @29
            parameters
              requiredPositional final this.foo @36
                type: int
                field: self::@class::A::@field::foo
        accessors
          synthetic get foo @-1
            returnType: int
            id: getter_0
            variable: field_0
          set foo @48
            parameters
              requiredPositional newValue @56
                type: int
            returnType: void
            id: setter_0
            variable: field_0
''');
  }

  test_class_field_formal_param_inferred_type_implicit() async {
    var library = await buildLibrary('class C extends D { var v; C(this.v); }'
        ' abstract class D { int get v; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        fields
          v @24
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          @27
            parameters
              requiredPositional final this.v @34
                type: int
                field: self::@class::C::@field::v
            superConstructor: self::@class::D::@constructor::new
        accessors
          synthetic get v @-1
            returnType: int
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: int
            returnType: void
      abstract class D @55
        fields
          synthetic v @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract get v @67
            returnType: int
''');
  }

  test_class_field_implicit_type() async {
    var library = await buildLibrary('class C { var x; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_field_implicit_type_late() async {
    var library = await buildLibrary('class C { late var x; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          late x @19
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_explicit_initialized() async {
    var library = await buildLibrary('class C { num v = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          v @14
            type: num
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get v @-1
            returnType: num
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: num
            returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_initialized() async {
    var library = await buildLibrary('class C { var v = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          v @14
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get v @-1
            returnType: int
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: int
            returnType: void
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_uninitialized() async {
    var library = await buildLibrary(
        'class C extends D { var v; } abstract class D { int get v; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        fields
          v @24
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        accessors
          synthetic get v @-1
            returnType: int
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: int
            returnType: void
      abstract class D @44
        fields
          synthetic v @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract get v @56
            returnType: int
''');
  }

  test_class_field_inferred_type_nonStatic_inherited_resolveInitializer() async {
    var library = await buildLibrary(r'''
const a = 0;
abstract class A {
  const A();
  List<int> get f;
}
class B extends A {
  const B();
  final f = [a];
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @28
        fields
          synthetic f @-1
            type: List<int>
        constructors
          const @40
        accessors
          abstract get f @61
            returnType: List<int>
      class B @72
        supertype: A
        fields
          final f @107
            type: List<int>
            shouldUseTypeForInitializerInference: true
            constantInitializer
              ListLiteral
                leftBracket: [ @111
                elements
                  SimpleIdentifier
                    token: a @112
                    staticElement: self::@getter::a
                    staticType: int
                rightBracket: ] @113
                staticType: List<int>
        constructors
          const @94
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get f @-1
            returnType: List<int>
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_class_field_inferred_type_static_implicit_initialized() async {
    var library = await buildLibrary('class C { static var v = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static v @21
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic static get v @-1
            returnType: int
          synthetic static set v @-1
            parameters
              requiredPositional _v @-1
                type: int
            returnType: void
''');
  }

  test_class_field_inheritedContextType_double() async {
    var library = await buildLibrary('''
abstract class A {
  const A();
  double get foo;
}
class B extends A {
  const B();
  final foo = 2;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        fields
          synthetic foo @-1
            type: double
        constructors
          const @27
        accessors
          abstract get foo @45
            returnType: double
      class B @58
        supertype: A
        fields
          final foo @93
            type: double
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 2 @99
                staticType: double
        constructors
          const @80
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get foo @-1
            returnType: double
''');
  }

  test_class_field_isPromotable_hasGetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasGetter_abstract() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasGetter_inPart() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @21
        fields
          final _foo @38
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasGetter_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasNotFinalField() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasNotFinalField_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_hasSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  set _field(int? _) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_language217() async {
    var library = await buildLibrary(r'''
// @dart = 2.19
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        fields
          final _foo @39
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInMixin() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B {
  final int? _foo = 0;
}

/// `_foo` is implemented in [M].
class C with M implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
      class B @90
        fields
          final promotable _foo @107
            type: int?
            shouldUseTypeForInitializerInference: true
    mixins
      mixin M @54
        superclassConstraints
          Object
        fields
          final promotable _foo @71
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInSuperclass() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

class C {
  final int? _foo = 0;
}

/// `_foo` is implemented in [B].
class D extends B implements C {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
      class B @54
        fields
          final promotable _foo @71
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inClassTypeAlias() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

mixin M {
  dynamic noSuchMethod(Invocation invocation) {}
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class E = Object with M implements B;
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inEnum() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
enum E implements B {
  v;
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
      class B @54
        fields
          final _foo @71
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_getter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inDifferentLibrary() async {
    newFile('$testPackageLibPath/a.dart', r'''
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
import 'a.dart';

class A {
  final int? _foo;
  A(this._foo);
}

/// Has a noSuchMethod thrower for B._field, but since private names in
/// different libraries are distinct, this has no effect on promotion of
/// C._field.
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class A @24
        fields
          final promotable _foo @41
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inheritedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B extends A {
  A(super.value);
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_mixedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B with M {}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
    mixins
      mixin M @54
        superclassConstraints
          Object
        fields
          final _foo @71
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_unusedMixin() async {
    // Mixins are implicitly abstract so the presence of a mixin that inherits
    // a field into its interface, and doesn't implement it, doesn't mean that
    // a noSuchMethod forwarder created for it. So,  this does not block that
    // field from promoting.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M implements A {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_notFinal() async {
    var library = await buildLibrary(r'''
class A {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          _foo @17
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_notPrivate() async {
    var library = await buildLibrary(r'''
class A {
  int? field;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          field @17
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_class_field_isPromotable_typeInference() async {
    // We decide that `_foo` is promotable before inferring the type of `bar`.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  final bar = _foo != null ? _foo : 0;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final promotable _foo @23
            type: int?
            shouldUseTypeForInitializerInference: true
          final bar @37
            type: int
            shouldUseTypeForInitializerInference: false
''');
  }

  test_class_field_propagatedType_const_noDep() async {
    var library = await buildLibrary('''
class C {
  static const x = 0;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const x @25
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 0 @29
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get x @-1
            returnType: int
''');
  }

  test_class_field_propagatedType_final_dep_inLib() async {
    addSource('$testPackageLibPath/a.dart', 'final a = 1;');
    var library = await buildLibrary('''
import "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class C @23
        fields
          final b @35
            type: double
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get b @-1
            returnType: double
''');
  }

  test_class_field_propagatedType_final_dep_inPart() async {
    addSource('$testPackageLibPath/a.dart', 'part of lib; final a = 1;');
    var library = await buildLibrary('''
library lib;
part "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  name: lib
  nameOffset: 8
  definingUnit
    classes
      class C @34
        fields
          final b @46
            type: double
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get b @-1
            returnType: double
  parts
    package:test/a.dart
      topLevelVariables
        static final a @19
          type: int
          shouldUseTypeForInitializerInference: false
      accessors
        synthetic static get a @-1
          returnType: int
''');
  }

  test_class_field_propagatedType_final_noDep_instance() async {
    var library = await buildLibrary('''
class C {
  final x = 0;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
''');
  }

  test_class_field_propagatedType_final_noDep_static() async {
    var library = await buildLibrary('''
class C {
  static final x = 0;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static final x @25
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic static get x @-1
            returnType: int
''');
  }

  test_class_field_static() async {
    var library = await buildLibrary('class C { static int i; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static i @21
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic static get i @-1
            returnType: int
          synthetic static set i @-1
            parameters
              requiredPositional _i @-1
                type: int
            returnType: void
''');
  }

  test_class_field_static_final_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  static final f = 0;
  const C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static final f @25
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          const @40
        accessors
          synthetic static get f @-1
            returnType: int
''');
  }

  test_class_field_static_final_untyped() async {
    var library = await buildLibrary('class C { static final x = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static final x @23
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic static get x @-1
            returnType: int
''');
  }

  test_class_field_static_late() async {
    var library = await buildLibrary('class C { static late int i; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static late i @26
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic static get i @-1
            returnType: int
          synthetic static set i @-1
            parameters
              requiredPositional _i @-1
                type: int
            returnType: void
''');
  }

  test_class_field_type_explicit() async {
    var library = await buildLibrary(r'''
class C {
  int a = 0;
}
''');

    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          a @16
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get a @-1
            returnType: int
          synthetic set a @-1
            parameters
              requiredPositional _a @-1
                type: int
            returnType: void
''');
  }

  test_class_field_type_inferred_fromInitializer() async {
    var library = await buildLibrary(r'''
class C {
  var foo = 0;
}
''');

    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
''');
  }

  test_class_field_type_inferred_fromSuper() async {
    var library = await buildLibrary(r'''
abstract class A {
  int get foo;
}

class B extends A {
  final foo = 0;
}
''');

    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract get foo @29
            returnType: int
      class B @43
        supertype: A
        fields
          final foo @65
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get foo @-1
            returnType: int
''');
  }

  test_class_field_type_inferred_Never() async {
    var library = await buildLibrary(r'''
class C {
  var a = throw 42;
}
''');

    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          a @16
            type: Never
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get a @-1
            returnType: Never
          synthetic set a @-1
            parameters
              requiredPositional _a @-1
                type: Never
            returnType: void
''');
  }

  test_class_field_type_inferred_nonNullify() async {
    addSource('$testPackageLibPath/a.dart', '''
// @dart = 2.7
var a = 0;
''');

    var library = await buildLibrary(r'''
import 'a.dart';
class C {
  var b = a;
}
''');

    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class C @23
        fields
          b @33
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get b @-1
            returnType: int
          synthetic set b @-1
            parameters
              requiredPositional _b @-1
                type: int
            returnType: void
''');
  }

  test_class_field_typed() async {
    var library = await buildLibrary('class C { int x = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_field_untyped() async {
    var library = await buildLibrary('class C { var x = 0; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          x @14
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
''');
  }

  test_class_fields() async {
    var library = await buildLibrary('class C { int i; int j; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          i @14
            type: int
            shouldUseTypeForInitializerInference: true
          j @21
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get i @-1
            returnType: int
          synthetic set i @-1
            parameters
              requiredPositional _i @-1
                type: int
            returnType: void
          synthetic get j @-1
            returnType: int
          synthetic set j @-1
            parameters
              requiredPositional _j @-1
                type: int
            returnType: void
''');
  }

  test_class_fields_late() async {
    var library = await buildLibrary('''
class C {
  late int foo;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          late foo @21
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
''');
  }

  test_class_fields_late_final() async {
    var library = await buildLibrary('''
class C {
  late final int foo;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          late final foo @27
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
''');
  }

  test_class_fields_late_final_initialized() async {
    var library = await buildLibrary('''
class C {
  late final int foo = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          late final foo @27
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
''');
  }

  test_class_fields_late_inference_usingSuper_methodInvocation() async {
    var library = await buildLibrary('''
class A {
  int foo() => 0;
}

class B extends A {
  late var f = super.foo();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          foo @16
            returnType: int
      class B @37
        supertype: A
        fields
          late f @62
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get f @-1
            returnType: int
          synthetic set f @-1
            parameters
              requiredPositional _f @-1
                type: int
            returnType: void
''');
  }

  test_class_fields_late_inference_usingSuper_propertyAccess() async {
    var library = await buildLibrary('''
class A {
  int get foo => 0;
}

class B extends A {
  late var f = super.foo;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get foo @20
            returnType: int
      class B @39
        supertype: A
        fields
          late f @64
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get f @-1
            returnType: int
          synthetic set f @-1
            parameters
              requiredPositional _f @-1
                type: int
            returnType: void
''');
  }

  test_class_final() async {
    var library = await buildLibrary('final class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      final class C @12
        constructors
          synthetic @-1
''');
  }

  test_class_getter_abstract() async {
    var library = await buildLibrary('abstract class C { int get x; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract get x @27
            returnType: int
''');
  }

  test_class_getter_external() async {
    var library = await buildLibrary('class C { external int get x; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          external get x @27
            returnType: int
''');
  }

  test_class_getter_implicit_return_type() async {
    var library = await buildLibrary('class C { get x => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          get x @14
            returnType: dynamic
''');
  }

  test_class_getter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get foo @20 invokesSuperSelf
            returnType: int
''');
  }

  test_class_getter_invokesSuperSelf_getter_nestedInAssignment() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    (super.foo).foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get foo @20 invokesSuperSelf
            returnType: int
''');
  }

  test_class_getter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get foo @20
            returnType: int
''');
  }

  test_class_getter_native() async {
    var library = await buildLibrary('''
class C {
  int get x() native;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          external get x @20
            returnType: int
''');
  }

  test_class_getter_static() async {
    var library = await buildLibrary('class C { static int get x => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic static x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          static get x @25
            returnType: int
''');
  }

  test_class_getters() async {
    var library =
        await buildLibrary('class C { int get x => null; get y => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
          synthetic y @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          get x @18
            returnType: int
          get y @33
            returnType: dynamic
''');
  }

  test_class_implicitField_getterFirst() async {
    var library = await buildLibrary('''
class C {
  int get x => 0;
  void set x(int value) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get x @20
            returnType: int
          set x @39
            parameters
              requiredPositional value @45
                type: int
            returnType: void
''');
  }

  test_class_implicitField_setterFirst() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  int get x => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set x @21
            parameters
              requiredPositional value @27
                type: int
            returnType: void
          get x @47
            returnType: int
''');
  }

  test_class_interface() async {
    var library = await buildLibrary('interface class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      interface class C @16
        constructors
          synthetic @-1
''');
  }

  test_class_interfaces() async {
    var library = await buildLibrary('''
class C implements D, E {}
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        interfaces
          D
          E
        constructors
          synthetic @-1
      class D @33
        constructors
          synthetic @-1
      class E @44
        constructors
          synthetic @-1
''');
  }

  test_class_interfaces_Function() async {
    var library = await buildLibrary('''
class A {}
class B {}
class C implements A, Function, B {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        constructors
          synthetic @-1
      class C @28
        interfaces
          A
          B
        constructors
          synthetic @-1
''');
  }

  test_class_interfaces_unresolved() async {
    var library = await buildLibrary(
        'class C implements X, Y, Z {} class X {} class Z {}',
        allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        interfaces
          X
          Z
        constructors
          synthetic @-1
      class X @36
        constructors
          synthetic @-1
      class Z @47
        constructors
          synthetic @-1
''');
  }

  test_class_method_abstract() async {
    var library = await buildLibrary('abstract class C { f(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        constructors
          synthetic @-1
        methods
          abstract f @19
            returnType: dynamic
''');
  }

  test_class_method_async() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Future f() async {}
}
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    classes
      class C @27
        constructors
          synthetic @-1
        methods
          f @40 async
            returnType: Future<dynamic>
''');
  }

  test_class_method_asyncStar() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Stream f() async* {}
}
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    classes
      class C @27
        constructors
          synthetic @-1
        methods
          f @40 async*
            returnType: Stream<dynamic>
''');
  }

  test_class_method_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  f() {}
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @34
            documentationComment: /**\n   * Docs\n   */
            returnType: dynamic
''');
  }

  test_class_method_external() async {
    var library = await buildLibrary('class C { external f(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          external f @19
            returnType: dynamic
''');
  }

  test_class_method_hasImplicitReturnType_false() async {
    var library = await buildLibrary('''
class C {
  int m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isFalse);
  }

  test_class_method_hasImplicitReturnType_true() async {
    var library = await buildLibrary('''
class C {
  m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isTrue);
  }

  test_class_method_inferred_type_nonStatic_implicit_param() async {
    var library = await buildLibrary('class C extends D { void f(value) {} }'
        ' abstract class D { void f(int value); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        methods
          f @25
            parameters
              requiredPositional value @27
                type: int
            returnType: void
      abstract class D @54
        constructors
          synthetic @-1
        methods
          abstract f @63
            parameters
              requiredPositional value @69
                type: int
            returnType: void
''');
  }

  test_class_method_inferred_type_nonStatic_implicit_return() async {
    var library = await buildLibrary('''
class C extends D {
  f() => null;
}
abstract class D {
  int f();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        methods
          f @22
            returnType: int
      abstract class D @52
        constructors
          synthetic @-1
        methods
          abstract f @62
            returnType: int
''');
  }

  test_class_method_invokesSuperSelf() async {
    var library = await buildLibrary(r'''
class A {
  void foo() {
    super.foo();
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          foo @17 invokesSuperSelf
            returnType: void
''');
  }

  test_class_method_namedAsSupertype() async {
    var library = await buildLibrary(r'''
class A {}
class B extends A {
  void A() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
        methods
          A @38
            returnType: void
''');
  }

  test_class_method_native() async {
    var library = await buildLibrary('''
class C {
  int m() native;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          external m @16
            returnType: int
''');
  }

  test_class_method_params() async {
    var library = await buildLibrary('class C { f(x, y) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @10
            parameters
              requiredPositional x @12
                type: dynamic
              requiredPositional y @15
                type: dynamic
            returnType: dynamic
''');
  }

  test_class_method_static() async {
    var library = await buildLibrary('class C { static f() {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          static f @17
            returnType: dynamic
''');
  }

  test_class_method_syncStar() async {
    var library = await buildLibrary(r'''
class C {
  Iterable<int> f() sync* {
    yield 42;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @26 sync*
            returnType: Iterable<int>
''');
  }

  test_class_method_type_parameter() async {
    var library = await buildLibrary('class C { T f<T, U>(U u) => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @12
            typeParameters
              covariant T @14
                defaultType: dynamic
              covariant U @17
                defaultType: dynamic
            parameters
              requiredPositional u @22
                type: U
            returnType: T
''');
  }

  test_class_method_type_parameter_in_generic_class() async {
    var library = await buildLibrary('''
class C<T, U> {
  V f<V, W>(T t, U u, W w) => null;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          f @20
            typeParameters
              covariant V @22
                defaultType: dynamic
              covariant W @25
                defaultType: dynamic
            parameters
              requiredPositional t @30
                type: T
              requiredPositional u @35
                type: U
              requiredPositional w @40
                type: W
            returnType: V
''');
  }

  test_class_method_type_parameter_with_function_typed_parameter() async {
    var library = await buildLibrary('class C { void f<T, U>(T x(U u)) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @15
            typeParameters
              covariant T @17
                defaultType: dynamic
              covariant U @20
                defaultType: dynamic
            parameters
              requiredPositional x @25
                type: T Function(U)
                parameters
                  requiredPositional u @29
                    type: U
            returnType: void
''');
  }

  test_class_methods() async {
    var library = await buildLibrary('class C { f() {} g() {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          f @10
            returnType: dynamic
          g @17
            returnType: dynamic
''');
  }

  test_class_mixin_class() async {
    var library = await buildLibrary('mixin class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      mixin class C @12
        constructors
          synthetic @-1
''');
  }

  test_class_mixins() async {
    var library = await buildLibrary('''
class C extends D with E, F, G {}
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        mixins
          E
          F
          G
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
      class D @40
        constructors
          synthetic @-1
      class E @51
        constructors
          synthetic @-1
      class F @62
        constructors
          synthetic @-1
      class G @73
        constructors
          synthetic @-1
''');
  }

  test_class_mixins_generic() async {
    var library = await buildLibrary('''
class Z extends A with B<int>, C<double> {}
class A {}
class B<B1> {}
class C<C1> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class Z @6
        supertype: A
        mixins
          B<int>
          C<double>
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
      class A @50
        constructors
          synthetic @-1
      class B @61
        typeParameters
          covariant B1 @63
            defaultType: dynamic
        constructors
          synthetic @-1
      class C @76
        typeParameters
          covariant C1 @78
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_mixins_genericMixin_tooManyArguments() async {
    var library = await buildLibrary('''
mixin M<T> {}
class A extends Object with M<int, String> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @20
        supertype: Object
        mixins
          M<dynamic>
        constructors
          synthetic @-1
    mixins
      mixin M @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_class_mixins_typeParameter() async {
    var library = await buildLibrary('''
mixin M1 {}
mixin M2 {}
class A<T> extends Object with M1, T<int>, M2 {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @30
        typeParameters
          covariant T @32
            defaultType: dynamic
        supertype: Object
        mixins
          M1
          M2
        constructors
          synthetic @-1
    mixins
      mixin M1 @6
        superclassConstraints
          Object
      mixin M2 @18
        superclassConstraints
          Object
''');
  }

  test_class_mixins_unresolved() async {
    var library = await buildLibrary(
        'class C extends Object with X, Y, Z {} class X {} class Z {}',
        allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: Object
        mixins
          X
          Z
        constructors
          synthetic @-1
      class X @45
        constructors
          synthetic @-1
      class Z @56
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_circularity_via_typeAlias_recordType() async {
    var library = await buildLibrary('''
class C<T extends A> {}
typedef A = (C, int);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: dynamic
            defaultType: dynamic
        constructors
          synthetic @-1
    typeAliases
      notSimplyBounded A @32
        aliasedType: (C<dynamic>, int)
''');
  }

  test_class_notSimplyBounded_circularity_via_typedef() async {
    // C's type parameter T is not simply bounded because its bound, F, expands
    // to `dynamic F(C)`, which refers to C.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(C value);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: dynamic
            defaultType: dynamic
        constructors
          synthetic @-1
    typeAliases
      functionTypeAliasBased notSimplyBounded F @32
        aliasedType: dynamic Function(C<dynamic>)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional value @36
              type: C<dynamic>
          returnType: dynamic
''');
  }

  test_class_notSimplyBounded_circularity_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // C, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends C<dynamic>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: C<dynamic>
            defaultType: C<dynamic>
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_class() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends C> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: D<dynamic>
            defaultType: dynamic
        constructors
          synthetic @-1
      notSimplyBounded class D @30
        typeParameters
          covariant T @32
            bound: C<dynamic>
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_functionType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = void Function();
typedef D<T extends C> = void Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded C @8
        typeParameters
          unrelated T @10
            bound: dynamic
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
      notSimplyBounded D @50
        typeParameters
          unrelated T @52
            bound: dynamic
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_interfaceType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = List<T>;
typedef D<T extends C> = List<T>;
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded C @8
        typeParameters
          covariant T @10
            bound: dynamic
            defaultType: dynamic
        aliasedType: List<T>
      notSimplyBounded D @42
        typeParameters
          covariant T @44
            bound: dynamic
            defaultType: dynamic
        aliasedType: List<T>
''');
  }

  test_class_notSimplyBounded_complex_by_reference_to_cycle() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends D> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: D<dynamic>
            defaultType: D<dynamic>
        constructors
          synthetic @-1
      notSimplyBounded class D @30
        typeParameters
          covariant T @32
            bound: D<dynamic>
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_complex_by_use_of_parameter() async {
    var library = await buildLibrary('''
class C<T extends D<T>> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: D<T>
            defaultType: D<dynamic>
        constructors
          synthetic @-1
      class D @33
        typeParameters
          covariant T @35
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_dependency_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // non-simply-bounded type D, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends D<dynamic>> {}
class D<T extends D<T>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: D<dynamic>
            defaultType: D<dynamic>
        constructors
          synthetic @-1
      notSimplyBounded class D @39
        typeParameters
          covariant T @41
            bound: D<T>
            defaultType: D<dynamic>
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_parameter_type() async {
    var library = await buildLibrary('''
class C<T extends void Function(T)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: void Function(T)
            defaultType: void Function(Never)
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_parameter_type_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
class C<T extends void Function(T)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @21
        typeParameters
          covariant T @23
            bound: void Function(T*)*
            defaultType: void Function(Null*)*
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_return_type() async {
    var library = await buildLibrary('''
class C<T extends T Function()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: T Function()
            defaultType: dynamic Function()
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_function_typed_bound_simple() async {
    var library = await buildLibrary('''
class C<T extends void Function()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: void Function()
            defaultType: void Function()
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_refers_to_circular_typedef() async {
    // C's type parameter T has a bound of F, which is a circular typedef.  This
    // is illegal in Dart, but we need to make sure it doesn't lead to a crash
    // or infinite loop.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(G value);
typedef G(F value);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: dynamic
            defaultType: dynamic
        constructors
          synthetic @-1
    typeAliases
      functionTypeAliasBased notSimplyBounded F @32
        aliasedType: dynamic Function(dynamic)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional value @36
              type: dynamic
          returnType: dynamic
      functionTypeAliasBased notSimplyBounded G @52
        aliasedType: dynamic Function(dynamic)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional value @56
              type: dynamic
          returnType: dynamic
''');
  }

  test_class_notSimplyBounded_self() async {
    var library = await buildLibrary('''
class C<T extends C> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: C<dynamic>
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_simple_because_non_generic() async {
    // If no type parameters are specified, then the class is simply bounded, so
    // there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_simple_by_lack_of_cycles() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: D<dynamic>
            defaultType: D<dynamic>
        constructors
          synthetic @-1
      class D @30
        typeParameters
          covariant T @32
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_notSimplyBounded_simple_by_syntax() async {
    // If no bounds are specified, then the class is simply bounded by syntax
    // alone, so there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_operator() async {
    var library =
        await buildLibrary('class C { C operator+(C other) => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          + @20
            parameters
              requiredPositional other @24
                type: C
            returnType: C
''');
  }

  test_class_operator_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator==(Object other) => false;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          == @25
            parameters
              requiredPositional other @35
                type: Object
            returnType: bool
''');
  }

  test_class_operator_external() async {
    var library =
        await buildLibrary('class C { external C operator+(C other); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          external + @29
            parameters
              requiredPositional other @33
                type: C
            returnType: C
''');
  }

  test_class_operator_greater_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator>=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          >= @25
            parameters
              requiredPositional other @30
                type: C
            returnType: bool
''');
  }

  test_class_operator_index() async {
    var library =
        await buildLibrary('class C { bool operator[](int i) => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          [] @23
            parameters
              requiredPositional i @30
                type: int
            returnType: bool
''');
  }

  test_class_operator_index_set() async {
    var library = await buildLibrary('''
class C {
  void operator[]=(int i, bool v) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          []= @25
            parameters
              requiredPositional i @33
                type: int
              requiredPositional v @41
                type: bool
            returnType: void
''');
  }

  test_class_operator_less_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator<=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          <= @25
            parameters
              requiredPositional other @30
                type: C
            returnType: bool
''');
  }

  test_class_ref_nullability_none() async {
    var library = await buildLibrary('''
class C {}
C c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
    topLevelVariables
      static c @13
        type: C
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C
        returnType: void
''');
  }

  test_class_ref_nullability_question() async {
    var library = await buildLibrary('''
class C {}
C? c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
    topLevelVariables
      static c @14
        type: C?
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C?
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C?
        returnType: void
''');
  }

  test_class_ref_nullability_star() async {
    var library = await buildLibrary('''
// @dart = 2.9
class C {}
C c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @21
        constructors
          synthetic @-1
    topLevelVariables
      static c @28
        type: C*
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C*
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C*
        returnType: void
''');
  }

  test_class_sealed() async {
    var library = await buildLibrary('sealed class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract sealed class C @13
        constructors
          synthetic @-1
''');
  }

  test_class_sealed_induced_base_extends_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      base class A @11
        constructors
          synthetic @-1
      abstract sealed base class B @29
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_sealed_induced_base_implements_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      base class A @11
        constructors
          synthetic @-1
      abstract sealed base class B @29
        interfaces
          A
        constructors
          synthetic @-1
''');
  }

  test_class_sealed_induced_base_implements_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      final class A @12
        constructors
          synthetic @-1
      abstract sealed base class B @30
        interfaces
          A
        constructors
          synthetic @-1
''');
  }

  test_class_sealed_induced_final_extends_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      final class A @12
        constructors
          synthetic @-1
      abstract sealed final class B @30
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_sealed_induced_final_with_base_mixin() async {
    var library = await buildLibrary('''
base mixin A {}
interface class B {}
sealed class C extends B with A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      interface class B @32
        constructors
          synthetic @-1
      abstract sealed final class C @50
        supertype: B
        mixins
          A
        constructors
          synthetic @-1
            superConstructor: self::@class::B::@constructor::new
    mixins
      base mixin A @11
        superclassConstraints
          Object
''');
  }

  test_class_sealed_induced_interface_extends_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      interface class A @16
        constructors
          synthetic @-1
      abstract sealed interface class B @34
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_class_sealed_induced_none_implements_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      interface class A @16
        constructors
          synthetic @-1
      abstract sealed class B @34
        interfaces
          A
        constructors
          synthetic @-1
''');
  }

  test_class_setter_abstract() async {
    var library =
        await buildLibrary('abstract class C { void set x(int value); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class C @15
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract set x @28
            parameters
              requiredPositional value @34
                type: int
            returnType: void
''');
  }

  test_class_setter_covariant() async {
    var library =
        await buildLibrary('class C { void set x(covariant int value); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract set x @19
            parameters
              requiredPositional covariant value @35
                type: int
            returnType: void
''');
  }

  test_class_setter_external() async {
    var library =
        await buildLibrary('class C { external void set x(int value); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          external set x @28
            parameters
              requiredPositional value @34
                type: int
            returnType: void
''');
  }

  test_class_setter_implicit_param_type() async {
    var library = await buildLibrary('class C { void set x(value) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @19
            parameters
              requiredPositional value @21
                type: dynamic
            returnType: void
''');
  }

  test_class_setter_implicit_return_type() async {
    var library = await buildLibrary('class C { set x(int value) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set x @14
            parameters
              requiredPositional value @20
                type: int
            returnType: void
''');
  }

  test_class_setter_inferred_type_conflictingInheritance() async {
    var library = await buildLibrary('''
class A {
  int t;
}
class B extends A {
  double t;
}
class C extends A implements B {
}
class D extends C {
  void set t(p) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          t @16
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get t @-1
            returnType: int
          synthetic set t @-1
            parameters
              requiredPositional _t @-1
                type: int
            returnType: void
      class B @27
        supertype: A
        fields
          t @50
            type: double
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
        accessors
          synthetic get t @-1
            returnType: double
          synthetic set t @-1
            parameters
              requiredPositional _t @-1
                type: double
            returnType: void
      class C @61
        supertype: A
        interfaces
          B
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
      class D @96
        supertype: C
        fields
          synthetic t @-1
            type: dynamic
        constructors
          synthetic @-1
            superConstructor: self::@class::C::@constructor::new
        accessors
          set t @121
            parameters
              requiredPositional p @123
                type: dynamic
            returnType: void
''');
  }

  test_class_setter_inferred_type_nonStatic_implicit_param() async {
    var library =
        await buildLibrary('class C extends D { void set f(value) {} }'
            ' abstract class D { void set f(int value); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        fields
          synthetic f @-1
            type: int
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        accessors
          set f @29
            parameters
              requiredPositional value @31
                type: int
            returnType: void
      abstract class D @58
        fields
          synthetic f @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract set f @71
            parameters
              requiredPositional value @77
                type: int
            returnType: void
''');
  }

  test_class_setter_inferred_type_static_implicit_return() async {
    var library = await buildLibrary('''
class C {
  static set f(int value) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic static f @-1
            type: int
        constructors
          synthetic @-1
        accessors
          static set f @23
            parameters
              requiredPositional value @29
                type: int
            returnType: void
''');
  }

  test_class_setter_invalid_named_parameter() async {
    var library = await buildLibrary('class C { void set x({a}) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @19
            parameters
              optionalNamed default a @22
                type: dynamic
            returnType: void
''');
  }

  test_class_setter_invalid_no_parameter() async {
    var library = await buildLibrary('class C { void set x() {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @19
            returnType: void
''');
  }

  test_class_setter_invalid_optional_parameter() async {
    var library = await buildLibrary('class C { void set x([a]) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @19
            parameters
              optionalPositional default a @22
                type: dynamic
            returnType: void
''');
  }

  test_class_setter_invalid_too_many_parameters() async {
    var library = await buildLibrary('class C { void set x(a, b) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @19
            parameters
              requiredPositional a @21
                type: dynamic
              requiredPositional b @24
                type: dynamic
            returnType: void
''');
  }

  test_class_setter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set foo @16
            parameters
              requiredPositional _ @24
                type: int
            returnType: void
''');
  }

  test_class_setter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set foo @16 invokesSuperSelf
            parameters
              requiredPositional _ @24
                type: int
            returnType: void
''');
  }

  test_class_setter_native() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) native;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          external set x @21
            parameters
              requiredPositional value @27
                type: int
            returnType: void
''');
  }

  test_class_setter_static() async {
    var library =
        await buildLibrary('class C { static void set x(int value) {} }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic static x @-1
            type: int
        constructors
          synthetic @-1
        accessors
          static set x @26
            parameters
              requiredPositional value @32
                type: int
            returnType: void
''');
  }

  test_class_setters() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  set y(value) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: int
          synthetic y @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @21
            parameters
              requiredPositional value @27
                type: int
            returnType: void
          set y @43
            parameters
              requiredPositional value @45
                type: dynamic
            returnType: void
''');
  }

  test_class_supertype() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
      class B @17
        supertype: A
''');
  }

  test_class_supertype_dynamic() async {
    var library = await buildLibrary('''
class A extends dynamic {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
''');
  }

  test_class_supertype_genericClass() async {
    var library = await buildLibrary('''
class C extends D<int, double> {}
class D<T1, T2> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D<int, double>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::D::@constructor::new
              substitution: {T1: int, T2: double}
      class D @40
        typeParameters
          covariant T1 @42
            defaultType: dynamic
          covariant T2 @46
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_supertype_genericClass_tooManyArguments() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<int, String> {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
      class B @20
        supertype: A<dynamic>
''');
  }

  test_class_supertype_typeArguments_self() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<B> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class B @20
        supertype: A<B>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: B}
''');
  }

  test_class_supertype_typeParameter() async {
    var library = await buildLibrary('''
class A<T> extends T<int> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_supertype_unresolved() async {
    var library = await buildLibrary('class C extends D {}', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_bound() async {
    var library = await buildLibrary('''
class C<T extends Object, U extends D> {}
class D {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: Object
            defaultType: Object
          covariant U @26
            bound: D
            defaultType: D
        constructors
          synthetic @-1
      class D @48
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_cycle_1of1() async {
    var library = await buildLibrary('class C<T extends T> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: dynamic
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_cycle_2of3() async {
    var library = await buildLibrary(r'''
class C<T extends V, U, V extends T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: dynamic
            defaultType: dynamic
          covariant U @21
            defaultType: dynamic
          covariant V @24
            bound: dynamic
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType() async {
    var library = await buildLibrary(r'''
class A<T extends void Function(A)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @6
        typeParameters
          covariant T @8
            bound: void Function(A<dynamic>)
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType2() async {
    var library = await buildLibrary(r'''
class C<T extends void Function<U extends C>()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: void Function<U extends C<dynamic>>()
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_contravariant_legacy() async {
    var library = await buildLibrary(r'''
// @dart = 2.9
typedef F<X> = void Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @55
        typeParameters
          covariant X @57
            bound: void Function(X*)*
              alias: self::@typeAlias::F
                typeArguments
                  X*
            defaultType: void Function(Null*)*
              alias: self::@typeAlias::F
                typeArguments
                  Null*
        constructors
          synthetic @-1
    typeAliases
      F @23
        typeParameters
          contravariant X @25
            defaultType: dynamic
        aliasedType: void Function(X*)*
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional @-1
              type: X*
          returnType: void
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_contravariant_nullSafe() async {
    var library = await buildLibrary(r'''
typedef F<X> = void Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @40
        typeParameters
          covariant X @42
            bound: void Function(X)
              alias: self::@typeAlias::F
                typeArguments
                  X
            defaultType: void Function(Never)
              alias: self::@typeAlias::F
                typeArguments
                  Never
        constructors
          synthetic @-1
    typeAliases
      F @8
        typeParameters
          contravariant X @10
            defaultType: dynamic
        aliasedType: void Function(X)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional @-1
              type: X
          returnType: void
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_covariant_nullSafe() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function();

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @36
        typeParameters
          covariant X @38
            bound: X Function()
              alias: self::@typeAlias::F
                typeArguments
                  X
            defaultType: dynamic Function()
              alias: self::@typeAlias::F
                typeArguments
                  dynamic
        constructors
          synthetic @-1
    typeAliases
      F @8
        typeParameters
          covariant X @10
            defaultType: dynamic
        aliasedType: X Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: X
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant_legacy() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @37
        typeParameters
          covariant X @39
            bound: X Function(X)
              alias: self::@typeAlias::F
                typeArguments
                  X
            defaultType: dynamic Function(dynamic)
              alias: self::@typeAlias::F
                typeArguments
                  dynamic
        constructors
          synthetic @-1
    typeAliases
      F @8
        typeParameters
          invariant X @10
            defaultType: dynamic
        aliasedType: X Function(X)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional @-1
              type: X
          returnType: X
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant_nullSafe() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @37
        typeParameters
          covariant X @39
            bound: X Function(X)
              alias: self::@typeAlias::F
                typeArguments
                  X
            defaultType: dynamic Function(dynamic)
              alias: self::@typeAlias::F
                typeArguments
                  dynamic
        constructors
          synthetic @-1
    typeAliases
      F @8
        typeParameters
          invariant X @10
            defaultType: dynamic
        aliasedType: X Function(X)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional @-1
              type: X
          returnType: X
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_both_legacy() async {
    var library = await buildLibrary(r'''
// @dart = 2.9
class A<X extends X Function(X)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @21
        typeParameters
          covariant X @23
            bound: X* Function(X*)*
            defaultType: dynamic Function(Null*)*
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_both_nullSafe() async {
    var library = await buildLibrary(r'''
class A<X extends X Function(X)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @6
        typeParameters
          covariant X @8
            bound: X Function(X)
            defaultType: dynamic Function(Never)
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_contravariant_legacy() async {
    var library = await buildLibrary(r'''
// @dart = 2.9
class A<X extends void Function(X)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @21
        typeParameters
          covariant X @23
            bound: void Function(X*)*
            defaultType: void Function(Null*)*
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_contravariant_nullSafe() async {
    var library = await buildLibrary(r'''
class A<X extends void Function(X)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @6
        typeParameters
          covariant X @8
            bound: void Function(X)
            defaultType: void Function(Never)
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant_legacy() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @6
        typeParameters
          covariant X @8
            bound: X Function()
            defaultType: dynamic Function()
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant_nullSafe() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class A @6
        typeParameters
          covariant X @8
            bound: X Function()
            defaultType: dynamic Function()
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_contravariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = List<void Function(X)>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class B @46
        typeParameters
          covariant X @48
            bound: List<void Function(X)>
              alias: self::@typeAlias::A
                typeArguments
                  X
            defaultType: List<void Function(Never)>
              alias: self::@typeAlias::A
                typeArguments
                  Never
        constructors
          synthetic @-1
    typeAliases
      A @8
        typeParameters
          contravariant X @10
            defaultType: dynamic
        aliasedType: List<void Function(X)>
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_covariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = Map<X, int>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class B @35
        typeParameters
          covariant X @37
            bound: Map<X, int>
              alias: self::@typeAlias::A
                typeArguments
                  X
            defaultType: Map<dynamic, int>
              alias: self::@typeAlias::A
                typeArguments
                  dynamic
        constructors
          synthetic @-1
    typeAliases
      A @8
        typeParameters
          covariant X @10
            defaultType: dynamic
        aliasedType: Map<X, int>
''');
  }

  test_class_typeParameters_f_bound_complex() async {
    var library = await buildLibrary('class C<T extends List<U>, U> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: List<U>
            defaultType: List<dynamic>
          covariant U @27
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_f_bound_simple() async {
    var library = await buildLibrary('class C<T extends U, U> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: U
            defaultType: dynamic
          covariant U @21
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_variance_contravariant() async {
    var library = await buildLibrary('class C<in T> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          contravariant T @11
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_variance_covariant() async {
    var library = await buildLibrary('class C<out T> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @12
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_variance_invariant() async {
    var library = await buildLibrary('class C<inout T> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          invariant T @14
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_class_typeParameters_variance_multiple() async {
    var library = await buildLibrary('class C<inout T, in U, out V> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          invariant T @14
            defaultType: dynamic
          contravariant U @20
            defaultType: dynamic
          covariant V @27
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_classAlias() async {
    var library = await buildLibrary('''
class C = D with E, F, G;
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @6
        supertype: D
        mixins
          E
          F
          G
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @32
        constructors
          synthetic @-1
      class E @43
        constructors
          synthetic @-1
      class F @54
        constructors
          synthetic @-1
      class G @65
        constructors
          synthetic @-1
''');
  }

  test_classAlias_abstract() async {
    var library = await buildLibrary('''
abstract class C = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class alias C @15
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @35
        constructors
          synthetic @-1
      class E @46
        constructors
          synthetic @-1
''');
  }

  test_classAlias_base() async {
    var library = await buildLibrary('''
base class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      base class alias C @11
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M @36
        superclassConstraints
          Object
''');
  }

  test_classAlias_constructors_default() async {
    var library = await buildLibrary('''
class A {}
mixin class M {}
class X = A with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      mixin class M @23
        constructors
          synthetic @-1
      class alias X @34
        supertype: A
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_classAlias_constructors_dependencies() async {
    var library = await buildLibrary('''
class A {
  A(int i);
}
mixin class M1 {}
mixin class M2 {}

class C2 = C1 with M2;
class C1 = A with M1;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional i @18
                type: int
      mixin class M1 @36
        constructors
          synthetic @-1
      mixin class M2 @54
        constructors
          synthetic @-1
      class alias C2 @67
        supertype: C1
        mixins
          M2
        constructors
          synthetic @-1
            parameters
              requiredPositional i @-1
                type: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: i @-1
                      staticElement: self::@class::C2::@constructor::new::@parameter::i
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: self::@class::C1::@constructor::new
            superConstructor: self::@class::C1::@constructor::new
      class alias C1 @90
        supertype: A
        mixins
          M1
        constructors
          synthetic @-1
            parameters
              requiredPositional i @-1
                type: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: i @-1
                      staticElement: self::@class::C1::@constructor::new::@parameter::i
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
''');
  }

  test_classAlias_constructors_optionalParameters() async {
    var library = await buildLibrary('''
class A {
  A.c1(int a);
  A.c2(int a, [int? b, int c = 0]);
  A.c3(int a, {int? b, int c = 0});
}

mixin M {}

class C = A with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          c1 @14
            periodOffset: 13
            nameEnd: 16
            parameters
              requiredPositional a @21
                type: int
          c2 @29
            periodOffset: 28
            nameEnd: 31
            parameters
              requiredPositional a @36
                type: int
              optionalPositional default b @45
                type: int?
              optionalPositional default c @52
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 0 @56
                    staticType: int
          c3 @65
            periodOffset: 64
            nameEnd: 67
            parameters
              requiredPositional a @72
                type: int
              optionalNamed default b @81
                type: int?
              optionalNamed default c @88
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 0 @92
                    staticType: int
      class alias C @118
        supertype: A
        mixins
          M
        constructors
          synthetic c1 @-1
            parameters
              requiredPositional a @-1
                type: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: c1 @-1
                  staticElement: self::@class::A::@constructor::c1
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: a @-1
                      staticElement: self::@class::C::@constructor::c1::@parameter::a
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::c1
            superConstructor: self::@class::A::@constructor::c1
          synthetic c2 @-1
            parameters
              requiredPositional a @-1
                type: int
              optionalPositional default b @-1
                type: int?
              optionalPositional default c @-1
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 0 @56
                    staticType: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: c2 @-1
                  staticElement: self::@class::A::@constructor::c2
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: a @-1
                      staticElement: self::@class::C::@constructor::c2::@parameter::a
                      staticType: int
                    SimpleIdentifier
                      token: b @-1
                      staticElement: self::@class::C::@constructor::c2::@parameter::b
                      staticType: int?
                    SimpleIdentifier
                      token: c @-1
                      staticElement: self::@class::C::@constructor::c2::@parameter::c
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::c2
            superConstructor: self::@class::A::@constructor::c2
          synthetic c3 @-1
            parameters
              requiredPositional a @-1
                type: int
              optionalNamed default b @-1
                type: int?
              optionalNamed default c @-1
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 0 @92
                    staticType: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: c3 @-1
                  staticElement: self::@class::A::@constructor::c3
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: a @-1
                      staticElement: self::@class::C::@constructor::c3::@parameter::a
                      staticType: int
                    SimpleIdentifier
                      token: b @-1
                      staticElement: self::@class::C::@constructor::c3::@parameter::b
                      staticType: int?
                    SimpleIdentifier
                      token: c @-1
                      staticElement: self::@class::C::@constructor::c3::@parameter::c
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::c3
            superConstructor: self::@class::A::@constructor::c3
    mixins
      mixin M @106
        superclassConstraints
          Object
''');
  }

  test_classAlias_constructors_requiredParameters() async {
    var library = await buildLibrary('''
class A<T extends num> {
  A(T x, T y);
}

mixin M {}

class B<E extends num> = A<E> with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            bound: num
            defaultType: num
        constructors
          @27
            parameters
              requiredPositional x @31
                type: T
              requiredPositional y @36
                type: T
      class alias B @61
        typeParameters
          covariant E @63
            bound: num
            defaultType: num
        supertype: A<E>
        mixins
          M
        constructors
          synthetic @-1
            parameters
              requiredPositional x @-1
                type: E
              requiredPositional y @-1
                type: E
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::B::@constructor::new::@parameter::x
                      staticType: E
                    SimpleIdentifier
                      token: y @-1
                      staticElement: self::@class::B::@constructor::new::@parameter::y
                      staticType: E
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: E}
    mixins
      mixin M @49
        superclassConstraints
          Object
''');
  }

  test_classAlias_documented() async {
    var library = await buildLibrary('''
/**
 * Docs
 */
class C = D with E;

class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @22
        documentationComment: /**\n * Docs\n */
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @43
        constructors
          synthetic @-1
      class E @54
        constructors
          synthetic @-1
''');
  }

  test_classAlias_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// aaa
/// b
/// cc
class C = D with E;

class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @27
        documentationComment: /// aaa\n/// b\n/// cc
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @48
        constructors
          synthetic @-1
      class E @59
        constructors
          synthetic @-1
''');
  }

  test_classAlias_documented_withLeadingNonDocumentation() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
class C = D with E;

class D {}
class E {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @66
        documentationComment: /**\n * Docs\n */
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @87
        constructors
          synthetic @-1
      class E @98
        constructors
          synthetic @-1
''');
  }

  test_classAlias_final() async {
    var library = await buildLibrary('''
final class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      final class alias C @12
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M @37
        superclassConstraints
          Object
''');
  }

  test_classAlias_generic() async {
    var library = await buildLibrary('''
class Z = A with B<int>, C<double>;
class A {}
class B<B1> {}
class C<C1> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias Z @6
        supertype: A
        mixins
          B<int>
          C<double>
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
      class A @42
        constructors
          synthetic @-1
      class B @53
        typeParameters
          covariant B1 @55
            defaultType: dynamic
        constructors
          synthetic @-1
      class C @68
        typeParameters
          covariant C1 @70
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_classAlias_interface() async {
    var library = await buildLibrary('''
interface class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      interface class alias C @16
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M @41
        superclassConstraints
          Object
''');
  }

  test_classAlias_invalid_extendsEnum() async {
    newFile('$testPackageLibPath/a.dart', r'''
enum E { v }
mixin M {}
''');

    var library = await buildLibrary('''
import 'a.dart';
class A = E with M;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class alias A @23
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
''');
  }

  test_classAlias_invalid_extendsMixin() async {
    var library = await buildLibrary('''
mixin M1 {}
mixin M2 {}
class A = M1 with M2;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias A @30
        supertype: Object
        mixins
          M2
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M1 @6
        superclassConstraints
          Object
      mixin M2 @18
        superclassConstraints
          Object
''');
  }

  test_classAlias_mixin_class() async {
    var library = await buildLibrary('''
mixin class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      mixin class alias C @12
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M @37
        superclassConstraints
          Object
''');
  }

  test_classAlias_notSimplyBounded_self() async {
    var library = await buildLibrary('''
class C<T extends C> = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class alias C @6
        typeParameters
          covariant T @8
            bound: C<dynamic>
            defaultType: dynamic
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @39
        constructors
          synthetic @-1
      class E @50
        constructors
          synthetic @-1
''');
  }

  test_classAlias_notSimplyBounded_simple_no_type_parameter_bound() async {
    // If no bounds are specified, then the class is simply bounded by syntax
    // alone, so there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C<T> = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @29
        constructors
          synthetic @-1
      class E @40
        constructors
          synthetic @-1
''');
  }

  test_classAlias_notSimplyBounded_simple_non_generic() async {
    // If no type parameters are specified, then the class is simply bounded, so
    // there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @6
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @26
        constructors
          synthetic @-1
      class E @37
        constructors
          synthetic @-1
''');
  }

  test_classAlias_sealed() async {
    var library = await buildLibrary('''
sealed class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract sealed class alias C @13
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    mixins
      mixin M @38
        superclassConstraints
          Object
''');
  }

  test_classAlias_with_const_constructors() async {
    addSource('$testPackageLibPath/a.dart', r'''
class Base {
  const Base._priv();
  const Base();
  const Base.named();
}
''');
    var library = await buildLibrary('''
import "a.dart";
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class M @23
        constructors
          synthetic @-1
      class alias MixinApp @34
        supertype: Base
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::new
            superConstructor: package:test/a.dart::@class::Base::@constructor::new
          synthetic const named @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: named @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::named
            superConstructor: package:test/a.dart::@class::Base::@constructor::named
''');
  }

  test_classAlias_with_forwarding_constructors() async {
    addSource('$testPackageLibPath/a.dart', r'''
class Base {
  bool x = true;
  Base._priv();
  Base();
  Base.noArgs();
  Base.requiredArg(x);
  Base.positionalArg([bool x = true]);
  Base.positionalArg2([this.x = true]);
  Base.namedArg({int x = 42});
  Base.namedArg2({this.x = true});
  factory Base.fact() => Base();
  factory Base.fact2() = Base.noArgs;
}
''');
    var library = await buildLibrary('''
import "a.dart";
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class M @23
        constructors
          synthetic @-1
      class alias MixinApp @34
        supertype: Base
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::new
            superConstructor: package:test/a.dart::@class::Base::@constructor::new
          synthetic noArgs @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: noArgs @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::noArgs
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::noArgs
            superConstructor: package:test/a.dart::@class::Base::@constructor::noArgs
          synthetic requiredArg @-1
            parameters
              requiredPositional x @-1
                type: dynamic
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: requiredArg @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::requiredArg
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::MixinApp::@constructor::requiredArg::@parameter::x
                      staticType: dynamic
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::requiredArg
            superConstructor: package:test/a.dart::@class::Base::@constructor::requiredArg
          synthetic positionalArg @-1
            parameters
              optionalPositional default x @-1
                type: bool
                constantInitializer
                  BooleanLiteral
                    literal: true @127
                    staticType: bool
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: positionalArg @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::positionalArg
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::MixinApp::@constructor::positionalArg::@parameter::x
                      staticType: bool
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::positionalArg
            superConstructor: package:test/a.dart::@class::Base::@constructor::positionalArg
          synthetic positionalArg2 @-1
            parameters
              optionalPositional default final x @-1
                type: bool
                constantInitializer
                  BooleanLiteral
                    literal: true @167
                    staticType: bool
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: positionalArg2 @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::positionalArg2
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::MixinApp::@constructor::positionalArg2::@parameter::x
                      staticType: bool
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::positionalArg2
            superConstructor: package:test/a.dart::@class::Base::@constructor::positionalArg2
          synthetic namedArg @-1
            parameters
              optionalNamed default x @-1
                type: int
                constantInitializer
                  IntegerLiteral
                    literal: 42 @200
                    staticType: int
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: namedArg @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::namedArg
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::MixinApp::@constructor::namedArg::@parameter::x
                      staticType: int
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::namedArg
            superConstructor: package:test/a.dart::@class::Base::@constructor::namedArg
          synthetic namedArg2 @-1
            parameters
              optionalNamed default final x @-1
                type: bool
                constantInitializer
                  BooleanLiteral
                    literal: true @233
                    staticType: bool
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: namedArg2 @-1
                  staticElement: package:test/a.dart::@class::Base::@constructor::namedArg2
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: x @-1
                      staticElement: self::@class::MixinApp::@constructor::namedArg2::@parameter::x
                      staticType: bool
                  rightParenthesis: ) @0
                staticElement: package:test/a.dart::@class::Base::@constructor::namedArg2
            superConstructor: package:test/a.dart::@class::Base::@constructor::namedArg2
''');
  }

  test_classAlias_with_forwarding_constructors_type_substitution() async {
    var library = await buildLibrary('''
class Base<T> {
  Base.ctor(T t, List<T> l);
}
class M {}
class MixinApp = Base with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class Base @6
        typeParameters
          covariant T @11
            defaultType: dynamic
        constructors
          ctor @23
            periodOffset: 22
            nameEnd: 27
            parameters
              requiredPositional t @30
                type: T
              requiredPositional l @41
                type: List<T>
      class M @53
        constructors
          synthetic @-1
      class alias MixinApp @64
        supertype: Base<dynamic>
        mixins
          M
        constructors
          synthetic ctor @-1
            parameters
              requiredPositional t @-1
                type: dynamic
              requiredPositional l @-1
                type: List<dynamic>
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: ctor @-1
                  staticElement: self::@class::Base::@constructor::ctor
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: t @-1
                      staticElement: self::@class::MixinApp::@constructor::ctor::@parameter::t
                      staticType: dynamic
                    SimpleIdentifier
                      token: l @-1
                      staticElement: self::@class::MixinApp::@constructor::ctor::@parameter::l
                      staticType: List<dynamic>
                  rightParenthesis: ) @0
                staticElement: self::@class::Base::@constructor::ctor
            superConstructor: ConstructorMember
              base: self::@class::Base::@constructor::ctor
              substitution: {T: dynamic}
''');
  }

  test_classAlias_with_forwarding_constructors_type_substitution_complex() async {
    var library = await buildLibrary('''
class Base<T> {
  Base.ctor(T t, List<T> l);
}
class M {}
class MixinApp<U> = Base<List<U>> with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class Base @6
        typeParameters
          covariant T @11
            defaultType: dynamic
        constructors
          ctor @23
            periodOffset: 22
            nameEnd: 27
            parameters
              requiredPositional t @30
                type: T
              requiredPositional l @41
                type: List<T>
      class M @53
        constructors
          synthetic @-1
      class alias MixinApp @64
        typeParameters
          covariant U @73
            defaultType: dynamic
        supertype: Base<List<U>>
        mixins
          M
        constructors
          synthetic ctor @-1
            parameters
              requiredPositional t @-1
                type: List<U>
              requiredPositional l @-1
                type: List<List<U>>
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: ctor @-1
                  staticElement: self::@class::Base::@constructor::ctor
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleIdentifier
                      token: t @-1
                      staticElement: self::@class::MixinApp::@constructor::ctor::@parameter::t
                      staticType: List<U>
                    SimpleIdentifier
                      token: l @-1
                      staticElement: self::@class::MixinApp::@constructor::ctor::@parameter::l
                      staticType: List<List<U>>
                  rightParenthesis: ) @0
                staticElement: self::@class::Base::@constructor::ctor
            superConstructor: ConstructorMember
              base: self::@class::Base::@constructor::ctor
              substitution: {T: List<U>}
''');
  }

  test_classAlias_with_mixin_members() async {
    var library = await buildLibrary('''
class C = D with E;
class D {}
class E {
  int get a => null;
  void set b(int i) {}
  void f() {}
  int x;
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @6
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @26
        constructors
          synthetic @-1
      class E @37
        fields
          x @105
            type: int
            shouldUseTypeForInitializerInference: true
          synthetic a @-1
            type: int
          synthetic b @-1
            type: int
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
          get a @51
            returnType: int
          set b @73
            parameters
              requiredPositional i @79
                type: int
            returnType: void
        methods
          f @92
            returnType: void
''');
  }

  test_classes() async {
    var library = await buildLibrary('class C {} class D {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
      class D @17
        constructors
          synthetic @-1
''');
  }

  test_closure_executable_with_return_type_from_closure() async {
    var library = await buildLibrary('''
f() {
  print(() {});
  print(() => () => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        returnType: dynamic
''');
  }

  test_closure_generic() async {
    var library = await buildLibrary(r'''
final f = <U, V>(U x, V y) => y;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static final f @6
        type: V Function<U, V>(U, V)
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get f @-1
        returnType: V Function<U, V>(U, V)
''');
  }

  test_closure_in_variable_declaration_in_part() async {
    addSource('$testPackageLibPath/a.dart',
        'part of lib; final f = (int i) => i.toDouble();');
    var library = await buildLibrary('''
library lib;
part "a.dart";
''');
    checkElementText(library, r'''
library
  name: lib
  nameOffset: 8
  definingUnit
  parts
    package:test/a.dart
      topLevelVariables
        static final f @19
          type: double Function(int)
          shouldUseTypeForInitializerInference: false
      accessors
        synthetic static get f @-1
          returnType: double Function(int)
''');
  }

  test_codeRange_class() async {
    var library = await buildLibrary('''
class Raw {}

/// Comment 1.
/// Comment 2.
class HasDocComment {}

@Object()
class HasAnnotation {}

@Object()
/// Comment 1.
/// Comment 2.
class AnnotationThenComment {}

/// Comment 1.
/// Comment 2.
@Object()
class CommentThenAnnotation {}

/// Comment 1.
@Object()
/// Comment 2.
class CommentAroundAnnotation {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class Raw @6
        codeOffset: 0
        codeLength: 12
        constructors
          synthetic @-1
      class HasDocComment @50
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 14
        codeLength: 52
        constructors
          synthetic @-1
      class HasAnnotation @84
        metadata
          Annotation
            atSign: @ @68
            name: SimpleIdentifier
              token: Object @69
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @75
              rightParenthesis: ) @76
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 68
        codeLength: 32
        constructors
          synthetic @-1
      class AnnotationThenComment @148
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @102
            name: SimpleIdentifier
              token: Object @103
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @109
              rightParenthesis: ) @110
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 102
        codeLength: 70
        constructors
          synthetic @-1
      class CommentThenAnnotation @220
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @204
            name: SimpleIdentifier
              token: Object @205
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @211
              rightParenthesis: ) @212
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 174
        codeLength: 70
        constructors
          synthetic @-1
      class CommentAroundAnnotation @292
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @261
            name: SimpleIdentifier
              token: Object @262
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @268
              rightParenthesis: ) @269
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 261
        codeLength: 57
        constructors
          synthetic @-1
''');
  }

  test_codeRange_class_namedMixin() async {
    var library = await buildLibrary('''
class A {}

class B {}

class Raw = Object with A, B;

/// Comment 1.
/// Comment 2.
class HasDocComment = Object with A, B;

@Object()
class HasAnnotation = Object with A, B;

@Object()
/// Comment 1.
/// Comment 2.
class AnnotationThenComment = Object with A, B;

/// Comment 1.
/// Comment 2.
@Object()
class CommentThenAnnotation = Object with A, B;

/// Comment 1.
@Object()
/// Comment 2.
class CommentAroundAnnotation = Object with A, B;
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        codeOffset: 0
        codeLength: 10
        constructors
          synthetic @-1
      class B @18
        codeOffset: 12
        codeLength: 10
        constructors
          synthetic @-1
      class alias Raw @30
        codeOffset: 24
        codeLength: 29
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias HasDocComment @91
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 55
        codeLength: 69
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias HasAnnotation @142
        metadata
          Annotation
            atSign: @ @126
            name: SimpleIdentifier
              token: Object @127
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @133
              rightParenthesis: ) @134
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 126
        codeLength: 49
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias AnnotationThenComment @223
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @177
            name: SimpleIdentifier
              token: Object @178
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @184
              rightParenthesis: ) @185
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 177
        codeLength: 87
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias CommentThenAnnotation @312
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @296
            name: SimpleIdentifier
              token: Object @297
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @303
              rightParenthesis: ) @304
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 266
        codeLength: 87
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias CommentAroundAnnotation @401
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @370
            name: SimpleIdentifier
              token: Object @371
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @377
              rightParenthesis: ) @378
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 370
        codeLength: 74
        supertype: Object
        mixins
          A
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
''');
  }

  test_codeRange_constructor() async {
    var library = await buildLibrary('''
class C {
  C();

  C.raw() {}

  /// Comment 1.
  /// Comment 2.
  C.hasDocComment() {}

  @Object()
  C.hasAnnotation() {}

  @Object()
  /// Comment 1.
  /// Comment 2.
  C.annotationThenComment() {}

  /// Comment 1.
  /// Comment 2.
  @Object()
  C.commentThenAnnotation() {}

  /// Comment 1.
  @Object()
  /// Comment 2.
  C.commentAroundAnnotation() {}
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        codeOffset: 0
        codeLength: 362
        constructors
          @12
            codeOffset: 12
            codeLength: 4
          raw @22
            codeOffset: 20
            codeLength: 10
            periodOffset: 21
            nameEnd: 25
          hasDocComment @70
            documentationComment: /// Comment 1.\n/// Comment 2.
            codeOffset: 34
            codeLength: 54
            periodOffset: 69
            nameEnd: 83
          hasAnnotation @106
            metadata
              Annotation
                atSign: @ @92
                name: SimpleIdentifier
                  token: Object @93
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @99
                  rightParenthesis: ) @100
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 92
            codeLength: 32
            periodOffset: 105
            nameEnd: 119
          annotationThenComment @176
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @128
                name: SimpleIdentifier
                  token: Object @129
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @135
                  rightParenthesis: ) @136
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 128
            codeLength: 74
            periodOffset: 175
            nameEnd: 197
          commentThenAnnotation @254
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @240
                name: SimpleIdentifier
                  token: Object @241
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @247
                  rightParenthesis: ) @248
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 206
            codeLength: 74
            periodOffset: 253
            nameEnd: 275
          commentAroundAnnotation @332
            documentationComment: /// Comment 2.
            metadata
              Annotation
                atSign: @ @301
                name: SimpleIdentifier
                  token: Object @302
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @308
                  rightParenthesis: ) @309
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 301
            codeLength: 59
            periodOffset: 331
            nameEnd: 355
''');
  }

  test_codeRange_constructor_factory() async {
    var library = await buildLibrary('''
class C {
  factory C() => throw 0;

  factory C.raw() => throw 0;

  /// Comment 1.
  /// Comment 2.
  factory C.hasDocComment() => throw 0;

  @Object()
  factory C.hasAnnotation() => throw 0;

  @Object()
  /// Comment 1.
  /// Comment 2.
  factory C.annotationThenComment() => throw 0;

  /// Comment 1.
  /// Comment 2.
  @Object()
  factory C.commentThenAnnotation() => throw 0;

  /// Comment 1.
  @Object()
  /// Comment 2.
  factory C.commentAroundAnnotation() => throw 0;
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        codeOffset: 0
        codeLength: 483
        constructors
          factory @20
            codeOffset: 12
            codeLength: 23
          factory raw @49
            codeOffset: 39
            codeLength: 27
            periodOffset: 48
            nameEnd: 52
          factory hasDocComment @114
            documentationComment: /// Comment 1.\n/// Comment 2.
            codeOffset: 70
            codeLength: 71
            periodOffset: 113
            nameEnd: 127
          factory hasAnnotation @167
            metadata
              Annotation
                atSign: @ @145
                name: SimpleIdentifier
                  token: Object @146
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @152
                  rightParenthesis: ) @153
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 145
            codeLength: 49
            periodOffset: 166
            nameEnd: 180
          factory annotationThenComment @254
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @198
                name: SimpleIdentifier
                  token: Object @199
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @205
                  rightParenthesis: ) @206
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 198
            codeLength: 91
            periodOffset: 253
            nameEnd: 275
          factory commentThenAnnotation @349
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @327
                name: SimpleIdentifier
                  token: Object @328
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @334
                  rightParenthesis: ) @335
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 293
            codeLength: 91
            periodOffset: 348
            nameEnd: 370
          factory commentAroundAnnotation @444
            documentationComment: /// Comment 2.
            metadata
              Annotation
                atSign: @ @405
                name: SimpleIdentifier
                  token: Object @406
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @412
                  rightParenthesis: ) @413
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 405
            codeLength: 76
            periodOffset: 443
            nameEnd: 467
''');
  }

  test_codeRange_enum() async {
    var library = await buildLibrary('''
enum E {
  aaa, bbb, ccc
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        codeOffset: 0
        codeLength: 26
        supertype: Enum
        fields
          static const enumConstant aaa @11
            codeOffset: 11
            codeLength: 3
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant bbb @16
            codeOffset: 16
            codeLength: 3
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant ccc @21
            codeOffset: 21
            codeLength: 3
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: aaa @-1
                    staticElement: self::@enum::E::@getter::aaa
                    staticType: E
                  SimpleIdentifier
                    token: bbb @-1
                    staticElement: self::@enum::E::@getter::bbb
                    staticType: E
                  SimpleIdentifier
                    token: ccc @-1
                    staticElement: self::@enum::E::@getter::ccc
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get aaa @-1
            returnType: E
          synthetic static get bbb @-1
            returnType: E
          synthetic static get ccc @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_codeRange_extensions() async {
    var library = await buildLibrary('''
class A {}

extension Raw on A {}

/// Comment 1.
/// Comment 2.
extension HasDocComment on A {}

@Object()
extension HasAnnotation on A {}

@Object()
/// Comment 1.
/// Comment 2.
extension AnnotationThenComment on A {}

/// Comment 1.
/// Comment 2.
@Object()
extension CommentThenAnnotation on A {}

/// Comment 1.
@Object()
/// Comment 2.
extension CommentAroundAnnotation on A {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        codeOffset: 0
        codeLength: 10
        constructors
          synthetic @-1
    extensions
      Raw @22
        codeOffset: 12
        codeLength: 21
        extendedType: A
      HasDocComment @75
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 35
        codeLength: 61
        extendedType: A
      HasAnnotation @118
        metadata
          Annotation
            atSign: @ @98
            name: SimpleIdentifier
              token: Object @99
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @105
              rightParenthesis: ) @106
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 98
        codeLength: 41
        extendedType: A
      AnnotationThenComment @191
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @141
            name: SimpleIdentifier
              token: Object @142
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @148
              rightParenthesis: ) @149
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 141
        codeLength: 79
        extendedType: A
      CommentThenAnnotation @272
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @252
            name: SimpleIdentifier
              token: Object @253
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @259
              rightParenthesis: ) @260
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 222
        codeLength: 79
        extendedType: A
      CommentAroundAnnotation @353
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @318
            name: SimpleIdentifier
              token: Object @319
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @325
              rightParenthesis: ) @326
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 318
        codeLength: 66
        extendedType: A
''');
  }

  test_codeRange_field() async {
    var library = await buildLibrary('''
class C {
  int withInit = 1;

  int withoutInit;

  int multiWithInit = 2, multiWithoutInit, multiWithInit2 = 3;
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        codeOffset: 0
        codeLength: 115
        fields
          withInit @16
            codeOffset: 12
            codeLength: 16
            type: int
            shouldUseTypeForInitializerInference: true
          withoutInit @37
            codeOffset: 33
            codeLength: 15
            type: int
            shouldUseTypeForInitializerInference: true
          multiWithInit @57
            codeOffset: 53
            codeLength: 21
            type: int
            shouldUseTypeForInitializerInference: true
          multiWithoutInit @76
            codeOffset: 76
            codeLength: 16
            type: int
            shouldUseTypeForInitializerInference: true
          multiWithInit2 @94
            codeOffset: 94
            codeLength: 18
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get withInit @-1
            returnType: int
          synthetic set withInit @-1
            parameters
              requiredPositional _withInit @-1
                type: int
            returnType: void
          synthetic get withoutInit @-1
            returnType: int
          synthetic set withoutInit @-1
            parameters
              requiredPositional _withoutInit @-1
                type: int
            returnType: void
          synthetic get multiWithInit @-1
            returnType: int
          synthetic set multiWithInit @-1
            parameters
              requiredPositional _multiWithInit @-1
                type: int
            returnType: void
          synthetic get multiWithoutInit @-1
            returnType: int
          synthetic set multiWithoutInit @-1
            parameters
              requiredPositional _multiWithoutInit @-1
                type: int
            returnType: void
          synthetic get multiWithInit2 @-1
            returnType: int
          synthetic set multiWithInit2 @-1
            parameters
              requiredPositional _multiWithInit2 @-1
                type: int
            returnType: void
''');
  }

  test_codeRange_field_annotations() async {
    var library = await buildLibrary('''
class C {
  /// Comment 1.
  /// Comment 2.
  int hasDocComment, hasDocComment2;

  @Object()
  int hasAnnotation, hasAnnotation2;

  @Object()
  /// Comment 1.
  /// Comment 2.
  int annotationThenComment, annotationThenComment2;

  /// Comment 1.
  /// Comment 2.
  @Object()
  int commentThenAnnotation, commentThenAnnotation2;

  /// Comment 1.
  @Object()
  /// Comment 2.
  int commentAroundAnnotation, commentAroundAnnotation2;
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        codeOffset: 0
        codeLength: 436
        fields
          hasDocComment @50
            documentationComment: /// Comment 1.\n/// Comment 2.
            codeOffset: 12
            codeLength: 51
            type: int
            shouldUseTypeForInitializerInference: true
          hasDocComment2 @65
            documentationComment: /// Comment 1.\n/// Comment 2.
            codeOffset: 65
            codeLength: 14
            type: int
            shouldUseTypeForInitializerInference: true
          hasAnnotation @100
            metadata
              Annotation
                atSign: @ @84
                name: SimpleIdentifier
                  token: Object @85
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @91
                  rightParenthesis: ) @92
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 84
            codeLength: 29
            type: int
            shouldUseTypeForInitializerInference: true
          hasAnnotation2 @115
            metadata
              Annotation
                atSign: @ @84
                name: SimpleIdentifier
                  token: Object @85
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @91
                  rightParenthesis: ) @92
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 115
            codeLength: 14
            type: int
            shouldUseTypeForInitializerInference: true
          annotationThenComment @184
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @134
                name: SimpleIdentifier
                  token: Object @135
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @141
                  rightParenthesis: ) @142
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 134
            codeLength: 71
            type: int
            shouldUseTypeForInitializerInference: true
          annotationThenComment2 @207
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @134
                name: SimpleIdentifier
                  token: Object @135
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @141
                  rightParenthesis: ) @142
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 207
            codeLength: 22
            type: int
            shouldUseTypeForInitializerInference: true
          commentThenAnnotation @284
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @268
                name: SimpleIdentifier
                  token: Object @269
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @275
                  rightParenthesis: ) @276
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 234
            codeLength: 71
            type: int
            shouldUseTypeForInitializerInference: true
          commentThenAnnotation2 @307
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @268
                name: SimpleIdentifier
                  token: Object @269
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @275
                  rightParenthesis: ) @276
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 307
            codeLength: 22
            type: int
            shouldUseTypeForInitializerInference: true
          commentAroundAnnotation @384
            documentationComment: /// Comment 2.
            metadata
              Annotation
                atSign: @ @351
                name: SimpleIdentifier
                  token: Object @352
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @358
                  rightParenthesis: ) @359
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 351
            codeLength: 56
            type: int
            shouldUseTypeForInitializerInference: true
          commentAroundAnnotation2 @409
            documentationComment: /// Comment 2.
            metadata
              Annotation
                atSign: @ @351
                name: SimpleIdentifier
                  token: Object @352
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @358
                  rightParenthesis: ) @359
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 409
            codeLength: 24
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get hasDocComment @-1
            returnType: int
          synthetic set hasDocComment @-1
            parameters
              requiredPositional _hasDocComment @-1
                type: int
            returnType: void
          synthetic get hasDocComment2 @-1
            returnType: int
          synthetic set hasDocComment2 @-1
            parameters
              requiredPositional _hasDocComment2 @-1
                type: int
            returnType: void
          synthetic get hasAnnotation @-1
            returnType: int
          synthetic set hasAnnotation @-1
            parameters
              requiredPositional _hasAnnotation @-1
                type: int
            returnType: void
          synthetic get hasAnnotation2 @-1
            returnType: int
          synthetic set hasAnnotation2 @-1
            parameters
              requiredPositional _hasAnnotation2 @-1
                type: int
            returnType: void
          synthetic get annotationThenComment @-1
            returnType: int
          synthetic set annotationThenComment @-1
            parameters
              requiredPositional _annotationThenComment @-1
                type: int
            returnType: void
          synthetic get annotationThenComment2 @-1
            returnType: int
          synthetic set annotationThenComment2 @-1
            parameters
              requiredPositional _annotationThenComment2 @-1
                type: int
            returnType: void
          synthetic get commentThenAnnotation @-1
            returnType: int
          synthetic set commentThenAnnotation @-1
            parameters
              requiredPositional _commentThenAnnotation @-1
                type: int
            returnType: void
          synthetic get commentThenAnnotation2 @-1
            returnType: int
          synthetic set commentThenAnnotation2 @-1
            parameters
              requiredPositional _commentThenAnnotation2 @-1
                type: int
            returnType: void
          synthetic get commentAroundAnnotation @-1
            returnType: int
          synthetic set commentAroundAnnotation @-1
            parameters
              requiredPositional _commentAroundAnnotation @-1
                type: int
            returnType: void
          synthetic get commentAroundAnnotation2 @-1
            returnType: int
          synthetic set commentAroundAnnotation2 @-1
            parameters
              requiredPositional _commentAroundAnnotation2 @-1
                type: int
            returnType: void
''');
  }

  test_codeRange_function() async {
    var library = await buildLibrary('''
void raw() {}

/// Comment 1.
/// Comment 2.
void hasDocComment() {}

@Object()
void hasAnnotation() {}

@Object()
/// Comment 1.
/// Comment 2.
void annotationThenComment() {}

/// Comment 1.
/// Comment 2.
@Object()
void commentThenAnnotation() {}

/// Comment 1.
@Object()
/// Comment 2.
void commentAroundAnnotation() {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    functions
      raw @5
        codeOffset: 0
        codeLength: 13
        returnType: void
      hasDocComment @50
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 15
        codeLength: 53
        returnType: void
      hasAnnotation @85
        metadata
          Annotation
            atSign: @ @70
            name: SimpleIdentifier
              token: Object @71
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @77
              rightParenthesis: ) @78
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 70
        codeLength: 33
        returnType: void
      annotationThenComment @150
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @105
            name: SimpleIdentifier
              token: Object @106
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @112
              rightParenthesis: ) @113
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 105
        codeLength: 71
        returnType: void
      commentThenAnnotation @223
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @208
            name: SimpleIdentifier
              token: Object @209
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @215
              rightParenthesis: ) @216
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 178
        codeLength: 71
        returnType: void
      commentAroundAnnotation @296
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @266
            name: SimpleIdentifier
              token: Object @267
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @273
              rightParenthesis: ) @274
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 266
        codeLength: 58
        returnType: void
''');
  }

  test_codeRange_functionTypeAlias() async {
    var library = await buildLibrary('''
typedef Raw();

/// Comment 1.
/// Comment 2.
typedef HasDocComment();

@Object()
typedef HasAnnotation();

@Object()
/// Comment 1.
/// Comment 2.
typedef AnnotationThenComment();

/// Comment 1.
/// Comment 2.
@Object()
typedef CommentThenAnnotation();

/// Comment 1.
@Object()
/// Comment 2.
typedef CommentAroundAnnotation();
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased Raw @8
        codeOffset: 0
        codeLength: 14
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      functionTypeAliasBased HasDocComment @54
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 16
        codeLength: 54
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      functionTypeAliasBased HasAnnotation @90
        metadata
          Annotation
            atSign: @ @72
            name: SimpleIdentifier
              token: Object @73
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @79
              rightParenthesis: ) @80
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 72
        codeLength: 34
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      functionTypeAliasBased AnnotationThenComment @156
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @108
            name: SimpleIdentifier
              token: Object @109
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @115
              rightParenthesis: ) @116
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 108
        codeLength: 72
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      functionTypeAliasBased CommentThenAnnotation @230
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @212
            name: SimpleIdentifier
              token: Object @213
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @219
              rightParenthesis: ) @220
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 182
        codeLength: 72
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      functionTypeAliasBased CommentAroundAnnotation @304
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @271
            name: SimpleIdentifier
              token: Object @272
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @278
              rightParenthesis: ) @279
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 271
        codeLength: 59
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
''');
  }

  test_codeRange_genericTypeAlias() async {
    var library = await buildLibrary('''
typedef Raw = Function();

/// Comment 1.
/// Comment 2.
typedef HasDocComment = Function();

@Object()
typedef HasAnnotation = Function();

@Object()
/// Comment 1.
/// Comment 2.
typedef AnnotationThenComment = Function();

/// Comment 1.
/// Comment 2.
@Object()
typedef CommentThenAnnotation = Function();

/// Comment 1.
@Object()
/// Comment 2.
typedef CommentAroundAnnotation = Function();
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      Raw @8
        codeOffset: 0
        codeLength: 25
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      HasDocComment @65
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 27
        codeLength: 65
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      HasAnnotation @112
        metadata
          Annotation
            atSign: @ @94
            name: SimpleIdentifier
              token: Object @95
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @101
              rightParenthesis: ) @102
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 94
        codeLength: 45
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      AnnotationThenComment @189
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @141
            name: SimpleIdentifier
              token: Object @142
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @148
              rightParenthesis: ) @149
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 141
        codeLength: 83
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      CommentThenAnnotation @274
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @256
            name: SimpleIdentifier
              token: Object @257
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @263
              rightParenthesis: ) @264
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 226
        codeLength: 83
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      CommentAroundAnnotation @359
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @326
            name: SimpleIdentifier
              token: Object @327
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @333
              rightParenthesis: ) @334
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 326
        codeLength: 70
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
''');
  }

  test_codeRange_method() async {
    var library = await buildLibrary('''
class C {
  void raw() {}

  /// Comment 1.
  /// Comment 2.
  void hasDocComment() {}

  @Object()
  void hasAnnotation() {}

  @Object()
  /// Comment 1.
  /// Comment 2.
  void annotationThenComment() {}

  /// Comment 1.
  /// Comment 2.
  @Object()
  void commentThenAnnotation() {}

  /// Comment 1.
  @Object()
  /// Comment 2.
  void commentAroundAnnotation() {}
}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        codeOffset: 0
        codeLength: 372
        constructors
          synthetic @-1
        methods
          raw @17
            codeOffset: 12
            codeLength: 13
            returnType: void
          hasDocComment @68
            documentationComment: /// Comment 1.\n/// Comment 2.
            codeOffset: 29
            codeLength: 57
            returnType: void
          hasAnnotation @107
            metadata
              Annotation
                atSign: @ @90
                name: SimpleIdentifier
                  token: Object @91
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @97
                  rightParenthesis: ) @98
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 90
            codeLength: 35
            returnType: void
          annotationThenComment @180
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @129
                name: SimpleIdentifier
                  token: Object @130
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @136
                  rightParenthesis: ) @137
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 129
            codeLength: 77
            returnType: void
          commentThenAnnotation @261
            documentationComment: /// Comment 1.\n/// Comment 2.
            metadata
              Annotation
                atSign: @ @244
                name: SimpleIdentifier
                  token: Object @245
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @251
                  rightParenthesis: ) @252
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 210
            codeLength: 77
            returnType: void
          commentAroundAnnotation @342
            documentationComment: /// Comment 2.
            metadata
              Annotation
                atSign: @ @308
                name: SimpleIdentifier
                  token: Object @309
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @315
                  rightParenthesis: ) @316
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 308
            codeLength: 62
            returnType: void
''');
  }

  test_codeRange_parameter() async {
    var library = await buildLibrary('''
main({int a = 1, int b, int c = 2}) {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    functions
      main @0
        codeOffset: 0
        codeLength: 38
        parameters
          optionalNamed default a @10
            type: int
            codeOffset: 6
            codeLength: 9
            constantInitializer
              IntegerLiteral
                literal: 1 @14
                staticType: int
          optionalNamed default b @21
            type: int
            codeOffset: 17
            codeLength: 5
          optionalNamed default c @28
            type: int
            codeOffset: 24
            codeLength: 9
            constantInitializer
              IntegerLiteral
                literal: 2 @32
                staticType: int
        returnType: dynamic
''');
  }

  test_codeRange_parameter_annotations() async {
    var library = await buildLibrary('''
main(@Object() int a, int b, @Object() int c) {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    functions
      main @0
        codeOffset: 0
        codeLength: 48
        parameters
          requiredPositional a @19
            type: int
            metadata
              Annotation
                atSign: @ @5
                name: SimpleIdentifier
                  token: Object @6
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @12
                  rightParenthesis: ) @13
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 5
            codeLength: 15
          requiredPositional b @26
            type: int
            codeOffset: 22
            codeLength: 5
          requiredPositional c @43
            type: int
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: Object @30
                  staticElement: dart:core::@class::Object
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @36
                  rightParenthesis: ) @37
                element: dart:core::@class::Object::@constructor::new
            codeOffset: 29
            codeLength: 15
        returnType: dynamic
''');
  }

  test_codeRange_topLevelVariable() async {
    var library = await buildLibrary('''
int withInit = 1 + 2 * 3;

int withoutInit;

int multiWithInit = 2, multiWithoutInit, multiWithInit2 = 3;
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static withInit @4
        codeOffset: 0
        codeLength: 24
        type: int
        shouldUseTypeForInitializerInference: true
      static withoutInit @31
        codeOffset: 27
        codeLength: 15
        type: int
        shouldUseTypeForInitializerInference: true
      static multiWithInit @49
        codeOffset: 45
        codeLength: 21
        type: int
        shouldUseTypeForInitializerInference: true
      static multiWithoutInit @68
        codeOffset: 68
        codeLength: 16
        type: int
        shouldUseTypeForInitializerInference: true
      static multiWithInit2 @86
        codeOffset: 86
        codeLength: 18
        type: int
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get withInit @-1
        returnType: int
      synthetic static set withInit @-1
        parameters
          requiredPositional _withInit @-1
            type: int
        returnType: void
      synthetic static get withoutInit @-1
        returnType: int
      synthetic static set withoutInit @-1
        parameters
          requiredPositional _withoutInit @-1
            type: int
        returnType: void
      synthetic static get multiWithInit @-1
        returnType: int
      synthetic static set multiWithInit @-1
        parameters
          requiredPositional _multiWithInit @-1
            type: int
        returnType: void
      synthetic static get multiWithoutInit @-1
        returnType: int
      synthetic static set multiWithoutInit @-1
        parameters
          requiredPositional _multiWithoutInit @-1
            type: int
        returnType: void
      synthetic static get multiWithInit2 @-1
        returnType: int
      synthetic static set multiWithInit2 @-1
        parameters
          requiredPositional _multiWithInit2 @-1
            type: int
        returnType: void
''');
  }

  test_codeRange_topLevelVariable_annotations() async {
    var library = await buildLibrary('''
/// Comment 1.
/// Comment 2.
int hasDocComment, hasDocComment2;

@Object()
int hasAnnotation, hasAnnotation2;

@Object()
/// Comment 1.
/// Comment 2.
int annotationThenComment, annotationThenComment2;

/// Comment 1.
/// Comment 2.
@Object()
int commentThenAnnotation, commentThenAnnotation2;

/// Comment 1.
@Object()
/// Comment 2.
int commentAroundAnnotation, commentAroundAnnotation2;
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static hasDocComment @34
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 0
        codeLength: 47
        type: int
        shouldUseTypeForInitializerInference: true
      static hasDocComment2 @49
        documentationComment: /// Comment 1.\n/// Comment 2.
        codeOffset: 49
        codeLength: 14
        type: int
        shouldUseTypeForInitializerInference: true
      static hasAnnotation @80
        metadata
          Annotation
            atSign: @ @66
            name: SimpleIdentifier
              token: Object @67
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @73
              rightParenthesis: ) @74
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 66
        codeLength: 27
        type: int
        shouldUseTypeForInitializerInference: true
      static hasAnnotation2 @95
        metadata
          Annotation
            atSign: @ @66
            name: SimpleIdentifier
              token: Object @67
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @73
              rightParenthesis: ) @74
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 95
        codeLength: 14
        type: int
        shouldUseTypeForInitializerInference: true
      static annotationThenComment @156
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @112
            name: SimpleIdentifier
              token: Object @113
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @119
              rightParenthesis: ) @120
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 112
        codeLength: 65
        type: int
        shouldUseTypeForInitializerInference: true
      static annotationThenComment2 @179
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @112
            name: SimpleIdentifier
              token: Object @113
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @119
              rightParenthesis: ) @120
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 179
        codeLength: 22
        type: int
        shouldUseTypeForInitializerInference: true
      static commentThenAnnotation @248
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @234
            name: SimpleIdentifier
              token: Object @235
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @241
              rightParenthesis: ) @242
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 204
        codeLength: 65
        type: int
        shouldUseTypeForInitializerInference: true
      static commentThenAnnotation2 @271
        documentationComment: /// Comment 1.\n/// Comment 2.
        metadata
          Annotation
            atSign: @ @234
            name: SimpleIdentifier
              token: Object @235
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @241
              rightParenthesis: ) @242
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 271
        codeLength: 22
        type: int
        shouldUseTypeForInitializerInference: true
      static commentAroundAnnotation @340
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @311
            name: SimpleIdentifier
              token: Object @312
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @318
              rightParenthesis: ) @319
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 311
        codeLength: 52
        type: int
        shouldUseTypeForInitializerInference: true
      static commentAroundAnnotation2 @365
        documentationComment: /// Comment 2.
        metadata
          Annotation
            atSign: @ @311
            name: SimpleIdentifier
              token: Object @312
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @318
              rightParenthesis: ) @319
            element: dart:core::@class::Object::@constructor::new
        codeOffset: 365
        codeLength: 24
        type: int
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get hasDocComment @-1
        returnType: int
      synthetic static set hasDocComment @-1
        parameters
          requiredPositional _hasDocComment @-1
            type: int
        returnType: void
      synthetic static get hasDocComment2 @-1
        returnType: int
      synthetic static set hasDocComment2 @-1
        parameters
          requiredPositional _hasDocComment2 @-1
            type: int
        returnType: void
      synthetic static get hasAnnotation @-1
        returnType: int
      synthetic static set hasAnnotation @-1
        parameters
          requiredPositional _hasAnnotation @-1
            type: int
        returnType: void
      synthetic static get hasAnnotation2 @-1
        returnType: int
      synthetic static set hasAnnotation2 @-1
        parameters
          requiredPositional _hasAnnotation2 @-1
            type: int
        returnType: void
      synthetic static get annotationThenComment @-1
        returnType: int
      synthetic static set annotationThenComment @-1
        parameters
          requiredPositional _annotationThenComment @-1
            type: int
        returnType: void
      synthetic static get annotationThenComment2 @-1
        returnType: int
      synthetic static set annotationThenComment2 @-1
        parameters
          requiredPositional _annotationThenComment2 @-1
            type: int
        returnType: void
      synthetic static get commentThenAnnotation @-1
        returnType: int
      synthetic static set commentThenAnnotation @-1
        parameters
          requiredPositional _commentThenAnnotation @-1
            type: int
        returnType: void
      synthetic static get commentThenAnnotation2 @-1
        returnType: int
      synthetic static set commentThenAnnotation2 @-1
        parameters
          requiredPositional _commentThenAnnotation2 @-1
            type: int
        returnType: void
      synthetic static get commentAroundAnnotation @-1
        returnType: int
      synthetic static set commentAroundAnnotation @-1
        parameters
          requiredPositional _commentAroundAnnotation @-1
            type: int
        returnType: void
      synthetic static get commentAroundAnnotation2 @-1
        returnType: int
      synthetic static set commentAroundAnnotation2 @-1
        parameters
          requiredPositional _commentAroundAnnotation2 @-1
            type: int
        returnType: void
''');
  }

  test_codeRange_type_parameter() async {
    var library = await buildLibrary('''
class A<T> {}
void f<U extends num> {}
''');
    configuration.withCodeRanges = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        codeOffset: 0
        codeLength: 13
        typeParameters
          covariant T @8
            codeOffset: 8
            codeLength: 1
            defaultType: dynamic
        constructors
          synthetic @-1
    functions
      f @19
        codeOffset: 14
        codeLength: 24
        typeParameters
          covariant U @21
            codeOffset: 21
            codeLength: 13
            bound: num
            defaultType: num
        returnType: void
''');
  }

  test_compilationUnit_nnbd_disabled_via_dart_directive() async {
    var library = await buildLibrary('''
// @dart = 2.9
''');
    expect(library.isNonNullableByDefault, isFalse);
  }

  test_compilationUnit_nnbd_enabled() async {
    var library = await buildLibrary('');
    expect(library.isNonNullableByDefault, isTrue);
  }

  test_const_asExpression() async {
    var library = await buildLibrary('''
const num a = 0;
const b = a as int;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @10
        type: num
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @14
            staticType: int
      static const b @23
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          AsExpression
            expression: SimpleIdentifier
              token: a @27
              staticElement: self::@getter::a
              staticType: num
            asOperator: as @29
            type: NamedType
              name: int @32
              element: dart:core::@class::int
              type: int
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: num
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_assignmentExpression() async {
    var library = await buildLibrary(r'''
const a = 0;
const b = (a += 1);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ParenthesizedExpression
            leftParenthesis: ( @23
            expression: AssignmentExpression
              leftHandSide: SimpleIdentifier
                token: a @24
                staticElement: <null>
                staticType: null
              operator: += @26
              rightHandSide: IntegerLiteral
                literal: 1 @29
                staticType: int
              readElement: self::@getter::a
              readType: int
              writeElement: self::@getter::a
              writeType: InvalidType
              staticElement: dart:core::@class::num::@method::+
              staticType: int
            rightParenthesis: ) @30
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_cascadeExpression() async {
    var library = await buildLibrary(r'''
const a = 0..isEven..abs();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          CascadeExpression
            target: IntegerLiteral
              literal: 0 @10
              staticType: int
            cascadeSections
              PropertyAccess
                operator: .. @11
                propertyName: SimpleIdentifier
                  token: isEven @13
                  staticElement: dart:core::@class::int::@getter::isEven
                  staticType: bool
                staticType: bool
              MethodInvocation
                operator: .. @19
                methodName: SimpleIdentifier
                  token: abs @21
                  staticElement: dart:core::@class::int::@method::abs
                  staticType: int Function()
                argumentList: ArgumentList
                  leftParenthesis: ( @24
                  rightParenthesis: ) @25
                staticInvokeType: int Function()
                staticType: int
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_const_classField() async {
    var library = await buildLibrary(r'''
class C {
  static const int f1 = 1;
  static const int f2 = C.f1, f3 = C.f2;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const f1 @29
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 1 @34
                staticType: int
          static const f2 @56
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: C @61
                  staticElement: self::@class::C
                  staticType: null
                period: . @62
                identifier: SimpleIdentifier
                  token: f1 @63
                  staticElement: self::@class::C::@getter::f1
                  staticType: int
                staticElement: self::@class::C::@getter::f1
                staticType: int
          static const f3 @67
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: C @72
                  staticElement: self::@class::C
                  staticType: null
                period: . @73
                identifier: SimpleIdentifier
                  token: f2 @74
                  staticElement: self::@class::C::@getter::f2
                  staticType: int
                staticElement: self::@class::C::@getter::f2
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get f1 @-1
            returnType: int
          synthetic static get f2 @-1
            returnType: int
          synthetic static get f3 @-1
            returnType: int
''');
  }

  test_const_constructor_inferred_args() async {
    var library = await buildLibrary('''
class C<T> {
  final T t;
  const C(this.t);
  const C.named(this.t);
}
const Object x = const C(0);
const Object y = const C.named(0);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        fields
          final t @23
            type: T
            shouldUseTypeForInitializerInference: true
        constructors
          const @34
            parameters
              requiredPositional final this.t @41
                type: T
                field: self::@class::C::@field::t
          const named @55
            periodOffset: 54
            nameEnd: 60
            parameters
              requiredPositional final this.t @66
                type: T
                field: self::@class::C::@field::t
        accessors
          synthetic get t @-1
            returnType: T
    topLevelVariables
      static const x @85
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          InstanceCreationExpression
            keyword: const @89
            constructorName: ConstructorName
              type: NamedType
                name: C @95
                element: self::@class::C
                type: C<int>
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::new
                substitution: {T: int}
            argumentList: ArgumentList
              leftParenthesis: ( @96
              arguments
                IntegerLiteral
                  literal: 0 @97
                  staticType: int
              rightParenthesis: ) @98
            staticType: C<int>
      static const y @114
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          InstanceCreationExpression
            keyword: const @118
            constructorName: ConstructorName
              type: NamedType
                name: C @124
                element: self::@class::C
                type: C<int>
              period: . @125
              name: SimpleIdentifier
                token: named @126
                staticElement: ConstructorMember
                  base: self::@class::C::@constructor::named
                  substitution: {T: dynamic}
                staticType: null
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::named
                substitution: {T: int}
            argumentList: ArgumentList
              leftParenthesis: ( @131
              arguments
                IntegerLiteral
                  literal: 0 @132
                  staticType: int
              rightParenthesis: ) @133
            staticType: C<int>
    accessors
      synthetic static get x @-1
        returnType: Object
      synthetic static get y @-1
        returnType: Object
''');
    var x = library.definingCompilationUnit.topLevelVariables[0];
    var xExpr = x.constantInitializer as InstanceCreationExpression;
    var xType = xExpr.constructorName.staticElement!.returnType;
    _assertTypeStr(
      xType,
      'C<int>',
    );
    var y = library.definingCompilationUnit.topLevelVariables[0];
    var yExpr = y.constantInitializer as InstanceCreationExpression;
    var yType = yExpr.constructorName.staticElement!.returnType;
    _assertTypeStr(yType, 'C<int>');
  }

  test_const_constructorReference() async {
    var library = await buildLibrary(r'''
class A {
  A.named();
}
const v = A.named;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          named @14
            periodOffset: 13
            nameEnd: 19
    topLevelVariables
      static const v @31
        type: A Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ConstructorReference
            constructorName: ConstructorName
              type: NamedType
                name: A @35
                element: self::@class::A
                type: null
              period: . @36
              name: SimpleIdentifier
                token: named @37
                staticElement: self::@class::A::@constructor::named
                staticType: null
              staticElement: self::@class::A::@constructor::named
            staticType: A Function()
    accessors
      synthetic static get v @-1
        returnType: A Function()
''');
  }

  test_const_finalField_hasConstConstructor() async {
    var library = await buildLibrary(r'''
class C {
  final int f = 42;
  const C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final f @22
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 42 @26
                staticType: int
        constructors
          const @38
        accessors
          synthetic get f @-1
            returnType: int
''');
  }

  test_const_functionExpression_typeArgumentTypes() async {
    var library = await buildLibrary('''
void f<T>(T a) {}

const void Function(int) v = f;
''');
    checkElementText(library, '''
library
  definingUnit
    topLevelVariables
      static const v @44
        type: void Function(int)
        shouldUseTypeForInitializerInference: true
        constantInitializer
          FunctionReference
            function: SimpleIdentifier
              token: f @48
              staticElement: self::@function::f
              staticType: void Function<T>(T)
            staticType: void Function(int)
            typeArgumentTypes
              int
    accessors
      synthetic static get v @-1
        returnType: void Function(int)
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        parameters
          requiredPositional a @12
            type: T
        returnType: void
''');
  }

  test_const_functionReference() async {
    var library = await buildLibrary(r'''
void f<T>(T a) {}
const v = f<int>;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @24
        type: void Function(int)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          FunctionReference
            function: SimpleIdentifier
              token: f @28
              staticElement: self::@function::f
              staticType: void Function<T>(T)
            typeArguments: TypeArgumentList
              leftBracket: < @29
              arguments
                NamedType
                  name: int @30
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @33
            staticType: void Function(int)
            typeArgumentTypes
              int
    accessors
      synthetic static get v @-1
        returnType: void Function(int)
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        parameters
          requiredPositional a @12
            type: T
        returnType: void
''');
  }

  test_const_indexExpression() async {
    var library = await buildLibrary(r'''
const a = [0];
const b = 0;
const c = a[b];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: List<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            leftBracket: [ @10
            elements
              IntegerLiteral
                literal: 0 @11
                staticType: int
            rightBracket: ] @12
            staticType: List<int>
      static const b @21
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @25
            staticType: int
      static const c @34
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IndexExpression
            target: SimpleIdentifier
              token: a @38
              staticElement: self::@getter::a
              staticType: List<int>
            leftBracket: [ @39
            index: SimpleIdentifier
              token: b @40
              staticElement: self::@getter::b
              staticType: int
            rightBracket: ] @41
            staticElement: MethodMember
              base: dart:core::@class::List::@method::[]
              substitution: {E: int}
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: List<int>
      synthetic static get b @-1
        returnType: int
      synthetic static get c @-1
        returnType: int
''');
  }

  test_const_inference_downward_list() async {
    var library = await buildLibrary('''
class P<T> {
  const P();
}

class P1<T> extends P<T> {
  const P1();
}

class P2<T> extends P<T> {
  const P2();
}

const List<P> values = [
  P1(),
  P2<int>(),
];
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class P @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class P1 @35
        typeParameters
          covariant T @38
            defaultType: dynamic
        supertype: P<T>
        constructors
          const @64
            superConstructor: ConstructorMember
              base: self::@class::P::@constructor::new
              substitution: {T: T}
      class P2 @79
        typeParameters
          covariant T @82
            defaultType: dynamic
        supertype: P<T>
        constructors
          const @108
            superConstructor: ConstructorMember
              base: self::@class::P::@constructor::new
              substitution: {T: T}
    topLevelVariables
      static const values @131
        type: List<P<dynamic>>
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            leftBracket: [ @140
            elements
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: P1 @144
                    element: self::@class::P1
                    type: P1<dynamic>
                  staticElement: ConstructorMember
                    base: self::@class::P1::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @146
                  rightParenthesis: ) @147
                staticType: P1<dynamic>
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: P2 @152
                    typeArguments: TypeArgumentList
                      leftBracket: < @154
                      arguments
                        NamedType
                          name: int @155
                          element: dart:core::@class::int
                          type: int
                      rightBracket: > @158
                    element: self::@class::P2
                    type: P2<int>
                  staticElement: ConstructorMember
                    base: self::@class::P2::@constructor::new
                    substitution: {T: int}
                argumentList: ArgumentList
                  leftParenthesis: ( @159
                  rightParenthesis: ) @160
                staticType: P2<int>
            rightBracket: ] @163
            staticType: List<P<dynamic>>
    accessors
      synthetic static get values @-1
        returnType: List<P<dynamic>>
''');
  }

  test_const_invalid_field_const() async {
    var library = await buildLibrary(r'''
class C {
  static const f = 1 + foo();
}
int foo() => 42;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const f @25
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @29
                  staticType: int
                operator: + @31
                rightOperand: MethodInvocation
                  methodName: SimpleIdentifier
                    token: foo @33
                    staticElement: self::@function::foo
                    staticType: int Function()
                  argumentList: ArgumentList
                    leftParenthesis: ( @36
                    rightParenthesis: ) @37
                  staticInvokeType: int Function()
                  staticType: int
                staticElement: dart:core::@class::num::@method::+
                staticInvokeType: num Function(num)
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get f @-1
            returnType: int
    functions
      foo @46
        returnType: int
''');
  }

  test_const_invalid_field_final() async {
    var library = await buildLibrary(r'''
class C {
  final f = 1 + foo();
}
int foo() => 42;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final f @18
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get f @-1
            returnType: int
    functions
      foo @39
        returnType: int
''');
  }

  test_const_invalid_functionExpression() async {
    var library = await buildLibrary('''
const v = () { return 0; };
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: int Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: _notSerializableExpression @-1
            staticElement: <null>
            staticType: null
    accessors
      synthetic static get v @-1
        returnType: int Function()
''');
  }

  test_const_invalid_functionExpression_constructorFieldInitializer() async {
    var library = await buildLibrary('''
class A {
  final Object? foo;
  const A() : foo = (() => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final foo @26
            type: Object?
            shouldUseTypeForInitializerInference: true
        constructors
          const @39
            constantInitializers
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: foo @45
                  staticElement: self::@class::A::@field::foo
                  staticType: null
                equals: = @49
                expression: SimpleIdentifier
                  token: _notSerializableExpression @-1
                  staticElement: <null>
                  staticType: null
        accessors
          synthetic get foo @-1
            returnType: Object?
''');
  }

  test_const_invalid_functionExpression_nested() async {
    var library = await buildLibrary('''
const v = () { return 0; } + 2;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: _notSerializableExpression @-1
            staticElement: <null>
            staticType: null
    accessors
      synthetic static get v @-1
        returnType: InvalidType
''');
  }

  test_const_invalid_functionExpression_redirectingConstructorInvocation() async {
    var library = await buildLibrary('''
class A {
  const A(Object a, Object b);
  const A.named() : this(0, () => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional a @27
                type: Object
              requiredPositional b @37
                type: Object
          const named @51
            periodOffset: 50
            nameEnd: 56
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @61
                argumentList: ArgumentList
                  leftParenthesis: ( @65
                  arguments
                    IntegerLiteral
                      literal: 0 @66
                      staticType: int
                    SimpleIdentifier
                      token: _notSerializableExpression @-1
                      staticElement: <null>
                      staticType: null
                  rightParenthesis: ) @76
                staticElement: self::@class::A::@constructor::new
            redirectedConstructor: self::@class::A::@constructor::new
''');
  }

  test_const_invalid_functionExpression_superConstructorInvocation() async {
    var library = await buildLibrary('''
class A {
  const A(Object a, Object b);
}
class B extends A {
  const B() : super(0, () => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional a @27
                type: Object
              requiredPositional b @37
                type: Object
      class B @49
        supertype: A
        constructors
          const @71
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @77
                argumentList: ArgumentList
                  leftParenthesis: ( @82
                  arguments
                    IntegerLiteral
                      literal: 0 @83
                      staticType: int
                    SimpleIdentifier
                      token: _notSerializableExpression @-1
                      staticElement: <null>
                      staticType: null
                  rightParenthesis: ) @93
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
''');
  }

  @FailingTest(issue: 'https://github.com/dart-lang/sdk/issues/44522')
  test_const_invalid_intLiteral() async {
    var library = await buildLibrary(r'''
const int x = 0x;
''', allowErrors: true);
    checkElementText(library, r'''
const int x = 0;
''');
  }

  test_const_invalid_methodInvocation() async {
    var library = await buildLibrary(r'''
const a = 'abc'.codeUnitAt(0);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          MethodInvocation
            target: SimpleStringLiteral
              literal: 'abc' @10
            operator: . @15
            methodName: SimpleIdentifier
              token: codeUnitAt @16
              staticElement: dart:core::@class::String::@method::codeUnitAt
              staticType: int Function(int)
            argumentList: ArgumentList
              leftParenthesis: ( @26
              arguments
                IntegerLiteral
                  literal: 0 @27
                  staticType: int
              rightParenthesis: ) @28
            staticInvokeType: int Function(int)
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_const_invalid_patternAssignment() async {
    var library = await buildLibrary('''
const v = (a,) = (0,);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: (int)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: _notSerializableExpression @-1
            staticElement: <null>
            staticType: null
    accessors
      synthetic static get v @-1
        returnType: (int)
''');
  }

  test_const_invalid_topLevel() async {
    var library = await buildLibrary(r'''
const v = 1 + foo();
int foo() => 42;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @10
              staticType: int
            operator: + @12
            rightOperand: MethodInvocation
              methodName: SimpleIdentifier
                token: foo @14
                staticElement: self::@function::foo
                staticType: int Function()
              argumentList: ArgumentList
                leftParenthesis: ( @17
                rightParenthesis: ) @18
              staticInvokeType: int Function()
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
    functions
      foo @25
        returnType: int
''');
  }

  test_const_invalid_topLevel_switchExpression() async {
    var library = await buildLibrary(r'''
const a = 0 + switch (true) {_ => 1};
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: _notSerializableExpression @-1
            staticElement: <null>
            staticType: null
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_const_invalid_typeMismatch() async {
    var library = await buildLibrary(r'''
const int a = 0;
const bool b = a + 5;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @10
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @14
            staticType: int
      static const b @28
        type: bool
        shouldUseTypeForInitializerInference: true
        constantInitializer
          BinaryExpression
            leftOperand: SimpleIdentifier
              token: a @32
              staticElement: self::@getter::a
              staticType: int
            operator: + @34
            rightOperand: IntegerLiteral
              literal: 5 @36
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: bool
''');
  }

  test_const_invokeConstructor_generic_named() async {
    var library = await buildLibrary(r'''
class C<K, V> {
  const C.named(K k, V v);
}
const V = const C<int, String>.named(1, '222');
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant K @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        constructors
          const named @26
            periodOffset: 25
            nameEnd: 31
            parameters
              requiredPositional k @34
                type: K
              requiredPositional v @39
                type: V
    topLevelVariables
      static const V @51
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @55
            constructorName: ConstructorName
              type: NamedType
                name: C @61
                typeArguments: TypeArgumentList
                  leftBracket: < @62
                  arguments
                    NamedType
                      name: int @63
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @68
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @74
                element: self::@class::C
                type: C<int, String>
              period: . @75
              name: SimpleIdentifier
                token: named @76
                staticElement: ConstructorMember
                  base: self::@class::C::@constructor::named
                  substitution: {K: int, V: String}
                staticType: null
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::named
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @81
              arguments
                IntegerLiteral
                  literal: 1 @82
                  staticType: int
                SimpleStringLiteral
                  literal: '222' @85
              rightParenthesis: ) @90
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_generic_named_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C<K, V> {
  const C.named(K k, V v);
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = const C<int, String>.named(1, '222');
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @27
            constructorName: ConstructorName
              type: NamedType
                name: C @33
                typeArguments: TypeArgumentList
                  leftBracket: < @34
                  arguments
                    NamedType
                      name: int @35
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @40
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @46
                element: package:test/a.dart::@class::C
                type: C<int, String>
              period: . @47
              name: SimpleIdentifier
                token: named @48
                staticElement: ConstructorMember
                  base: package:test/a.dart::@class::C::@constructor::named
                  substitution: {K: int, V: String}
                staticType: null
              staticElement: ConstructorMember
                base: package:test/a.dart::@class::C::@constructor::named
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @53
              arguments
                IntegerLiteral
                  literal: 1 @54
                  staticType: int
                SimpleStringLiteral
                  literal: '222' @57
              rightParenthesis: ) @62
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_generic_named_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C<K, V> {
  const C.named(K k, V v);
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C<int, String>.named(1, '222');
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                typeArguments: TypeArgumentList
                  leftBracket: < @41
                  arguments
                    NamedType
                      name: int @42
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @47
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @53
                element: package:test/a.dart::@class::C
                type: C<int, String>
              period: . @54
              name: SimpleIdentifier
                token: named @55
                staticElement: ConstructorMember
                  base: package:test/a.dart::@class::C::@constructor::named
                  substitution: {K: int, V: String}
                staticType: null
              staticElement: ConstructorMember
                base: package:test/a.dart::@class::C::@constructor::named
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @60
              arguments
                IntegerLiteral
                  literal: 1 @61
                  staticType: int
                SimpleStringLiteral
                  literal: '222' @64
              rightParenthesis: ) @69
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_generic_noTypeArguments() async {
    var library = await buildLibrary(r'''
class C<K, V> {
  const C();
}
const V = const C();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant K @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        constructors
          const @24
    topLevelVariables
      static const V @37
        type: C<dynamic, dynamic>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @41
            constructorName: ConstructorName
              type: NamedType
                name: C @47
                element: self::@class::C
                type: C<dynamic, dynamic>
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::new
                substitution: {K: dynamic, V: dynamic}
            argumentList: ArgumentList
              leftParenthesis: ( @48
              rightParenthesis: ) @49
            staticType: C<dynamic, dynamic>
    accessors
      synthetic static get V @-1
        returnType: C<dynamic, dynamic>
''');
  }

  test_const_invokeConstructor_generic_unnamed() async {
    var library = await buildLibrary(r'''
class C<K, V> {
  const C();
}
const V = const C<int, String>();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant K @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        constructors
          const @24
    topLevelVariables
      static const V @37
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @41
            constructorName: ConstructorName
              type: NamedType
                name: C @47
                typeArguments: TypeArgumentList
                  leftBracket: < @48
                  arguments
                    NamedType
                      name: int @49
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @54
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @60
                element: self::@class::C
                type: C<int, String>
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::new
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @61
              rightParenthesis: ) @62
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_generic_unnamed_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C<K, V> {
  const C();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = const C<int, String>();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @27
            constructorName: ConstructorName
              type: NamedType
                name: C @33
                typeArguments: TypeArgumentList
                  leftBracket: < @34
                  arguments
                    NamedType
                      name: int @35
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @40
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @46
                element: package:test/a.dart::@class::C
                type: C<int, String>
              staticElement: ConstructorMember
                base: package:test/a.dart::@class::C::@constructor::new
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @47
              rightParenthesis: ) @48
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_generic_unnamed_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C<K, V> {
  const C();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C<int, String>();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: C<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                typeArguments: TypeArgumentList
                  leftBracket: < @41
                  arguments
                    NamedType
                      name: int @42
                      element: dart:core::@class::int
                      type: int
                    NamedType
                      name: String @47
                      element: dart:core::@class::String
                      type: String
                  rightBracket: > @53
                element: package:test/a.dart::@class::C
                type: C<int, String>
              staticElement: ConstructorMember
                base: package:test/a.dart::@class::C::@constructor::new
                substitution: {K: int, V: String}
            argumentList: ArgumentList
              leftParenthesis: ( @54
              rightParenthesis: ) @55
            staticType: C<int, String>
    accessors
      synthetic static get V @-1
        returnType: C<int, String>
''');
  }

  test_const_invokeConstructor_named() async {
    var library = await buildLibrary(r'''
class C {
  const C.named(bool a, int b, int c, {String d, double e});
}
const V = const C.named(true, 1, 2, d: 'ccc', e: 3.4);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const named @20
            periodOffset: 19
            nameEnd: 25
            parameters
              requiredPositional a @31
                type: bool
              requiredPositional b @38
                type: int
              requiredPositional c @45
                type: int
              optionalNamed default d @56
                type: String
              optionalNamed default e @66
                type: double
    topLevelVariables
      static const V @79
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @83
            constructorName: ConstructorName
              type: NamedType
                name: C @89
                element: self::@class::C
                type: C
              period: . @90
              name: SimpleIdentifier
                token: named @91
                staticElement: self::@class::C::@constructor::named
                staticType: null
              staticElement: self::@class::C::@constructor::named
            argumentList: ArgumentList
              leftParenthesis: ( @96
              arguments
                BooleanLiteral
                  literal: true @97
                  staticType: bool
                IntegerLiteral
                  literal: 1 @103
                  staticType: int
                IntegerLiteral
                  literal: 2 @106
                  staticType: int
                NamedExpression
                  name: Label
                    label: SimpleIdentifier
                      token: d @109
                      staticElement: self::@class::C::@constructor::named::@parameter::d
                      staticType: null
                    colon: : @110
                  expression: SimpleStringLiteral
                    literal: 'ccc' @112
                NamedExpression
                  name: Label
                    label: SimpleIdentifier
                      token: e @119
                      staticElement: self::@class::C::@constructor::named::@parameter::e
                      staticType: null
                    colon: : @120
                  expression: DoubleLiteral
                    literal: 3.4 @122
                    staticType: double
              rightParenthesis: ) @125
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_named_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  const C.named();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = const C.named();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @27
            constructorName: ConstructorName
              type: NamedType
                name: C @33
                element: package:test/a.dart::@class::C
                type: C
              period: . @34
              name: SimpleIdentifier
                token: named @35
                staticElement: package:test/a.dart::@class::C::@constructor::named
                staticType: null
              staticElement: package:test/a.dart::@class::C::@constructor::named
            argumentList: ArgumentList
              leftParenthesis: ( @40
              rightParenthesis: ) @41
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_named_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  const C.named();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C.named();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                element: package:test/a.dart::@class::C
                type: C
              period: . @41
              name: SimpleIdentifier
                token: named @42
                staticElement: package:test/a.dart::@class::C::@constructor::named
                staticType: null
              staticElement: package:test/a.dart::@class::C::@constructor::named
            argumentList: ArgumentList
              leftParenthesis: ( @47
              rightParenthesis: ) @48
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_named_unresolved() async {
    var library = await buildLibrary(r'''
class C {}
const V = const C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
    topLevelVariables
      static const V @17
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @21
            constructorName: ConstructorName
              type: NamedType
                name: C @27
                element: self::@class::C
                type: C
              period: . @28
              name: SimpleIdentifier
                token: named @29
                staticElement: <null>
                staticType: null
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @34
              rightParenthesis: ) @35
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_named_unresolved2() async {
    var library = await buildLibrary(r'''
const V = const C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @10
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: C @16
                  period: . @17
                  element: <null>
                name: named @18
                element: <null>
                type: InvalidType
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @23
              rightParenthesis: ) @24
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_invokeConstructor_named_unresolved3() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                element: package:test/a.dart::@class::C
                type: C
              period: . @41
              name: SimpleIdentifier
                token: named @42
                staticElement: <null>
                staticType: null
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @47
              rightParenthesis: ) @48
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_named_unresolved4() async {
    addSource('$testPackageLibPath/a.dart', '');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                element: <null>
                type: InvalidType
              period: . @41
              name: SimpleIdentifier
                token: named @42
                staticElement: <null>
                staticType: null
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @47
              rightParenthesis: ) @48
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_invokeConstructor_named_unresolved5() async {
    var library = await buildLibrary(r'''
const V = const p.C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @10
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @16
                  period: . @17
                  element: <null>
                name: C @18
                element: <null>
                type: InvalidType
              period: . @19
              name: SimpleIdentifier
                token: named @20
                staticElement: <null>
                staticType: null
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @25
              rightParenthesis: ) @26
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_invokeConstructor_named_unresolved6() async {
    var library = await buildLibrary(r'''
class C<T> {}
const V = const C.named();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
    topLevelVariables
      static const V @20
        type: C<dynamic>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @24
            constructorName: ConstructorName
              type: NamedType
                name: C @30
                element: self::@class::C
                type: C<dynamic>
              period: . @31
              name: SimpleIdentifier
                token: named @32
                staticElement: <null>
                staticType: null
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @37
              rightParenthesis: ) @38
            staticType: C<dynamic>
    accessors
      synthetic static get V @-1
        returnType: C<dynamic>
''');
  }

  test_const_invokeConstructor_unnamed() async {
    var library = await buildLibrary(r'''
class C {
  const C();
}
const V = const C();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const @18
    topLevelVariables
      static const V @31
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @35
            constructorName: ConstructorName
              type: NamedType
                name: C @41
                element: self::@class::C
                type: C
              staticElement: self::@class::C::@constructor::new
            argumentList: ArgumentList
              leftParenthesis: ( @42
              rightParenthesis: ) @43
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_unnamed_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  const C();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = const C();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @27
            constructorName: ConstructorName
              type: NamedType
                name: C @33
                element: package:test/a.dart::@class::C
                type: C
              staticElement: package:test/a.dart::@class::C::@constructor::new
            argumentList: ArgumentList
              leftParenthesis: ( @34
              rightParenthesis: ) @35
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_unnamed_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  const C();
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                element: package:test/a.dart::@class::C
                type: C
              staticElement: package:test/a.dart::@class::C::@constructor::new
            argumentList: ArgumentList
              leftParenthesis: ( @41
              rightParenthesis: ) @42
            staticType: C
    accessors
      synthetic static get V @-1
        returnType: C
''');
  }

  test_const_invokeConstructor_unnamed_unresolved() async {
    var library = await buildLibrary(r'''
const V = const C();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @10
            constructorName: ConstructorName
              type: NamedType
                name: C @16
                element: <null>
                type: InvalidType
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @17
              rightParenthesis: ) @18
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_invokeConstructor_unnamed_unresolved2() async {
    addSource('$testPackageLibPath/a.dart', '');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = const p.C();
''', allowErrors: true);
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @32
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @38
                  period: . @39
                  element: self::@prefix::p
                name: C @40
                element: <null>
                type: InvalidType
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @41
              rightParenthesis: ) @42
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_invokeConstructor_unnamed_unresolved3() async {
    var library = await buildLibrary(r'''
const V = const p.C();
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @10
            constructorName: ConstructorName
              type: NamedType
                importPrefix: ImportPrefixReference
                  name: p @16
                  period: . @17
                  element: <null>
                name: C @18
                element: <null>
                type: InvalidType
              staticElement: <null>
            argumentList: ArgumentList
              leftParenthesis: ( @19
              rightParenthesis: ) @20
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_isExpression() async {
    var library = await buildLibrary('''
const a = 0;
const b = a is int;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IsExpression
            expression: SimpleIdentifier
              token: a @23
              staticElement: self::@getter::a
              staticType: int
            isOperator: is @25
            type: NamedType
              name: int @28
              element: dart:core::@class::int
              type: int
            staticType: bool
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: bool
''');
  }

  test_const_length_ofClassConstField() async {
    var library = await buildLibrary(r'''
class C {
  static const String F = '';
}
const int v = C.F.length;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const F @32
            type: String
            shouldUseTypeForInitializerInference: true
            constantInitializer
              SimpleStringLiteral
                literal: '' @36
        constructors
          synthetic @-1
        accessors
          synthetic static get F @-1
            returnType: String
    topLevelVariables
      static const v @52
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: C @56
                staticElement: self::@class::C
                staticType: null
              period: . @57
              identifier: SimpleIdentifier
                token: F @58
                staticElement: self::@class::C::@getter::F
                staticType: String
              staticElement: self::@class::C::@getter::F
              staticType: String
            operator: . @59
            propertyName: SimpleIdentifier
              token: length @60
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofClassConstField_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static const String F = '';
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const int v = C.F.length;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const v @27
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: C @31
                staticElement: package:test/a.dart::@class::C
                staticType: null
              period: . @32
              identifier: SimpleIdentifier
                token: F @33
                staticElement: package:test/a.dart::@class::C::@getter::F
                staticType: String
              staticElement: package:test/a.dart::@class::C::@getter::F
              staticType: String
            operator: . @34
            propertyName: SimpleIdentifier
              token: length @35
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofClassConstField_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static const String F = '';
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const int v = p.C.F.length;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const v @32
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          PropertyAccess
            target: PropertyAccess
              target: PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: p @36
                  staticElement: self::@prefix::p
                  staticType: null
                period: . @37
                identifier: SimpleIdentifier
                  token: C @38
                  staticElement: package:test/a.dart::@class::C
                  staticType: null
                staticElement: package:test/a.dart::@class::C
                staticType: null
              operator: . @39
              propertyName: SimpleIdentifier
                token: F @40
                staticElement: package:test/a.dart::@class::C::@getter::F
                staticType: String
              staticType: String
            operator: . @41
            propertyName: SimpleIdentifier
              token: length @42
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofStringLiteral() async {
    var library = await buildLibrary(r'''
const v = 'abc'.length;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PropertyAccess
            target: SimpleStringLiteral
              literal: 'abc' @10
            operator: . @15
            propertyName: SimpleIdentifier
              token: length @16
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofTopLevelVariable() async {
    var library = await buildLibrary(r'''
const String S = 'abc';
const v = S.length;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const S @13
        type: String
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SimpleStringLiteral
            literal: 'abc' @17
      static const v @30
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: S @34
              staticElement: self::@getter::S
              staticType: String
            period: . @35
            identifier: SimpleIdentifier
              token: length @36
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticElement: dart:core::@class::String::@getter::length
            staticType: int
    accessors
      synthetic static get S @-1
        returnType: String
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofTopLevelVariable_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
const String S = 'abc';
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const v = S.length;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const v @23
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: S @27
              staticElement: package:test/a.dart::@getter::S
              staticType: String
            period: . @28
            identifier: SimpleIdentifier
              token: length @29
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticElement: dart:core::@class::String::@getter::length
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_ofTopLevelVariable_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
const String S = 'abc';
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const v = p.S.length;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const v @28
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: p @32
                staticElement: self::@prefix::p
                staticType: null
              period: . @33
              identifier: SimpleIdentifier
                token: S @34
                staticElement: package:test/a.dart::@getter::S
                staticType: String
              staticElement: package:test/a.dart::@getter::S
              staticType: String
            operator: . @35
            propertyName: SimpleIdentifier
              token: length @36
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v @-1
        returnType: int
''');
  }

  test_const_length_staticMethod() async {
    var library = await buildLibrary(r'''
class C {
  static int length() => 42;
}
const v = C.length;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          static length @23
            returnType: int
    topLevelVariables
      static const v @47
        type: int Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @51
              staticElement: self::@class::C
              staticType: null
            period: . @52
            identifier: SimpleIdentifier
              token: length @53
              staticElement: self::@class::C::@method::length
              staticType: int Function()
            staticElement: self::@class::C::@method::length
            staticType: int Function()
    accessors
      synthetic static get v @-1
        returnType: int Function()
''');
  }

  test_const_list_if() async {
    var library = await buildLibrary('''
const Object x = const <int>[if (true) 1];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: [ @28
            elements
              IfElement
                ifKeyword: if @29
                leftParenthesis: ( @32
                expression: BooleanLiteral
                  literal: true @33
                  staticType: bool
                rightParenthesis: ) @37
                thenElement: IntegerLiteral
                  literal: 1 @39
                  staticType: int
            rightBracket: ] @40
            staticType: List<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_list_if_else() async {
    var library = await buildLibrary('''
const Object x = const <int>[if (true) 1 else 2];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: [ @28
            elements
              IfElement
                ifKeyword: if @29
                leftParenthesis: ( @32
                expression: BooleanLiteral
                  literal: true @33
                  staticType: bool
                rightParenthesis: ) @37
                thenElement: IntegerLiteral
                  literal: 1 @39
                  staticType: int
                elseKeyword: else @41
                elseElement: IntegerLiteral
                  literal: 2 @46
                  staticType: int
            rightBracket: ] @47
            staticType: List<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_list_inferredType() async {
    // The summary needs to contain enough information so that when the constant
    // is resynthesized, the constant value can get the type that was computed
    // by type inference.
    var library = await buildLibrary('''
const Object x = const [1];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            constKeyword: const @17
            leftBracket: [ @23
            elements
              IntegerLiteral
                literal: 1 @24
                staticType: int
            rightBracket: ] @25
            staticType: List<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_list_spread() async {
    var library = await buildLibrary('''
const Object x = const <int>[...<int>[1]];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: [ @28
            elements
              SpreadElement
                spreadOperator: ... @29
                expression: ListLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @32
                    arguments
                      NamedType
                        name: int @33
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @36
                  leftBracket: [ @37
                  elements
                    IntegerLiteral
                      literal: 1 @38
                      staticType: int
                  rightBracket: ] @39
                  staticType: List<int>
            rightBracket: ] @40
            staticType: List<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_list_spread_null_aware() async {
    var library = await buildLibrary('''
const Object x = const <int>[...?<int>[1]];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: [ @28
            elements
              SpreadElement
                spreadOperator: ...? @29
                expression: ListLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @33
                    arguments
                      NamedType
                        name: int @34
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @37
                  leftBracket: [ @38
                  elements
                    IntegerLiteral
                      literal: 1 @39
                      staticType: int
                  rightBracket: ] @40
                  staticType: List<int>
            rightBracket: ] @41
            staticType: List<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_map_if() async {
    var library = await buildLibrary('''
const Object x = const <int, int>{if (true) 1: 2};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: int @29
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @32
            leftBracket: { @33
            elements
              IfElement
                ifKeyword: if @34
                leftParenthesis: ( @37
                expression: BooleanLiteral
                  literal: true @38
                  staticType: bool
                rightParenthesis: ) @42
                thenElement: MapLiteralEntry
                  key: IntegerLiteral
                    literal: 1 @44
                    staticType: int
                  separator: : @45
                  value: IntegerLiteral
                    literal: 2 @47
                    staticType: int
            rightBracket: } @48
            isMap: true
            staticType: Map<int, int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_map_inferredType() async {
    // The summary needs to contain enough information so that when the constant
    // is resynthesized, the constant value can get the type that was computed
    // by type inference.
    var library = await buildLibrary('''
const Object x = const {1: 1.0};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            leftBracket: { @23
            elements
              MapLiteralEntry
                key: IntegerLiteral
                  literal: 1 @24
                  staticType: int
                separator: : @25
                value: DoubleLiteral
                  literal: 1.0 @27
                  staticType: double
            rightBracket: } @30
            isMap: true
            staticType: Map<int, double>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_map_spread() async {
    var library = await buildLibrary('''
const Object x = const <int, int>{...<int, int>{1: 2}};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: int @29
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @32
            leftBracket: { @33
            elements
              SpreadElement
                spreadOperator: ... @34
                expression: SetOrMapLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @37
                    arguments
                      NamedType
                        name: int @38
                        element: dart:core::@class::int
                        type: int
                      NamedType
                        name: int @43
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @46
                  leftBracket: { @47
                  elements
                    MapLiteralEntry
                      key: IntegerLiteral
                        literal: 1 @48
                        staticType: int
                      separator: : @49
                      value: IntegerLiteral
                        literal: 2 @51
                        staticType: int
                  rightBracket: } @52
                  isMap: true
                  staticType: Map<int, int>
            rightBracket: } @53
            isMap: true
            staticType: Map<int, int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_map_spread_null_aware() async {
    var library = await buildLibrary('''
const Object x = const <int, int>{...?<int, int>{1: 2}};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: int @29
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @32
            leftBracket: { @33
            elements
              SpreadElement
                spreadOperator: ...? @34
                expression: SetOrMapLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @38
                    arguments
                      NamedType
                        name: int @39
                        element: dart:core::@class::int
                        type: int
                      NamedType
                        name: int @44
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @47
                  leftBracket: { @48
                  elements
                    MapLiteralEntry
                      key: IntegerLiteral
                        literal: 1 @49
                        staticType: int
                      separator: : @50
                      value: IntegerLiteral
                        literal: 2 @52
                        staticType: int
                  rightBracket: } @53
                  isMap: true
                  staticType: Map<int, int>
            rightBracket: } @54
            isMap: true
            staticType: Map<int, int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_methodInvocation() async {
    var library = await buildLibrary(r'''
T f<T>(T a) => a;
const b = f<int>(0);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const b @24
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          MethodInvocation
            methodName: SimpleIdentifier
              token: f @28
              staticElement: self::@function::f
              staticType: T Function<T>(T)
            typeArguments: TypeArgumentList
              leftBracket: < @29
              arguments
                NamedType
                  name: int @30
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @33
            argumentList: ArgumentList
              leftParenthesis: ( @34
              arguments
                IntegerLiteral
                  literal: 0 @35
                  staticType: int
              rightParenthesis: ) @36
            staticInvokeType: int Function(int)
            staticType: int
            typeArgumentTypes
              int
    accessors
      synthetic static get b @-1
        returnType: int
    functions
      f @2
        typeParameters
          covariant T @4
            defaultType: dynamic
        parameters
          requiredPositional a @9
            type: T
        returnType: T
''');
  }

  test_const_parameterDefaultValue_initializingFormal_functionTyped() async {
    var library = await buildLibrary(r'''
class C {
  final x;
  const C({this.x: foo});
}
int foo() => 42;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            parameters
              optionalNamed default final this.x @37
                type: dynamic
                constantInitializer
                  SimpleIdentifier
                    token: foo @40
                    staticElement: self::@function::foo
                    staticType: int Function()
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
    functions
      foo @53
        returnType: int
''');
  }

  test_const_parameterDefaultValue_initializingFormal_named() async {
    var library = await buildLibrary(r'''
class C {
  final x;
  const C({this.x: 1 + 2});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            parameters
              optionalNamed default final this.x @37
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @40
                      staticType: int
                    operator: + @42
                    rightOperand: IntegerLiteral
                      literal: 2 @44
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_const_parameterDefaultValue_initializingFormal_positional() async {
    var library = await buildLibrary(r'''
class C {
  final x;
  const C([this.x = 1 + 2]);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @18
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @29
            parameters
              optionalPositional default final this.x @37
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @41
                      staticType: int
                    operator: + @43
                    rightOperand: IntegerLiteral
                      literal: 2 @45
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_const_parameterDefaultValue_normal() async {
    var library = await buildLibrary(r'''
class C {
  const C.positional([p = 1 + 2]);
  const C.named({p: 1 + 2});
  void methodPositional([p = 1 + 2]) {}
  void methodPositionalWithoutDefault([p]) {}
  void methodNamed({p: 1 + 2}) {}
  void methodNamedWithoutDefault({p}) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          const positional @20
            periodOffset: 19
            nameEnd: 30
            parameters
              optionalPositional default p @32
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @36
                      staticType: int
                    operator: + @38
                    rightOperand: IntegerLiteral
                      literal: 2 @40
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
          const named @55
            periodOffset: 54
            nameEnd: 60
            parameters
              optionalNamed default p @62
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @65
                      staticType: int
                    operator: + @67
                    rightOperand: IntegerLiteral
                      literal: 2 @69
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
        methods
          methodPositional @81
            parameters
              optionalPositional default p @99
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @103
                      staticType: int
                    operator: + @105
                    rightOperand: IntegerLiteral
                      literal: 2 @107
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
            returnType: void
          methodPositionalWithoutDefault @121
            parameters
              optionalPositional default p @153
                type: dynamic
            returnType: void
          methodNamed @167
            parameters
              optionalNamed default p @180
                type: dynamic
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @183
                      staticType: int
                    operator: + @185
                    rightOperand: IntegerLiteral
                      literal: 2 @187
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
            returnType: void
          methodNamedWithoutDefault @201
            parameters
              optionalNamed default p @228
                type: dynamic
            returnType: void
''');
  }

  test_const_postfixExpression_increment() async {
    var library = await buildLibrary(r'''
const a = 0;
const b = a++;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PostfixExpression
            operand: SimpleIdentifier
              token: a @23
              staticElement: <null>
              staticType: null
            operator: ++ @24
            readElement: self::@getter::a
            readType: int
            writeElement: self::@getter::a
            writeType: InvalidType
            staticElement: dart:core::@class::num::@method::+
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_postfixExpression_nullCheck() async {
    var library = await buildLibrary(r'''
const int? a = 0;
const b = a!;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @11
        type: int?
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @15
            staticType: int
      static const b @24
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PostfixExpression
            operand: SimpleIdentifier
              token: a @28
              staticElement: self::@getter::a
              staticType: int?
            operator: ! @29
            staticElement: <null>
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int?
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_prefixExpression_class_unaryMinus() async {
    var library = await buildLibrary(r'''
const a = 0;
const b = -a;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: - @23
            operand: SimpleIdentifier
              token: a @24
              staticElement: self::@getter::a
              staticType: int
            staticElement: dart:core::@class::int::@method::unary-
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_prefixExpression_extension_unaryMinus() async {
    addSource('$testPackageLibPath/a.dart', r'''
extension E on Object {
  int operator -() => 0;
}
const a = const Object();
''');
    var library = await buildLibrary('''
import 'a.dart';
const b = -a;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const b @23
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: - @27
            operand: SimpleIdentifier
              token: a @28
              staticElement: package:test/a.dart::@getter::a
              staticType: Object
            staticElement: package:test/a.dart::@extension::E::@method::unary-
            staticType: int
    accessors
      synthetic static get b @-1
        returnType: int
''');
  }

  test_const_prefixExpression_increment() async {
    var library = await buildLibrary(r'''
const a = 0;
const b = ++a;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: ++ @23
            operand: SimpleIdentifier
              token: a @25
              staticElement: <null>
              staticType: null
            readElement: self::@getter::a
            readType: int
            writeElement: self::@getter::a
            writeType: InvalidType
            staticElement: dart:core::@class::num::@method::+
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: int
''');
  }

  void test_const_recordLiteral() async {
    var library = await buildLibrary('''
const a = 0;
const b = (a, a: a);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: (int, {int a})
        shouldUseTypeForInitializerInference: false
        constantInitializer
          RecordLiteral
            leftParenthesis: ( @23
            fields
              SimpleIdentifier
                token: a @24
                staticElement: self::@getter::a
                staticType: int
              NamedExpression
                name: Label
                  label: SimpleIdentifier
                    token: a @27
                    staticElement: <null>
                    staticType: null
                  colon: : @28
                expression: SimpleIdentifier
                  token: a @30
                  staticElement: self::@getter::a
                  staticType: int
            rightParenthesis: ) @31
            staticType: (int, {int a})
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: (int, {int a})
''');
  }

  void test_const_recordLiteral_explicitConst() async {
    var library = await buildLibrary('''
const a = 0;
const b = const (a, a: a);
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static const b @19
        type: (int, {int a})
        shouldUseTypeForInitializerInference: false
        constantInitializer
          RecordLiteral
            constKeyword: const @23
            leftParenthesis: ( @29
            fields
              SimpleIdentifier
                token: a @30
                staticElement: self::@getter::a
                staticType: int
              NamedExpression
                name: Label
                  label: SimpleIdentifier
                    token: a @33
                    staticElement: <null>
                    staticType: null
                  colon: : @34
                expression: SimpleIdentifier
                  token: a @36
                  staticElement: self::@getter::a
                  staticType: int
            rightParenthesis: ) @37
            staticType: (int, {int a})
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get b @-1
        returnType: (int, {int a})
''');
  }

  test_const_reference_staticField() async {
    var library = await buildLibrary(r'''
class C {
  static const int F = 42;
}
const V = C.F;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const F @29
            type: int
            shouldUseTypeForInitializerInference: true
            constantInitializer
              IntegerLiteral
                literal: 42 @33
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get F @-1
            returnType: int
    topLevelVariables
      static const V @45
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @49
              staticElement: self::@class::C
              staticType: null
            period: . @50
            identifier: SimpleIdentifier
              token: F @51
              staticElement: self::@class::C::@getter::F
              staticType: int
            staticElement: self::@class::C::@getter::F
            staticType: int
    accessors
      synthetic static get V @-1
        returnType: int
''');
  }

  test_const_reference_staticField_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static const int F = 42;
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = C.F;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @27
              staticElement: package:test/a.dart::@class::C
              staticType: null
            period: . @28
            identifier: SimpleIdentifier
              token: F @29
              staticElement: package:test/a.dart::@class::C::@getter::F
              staticType: int
            staticElement: package:test/a.dart::@class::C::@getter::F
            staticType: int
    accessors
      synthetic static get V @-1
        returnType: int
''');
  }

  test_const_reference_staticField_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static const int F = 42;
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = p.C.F;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: p @32
                staticElement: self::@prefix::p
                staticType: null
              period: . @33
              identifier: SimpleIdentifier
                token: C @34
                staticElement: package:test/a.dart::@class::C
                staticType: null
              staticElement: package:test/a.dart::@class::C
              staticType: null
            operator: . @35
            propertyName: SimpleIdentifier
              token: F @36
              staticElement: package:test/a.dart::@class::C::@getter::F
              staticType: int
            staticType: int
    accessors
      synthetic static get V @-1
        returnType: int
''');
  }

  test_const_reference_staticMethod() async {
    var library = await buildLibrary(r'''
class C {
  static int m(int a, String b) => 42;
}
const V = C.m;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          static m @23
            parameters
              requiredPositional a @29
                type: int
              requiredPositional b @39
                type: String
            returnType: int
    topLevelVariables
      static const V @57
        type: int Function(int, String)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @61
              staticElement: self::@class::C
              staticType: null
            period: . @62
            identifier: SimpleIdentifier
              token: m @63
              staticElement: self::@class::C::@method::m
              staticType: int Function(int, String)
            staticElement: self::@class::C::@method::m
            staticType: int Function(int, String)
    accessors
      synthetic static get V @-1
        returnType: int Function(int, String)
''');
  }

  test_const_reference_staticMethod_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static int m(int a, String b) => 42;
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = C.m;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: int Function(int, String)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @27
              staticElement: package:test/a.dart::@class::C
              staticType: null
            period: . @28
            identifier: SimpleIdentifier
              token: m @29
              staticElement: package:test/a.dart::@class::C::@method::m
              staticType: int Function(int, String)
            staticElement: package:test/a.dart::@class::C::@method::m
            staticType: int Function(int, String)
    accessors
      synthetic static get V @-1
        returnType: int Function(int, String)
''');
  }

  test_const_reference_staticMethod_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {
  static int m(int a, String b) => 42;
}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = p.C.m;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: int Function(int, String)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: p @32
                staticElement: self::@prefix::p
                staticType: null
              period: . @33
              identifier: SimpleIdentifier
                token: C @34
                staticElement: package:test/a.dart::@class::C
                staticType: null
              staticElement: package:test/a.dart::@class::C
              staticType: null
            operator: . @35
            propertyName: SimpleIdentifier
              token: m @36
              staticElement: package:test/a.dart::@class::C::@method::m
              staticType: int Function(int, String)
            staticType: int Function(int, String)
    accessors
      synthetic static get V @-1
        returnType: int Function(int, String)
''');
  }

  test_const_reference_staticMethod_ofExtension() async {
    var library = await buildLibrary('''
class A {}
extension E on A {
  static void f() {}
}
const x = E.f;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
    extensions
      E @21
        extendedType: A
        methods
          static f @44
            returnType: void
    topLevelVariables
      static const x @59
        type: void Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: E @63
              staticElement: self::@extension::E
              staticType: null
            period: . @64
            identifier: SimpleIdentifier
              token: f @65
              staticElement: self::@extension::E::@method::f
              staticType: void Function()
            staticElement: self::@extension::E::@method::f
            staticType: void Function()
    accessors
      synthetic static get x @-1
        returnType: void Function()
''');
  }

  test_const_reference_topLevelFunction() async {
    var library = await buildLibrary(r'''
foo() {}
const V = foo;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @15
        type: dynamic Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: foo @19
            staticElement: self::@function::foo
            staticType: dynamic Function()
    accessors
      synthetic static get V @-1
        returnType: dynamic Function()
    functions
      foo @0
        returnType: dynamic
''');
  }

  test_const_reference_topLevelFunction_generic() async {
    var library = await buildLibrary(r'''
R foo<P, R>(P p) {}
const V = foo;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @26
        type: R Function<P, R>(P)
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: foo @30
            staticElement: self::@function::foo
            staticType: R Function<P, R>(P)
    accessors
      synthetic static get V @-1
        returnType: R Function<P, R>(P)
    functions
      foo @2
        typeParameters
          covariant P @6
            defaultType: dynamic
          covariant R @9
            defaultType: dynamic
        parameters
          requiredPositional p @14
            type: P
        returnType: R
''');
  }

  test_const_reference_topLevelFunction_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
foo() {}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const V = foo;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const V @23
        type: dynamic Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: foo @27
            staticElement: package:test/a.dart::@function::foo
            staticType: dynamic Function()
    accessors
      synthetic static get V @-1
        returnType: dynamic Function()
''');
  }

  test_const_reference_topLevelFunction_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
foo() {}
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const V = p.foo;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const V @28
        type: dynamic Function()
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: p @32
              staticElement: self::@prefix::p
              staticType: null
            period: . @33
            identifier: SimpleIdentifier
              token: foo @34
              staticElement: package:test/a.dart::@function::foo
              staticType: dynamic Function()
            staticElement: package:test/a.dart::@function::foo
            staticType: dynamic Function()
    accessors
      synthetic static get V @-1
        returnType: dynamic Function()
''');
  }

  test_const_reference_topLevelVariable() async {
    var library = await buildLibrary(r'''
const A = 1;
const B = A + 2;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const A @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 1 @10
            staticType: int
      static const B @19
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: SimpleIdentifier
              token: A @23
              staticElement: self::@getter::A
              staticType: int
            operator: + @25
            rightOperand: IntegerLiteral
              literal: 2 @27
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
    accessors
      synthetic static get A @-1
        returnType: int
      synthetic static get B @-1
        returnType: int
''');
  }

  test_const_reference_topLevelVariable_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
const A = 1;
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const B = A + 2;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const B @23
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: SimpleIdentifier
              token: A @27
              staticElement: package:test/a.dart::@getter::A
              staticType: int
            operator: + @29
            rightOperand: IntegerLiteral
              literal: 2 @31
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
    accessors
      synthetic static get B @-1
        returnType: int
''');
  }

  test_const_reference_topLevelVariable_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
const A = 1;
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const B = p.A + 2;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const B @28
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: p @32
                staticElement: self::@prefix::p
                staticType: null
              period: . @33
              identifier: SimpleIdentifier
                token: A @34
                staticElement: package:test/a.dart::@getter::A
                staticType: int
              staticElement: package:test/a.dart::@getter::A
              staticType: int
            operator: + @36
            rightOperand: IntegerLiteral
              literal: 2 @38
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
    accessors
      synthetic static get B @-1
        returnType: int
''');
  }

  test_const_reference_type() async {
    var library = await buildLibrary(r'''
class C {}
class D<T> {}
enum E {a, b, c}
typedef F(int a, String b);
const vDynamic = dynamic;
const vNull = Null;
const vObject = Object;
const vClass = C;
const vGenericClass = D;
const vEnum = E;
const vFunctionTypeAlias = F;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
      class D @17
        typeParameters
          covariant T @19
            defaultType: dynamic
        constructors
          synthetic @-1
    enums
      enum E @30
        supertype: Enum
        fields
          static const enumConstant a @33
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @36
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant c @39
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get c @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    typeAliases
      functionTypeAliasBased F @50
        aliasedType: dynamic Function(int, String)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @56
              type: int
            requiredPositional b @66
              type: String
          returnType: dynamic
    topLevelVariables
      static const vDynamic @76
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: dynamic @87
            staticElement: dynamic@-1
            staticType: Type
      static const vNull @102
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: Null @110
            staticElement: dart:core::@class::Null
            staticType: Type
      static const vObject @122
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: Object @132
            staticElement: dart:core::@class::Object
            staticType: Type
      static const vClass @146
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: C @155
            staticElement: self::@class::C
            staticType: Type
      static const vGenericClass @164
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: D @180
            staticElement: self::@class::D
            staticType: Type
      static const vEnum @189
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: E @197
            staticElement: self::@enum::E
            staticType: Type
      static const vFunctionTypeAlias @206
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: F @227
            staticElement: self::@typeAlias::F
            staticType: Type
    accessors
      synthetic static get vDynamic @-1
        returnType: Type
      synthetic static get vNull @-1
        returnType: Type
      synthetic static get vObject @-1
        returnType: Type
      synthetic static get vClass @-1
        returnType: Type
      synthetic static get vGenericClass @-1
        returnType: Type
      synthetic static get vEnum @-1
        returnType: Type
      synthetic static get vFunctionTypeAlias @-1
        returnType: Type
''');
  }

  test_const_reference_type_functionType() async {
    var library = await buildLibrary(r'''
typedef F();
class C {
  final f = <F>[];
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @19
        fields
          final f @31
            type: List<dynamic Function()>
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get f @-1
            returnType: List<dynamic Function()>
    typeAliases
      functionTypeAliasBased F @8
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
''');
  }

  test_const_reference_type_imported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {}
enum E {a, b, c}
typedef F(int a, String b);
''');
    var library = await buildLibrary(r'''
import 'a.dart';
const vClass = C;
const vEnum = E;
const vFunctionTypeAlias = F;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const vClass @23
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: C @32
            staticElement: package:test/a.dart::@class::C
            staticType: Type
      static const vEnum @41
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: E @49
            staticElement: package:test/a.dart::@enum::E
            staticType: Type
      static const vFunctionTypeAlias @58
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: F @79
            staticElement: package:test/a.dart::@typeAlias::F
            staticType: Type
    accessors
      synthetic static get vClass @-1
        returnType: Type
      synthetic static get vEnum @-1
        returnType: Type
      synthetic static get vFunctionTypeAlias @-1
        returnType: Type
''');
  }

  test_const_reference_type_imported_withPrefix() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C {}
enum E {a, b, c}
typedef F(int a, String b);
''');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const vClass = p.C;
const vEnum = p.E;
const vFunctionTypeAlias = p.F;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const vClass @28
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: p @37
              staticElement: self::@prefix::p
              staticType: null
            period: . @38
            identifier: SimpleIdentifier
              token: C @39
              staticElement: package:test/a.dart::@class::C
              staticType: Type
            staticElement: package:test/a.dart::@class::C
            staticType: Type
      static const vEnum @48
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: p @56
              staticElement: self::@prefix::p
              staticType: null
            period: . @57
            identifier: SimpleIdentifier
              token: E @58
              staticElement: package:test/a.dart::@enum::E
              staticType: Type
            staticElement: package:test/a.dart::@enum::E
            staticType: Type
      static const vFunctionTypeAlias @67
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: p @88
              staticElement: self::@prefix::p
              staticType: null
            period: . @89
            identifier: SimpleIdentifier
              token: F @90
              staticElement: package:test/a.dart::@typeAlias::F
              staticType: Type
            staticElement: package:test/a.dart::@typeAlias::F
            staticType: Type
    accessors
      synthetic static get vClass @-1
        returnType: Type
      synthetic static get vEnum @-1
        returnType: Type
      synthetic static get vFunctionTypeAlias @-1
        returnType: Type
''');
  }

  test_const_reference_type_typeParameter() async {
    var library = await buildLibrary(r'''
class C<T> {
  final f = <T>[];
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        fields
          final f @21
            type: List<T>
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get f @-1
            returnType: List<T>
''');
  }

  test_const_reference_unresolved_prefix0() async {
    var library = await buildLibrary(r'''
const V = foo;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const V @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: foo @10
            staticElement: <null>
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_reference_unresolved_prefix1() async {
    var library = await buildLibrary(r'''
class C {}
const V = C.foo;
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
    topLevelVariables
      static const V @17
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixedIdentifier
            prefix: SimpleIdentifier
              token: C @21
              staticElement: self::@class::C
              staticType: null
            period: . @22
            identifier: SimpleIdentifier
              token: foo @23
              staticElement: <null>
              staticType: InvalidType
            staticElement: <null>
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_reference_unresolved_prefix2() async {
    addSource('$testPackageLibPath/foo.dart', '''
class C {}
''');
    var library = await buildLibrary(r'''
import 'foo.dart' as p;
const V = p.C.foo;
''', allowErrors: true);
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as p @21
  definingUnit
    topLevelVariables
      static const V @30
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PropertyAccess
            target: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: p @34
                staticElement: self::@prefix::p
                staticType: null
              period: . @35
              identifier: SimpleIdentifier
                token: C @36
                staticElement: package:test/foo.dart::@class::C
                staticType: null
              staticElement: package:test/foo.dart::@class::C
              staticType: null
            operator: . @37
            propertyName: SimpleIdentifier
              token: foo @38
              staticElement: <null>
              staticType: InvalidType
            staticType: InvalidType
    accessors
      synthetic static get V @-1
        returnType: InvalidType
''');
  }

  test_const_set_if() async {
    var library = await buildLibrary('''
const Object x = const <int>{if (true) 1};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: { @28
            elements
              IfElement
                ifKeyword: if @29
                leftParenthesis: ( @32
                expression: BooleanLiteral
                  literal: true @33
                  staticType: bool
                rightParenthesis: ) @37
                thenElement: IntegerLiteral
                  literal: 1 @39
                  staticType: int
            rightBracket: } @40
            isMap: false
            staticType: Set<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_set_inferredType() async {
    // The summary needs to contain enough information so that when the constant
    // is resynthesized, the constant value can get the type that was computed
    // by type inference.
    var library = await buildLibrary('''
const Object x = const {1};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            leftBracket: { @23
            elements
              IntegerLiteral
                literal: 1 @24
                staticType: int
            rightBracket: } @25
            isMap: false
            staticType: Set<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_set_spread() async {
    var library = await buildLibrary('''
const Object x = const <int>{...<int>{1}};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: { @28
            elements
              SpreadElement
                spreadOperator: ... @29
                expression: SetOrMapLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @32
                    arguments
                      NamedType
                        name: int @33
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @36
                  leftBracket: { @37
                  elements
                    IntegerLiteral
                      literal: 1 @38
                      staticType: int
                  rightBracket: } @39
                  isMap: false
                  staticType: Set<int>
            rightBracket: } @40
            isMap: false
            staticType: Set<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_set_spread_null_aware() async {
    var library = await buildLibrary('''
const Object x = const <int>{...?<int>{1}};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const x @13
        type: Object
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @17
            typeArguments: TypeArgumentList
              leftBracket: < @23
              arguments
                NamedType
                  name: int @24
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @27
            leftBracket: { @28
            elements
              SpreadElement
                spreadOperator: ...? @29
                expression: SetOrMapLiteral
                  typeArguments: TypeArgumentList
                    leftBracket: < @33
                    arguments
                      NamedType
                        name: int @34
                        element: dart:core::@class::int
                        type: int
                    rightBracket: > @37
                  leftBracket: { @38
                  elements
                    IntegerLiteral
                      literal: 1 @39
                      staticType: int
                  rightBracket: } @40
                  isMap: false
                  staticType: Set<int>
            rightBracket: } @41
            isMap: false
            staticType: Set<int>
    accessors
      synthetic static get x @-1
        returnType: Object
''');
  }

  test_const_topLevel_binary() async {
    var library = await buildLibrary(r'''
const vEqual = 1 == 2;
const vAnd = true && false;
const vOr = false || true;
const vBitXor = 1 ^ 2;
const vBitAnd = 1 & 2;
const vBitOr = 1 | 2;
const vBitShiftLeft = 1 << 2;
const vBitShiftRight = 1 >> 2;
const vAdd = 1 + 2;
const vSubtract = 1 - 2;
const vMiltiply = 1 * 2;
const vDivide = 1 / 2;
const vFloorDivide = 1 ~/ 2;
const vModulo = 1 % 2;
const vGreater = 1 > 2;
const vGreaterEqual = 1 >= 2;
const vLess = 1 < 2;
const vLessEqual = 1 <= 2;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vEqual @6
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @15
              staticType: int
            operator: == @17
            rightOperand: IntegerLiteral
              literal: 2 @20
              staticType: int
            staticElement: dart:core::@class::num::@method::==
            staticInvokeType: bool Function(Object)
            staticType: bool
      static const vAnd @29
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: BooleanLiteral
              literal: true @36
              staticType: bool
            operator: && @41
            rightOperand: BooleanLiteral
              literal: false @44
              staticType: bool
            staticElement: <null>
            staticInvokeType: null
            staticType: bool
      static const vOr @57
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: BooleanLiteral
              literal: false @63
              staticType: bool
            operator: || @69
            rightOperand: BooleanLiteral
              literal: true @72
              staticType: bool
            staticElement: <null>
            staticInvokeType: null
            staticType: bool
      static const vBitXor @84
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @94
              staticType: int
            operator: ^ @96
            rightOperand: IntegerLiteral
              literal: 2 @98
              staticType: int
            staticElement: dart:core::@class::int::@method::^
            staticInvokeType: int Function(int)
            staticType: int
      static const vBitAnd @107
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @117
              staticType: int
            operator: & @119
            rightOperand: IntegerLiteral
              literal: 2 @121
              staticType: int
            staticElement: dart:core::@class::int::@method::&
            staticInvokeType: int Function(int)
            staticType: int
      static const vBitOr @130
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @139
              staticType: int
            operator: | @141
            rightOperand: IntegerLiteral
              literal: 2 @143
              staticType: int
            staticElement: dart:core::@class::int::@method::|
            staticInvokeType: int Function(int)
            staticType: int
      static const vBitShiftLeft @152
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @168
              staticType: int
            operator: << @170
            rightOperand: IntegerLiteral
              literal: 2 @173
              staticType: int
            staticElement: dart:core::@class::int::@method::<<
            staticInvokeType: int Function(int)
            staticType: int
      static const vBitShiftRight @182
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @199
              staticType: int
            operator: >> @201
            rightOperand: IntegerLiteral
              literal: 2 @204
              staticType: int
            staticElement: dart:core::@class::int::@method::>>
            staticInvokeType: int Function(int)
            staticType: int
      static const vAdd @213
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @220
              staticType: int
            operator: + @222
            rightOperand: IntegerLiteral
              literal: 2 @224
              staticType: int
            staticElement: dart:core::@class::num::@method::+
            staticInvokeType: num Function(num)
            staticType: int
      static const vSubtract @233
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @245
              staticType: int
            operator: - @247
            rightOperand: IntegerLiteral
              literal: 2 @249
              staticType: int
            staticElement: dart:core::@class::num::@method::-
            staticInvokeType: num Function(num)
            staticType: int
      static const vMiltiply @258
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @270
              staticType: int
            operator: * @272
            rightOperand: IntegerLiteral
              literal: 2 @274
              staticType: int
            staticElement: dart:core::@class::num::@method::*
            staticInvokeType: num Function(num)
            staticType: int
      static const vDivide @283
        type: double
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @293
              staticType: int
            operator: / @295
            rightOperand: IntegerLiteral
              literal: 2 @297
              staticType: int
            staticElement: dart:core::@class::num::@method::/
            staticInvokeType: double Function(num)
            staticType: double
      static const vFloorDivide @306
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @321
              staticType: int
            operator: ~/ @323
            rightOperand: IntegerLiteral
              literal: 2 @326
              staticType: int
            staticElement: dart:core::@class::num::@method::~/
            staticInvokeType: int Function(num)
            staticType: int
      static const vModulo @335
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @345
              staticType: int
            operator: % @347
            rightOperand: IntegerLiteral
              literal: 2 @349
              staticType: int
            staticElement: dart:core::@class::num::@method::%
            staticInvokeType: num Function(num)
            staticType: int
      static const vGreater @358
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @369
              staticType: int
            operator: > @371
            rightOperand: IntegerLiteral
              literal: 2 @373
              staticType: int
            staticElement: dart:core::@class::num::@method::>
            staticInvokeType: bool Function(num)
            staticType: bool
      static const vGreaterEqual @382
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @398
              staticType: int
            operator: >= @400
            rightOperand: IntegerLiteral
              literal: 2 @403
              staticType: int
            staticElement: dart:core::@class::num::@method::>=
            staticInvokeType: bool Function(num)
            staticType: bool
      static const vLess @412
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @420
              staticType: int
            operator: < @422
            rightOperand: IntegerLiteral
              literal: 2 @424
              staticType: int
            staticElement: dart:core::@class::num::@method::<
            staticInvokeType: bool Function(num)
            staticType: bool
      static const vLessEqual @433
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @446
              staticType: int
            operator: <= @448
            rightOperand: IntegerLiteral
              literal: 2 @451
              staticType: int
            staticElement: dart:core::@class::num::@method::<=
            staticInvokeType: bool Function(num)
            staticType: bool
    accessors
      synthetic static get vEqual @-1
        returnType: bool
      synthetic static get vAnd @-1
        returnType: bool
      synthetic static get vOr @-1
        returnType: bool
      synthetic static get vBitXor @-1
        returnType: int
      synthetic static get vBitAnd @-1
        returnType: int
      synthetic static get vBitOr @-1
        returnType: int
      synthetic static get vBitShiftLeft @-1
        returnType: int
      synthetic static get vBitShiftRight @-1
        returnType: int
      synthetic static get vAdd @-1
        returnType: int
      synthetic static get vSubtract @-1
        returnType: int
      synthetic static get vMiltiply @-1
        returnType: int
      synthetic static get vDivide @-1
        returnType: double
      synthetic static get vFloorDivide @-1
        returnType: int
      synthetic static get vModulo @-1
        returnType: int
      synthetic static get vGreater @-1
        returnType: bool
      synthetic static get vGreaterEqual @-1
        returnType: bool
      synthetic static get vLess @-1
        returnType: bool
      synthetic static get vLessEqual @-1
        returnType: bool
''');
  }

  test_const_topLevel_conditional() async {
    var library = await buildLibrary(r'''
const vConditional = (1 == 2) ? 11 : 22;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vConditional @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ConditionalExpression
            condition: ParenthesizedExpression
              leftParenthesis: ( @21
              expression: BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @22
                  staticType: int
                operator: == @24
                rightOperand: IntegerLiteral
                  literal: 2 @27
                  staticType: int
                staticElement: dart:core::@class::num::@method::==
                staticInvokeType: bool Function(Object)
                staticType: bool
              rightParenthesis: ) @28
              staticType: bool
            question: ? @30
            thenExpression: IntegerLiteral
              literal: 11 @32
              staticType: int
            colon: : @35
            elseExpression: IntegerLiteral
              literal: 22 @37
              staticType: int
            staticType: int
    accessors
      synthetic static get vConditional @-1
        returnType: int
''');
  }

  test_const_topLevel_identical() async {
    var library = await buildLibrary(r'''
const vIdentical = (1 == 2) ? 11 : 22;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vIdentical @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ConditionalExpression
            condition: ParenthesizedExpression
              leftParenthesis: ( @19
              expression: BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @20
                  staticType: int
                operator: == @22
                rightOperand: IntegerLiteral
                  literal: 2 @25
                  staticType: int
                staticElement: dart:core::@class::num::@method::==
                staticInvokeType: bool Function(Object)
                staticType: bool
              rightParenthesis: ) @26
              staticType: bool
            question: ? @28
            thenExpression: IntegerLiteral
              literal: 11 @30
              staticType: int
            colon: : @33
            elseExpression: IntegerLiteral
              literal: 22 @35
              staticType: int
            staticType: int
    accessors
      synthetic static get vIdentical @-1
        returnType: int
''');
  }

  test_const_topLevel_ifNull() async {
    var library = await buildLibrary(r'''
const vIfNull = 1 ?? 2.0;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vIfNull @6
        type: num
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @16
              staticType: int
            operator: ?? @18
            rightOperand: DoubleLiteral
              literal: 2.0 @21
              staticType: double
            staticElement: <null>
            staticInvokeType: null
            staticType: num
    accessors
      synthetic static get vIfNull @-1
        returnType: num
''');
  }

  test_const_topLevel_literal() async {
    var library = await buildLibrary(r'''
const vNull = null;
const vBoolFalse = false;
const vBoolTrue = true;
const vIntPositive = 1;
const vIntNegative = -2;
const vIntLong1 = 0x7FFFFFFFFFFFFFFF;
const vIntLong2 = 0xFFFFFFFFFFFFFFFF;
const vIntLong3 = 0x8000000000000000;
const vDouble = 2.3;
const vString = 'abc';
const vStringConcat = 'aaa' 'bbb';
const vStringInterpolation = 'aaa ${true} ${42} bbb';
const vSymbol = #aaa.bbb.ccc;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vNull @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @14
            staticType: Null
      static const vBoolFalse @26
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BooleanLiteral
            literal: false @39
            staticType: bool
      static const vBoolTrue @52
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BooleanLiteral
            literal: true @64
            staticType: bool
      static const vIntPositive @76
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 1 @91
            staticType: int
      static const vIntNegative @100
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: - @115
            operand: IntegerLiteral
              literal: 2 @116
              staticType: int
            staticElement: dart:core::@class::int::@method::unary-
            staticType: int
      static const vIntLong1 @125
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0x7FFFFFFFFFFFFFFF @137
            staticType: int
      static const vIntLong2 @163
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0xFFFFFFFFFFFFFFFF @175
            staticType: int
      static const vIntLong3 @201
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0x8000000000000000 @213
            staticType: int
      static const vDouble @239
        type: double
        shouldUseTypeForInitializerInference: false
        constantInitializer
          DoubleLiteral
            literal: 2.3 @249
            staticType: double
      static const vString @260
        type: String
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleStringLiteral
            literal: 'abc' @270
      static const vStringConcat @283
        type: String
        shouldUseTypeForInitializerInference: false
        constantInitializer
          AdjacentStrings
            strings
              SimpleStringLiteral
                literal: 'aaa' @299
              SimpleStringLiteral
                literal: 'bbb' @305
            staticType: String
            stringValue: aaabbb
      static const vStringInterpolation @318
        type: String
        shouldUseTypeForInitializerInference: false
        constantInitializer
          StringInterpolation
            elements
              InterpolationString
                contents: 'aaa  @341
              InterpolationExpression
                leftBracket: ${ @346
                expression: BooleanLiteral
                  literal: true @348
                  staticType: bool
                rightBracket: } @352
              InterpolationString
                contents:   @353
              InterpolationExpression
                leftBracket: ${ @354
                expression: IntegerLiteral
                  literal: 42 @356
                  staticType: int
                rightBracket: } @358
              InterpolationString
                contents:  bbb' @359
            staticType: String
            stringValue: null
      static const vSymbol @372
        type: Symbol
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SymbolLiteral
            poundSign: # @382
            components
              aaa
                offset: 383
              bbb
                offset: 387
              ccc
                offset: 391
    accessors
      synthetic static get vNull @-1
        returnType: dynamic
      synthetic static get vBoolFalse @-1
        returnType: bool
      synthetic static get vBoolTrue @-1
        returnType: bool
      synthetic static get vIntPositive @-1
        returnType: int
      synthetic static get vIntNegative @-1
        returnType: int
      synthetic static get vIntLong1 @-1
        returnType: int
      synthetic static get vIntLong2 @-1
        returnType: int
      synthetic static get vIntLong3 @-1
        returnType: int
      synthetic static get vDouble @-1
        returnType: double
      synthetic static get vString @-1
        returnType: String
      synthetic static get vStringConcat @-1
        returnType: String
      synthetic static get vStringInterpolation @-1
        returnType: String
      synthetic static get vSymbol @-1
        returnType: Symbol
''');
  }

  test_const_topLevel_methodInvocation_questionPeriod() async {
    var library = await buildLibrary(r'''
const int? a = 0;
const b = a?.toString();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @11
        type: int?
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @15
            staticType: int
      static const b @24
        type: String?
        shouldUseTypeForInitializerInference: false
        constantInitializer
          MethodInvocation
            target: SimpleIdentifier
              token: a @28
              staticElement: self::@getter::a
              staticType: int?
            operator: ?. @29
            methodName: SimpleIdentifier
              token: toString @31
              staticElement: dart:core::@class::int::@method::toString
              staticType: String Function()
            argumentList: ArgumentList
              leftParenthesis: ( @39
              rightParenthesis: ) @40
            staticInvokeType: String Function()
            staticType: String?
    accessors
      synthetic static get a @-1
        returnType: int?
      synthetic static get b @-1
        returnType: String?
''');
  }

  test_const_topLevel_methodInvocation_questionPeriodPeriod() async {
    var library = await buildLibrary(r'''
const int? a = 0;
const b = a?..toString();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @11
        type: int?
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @15
            staticType: int
      static const b @24
        type: int?
        shouldUseTypeForInitializerInference: false
        constantInitializer
          CascadeExpression
            target: SimpleIdentifier
              token: a @28
              staticElement: self::@getter::a
              staticType: int?
            cascadeSections
              MethodInvocation
                operator: ?.. @29
                methodName: SimpleIdentifier
                  token: toString @32
                  staticElement: dart:core::@class::int::@method::toString
                  staticType: String Function()
                argumentList: ArgumentList
                  leftParenthesis: ( @40
                  rightParenthesis: ) @41
                staticInvokeType: String Function()
                staticType: String
            staticType: int?
    accessors
      synthetic static get a @-1
        returnType: int?
      synthetic static get b @-1
        returnType: int?
''');
  }

  test_const_topLevel_nullSafe_nullAware_propertyAccess() async {
    var library = await buildLibrary(r'''
const String? a = '';

const List<int?> b = [
  a?.length,
];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @14
        type: String?
        shouldUseTypeForInitializerInference: true
        constantInitializer
          SimpleStringLiteral
            literal: '' @18
      static const b @40
        type: List<int?>
        shouldUseTypeForInitializerInference: true
        constantInitializer
          ListLiteral
            leftBracket: [ @44
            elements
              PropertyAccess
                target: SimpleIdentifier
                  token: a @48
                  staticElement: self::@getter::a
                  staticType: String?
                operator: ?. @49
                propertyName: SimpleIdentifier
                  token: length @51
                  staticElement: dart:core::@class::String::@getter::length
                  staticType: int
                staticType: int?
            rightBracket: ] @59
            staticType: List<int?>
    accessors
      synthetic static get a @-1
        returnType: String?
      synthetic static get b @-1
        returnType: List<int?>
''');
  }

  test_const_topLevel_parenthesis() async {
    var library = await buildLibrary(r'''
const int v1 = (1 + 2) * 3;
const int v2 = -(1 + 2);
const int v3 = ('aaa' + 'bbb').length;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v1 @10
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          BinaryExpression
            leftOperand: ParenthesizedExpression
              leftParenthesis: ( @15
              expression: BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @16
                  staticType: int
                operator: + @18
                rightOperand: IntegerLiteral
                  literal: 2 @20
                  staticType: int
                staticElement: dart:core::@class::num::@method::+
                staticInvokeType: num Function(num)
                staticType: int
              rightParenthesis: ) @21
              staticType: int
            operator: * @23
            rightOperand: IntegerLiteral
              literal: 3 @25
              staticType: int
            staticElement: dart:core::@class::num::@method::*
            staticInvokeType: num Function(num)
            staticType: int
      static const v2 @38
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          PrefixExpression
            operator: - @43
            operand: ParenthesizedExpression
              leftParenthesis: ( @44
              expression: BinaryExpression
                leftOperand: IntegerLiteral
                  literal: 1 @45
                  staticType: int
                operator: + @47
                rightOperand: IntegerLiteral
                  literal: 2 @49
                  staticType: int
                staticElement: dart:core::@class::num::@method::+
                staticInvokeType: num Function(num)
                staticType: int
              rightParenthesis: ) @50
              staticType: int
            staticElement: dart:core::@class::int::@method::unary-
            staticType: int
      static const v3 @63
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          PropertyAccess
            target: ParenthesizedExpression
              leftParenthesis: ( @68
              expression: BinaryExpression
                leftOperand: SimpleStringLiteral
                  literal: 'aaa' @69
                operator: + @75
                rightOperand: SimpleStringLiteral
                  literal: 'bbb' @77
                staticElement: dart:core::@class::String::@method::+
                staticInvokeType: String Function(String)
                staticType: String
              rightParenthesis: ) @82
              staticType: String
            operator: . @83
            propertyName: SimpleIdentifier
              token: length @84
              staticElement: dart:core::@class::String::@getter::length
              staticType: int
            staticType: int
    accessors
      synthetic static get v1 @-1
        returnType: int
      synthetic static get v2 @-1
        returnType: int
      synthetic static get v3 @-1
        returnType: int
''');
  }

  test_const_topLevel_prefix() async {
    var library = await buildLibrary(r'''
const vNotEqual = 1 != 2;
const vNot = !true;
const vNegate = -1;
const vComplement = ~1;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vNotEqual @6
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          BinaryExpression
            leftOperand: IntegerLiteral
              literal: 1 @18
              staticType: int
            operator: != @20
            rightOperand: IntegerLiteral
              literal: 2 @23
              staticType: int
            staticElement: dart:core::@class::num::@method::==
            staticInvokeType: bool Function(Object)
            staticType: bool
      static const vNot @32
        type: bool
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: ! @39
            operand: BooleanLiteral
              literal: true @40
              staticType: bool
            staticElement: <null>
            staticType: bool
      static const vNegate @52
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: - @62
            operand: IntegerLiteral
              literal: 1 @63
              staticType: int
            staticElement: dart:core::@class::int::@method::unary-
            staticType: int
      static const vComplement @72
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          PrefixExpression
            operator: ~ @86
            operand: IntegerLiteral
              literal: 1 @87
              staticType: int
            staticElement: dart:core::@class::int::@method::~
            staticType: int
    accessors
      synthetic static get vNotEqual @-1
        returnType: bool
      synthetic static get vNot @-1
        returnType: bool
      synthetic static get vNegate @-1
        returnType: int
      synthetic static get vComplement @-1
        returnType: int
''');
  }

  test_const_topLevel_super() async {
    var library = await buildLibrary(r'''
const vSuper = super;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vSuper @6
        type: InvalidType
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SuperExpression
            superKeyword: super @15
            staticType: InvalidType
    accessors
      synthetic static get vSuper @-1
        returnType: InvalidType
''');
  }

  test_const_topLevel_this() async {
    var library = await buildLibrary(r'''
const vThis = this;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vThis @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ThisExpression
            thisKeyword: this @14
            staticType: dynamic
    accessors
      synthetic static get vThis @-1
        returnType: dynamic
''');
  }

  test_const_topLevel_throw() async {
    var library = await buildLibrary(r'''
const c = throw 42;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const c @6
        type: Never
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ThrowExpression
            throwKeyword: throw @10
            expression: IntegerLiteral
              literal: 42 @16
              staticType: int
            staticType: Never
    accessors
      synthetic static get c @-1
        returnType: Never
''');
  }

  test_const_topLevel_throw_legacy() async {
    var library = await buildLibrary(r'''
// @dart = 2.9
const c = throw 42;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const c @21
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ThrowExpression
            throwKeyword: throw @25
            expression: IntegerLiteral
              literal: 42 @31
              staticType: int*
            staticType: Never*
    accessors
      synthetic static get c @-1
        returnType: dynamic
''');
  }

  test_const_topLevel_typedList() async {
    var library = await buildLibrary(r'''
const vNull = const <Null>[];
const vDynamic = const <dynamic>[1, 2, 3];
const vInterfaceNoTypeParameters = const <int>[1, 2, 3];
const vInterfaceNoTypeArguments = const <List>[];
const vInterfaceWithTypeArguments = const <List<String>>[];
const vInterfaceWithTypeArguments2 = const <Map<int, List<String>>>[];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vNull @6
        type: List<Null>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @14
            typeArguments: TypeArgumentList
              leftBracket: < @20
              arguments
                NamedType
                  name: Null @21
                  element: dart:core::@class::Null
                  type: Null
              rightBracket: > @25
            leftBracket: [ @26
            rightBracket: ] @27
            staticType: List<Null>
      static const vDynamic @36
        type: List<dynamic>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @47
            typeArguments: TypeArgumentList
              leftBracket: < @53
              arguments
                NamedType
                  name: dynamic @54
                  element: dynamic@-1
                  type: dynamic
              rightBracket: > @61
            leftBracket: [ @62
            elements
              IntegerLiteral
                literal: 1 @63
                staticType: int
              IntegerLiteral
                literal: 2 @66
                staticType: int
              IntegerLiteral
                literal: 3 @69
                staticType: int
            rightBracket: ] @70
            staticType: List<dynamic>
      static const vInterfaceNoTypeParameters @79
        type: List<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @108
            typeArguments: TypeArgumentList
              leftBracket: < @114
              arguments
                NamedType
                  name: int @115
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @118
            leftBracket: [ @119
            elements
              IntegerLiteral
                literal: 1 @120
                staticType: int
              IntegerLiteral
                literal: 2 @123
                staticType: int
              IntegerLiteral
                literal: 3 @126
                staticType: int
            rightBracket: ] @127
            staticType: List<int>
      static const vInterfaceNoTypeArguments @136
        type: List<List<dynamic>>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @164
            typeArguments: TypeArgumentList
              leftBracket: < @170
              arguments
                NamedType
                  name: List @171
                  element: dart:core::@class::List
                  type: List<dynamic>
              rightBracket: > @175
            leftBracket: [ @176
            rightBracket: ] @177
            staticType: List<List<dynamic>>
      static const vInterfaceWithTypeArguments @186
        type: List<List<String>>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @216
            typeArguments: TypeArgumentList
              leftBracket: < @222
              arguments
                NamedType
                  name: List @223
                  typeArguments: TypeArgumentList
                    leftBracket: < @227
                    arguments
                      NamedType
                        name: String @228
                        element: dart:core::@class::String
                        type: String
                    rightBracket: > @234
                  element: dart:core::@class::List
                  type: List<String>
              rightBracket: > @235
            leftBracket: [ @236
            rightBracket: ] @237
            staticType: List<List<String>>
      static const vInterfaceWithTypeArguments2 @246
        type: List<Map<int, List<String>>>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @277
            typeArguments: TypeArgumentList
              leftBracket: < @283
              arguments
                NamedType
                  name: Map @284
                  typeArguments: TypeArgumentList
                    leftBracket: < @287
                    arguments
                      NamedType
                        name: int @288
                        element: dart:core::@class::int
                        type: int
                      NamedType
                        name: List @293
                        typeArguments: TypeArgumentList
                          leftBracket: < @297
                          arguments
                            NamedType
                              name: String @298
                              element: dart:core::@class::String
                              type: String
                          rightBracket: > @304
                        element: dart:core::@class::List
                        type: List<String>
                    rightBracket: > @305
                  element: dart:core::@class::Map
                  type: Map<int, List<String>>
              rightBracket: > @306
            leftBracket: [ @307
            rightBracket: ] @308
            staticType: List<Map<int, List<String>>>
    accessors
      synthetic static get vNull @-1
        returnType: List<Null>
      synthetic static get vDynamic @-1
        returnType: List<dynamic>
      synthetic static get vInterfaceNoTypeParameters @-1
        returnType: List<int>
      synthetic static get vInterfaceNoTypeArguments @-1
        returnType: List<List<dynamic>>
      synthetic static get vInterfaceWithTypeArguments @-1
        returnType: List<List<String>>
      synthetic static get vInterfaceWithTypeArguments2 @-1
        returnType: List<Map<int, List<String>>>
''');
  }

  test_const_topLevel_typedList_imported() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    var library = await buildLibrary(r'''
import 'a.dart';
const v = const <C>[];
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static const v @23
        type: List<C>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @27
            typeArguments: TypeArgumentList
              leftBracket: < @33
              arguments
                NamedType
                  name: C @34
                  element: package:test/a.dart::@class::C
                  type: C
              rightBracket: > @35
            leftBracket: [ @36
            rightBracket: ] @37
            staticType: List<C>
    accessors
      synthetic static get v @-1
        returnType: List<C>
''');
  }

  test_const_topLevel_typedList_importedWithPrefix() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    var library = await buildLibrary(r'''
import 'a.dart' as p;
const v = const <p.C>[];
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as p @19
  definingUnit
    topLevelVariables
      static const v @28
        type: List<C>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @32
            typeArguments: TypeArgumentList
              leftBracket: < @38
              arguments
                NamedType
                  importPrefix: ImportPrefixReference
                    name: p @39
                    period: . @40
                    element: self::@prefix::p
                  name: C @41
                  element: package:test/a.dart::@class::C
                  type: C
              rightBracket: > @42
            leftBracket: [ @43
            rightBracket: ] @44
            staticType: List<C>
    accessors
      synthetic static get v @-1
        returnType: List<C>
''');
  }

  test_const_topLevel_typedList_typedefArgument() async {
    var library = await buildLibrary(r'''
typedef int F(String id);
const v = const <F>[];
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @12
        aliasedType: int Function(String)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional id @21
              type: String
          returnType: int
    topLevelVariables
      static const v @32
        type: List<int Function(String)>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @36
            typeArguments: TypeArgumentList
              leftBracket: < @42
              arguments
                NamedType
                  name: F @43
                  element: self::@typeAlias::F
                  type: int Function(String)
                    alias: self::@typeAlias::F
              rightBracket: > @44
            leftBracket: [ @45
            rightBracket: ] @46
            staticType: List<int Function(String)>
    accessors
      synthetic static get v @-1
        returnType: List<int Function(String)>
''');
  }

  test_const_topLevel_typedMap() async {
    var library = await buildLibrary(r'''
const vDynamic1 = const <dynamic, int>{};
const vDynamic2 = const <int, dynamic>{};
const vInterface = const <int, String>{};
const vInterfaceWithTypeArguments = const <int, List<String>>{};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vDynamic1 @6
        type: Map<dynamic, int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @18
            typeArguments: TypeArgumentList
              leftBracket: < @24
              arguments
                NamedType
                  name: dynamic @25
                  element: dynamic@-1
                  type: dynamic
                NamedType
                  name: int @34
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @37
            leftBracket: { @38
            rightBracket: } @39
            isMap: true
            staticType: Map<dynamic, int>
      static const vDynamic2 @48
        type: Map<int, dynamic>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @60
            typeArguments: TypeArgumentList
              leftBracket: < @66
              arguments
                NamedType
                  name: int @67
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: dynamic @72
                  element: dynamic@-1
                  type: dynamic
              rightBracket: > @79
            leftBracket: { @80
            rightBracket: } @81
            isMap: true
            staticType: Map<int, dynamic>
      static const vInterface @90
        type: Map<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @103
            typeArguments: TypeArgumentList
              leftBracket: < @109
              arguments
                NamedType
                  name: int @110
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: String @115
                  element: dart:core::@class::String
                  type: String
              rightBracket: > @121
            leftBracket: { @122
            rightBracket: } @123
            isMap: true
            staticType: Map<int, String>
      static const vInterfaceWithTypeArguments @132
        type: Map<int, List<String>>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @162
            typeArguments: TypeArgumentList
              leftBracket: < @168
              arguments
                NamedType
                  name: int @169
                  element: dart:core::@class::int
                  type: int
                NamedType
                  name: List @174
                  typeArguments: TypeArgumentList
                    leftBracket: < @178
                    arguments
                      NamedType
                        name: String @179
                        element: dart:core::@class::String
                        type: String
                    rightBracket: > @185
                  element: dart:core::@class::List
                  type: List<String>
              rightBracket: > @186
            leftBracket: { @187
            rightBracket: } @188
            isMap: true
            staticType: Map<int, List<String>>
    accessors
      synthetic static get vDynamic1 @-1
        returnType: Map<dynamic, int>
      synthetic static get vDynamic2 @-1
        returnType: Map<int, dynamic>
      synthetic static get vInterface @-1
        returnType: Map<int, String>
      synthetic static get vInterfaceWithTypeArguments @-1
        returnType: Map<int, List<String>>
''');
  }

  test_const_topLevel_typedSet() async {
    var library = await buildLibrary(r'''
const vDynamic1 = const <dynamic>{};
const vInterface = const <int>{};
const vInterfaceWithTypeArguments = const <List<String>>{};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const vDynamic1 @6
        type: Set<dynamic>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @18
            typeArguments: TypeArgumentList
              leftBracket: < @24
              arguments
                NamedType
                  name: dynamic @25
                  element: dynamic@-1
                  type: dynamic
              rightBracket: > @32
            leftBracket: { @33
            rightBracket: } @34
            isMap: false
            staticType: Set<dynamic>
      static const vInterface @43
        type: Set<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @56
            typeArguments: TypeArgumentList
              leftBracket: < @62
              arguments
                NamedType
                  name: int @63
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @66
            leftBracket: { @67
            rightBracket: } @68
            isMap: false
            staticType: Set<int>
      static const vInterfaceWithTypeArguments @77
        type: Set<List<String>>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @107
            typeArguments: TypeArgumentList
              leftBracket: < @113
              arguments
                NamedType
                  name: List @114
                  typeArguments: TypeArgumentList
                    leftBracket: < @118
                    arguments
                      NamedType
                        name: String @119
                        element: dart:core::@class::String
                        type: String
                    rightBracket: > @125
                  element: dart:core::@class::List
                  type: List<String>
              rightBracket: > @126
            leftBracket: { @127
            rightBracket: } @128
            isMap: false
            staticType: Set<List<String>>
    accessors
      synthetic static get vDynamic1 @-1
        returnType: Set<dynamic>
      synthetic static get vInterface @-1
        returnType: Set<int>
      synthetic static get vInterfaceWithTypeArguments @-1
        returnType: Set<List<String>>
''');
  }

  test_const_topLevel_untypedList() async {
    var library = await buildLibrary(r'''
const v = const [1, 2, 3];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: List<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          ListLiteral
            constKeyword: const @10
            leftBracket: [ @16
            elements
              IntegerLiteral
                literal: 1 @17
                staticType: int
              IntegerLiteral
                literal: 2 @20
                staticType: int
              IntegerLiteral
                literal: 3 @23
                staticType: int
            rightBracket: ] @24
            staticType: List<int>
    accessors
      synthetic static get v @-1
        returnType: List<int>
''');
  }

  test_const_topLevel_untypedMap() async {
    var library = await buildLibrary(r'''
const v = const {0: 'aaa', 1: 'bbb', 2: 'ccc'};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: Map<int, String>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @10
            leftBracket: { @16
            elements
              MapLiteralEntry
                key: IntegerLiteral
                  literal: 0 @17
                  staticType: int
                separator: : @18
                value: SimpleStringLiteral
                  literal: 'aaa' @20
              MapLiteralEntry
                key: IntegerLiteral
                  literal: 1 @27
                  staticType: int
                separator: : @28
                value: SimpleStringLiteral
                  literal: 'bbb' @30
              MapLiteralEntry
                key: IntegerLiteral
                  literal: 2 @37
                  staticType: int
                separator: : @38
                value: SimpleStringLiteral
                  literal: 'ccc' @40
            rightBracket: } @45
            isMap: true
            staticType: Map<int, String>
    accessors
      synthetic static get v @-1
        returnType: Map<int, String>
''');
  }

  test_const_topLevel_untypedSet() async {
    var library = await buildLibrary(r'''
const v = const {0, 1, 2};
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: Set<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SetOrMapLiteral
            constKeyword: const @10
            leftBracket: { @16
            elements
              IntegerLiteral
                literal: 0 @17
                staticType: int
              IntegerLiteral
                literal: 1 @20
                staticType: int
              IntegerLiteral
                literal: 2 @23
                staticType: int
            rightBracket: } @24
            isMap: false
            staticType: Set<int>
    accessors
      synthetic static get v @-1
        returnType: Set<int>
''');
  }

  test_const_typeLiteral() async {
    var library = await buildLibrary(r'''
const v = List<int>;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const v @6
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          TypeLiteral
            type: NamedType
              name: List @10
              typeArguments: TypeArgumentList
                leftBracket: < @14
                arguments
                  NamedType
                    name: int @15
                    element: dart:core::@class::int
                    type: int
                rightBracket: > @18
              element: dart:core::@class::List
              type: List<int>
            staticType: Type
    accessors
      synthetic static get v @-1
        returnType: Type
''');
  }

  test_constExpr_pushReference_enum_field() async {
    var library = await buildLibrary('''
enum E {a, b, c}
final vValue = E.a;
final vValues = E.values;
final vIndex = E.a.index;
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @8
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant c @14
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get c @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static final vValue @23
        type: E
        shouldUseTypeForInitializerInference: false
      static final vValues @43
        type: List<E>
        shouldUseTypeForInitializerInference: false
      static final vIndex @69
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get vValue @-1
        returnType: E
      synthetic static get vValues @-1
        returnType: List<E>
      synthetic static get vIndex @-1
        returnType: int
''');
  }

  test_constExpr_pushReference_enum_method() async {
    var library = await buildLibrary('''
enum E {a}
final vToString = E.a.toString();
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @8
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static final vToString @17
        type: String
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get vToString @-1
        returnType: String
''');
  }

  test_constExpr_pushReference_field_simpleIdentifier() async {
    var library = await buildLibrary('''
class C {
  static const a = b;
  static const b = null;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const a @25
            type: dynamic
            shouldUseTypeForInitializerInference: false
            constantInitializer
              SimpleIdentifier
                token: b @29
                staticElement: self::@class::C::@getter::b
                staticType: dynamic
          static const b @47
            type: dynamic
            shouldUseTypeForInitializerInference: false
            constantInitializer
              NullLiteral
                literal: null @51
                staticType: Null
        constructors
          synthetic @-1
        accessors
          synthetic static get a @-1
            returnType: dynamic
          synthetic static get b @-1
            returnType: dynamic
''');
  }

  test_constExpr_pushReference_staticMethod_simpleIdentifier() async {
    var library = await buildLibrary('''
class C {
  static const a = m;
  static m() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          static const a @25
            type: dynamic Function()
            shouldUseTypeForInitializerInference: false
            constantInitializer
              SimpleIdentifier
                token: m @29
                staticElement: self::@class::C::@method::m
                staticType: dynamic Function()
        constructors
          synthetic @-1
        accessors
          synthetic static get a @-1
            returnType: dynamic Function()
        methods
          static m @41
            returnType: dynamic
''');
  }

  test_defaultValue_eliminateTypeParameters() async {
    var library = await buildLibrary('''
class A<T> {
  const X({List<T> a = const []});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          abstract X @21
            parameters
              optionalNamed default a @32
                type: List<T>
                constantInitializer
                  ListLiteral
                    constKeyword: const @36
                    leftBracket: [ @42
                    rightBracket: ] @43
                    staticType: List<Never>
            returnType: dynamic
''');
  }

  test_defaultValue_eliminateTypeParameters_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
class A<T> {
  const X({List<T> a = const []});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @21
        typeParameters
          covariant T @23
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          abstract X @36
            parameters
              optionalNamed default a @47
                type: List<T*>*
                constantInitializer
                  ListLiteral
                    constKeyword: const @51
                    leftBracket: [ @57
                    rightBracket: ] @58
                    staticType: List<Null*>*
            returnType: dynamic
''');
  }

  test_defaultValue_genericFunction() async {
    var library = await buildLibrary('''
typedef void F<T>(T v);

void defaultF<T>(T v) {}

class X {
  final F f;
  const X({this.f: defaultF});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class X @57
        fields
          final f @71
            type: void Function(dynamic)
              alias: self::@typeAlias::F
                typeArguments
                  dynamic
            shouldUseTypeForInitializerInference: true
        constructors
          const @82
            parameters
              optionalNamed default final this.f @90
                type: void Function(dynamic)
                  alias: self::@typeAlias::F
                    typeArguments
                      dynamic
                constantInitializer
                  FunctionReference
                    function: SimpleIdentifier
                      token: defaultF @93
                      staticElement: self::@function::defaultF
                      staticType: void Function<T>(T)
                    staticType: void Function(dynamic)
                    typeArgumentTypes
                      dynamic
                field: self::@class::X::@field::f
        accessors
          synthetic get f @-1
            returnType: void Function(dynamic)
              alias: self::@typeAlias::F
                typeArguments
                  dynamic
    typeAliases
      functionTypeAliasBased F @13
        typeParameters
          contravariant T @15
            defaultType: dynamic
        aliasedType: void Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional v @20
              type: T
          returnType: void
    functions
      defaultF @30
        typeParameters
          covariant T @39
            defaultType: dynamic
        parameters
          requiredPositional v @44
            type: T
        returnType: void
''');
  }

  test_defaultValue_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  void foo({a: const A<Function()>()}) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class B @34
        constructors
          synthetic @-1
        methods
          foo @45
            parameters
              optionalNamed default a @50
                type: dynamic
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @53
                    constructorName: ConstructorName
                      type: NamedType
                        name: A @59
                        typeArguments: TypeArgumentList
                          leftBracket: < @60
                          arguments
                            GenericFunctionType
                              functionKeyword: Function @61
                              parameters: FormalParameterList
                                leftParenthesis: ( @69
                                rightParenthesis: ) @70
                              declaredElement: GenericFunctionTypeElement
                                parameters
                                returnType: dynamic
                                type: dynamic Function()
                              type: dynamic Function()
                          rightBracket: > @71
                        element: self::@class::A
                        type: A<dynamic Function()>
                      staticElement: ConstructorMember
                        base: self::@class::A::@constructor::new
                        substitution: {T: dynamic Function()}
                    argumentList: ArgumentList
                      leftParenthesis: ( @72
                      rightParenthesis: ) @73
                    staticType: A<dynamic Function()>
            returnType: void
''');
  }

  test_defaultValue_inFunctionTypedFormalParameter() async {
    var library = await buildLibrary('''
void f( g({a: 0 is int}) ) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional g @8
            type: dynamic Function({dynamic a})
            parameters
              optionalNamed a @11
                type: dynamic
        returnType: void
''');
  }

  test_defaultValue_methodMember_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
void f([Comparator<T> compare = Comparable.compare]) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @20
        parameters
          optionalPositional default compare @37
            type: int* Function(InvalidType, InvalidType)*
              alias: dart:core::@typeAlias::Comparator
                typeArguments
                  InvalidType
            constantInitializer
              PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: Comparable @47
                  staticElement: dart:core::@class::Comparable
                  staticType: null
                period: . @57
                identifier: SimpleIdentifier
                  token: compare @58
                  staticElement: MethodMember
                    base: dart:core::@class::Comparable::@method::compare
                    isLegacy: true
                  staticType: int* Function(Comparable<dynamic>*, Comparable<dynamic>*)*
                staticElement: MethodMember
                  base: dart:core::@class::Comparable::@method::compare
                  isLegacy: true
                staticType: int* Function(Comparable<dynamic>*, Comparable<dynamic>*)*
        returnType: void
''');
  }

  test_defaultValue_recordLiteral_named() async {
    var library = await buildLibrary('''
void f({({int f1, bool f2}) x = (f1: 1, f2: true)}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          optionalNamed default x @28
            type: ({int f1, bool f2})
            constantInitializer
              RecordLiteral
                leftParenthesis: ( @32
                fields
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: f1 @33
                        staticElement: <null>
                        staticType: null
                      colon: : @35
                    expression: IntegerLiteral
                      literal: 1 @37
                      staticType: int
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: f2 @40
                        staticElement: <null>
                        staticType: null
                      colon: : @42
                    expression: BooleanLiteral
                      literal: true @44
                      staticType: bool
                rightParenthesis: ) @48
                staticType: ({int f1, bool f2})
        returnType: void
''');
  }

  test_defaultValue_recordLiteral_named_const() async {
    var library = await buildLibrary('''
void f({({int f1, bool f2}) x = const (f1: 1, f2: true)}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          optionalNamed default x @28
            type: ({int f1, bool f2})
            constantInitializer
              RecordLiteral
                constKeyword: const @32
                leftParenthesis: ( @38
                fields
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: f1 @39
                        staticElement: <null>
                        staticType: null
                      colon: : @41
                    expression: IntegerLiteral
                      literal: 1 @43
                      staticType: int
                  NamedExpression
                    name: Label
                      label: SimpleIdentifier
                        token: f2 @46
                        staticElement: <null>
                        staticType: null
                      colon: : @48
                    expression: BooleanLiteral
                      literal: true @50
                      staticType: bool
                rightParenthesis: ) @54
                staticType: ({int f1, bool f2})
        returnType: void
''');
  }

  test_defaultValue_recordLiteral_positional() async {
    var library = await buildLibrary('''
void f({(int, bool) x = (1, true)}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          optionalNamed default x @20
            type: (int, bool)
            constantInitializer
              RecordLiteral
                leftParenthesis: ( @24
                fields
                  IntegerLiteral
                    literal: 1 @25
                    staticType: int
                  BooleanLiteral
                    literal: true @28
                    staticType: bool
                rightParenthesis: ) @32
                staticType: (int, bool)
        returnType: void
''');
  }

  void test_defaultValue_recordLiteral_positional_const() async {
    var library = await buildLibrary('''
void f({(int, bool) x = const (1, true)}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          optionalNamed default x @20
            type: (int, bool)
            constantInitializer
              RecordLiteral
                constKeyword: const @24
                leftParenthesis: ( @30
                fields
                  IntegerLiteral
                    literal: 1 @31
                    staticType: int
                  BooleanLiteral
                    literal: true @34
                    staticType: bool
                rightParenthesis: ) @38
                staticType: (int, bool)
        returnType: void
''');
  }

  test_defaultValue_refersToExtension_method_inside() async {
    var library = await buildLibrary('''
class A {}
extension E on A {
  static void f() {}
  static void g([Object p = f]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
    extensions
      E @21
        extendedType: A
        methods
          static f @44
            returnType: void
          static g @65
            parameters
              optionalPositional default p @75
                type: Object
                constantInitializer
                  SimpleIdentifier
                    token: f @79
                    staticElement: self::@extension::E::@method::f
                    staticType: void Function()
            returnType: void
''');
  }

  test_defaultValue_refersToGenericClass() async {
    var library = await buildLibrary('''
class B<T1, T2> {
  const B();
}
class C {
  void foo([B<int, double> b = const B()]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T1 @8
            defaultType: dynamic
          covariant T2 @12
            defaultType: dynamic
        constructors
          const @26
      class C @39
        constructors
          synthetic @-1
        methods
          foo @50
            parameters
              optionalPositional default b @70
                type: B<int, double>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @74
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @80
                        element: self::@class::B
                        type: B<int, double>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T1: int, T2: double}
                    argumentList: ArgumentList
                      leftParenthesis: ( @81
                      rightParenthesis: ) @82
                    staticType: B<int, double>
            returnType: void
''');
  }

  test_defaultValue_refersToGenericClass_constructor() async {
    var library = await buildLibrary('''
class B<T> {
  const B();
}
class C<T> {
  const C([B<T> b = const B()]);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class C @34
        typeParameters
          covariant T @36
            defaultType: dynamic
        constructors
          const @49
            parameters
              optionalPositional default b @57
                type: B<T>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @61
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @67
                        element: self::@class::B
                        type: B<Never>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Never}
                    argumentList: ArgumentList
                      leftParenthesis: ( @68
                      rightParenthesis: ) @69
                    staticType: B<Never>
''');
  }

  test_defaultValue_refersToGenericClass_constructor2() async {
    var library = await buildLibrary('''
abstract class A<T> {}
class B<T> implements A<T> {
  const B();
}
class C<T> implements A<Iterable<T>> {
  const C([A<T> a = const B()]);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        typeParameters
          covariant T @17
            defaultType: dynamic
        constructors
          synthetic @-1
      class B @29
        typeParameters
          covariant T @31
            defaultType: dynamic
        interfaces
          A<T>
        constructors
          const @60
      class C @73
        typeParameters
          covariant T @75
            defaultType: dynamic
        interfaces
          A<Iterable<T>>
        constructors
          const @114
            parameters
              optionalPositional default a @122
                type: A<T>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @126
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @132
                        element: self::@class::B
                        type: B<Never>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Never}
                    argumentList: ArgumentList
                      leftParenthesis: ( @133
                      rightParenthesis: ) @134
                    staticType: B<Never>
''');
  }

  test_defaultValue_refersToGenericClass_constructor2_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
abstract class A<T> {}
class B<T> implements A<T> {
  const B();
}
class C<T> implements A<Iterable<T>> {
  const C([A<T> a = const B()]);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @30
        typeParameters
          covariant T @32
            defaultType: dynamic
        constructors
          synthetic @-1
      class B @44
        typeParameters
          covariant T @46
            defaultType: dynamic
        interfaces
          A<T*>*
        constructors
          const @75
      class C @88
        typeParameters
          covariant T @90
            defaultType: dynamic
        interfaces
          A<Iterable<T*>*>*
        constructors
          const @129
            parameters
              optionalPositional default a @137
                type: A<T*>*
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @141
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @147
                        element: self::@class::B
                        type: B<Null*>*
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Null*}
                    argumentList: ArgumentList
                      leftParenthesis: ( @148
                      rightParenthesis: ) @149
                    staticType: B<Null*>*
''');
  }

  test_defaultValue_refersToGenericClass_constructor_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
class B<T> {
  const B();
}
class C<T> {
  const C([B<T> b = const B()]);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @21
        typeParameters
          covariant T @23
            defaultType: dynamic
        constructors
          const @36
      class C @49
        typeParameters
          covariant T @51
            defaultType: dynamic
        constructors
          const @64
            parameters
              optionalPositional default b @72
                type: B<T*>*
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @76
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @82
                        element: self::@class::B
                        type: B<Null*>*
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Null*}
                    argumentList: ArgumentList
                      leftParenthesis: ( @83
                      rightParenthesis: ) @84
                    staticType: B<Null*>*
''');
  }

  test_defaultValue_refersToGenericClass_functionG() async {
    var library = await buildLibrary('''
class B<T> {
  const B();
}
void foo<T>([B<T> b = const B()]) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    functions
      foo @33
        typeParameters
          covariant T @37
            defaultType: dynamic
        parameters
          optionalPositional default b @46
            type: B<T>
            constantInitializer
              InstanceCreationExpression
                keyword: const @50
                constructorName: ConstructorName
                  type: NamedType
                    name: B @56
                    element: self::@class::B
                    type: B<Never>
                  staticElement: ConstructorMember
                    base: self::@class::B::@constructor::new
                    substitution: {T: Never}
                argumentList: ArgumentList
                  leftParenthesis: ( @57
                  rightParenthesis: ) @58
                staticType: B<Never>
        returnType: void
''');
  }

  test_defaultValue_refersToGenericClass_methodG() async {
    var library = await buildLibrary('''
class B<T> {
  const B();
}
class C {
  void foo<T>([B<T> b = const B()]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class C @34
        constructors
          synthetic @-1
        methods
          foo @45
            typeParameters
              covariant T @49
                defaultType: dynamic
            parameters
              optionalPositional default b @58
                type: B<T>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @62
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @68
                        element: self::@class::B
                        type: B<Never>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Never}
                    argumentList: ArgumentList
                      leftParenthesis: ( @69
                      rightParenthesis: ) @70
                    staticType: B<Never>
            returnType: void
''');
  }

  test_defaultValue_refersToGenericClass_methodG_classG() async {
    var library = await buildLibrary('''
class B<T1, T2> {
  const B();
}
class C<E1> {
  void foo<E2>([B<E1, E2> b = const B()]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T1 @8
            defaultType: dynamic
          covariant T2 @12
            defaultType: dynamic
        constructors
          const @26
      class C @39
        typeParameters
          covariant E1 @41
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          foo @54
            typeParameters
              covariant E2 @58
                defaultType: dynamic
            parameters
              optionalPositional default b @73
                type: B<E1, E2>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @77
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @83
                        element: self::@class::B
                        type: B<Never, Never>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T1: Never, T2: Never}
                    argumentList: ArgumentList
                      leftParenthesis: ( @84
                      rightParenthesis: ) @85
                    staticType: B<Never, Never>
            returnType: void
''');
  }

  test_defaultValue_refersToGenericClass_methodNG() async {
    var library = await buildLibrary('''
class B<T> {
  const B();
}
class C<T> {
  void foo([B<T> b = const B()]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class B @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class C @34
        typeParameters
          covariant T @36
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          foo @48
            parameters
              optionalPositional default b @58
                type: B<T>
                constantInitializer
                  InstanceCreationExpression
                    keyword: const @62
                    constructorName: ConstructorName
                      type: NamedType
                        name: B @68
                        element: self::@class::B
                        type: B<Never>
                      staticElement: ConstructorMember
                        base: self::@class::B::@constructor::new
                        substitution: {T: Never}
                    argumentList: ArgumentList
                      leftParenthesis: ( @69
                      rightParenthesis: ) @70
                    staticType: B<Never>
            returnType: void
''');
  }

  test_duplicateDeclaration_class() async {
    var library = await buildLibrary(r'''
class A {}
class A {
  var x;
}
class A {
  var y = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class A @17
        fields
          x @27
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
      class A @38
        fields
          y @48
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get y @-1
            returnType: int
          synthetic set y @-1
            parameters
              requiredPositional _y @-1
                type: int
            returnType: void
''');
  }

  test_duplicateDeclaration_classTypeAlias() async {
    var library = await buildLibrary(r'''
class A {}
class B {}
class X = A with M;
class X = B with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        constructors
          synthetic @-1
      class alias X @28
        supertype: A
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
      class alias X @48
        supertype: B
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::B::@constructor::new
            superConstructor: self::@class::B::@constructor::new
    mixins
      mixin M @68
        superclassConstraints
          Object
''');
  }

  test_duplicateDeclaration_enum() async {
    var library = await buildLibrary(r'''
enum E {a, b}
enum E {c, d, e}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @8
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
      enum E @19
        supertype: Enum
        fields
          static const enumConstant c @22
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant d @25
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant e @28
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                  SimpleIdentifier
                    token: d @-1
                    staticElement: self::@enum::E::@getter::d
                    staticType: E
                  SimpleIdentifier
                    token: e @-1
                    staticElement: self::@enum::E::@getter::e
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get c @-1
            returnType: E
          synthetic static get d @-1
            returnType: E
          synthetic static get e @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_duplicateDeclaration_extension() async {
    var library = await buildLibrary(r'''
class A {}
extension E on A {}
extension E on A {
  static var x;
}
extension E on A {
  static var y = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
    extensions
      E @21
        extendedType: A
      E @41
        extendedType: A
        fields
          static x @63
            type: dynamic
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic static get x @-1
            returnType: dynamic
          synthetic static set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
      E @78
        extendedType: A
        fields
          static y @100
            type: int
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic static get y @-1
            returnType: int
          synthetic static set y @-1
            parameters
              requiredPositional _y @-1
                type: int
            returnType: void
''');
  }

  test_duplicateDeclaration_function() async {
    var library = await buildLibrary(r'''
void f() {}
void f(int a) {}
void f([int b, double c]) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        returnType: void
      f @17
        parameters
          requiredPositional a @23
            type: int
        returnType: void
      f @34
        parameters
          optionalPositional default b @41
            type: int
          optionalPositional default c @51
            type: double
        returnType: void
''');
  }

  test_duplicateDeclaration_functionTypeAlias() async {
    var library = await buildLibrary(r'''
typedef void F();
typedef void F(int a);
typedef void F([int b, double c]);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @13
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
      functionTypeAliasBased F @31
        aliasedType: void Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @37
              type: int
          returnType: void
      functionTypeAliasBased F @54
        aliasedType: void Function([int, double])
        aliasedElement: GenericFunctionTypeElement
          parameters
            optionalPositional b @61
              type: int
            optionalPositional c @71
              type: double
          returnType: void
''');
  }

  test_duplicateDeclaration_mixin() async {
    var library = await buildLibrary(r'''
mixin A {}
mixin A {
  var x;
}
mixin A {
  var y = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin A @6
        superclassConstraints
          Object
      mixin A @17
        superclassConstraints
          Object
        fields
          x @27
            type: dynamic
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
      mixin A @38
        superclassConstraints
          Object
        fields
          y @48
            type: int
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic get y @-1
            returnType: int
          synthetic set y @-1
            parameters
              requiredPositional _y @-1
                type: int
            returnType: void
''');
  }

  test_duplicateDeclaration_topLevelVariable() async {
    var library = await buildLibrary(r'''
bool x;
var x;
var x = 1;
var x = 2.3;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static x @5
        type: bool
        shouldUseTypeForInitializerInference: true
      static x @12
        type: dynamic
        shouldUseTypeForInitializerInference: false
      static x @19
        type: int
        shouldUseTypeForInitializerInference: false
      static x @30
        type: double
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: bool
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: bool
        returnType: void
      synthetic static get x @-1
        returnType: dynamic
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: dynamic
        returnType: void
      synthetic static get x @-1
        returnType: int
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int
        returnType: void
      synthetic static get x @-1
        returnType: double
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: double
        returnType: void
''');
  }

  test_enum_constant_inference() async {
    var library = await buildLibrary(r'''
enum E<T> {
  int(1), string('2');
  const E(T a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        supertype: Enum
        fields
          static const enumConstant int @14
            type: E<int>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<int>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: int}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    IntegerLiteral
                      literal: 1 @18
                      staticType: int
                  rightParenthesis: ) @0
                staticType: E<int>
          static const enumConstant string @22
            type: E<String>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<String>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: String}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    SimpleStringLiteral
                      literal: '2' @29
                  rightParenthesis: ) @0
                staticType: E<String>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: int @-1
                    staticElement: self::@enum::E::@getter::int
                    staticType: E<int>
                  SimpleIdentifier
                    token: string @-1
                    staticElement: self::@enum::E::@getter::string
                    staticType: E<String>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          const @43
            parameters
              requiredPositional a @47
                type: T
        accessors
          synthetic static get int @-1
            returnType: E<int>
          synthetic static get string @-1
            returnType: E<String>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  /// Test that a constant named `_name` renames the synthetic `name` field.
  test_enum_constant_name() async {
    var library = await buildLibrary(r'''
enum E {
  _name;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant _name @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: _name @-1
                    staticElement: self::@enum::E::@getter::_name
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get _name @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constant_typeArguments() async {
    var library = await buildLibrary(r'''
enum E<T> {
  v<double>(42);
  const E(T a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        supertype: Enum
        fields
          static const enumConstant v @14
            type: E<double>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    typeArguments: TypeArgumentList
                      leftBracket: < @15
                      arguments
                        NamedType
                          name: double @16
                          element: dart:core::@class::double
                          type: double
                      rightBracket: > @22
                    element: self::@enum::E
                    type: E<double>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: double}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    IntegerLiteral
                      literal: 42 @24
                      staticType: double
                  rightParenthesis: ) @0
                staticType: E<double>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<double>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          const @37
            parameters
              requiredPositional a @41
                type: T
        accessors
          synthetic static get v @-1
            returnType: E<double>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_constant_underscore() async {
    var library = await buildLibrary('''
enum E {
  _
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant _ @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: _ @-1
                    staticElement: self::@enum::E::@getter::_
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get _ @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_factory_named() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  factory E.named() => v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          factory named @26
            periodOffset: 25
            nameEnd: 31
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_factory_unnamed() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  factory E() => v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          factory @24
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_fieldFormal_functionTyped_withReturnType() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  final x;
  const E(int this.x(double a));
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @22
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          const @33
            parameters
              requiredPositional final this.x @44
                type: int Function(double)
                parameters
                  requiredPositional a @53
                    type: double
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_enum_constructor_fieldFormal_multiple_matching_fields() async {
    var library = await buildLibrary('''
enum E {
  v;
  final int x;
  final String x;
  const E(this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @26
            type: int
            shouldUseTypeForInitializerInference: true
          final x @44
            type: String
            shouldUseTypeForInitializerInference: true
        constructors
          const @55
            parameters
              requiredPositional final this.x @62
                type: int
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: int
          synthetic get x @-1
            returnType: String
''');
  }

  test_enum_constructor_fieldFormal_no_matching_field() async {
    var library = await buildLibrary('''
enum E {
  v;
  const E(this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          const @22
            parameters
              requiredPositional final this.x @29
                type: dynamic
                field: <null>
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_fieldFormal_optionalNamed_defaultValue() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  final int x;
  const E({this.x = 1 + 2});
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @26
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          const @37
            parameters
              optionalNamed default final this.x @45
                type: int
                constantInitializer
                  BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @49
                      staticType: int
                    operator: + @51
                    rightOperand: IntegerLiteral
                      literal: 2 @53
                      staticType: int
                    staticElement: dart:core::@class::num::@method::+
                    staticInvokeType: num Function(num)
                    staticType: int
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: int
''');
  }

  test_enum_constructor_fieldFormal_typed_typed() async {
    var library = await buildLibrary('''
enum E {
  v;
  final num x;
  const E(int this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @26
            type: num
            shouldUseTypeForInitializerInference: true
        constructors
          const @37
            parameters
              requiredPositional final this.x @48
                type: int
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: num
''');
  }

  test_enum_constructor_fieldFormal_untyped_typed() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  final x;
  E(int this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @22
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @27
            parameters
              requiredPositional final this.x @38
                type: int
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_enum_constructor_fieldFormal_untyped_untyped() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  final x;
  E(this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final x @22
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @27
            parameters
              requiredPositional final this.x @34
                type: dynamic
                field: self::@enum::E::@field::x
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_enum_constructor_generative_named() async {
    var library = await buildLibrary(r'''
enum E {
  v.named(42);
  const E.named(int a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  period: . @0
                  name: SimpleIdentifier
                    token: named @-1
                    staticElement: self::@enum::E::@constructor::named
                    staticType: null
                  staticElement: self::@enum::E::@constructor::named
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    IntegerLiteral
                      literal: 42 @19
                      staticType: int
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          const named @34
            periodOffset: 33
            nameEnd: 39
            parameters
              requiredPositional a @44
                type: int
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_generative_unnamed() async {
    var library = await buildLibrary(r'''
enum E {
  v(42);
  const E(int a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  arguments
                    IntegerLiteral
                      literal: 42 @13
                      staticType: int
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          const @26
            parameters
              requiredPositional a @32
                type: int
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_constructor_initializer() async {
    var library = await buildLibrary(r'''
enum E<T> {
  v;
  final int x;
  const E(T? a) : assert(a is T), x = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        supertype: Enum
        fields
          static const enumConstant v @14
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
          final x @29
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          const @40
            parameters
              requiredPositional a @45
                type: T?
            constantInitializers
              AssertInitializer
                assertKeyword: assert @50
                leftParenthesis: ( @56
                condition: IsExpression
                  expression: SimpleIdentifier
                    token: a @57
                    staticElement: self::@enum::E::@constructor::new::@parameter::a
                    staticType: T?
                  isOperator: is @59
                  type: NamedType
                    name: T @62
                    element: T@7
                    type: T
                  staticType: bool
                rightParenthesis: ) @63
              ConstructorFieldInitializer
                fieldName: SimpleIdentifier
                  token: x @66
                  staticElement: self::@enum::E::@field::x
                  staticType: null
                equals: = @68
                expression: IntegerLiteral
                  literal: 0 @70
                  staticType: int
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
          synthetic get x @-1
            returnType: int
''');
  }

  test_enum_documented() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
enum E { v }''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @65
        documentationComment: /**\n * Docs\n */
        supertype: Enum
        fields
          static const enumConstant v @69
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_field() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  final foo = 42;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          final foo @22
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 42 @28
                staticType: int
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          synthetic get foo @-1
            returnType: int
''');
  }

  test_enum_field_isPromotable() async {
    var library = await buildLibrary(r'''
enum E {
  v(null);
  final int? _foo;
  E(this._foo);
}
''');
    configuration.forPromotableFields(
      enumNames: {'E'},
      fieldNames: {'_foo'},
    );
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          final promotable _foo @33
            type: int?
            shouldUseTypeForInitializerInference: true
''');
  }

  test_enum_getter() async {
    var library = await buildLibrary(r'''
enum E{
  v;
  int get foo => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @10
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          synthetic foo @-1
            type: int
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          get foo @23
            returnType: int
''');
  }

  test_enum_interfaces() async {
    var library = await buildLibrary(r'''
class I {}
enum E implements I {
  v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class I @6
        constructors
          synthetic @-1
    enums
      enum E @16
        supertype: Enum
        interfaces
          I
        fields
          static const enumConstant v @35
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_interfaces_generic() async {
    var library = await buildLibrary(r'''
class I<T> {}
enum E<U> implements I<U> {
  v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class I @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
    enums
      enum E @19
        typeParameters
          covariant U @21
            defaultType: dynamic
        supertype: Enum
        interfaces
          I<U>
        fields
          static const enumConstant v @44
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {U: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_interfaces_unresolved() async {
    var library = await buildLibrary('''
class X {}
class Z {}
enum E implements X, Y, Z {
  v
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class X @6
        constructors
          synthetic @-1
      class Z @17
        constructors
          synthetic @-1
    enums
      enum E @27
        supertype: Enum
        interfaces
          X
          Z
        fields
          static const enumConstant v @52
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_method() async {
    var library = await buildLibrary(r'''
enum E<T> {
  v;
  int foo<U>(T t, U u) => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        supertype: Enum
        fields
          static const enumConstant v @14
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
        methods
          foo @23
            typeParameters
              covariant U @27
                defaultType: dynamic
            parameters
              requiredPositional t @32
                type: T
              requiredPositional u @37
                type: U
            returnType: int
''');
  }

  test_enum_method_toString() async {
    var library = await buildLibrary(r'''
enum E {
  v;
  String toString() => 'E';
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
        methods
          toString @23
            returnType: String
''');
  }

  test_enum_mixins() async {
    var library = await buildLibrary(r'''
mixin M {}
enum E with M {
  v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @16
        supertype: Enum
        mixins
          M
        fields
          static const enumConstant v @29
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    mixins
      mixin M @6
        superclassConstraints
          Object
''');
  }

  test_enum_mixins_inference() async {
    var library = await buildLibrary(r'''
mixin M1<T> {}
mixin M2<T> on M1<T> {}
enum E with M1<int>, M2 {
  v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @44
        supertype: Enum
        mixins
          M1<int>
          M2<int>
        fields
          static const enumConstant v @67
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    mixins
      mixin M1 @6
        typeParameters
          covariant T @9
            defaultType: dynamic
        superclassConstraints
          Object
      mixin M2 @21
        typeParameters
          covariant T @24
            defaultType: dynamic
        superclassConstraints
          M1<T>
''');
  }

  test_enum_setter() async {
    var library = await buildLibrary(r'''
enum E{
  v;
  set foo(int _) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @10
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
          synthetic foo @-1
            type: int
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
          set foo @19
            parameters
              requiredPositional _ @27
                type: int
            returnType: void
''');
  }

  test_enum_typeParameters() async {
    var library = await buildLibrary('''
enum E<T> {
  v
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        supertype: Enum
        fields
          static const enumConstant v @14
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_bound() async {
    var library = await buildLibrary('''
enum E<T extends num, U extends T> {
  v
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      notSimplyBounded enum E @5
        typeParameters
          covariant T @7
            bound: num
            defaultType: num
          covariant U @22
            bound: T
            defaultType: num
        supertype: Enum
        fields
          static const enumConstant v @39
            type: E<num, num>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<num, num>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: num, U: num}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<num, num>
          synthetic static const values @-1
            type: List<E<num, num>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<num, num>
                rightBracket: ] @0
                staticType: List<E<num, num>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<num, num>
          synthetic static get values @-1
            returnType: List<E<num, num>>
''');
  }

  test_enum_typeParameters_cycle_1of1() async {
    var library = await buildLibrary('''
enum E<T extends T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      notSimplyBounded enum E @5
        typeParameters
          covariant T @7
            bound: dynamic
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_cycle_2of3() async {
    var library = await buildLibrary(r'''
enum E<T extends V, U extends num, V extends T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      notSimplyBounded enum E @5
        typeParameters
          covariant T @7
            bound: dynamic
            defaultType: dynamic
          covariant U @20
            bound: num
            defaultType: num
          covariant V @35
            bound: dynamic
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic, num, dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic, num, dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic, num, dynamic>>
''');
  }

  test_enum_typeParameters_defaultType_cycle_genericFunctionType() async {
    var library = await buildLibrary(r'''
enum E<T extends void Function(E)> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      notSimplyBounded enum E @5
        typeParameters
          covariant T @7
            bound: void Function(E<dynamic>)
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_variance_contravariant() async {
    var library = await buildLibrary('''
enum E<in T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          contravariant T @10
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_variance_covariant() async {
    var library = await buildLibrary('''
enum E<out T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          covariant T @11
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_variance_invariant() async {
    var library = await buildLibrary('''
enum E<inout T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          invariant T @13
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic>>
''');
  }

  test_enum_typeParameters_variance_multiple() async {
    var library = await buildLibrary('''
enum E<inout T, in U, out V> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        typeParameters
          invariant T @13
            defaultType: dynamic
          contravariant U @19
            defaultType: dynamic
          covariant V @26
            defaultType: dynamic
        supertype: Enum
        fields
          synthetic static const values @-1
            type: List<E<dynamic, dynamic, dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                rightBracket: ] @0
                staticType: List<E<dynamic, dynamic, dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get values @-1
            returnType: List<E<dynamic, dynamic, dynamic>>
''');
  }

  test_enum_value_documented() async {
    var library = await buildLibrary('''
enum E {
  /**
   * aaa
   */
  a,
  /// bbb
  b
}''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @32
            documentationComment: /**\n   * aaa\n   */
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @47
            documentationComment: /// bbb
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enum_value_documented_withMetadata() async {
    var library = await buildLibrary('''
enum E {
  /**
   * aaa
   */
  @annotation
  a,
  /// bbb
  @annotation
  b,
}

const int annotation = 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @46
            documentationComment: /**\n   * aaa\n   */
            metadata
              Annotation
                atSign: @ @32
                name: SimpleIdentifier
                  token: annotation @33
                  staticElement: self::@getter::annotation
                  staticType: null
                element: self::@getter::annotation
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @75
            documentationComment: /// bbb
            metadata
              Annotation
                atSign: @ @61
                name: SimpleIdentifier
                  token: annotation @62
                  staticElement: self::@getter::annotation
                  staticType: null
                element: self::@getter::annotation
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static const annotation @91
        type: int
        shouldUseTypeForInitializerInference: true
        constantInitializer
          IntegerLiteral
            literal: 0 @104
            staticType: int
    accessors
      synthetic static get annotation @-1
        returnType: int
''');
  }

  test_enum_values() async {
    var library = await buildLibrary('enum E { v1, v2 }');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v1 @9
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant v2 @13
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v1 @-1
                    staticElement: self::@enum::E::@getter::v1
                    staticType: E
                  SimpleIdentifier
                    token: v2 @-1
                    staticElement: self::@enum::E::@getter::v2
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v1 @-1
            returnType: E
          synthetic static get v2 @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_enums() async {
    var library = await buildLibrary('enum E1 { v1 } enum E2 { v2 }');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E1 @5
        supertype: Enum
        fields
          static const enumConstant v1 @10
            type: E1
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E1 @-1
                    element: self::@enum::E1
                    type: E1
                  staticElement: self::@enum::E1::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E1
          synthetic static const values @-1
            type: List<E1>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v1 @-1
                    staticElement: self::@enum::E1::@getter::v1
                    staticType: E1
                rightBracket: ] @0
                staticType: List<E1>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v1 @-1
            returnType: E1
          synthetic static get values @-1
            returnType: List<E1>
      enum E2 @20
        supertype: Enum
        fields
          static const enumConstant v2 @25
            type: E2
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E2 @-1
                    element: self::@enum::E2
                    type: E2
                  staticElement: self::@enum::E2::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E2
          synthetic static const values @-1
            type: List<E2>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v2 @-1
                    staticElement: self::@enum::E2::@getter::v2
                    staticType: E2
                rightBracket: ] @0
                staticType: List<E2>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v2 @-1
            returnType: E2
          synthetic static get values @-1
            returnType: List<E2>
''');
  }

  test_error_extendsEnum() async {
    var library = await buildLibrary('''
enum E {a, b, c}

class M {}

class A extends E {
  foo() {}
}

class B implements E, M {
  foo() {}
}

class C extends Object with E, M {
  foo() {}
}

class D = Object with M, E;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class M @24
        constructors
          synthetic @-1
      class A @36
        constructors
          synthetic @-1
        methods
          foo @52
            returnType: dynamic
      class B @70
        interfaces
          M
        constructors
          synthetic @-1
        methods
          foo @92
            returnType: dynamic
      class C @110
        supertype: Object
        mixins
          M
        constructors
          synthetic @-1
        methods
          foo @141
            returnType: dynamic
      class alias D @159
        supertype: Object
        mixins
          M
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @8
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant c @14
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get c @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_executable_parameter_type_typedef() async {
    var library = await buildLibrary(r'''
typedef F(int p);
main(F f) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @8
        aliasedType: dynamic Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional p @14
              type: int
          returnType: dynamic
    functions
      main @18
        parameters
          requiredPositional f @25
            type: dynamic Function(int)
              alias: self::@typeAlias::F
        returnType: dynamic
''');
  }

  test_export_class() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::C
  exportNamespace
    C: package:test/a.dart;C
''');
  }

  test_export_class_type_alias() async {
    addSource('$testPackageLibPath/a.dart', r'''
class C = _D with _E;
class _D {}
class _E {}
''');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::C
  exportNamespace
    C: package:test/a.dart;C
''');
  }

  test_export_configurations_useDefault() async {
    declaredVariables = {
      'dart.library.io': 'false',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/foo.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/foo.dart::@unit::package:test/foo.dart::@class::A
  exportNamespace
    A: package:test/foo.dart;A
''');
    expect(library.libraryExports[0].exportedLibrary!.source.shortName,
        'foo.dart');
  }

  test_export_configurations_useFirst() async {
    declaredVariables = {
      'dart.library.io': 'true',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/foo_io.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/foo_io.dart::@unit::package:test/foo_io.dart::@class::A
  exportNamespace
    A: package:test/foo_io.dart;A
''');
    expect(library.libraryExports[0].exportedLibrary!.source.shortName,
        'foo_io.dart');
  }

  test_export_configurations_useSecond() async {
    declaredVariables = {
      'dart.library.io': 'false',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/foo_html.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/foo_html.dart::@unit::package:test/foo_html.dart::@class::A
  exportNamespace
    A: package:test/foo_html.dart;A
''');
    final export = library.libraryExports[0];
    expect(export.exportedLibrary!.source.shortName, 'foo_html.dart');
  }

  test_export_cycle() async {
    addSource('$testPackageLibPath/a.dart', r'''
export 'test.dart';
class A {}
''');

    var library = await buildLibrary(r'''
export 'a.dart';
class X {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
    classes
      class X @23
        constructors
          synthetic @-1
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A: package:test/a.dart;A
    X: package:test/test.dart;X
''');
  }

  test_export_function() async {
    addSource('$testPackageLibPath/a.dart', 'f() {}');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@function::f
  exportNamespace
    f: package:test/a.dart;f
''');
  }

  test_export_getter() async {
    addSource('$testPackageLibPath/a.dart', 'get f() => null;');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_export_hide() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {}
class B {}
class C {}
class D {}
''');
    var library = await buildLibrary(r'''
export 'a.dart' hide A, C;
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
      combinators
        hide: A, C
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::B
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::D
  exportNamespace
    B: package:test/a.dart;B
    D: package:test/a.dart;D
''');
  }

  test_export_multiple_combinators() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {}
class B {}
class C {}
class D {}
''');
    var library = await buildLibrary(r'''
export 'a.dart' hide A show C;
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
      combinators
        hide: A
        show: C
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::C
  exportNamespace
    C: package:test/a.dart;C
''');
  }

  test_export_reexport() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {}
''');

    addSource('$testPackageLibPath/b.dart', r'''
export 'a.dart';
class B {}
''');

    addSource('$testPackageLibPath/c.dart', r'''
export 'a.dart';
class C {}
''');

    var library = await buildLibrary(r'''
export 'b.dart';
export 'c.dart';
class X {}
''');

    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/b.dart
    package:test/c.dart
  definingUnit
    classes
      class X @40
        constructors
          synthetic @-1
  exportedReferences
    exported[(0, 0), (0, 1)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A
    exported[(0, 0)] root::package:test/b.dart::@unit::package:test/b.dart::@class::B
    exported[(0, 1)] root::package:test/c.dart::@unit::package:test/c.dart::@class::C
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A: package:test/a.dart;A
    B: package:test/b.dart;B
    C: package:test/c.dart;C
    X: package:test/test.dart;X
''');
  }

  test_export_setter() async {
    addSource('$testPackageLibPath/a.dart', 'void set f(value) {}');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@setter::f
  exportNamespace
    f=: package:test/a.dart;f=
''');
  }

  test_export_show() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {}
class B {}
class C {}
class D {}
''');
    var library = await buildLibrary(r'''
export 'a.dart' show A, C;
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
      combinators
        show: A, C
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::C
  exportNamespace
    A: package:test/a.dart;A
    C: package:test/a.dart;C
''');
  }

  test_export_show_getter_setter() async {
    addSource('$testPackageLibPath/a.dart', '''
get f => null;
void set f(value) {}
''');
    var library = await buildLibrary('export "a.dart" show f;');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
      combinators
        show: f
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@getter::f
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@setter::f
  exportNamespace
    f: package:test/a.dart;f?
    f=: package:test/a.dart;f=
''');
  }

  test_export_typedef() async {
    addSource('$testPackageLibPath/a.dart', 'typedef F();');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@typeAlias::F
  exportNamespace
    F: package:test/a.dart;F
''');
  }

  test_export_uri() async {
    var library = await buildLibrary('''
export 'foo.dart';
''');

    final uri = library.libraryExports[0].uri as DirectiveUriWithLibrary;
    expect(uri.relativeUriString, 'foo.dart');
  }

  test_export_variable() async {
    addSource('$testPackageLibPath/a.dart', 'var x;');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@getter::x
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@setter::x
  exportNamespace
    x: package:test/a.dart;x?
    x=: package:test/a.dart;x=
''');
  }

  test_export_variable_const() async {
    addSource('$testPackageLibPath/a.dart', 'const x = 0;');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@getter::x
  exportNamespace
    x: package:test/a.dart;x?
''');
  }

  test_export_variable_final() async {
    addSource('$testPackageLibPath/a.dart', 'final x = 0;');
    var library = await buildLibrary('export "a.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
  exportedReferences
    exported[(0, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@getter::x
  exportNamespace
    x: package:test/a.dart;x?
''');
  }

  test_exportImport_configurations_useDefault() async {
    declaredVariables = {
      'dart.library.io': 'false',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    addSource('$testPackageLibPath/bar.dart', r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    var library = await buildLibrary(r'''
import 'bar.dart';
class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/bar.dart
  definingUnit
    classes
      class B @25
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo.dart');
  }

  test_exportImport_configurations_useFirst() async {
    declaredVariables = {
      'dart.library.io': 'true',
      'dart.library.html': 'false',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    addSource('$testPackageLibPath/bar.dart', r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    var library = await buildLibrary(r'''
import 'bar.dart';
class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/bar.dart
  definingUnit
    classes
      class B @25
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_io.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_io.dart');
  }

  test_exportImport_configurations_useSecond() async {
    declaredVariables = {
      'dart.library.io': 'false',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    addSource('$testPackageLibPath/bar.dart', r'''
export 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';
''');
    var library = await buildLibrary(r'''
import 'bar.dart';
class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/bar.dart
  definingUnit
    classes
      class B @25
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_html.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_html.dart');
  }

  test_exports() async {
    addSource('$testPackageLibPath/a.dart', 'library a;');
    addSource('$testPackageLibPath/b.dart', 'library b;');
    var library = await buildLibrary('export "a.dart"; export "b.dart";');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
    package:test/b.dart
  definingUnit
  exportedReferences
  exportNamespace
''');
  }

  test_exportScope_augmentation_class() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class A {}
''');
    var library = await buildLibrary(r'''
import augment 'a.dart';
class B {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class A @35
            constructors
              synthetic @-1
  definingUnit
    classes
      class B @31
        constructors
          synthetic @-1
  exportedReferences
    declared root::package:test/test.dart::@augmentation::package:test/a.dart::@class::A
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::B
  exportNamespace
    A: package:test/test.dart;package:test/a.dart;package:test/a.dart;A
    B: package:test/test.dart;B
''');
  }

  test_exportScope_augmentation_export() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
class B1 {}
class B2 {}
''');
    newFile('$testPackageLibPath/c.dart', r'''
class C {}
''');
    newFile('$testPackageLibPath/d.dart', r'''
library augment 'test.dart';
export 'a.dart';
''');
    newFile('$testPackageLibPath/e.dart', r'''
library augment 'test.dart';
export 'b.dart';
export 'c.dart';
''');
    var library = await buildLibrary(r'''
import augment 'd.dart';
import augment 'e.dart';
class X {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/d.dart
      exports
        package:test/a.dart
      definingUnit
    package:test/e.dart
      exports
        package:test/b.dart
        package:test/c.dart
      definingUnit
  definingUnit
    classes
      class X @56
        constructors
          synthetic @-1
  exportedReferences
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A
    exported[(2, 0)] root::package:test/b.dart::@unit::package:test/b.dart::@class::B1
    exported[(2, 0)] root::package:test/b.dart::@unit::package:test/b.dart::@class::B2
    exported[(2, 1)] root::package:test/c.dart::@unit::package:test/c.dart::@class::C
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A: package:test/a.dart;A
    B1: package:test/b.dart;B1
    B2: package:test/b.dart;B2
    C: package:test/c.dart;C
    X: package:test/test.dart;X
''');
  }

  test_exportScope_augmentation_export_hide() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A1 {}
class A2 {}
class A3 {}
class A4 {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
export 'a.dart' hide A2, A4;
''');
    var library = await buildLibrary(r'''
import augment 'b.dart';
class X {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      exports
        package:test/a.dart
          combinators
            hide: A2, A4
      definingUnit
  definingUnit
    classes
      class X @31
        constructors
          synthetic @-1
  exportedReferences
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A1
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A3
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A1: package:test/a.dart;A1
    A3: package:test/a.dart;A3
    X: package:test/test.dart;X
''');
  }

  test_exportScope_augmentation_export_show() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A1 {}
class A2 {}
class A3 {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
export 'a.dart' show A1, A3;
''');
    var library = await buildLibrary(r'''
import augment 'b.dart';
class X {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      exports
        package:test/a.dart
          combinators
            show: A1, A3
      definingUnit
  definingUnit
    classes
      class X @31
        constructors
          synthetic @-1
  exportedReferences
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A1
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A3
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A1: package:test/a.dart;A1
    A3: package:test/a.dart;A3
    X: package:test/test.dart;X
''');
  }

  test_exportScope_augmentation_nested_class() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
import augment 'b.dart';
class A {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'a.dart';
class B {}
''');
    var library = await buildLibrary(r'''
import augment 'a.dart';
class C {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      augmentationImports
        package:test/b.dart
          definingUnit
            classes
              class B @32
                constructors
                  synthetic @-1
      definingUnit
        classes
          class A @60
            constructors
              synthetic @-1
  definingUnit
    classes
      class C @31
        constructors
          synthetic @-1
  exportedReferences
    declared root::package:test/test.dart::@augmentation::package:test/a.dart::@class::A
    declared root::package:test/test.dart::@augmentation::package:test/b.dart::@class::B
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::C
  exportNamespace
    A: package:test/test.dart;package:test/a.dart;package:test/a.dart;A
    B: package:test/test.dart;package:test/a.dart;package:test/b.dart;package:test/b.dart;B
    C: package:test/test.dart;C
''');
  }

  test_exportScope_augmentation_nested_export() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');
    newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');
    newFile('$testPackageLibPath/c.dart', r'''
library augment 'test.dart';
import augment 'd.dart';
export 'a.dart';
''');
    newFile('$testPackageLibPath/d.dart', r'''
library augment 'c.dart';
export 'b.dart';
''');
    var library = await buildLibrary(r'''
import augment 'c.dart';
class X {}
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/c.dart
      exports
        package:test/a.dart
      augmentationImports
        package:test/d.dart
          exports
            package:test/b.dart
          definingUnit
      definingUnit
  definingUnit
    classes
      class X @31
        constructors
          synthetic @-1
  exportedReferences
    exported[(1, 0)] root::package:test/a.dart::@unit::package:test/a.dart::@class::A
    exported[(2, 0)] root::package:test/b.dart::@unit::package:test/b.dart::@class::B
    declared root::package:test/test.dart::@unit::package:test/test.dart::@class::X
  exportNamespace
    A: package:test/a.dart;A
    B: package:test/b.dart;B
    X: package:test/test.dart;X
''');
  }

  test_exportScope_augmentation_variable() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
int a = 0;
''');
    var library = await buildLibrary(r'''
import augment 'a.dart';
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        topLevelVariables
          static a @33
            type: int
            shouldUseTypeForInitializerInference: true
        accessors
          synthetic static get a @-1
            returnType: int
          synthetic static set a @-1
            parameters
              requiredPositional _a @-1
                type: int
            returnType: void
  definingUnit
  exportedReferences
    declared root::package:test/test.dart::@augmentation::package:test/a.dart::@getter::a
    declared root::package:test/test.dart::@augmentation::package:test/a.dart::@setter::a
  exportNamespace
    a: package:test/test.dart;package:test/a.dart;package:test/a.dart;a?
    a=: package:test/test.dart;package:test/a.dart;package:test/a.dart;a=
''');
  }

  test_exportScope_augmentation_variable_const() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
const a = 0;
''');
    var library = await buildLibrary(r'''
import augment 'a.dart';
''');
    configuration.withExportScope = true;
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        topLevelVariables
          static const a @35
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 0 @39
                staticType: int
        accessors
          synthetic static get a @-1
            returnType: int
  definingUnit
  exportedReferences
    declared root::package:test/test.dart::@augmentation::package:test/a.dart::@getter::a
  exportNamespace
    a: package:test/test.dart;package:test/a.dart;package:test/a.dart;a?
''');
  }

  test_expr_invalid_typeParameter_asPrefix() async {
    var library = await buildLibrary('''
class C<T> {
  final f = T.k;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        fields
          final f @21
            type: InvalidType
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get f @-1
            returnType: InvalidType
''');
  }

  test_extension_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// aaa
/// bbbb
/// cc
extension E on int {}''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @34
        documentationComment: /// aaa\n/// bbbb\n/// cc
        extendedType: int
''');
  }

  test_extension_field_inferredType_const() async {
    var library = await buildLibrary('''
extension E on int {
  static const x = 0;
}''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        extendedType: int
        fields
          static const x @36
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 0 @40
                staticType: int
        accessors
          synthetic static get x @-1
            returnType: int
''');
  }

  test_extension_typeParameters_hasBound() async {
    var library = await buildLibrary('''
extension E<T extends num> on int {}
''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        typeParameters
          covariant T @12
            bound: num
            defaultType: num
        extendedType: int
''');
  }

  test_extension_typeParameters_noBound() async {
    var library = await buildLibrary('''
extension E<T> on int {}
''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        typeParameters
          covariant T @12
            defaultType: dynamic
        extendedType: int
''');
  }

  test_function_async() async {
    var library = await buildLibrary(r'''
import 'dart:async';
Future f() async {}
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    functions
      f @28 async
        returnType: Future<dynamic>
''');
  }

  test_function_asyncStar() async {
    var library = await buildLibrary(r'''
import 'dart:async';
Stream f() async* {}
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    functions
      f @28 async*
        returnType: Stream<dynamic>
''');
  }

  test_function_documented() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
f() {}''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @60
        documentationComment: /**\n * Docs\n */
        returnType: dynamic
''');
  }

  test_function_entry_point() async {
    var library = await buildLibrary('main() {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      main @0
        returnType: dynamic
''');
  }

  test_function_entry_point_in_export() async {
    addSource('$testPackageLibPath/a.dart', 'library a; main() {}');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_function_entry_point_in_export_hidden() async {
    addSource('$testPackageLibPath/a.dart', 'library a; main() {}');
    var library = await buildLibrary('export "a.dart" hide main;');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
      combinators
        hide: main
  definingUnit
''');
  }

  test_function_entry_point_in_part() async {
    addSource('$testPackageLibPath/a.dart', 'part of my.lib; main() {}');
    var library = await buildLibrary('library my.lib; part "a.dart";');
    checkElementText(library, r'''
library
  name: my.lib
  nameOffset: 8
  definingUnit
  parts
    package:test/a.dart
      functions
        main @16
          returnType: dynamic
''');
  }

  test_function_external() async {
    var library = await buildLibrary('external f();');
    checkElementText(library, r'''
library
  definingUnit
    functions
      external f @9
        returnType: dynamic
''');
  }

  test_function_hasImplicitReturnType_false() async {
    var library = await buildLibrary('''
int f() => 0;
''');
    var f = library.definingCompilationUnit.functions.single;
    expect(f.hasImplicitReturnType, isFalse);
  }

  test_function_hasImplicitReturnType_true() async {
    var library = await buildLibrary('''
f() => 0;
''');
    var f = library.definingCompilationUnit.functions.single;
    expect(f.hasImplicitReturnType, isTrue);
  }

  test_function_parameter_const() async {
    var library = await buildLibrary('''
void f(const x) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional x @13
            type: dynamic
        returnType: void
''');
  }

  test_function_parameter_fieldFormal() async {
    var library = await buildLibrary('''
void f(int this.a) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional final this.a @16
            type: int
            field: <null>
        returnType: void
''');
  }

  test_function_parameter_fieldFormal_default() async {
    var library = await buildLibrary('''
void f({int this.a: 42}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          optionalNamed default final this.a @17
            type: int
            constantInitializer
              IntegerLiteral
                literal: 42 @20
                staticType: int
            field: <null>
        returnType: void
''');
  }

  test_function_parameter_fieldFormal_functionTyped() async {
    var library = await buildLibrary('''
void f(int this.a(int b)) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional final this.a @16
            type: int Function(int)
            parameters
              requiredPositional b @22
                type: int
            field: <null>
        returnType: void
''');
  }

  test_function_parameter_final() async {
    var library = await buildLibrary('f(final x) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional final x @8
            type: dynamic
        returnType: dynamic
''');
  }

  test_function_parameter_kind_named() async {
    var library = await buildLibrary('f({x}) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          optionalNamed default x @3
            type: dynamic
        returnType: dynamic
''');
  }

  test_function_parameter_kind_positional() async {
    var library = await buildLibrary('f([x]) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          optionalPositional default x @3
            type: dynamic
        returnType: dynamic
''');
  }

  test_function_parameter_kind_required() async {
    var library = await buildLibrary('f(x) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional x @2
            type: dynamic
        returnType: dynamic
''');
  }

  test_function_parameter_parameters() async {
    var library = await buildLibrary('f(g(x, y)) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional g @2
            type: dynamic Function(dynamic, dynamic)
            parameters
              requiredPositional x @4
                type: dynamic
              requiredPositional y @7
                type: dynamic
        returnType: dynamic
''');
  }

  test_function_parameter_return_type() async {
    var library = await buildLibrary('f(int g()) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional g @6
            type: int Function()
        returnType: dynamic
''');
  }

  test_function_parameter_return_type_void() async {
    var library = await buildLibrary('f(void g()) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional g @7
            type: void Function()
        returnType: dynamic
''');
  }

  test_function_parameter_type() async {
    var library = await buildLibrary('f(int i) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional i @6
            type: int
        returnType: dynamic
''');
  }

  test_function_parameter_type_typeParameter() async {
    var library = await buildLibrary('''
void f<T>(T a) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        parameters
          requiredPositional a @12
            type: T
        returnType: void
''');
  }

  test_function_parameter_type_unresolved() async {
    var library = await buildLibrary(r'''
void f(A a) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional a @9
            type: InvalidType
        returnType: void
''');
  }

  test_function_parameters() async {
    var library = await buildLibrary('f(x, y) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        parameters
          requiredPositional x @2
            type: dynamic
          requiredPositional y @5
            type: dynamic
        returnType: dynamic
''');
  }

  test_function_return_type_implicit() async {
    var library = await buildLibrary('f() => null;');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        returnType: dynamic
''');
  }

  test_function_return_type_unresolved() async {
    var library = await buildLibrary(r'''
A f() {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @2
        returnType: InvalidType
''');
  }

  test_function_return_type_void() async {
    var library = await buildLibrary('void f() {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        returnType: void
''');
  }

  test_function_returnType() async {
    var library = await buildLibrary('''
int f() => 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @4
        returnType: int
''');
  }

  test_function_returnType_typeParameter() async {
    var library = await buildLibrary('''
T f<T>() => throw 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @2
        typeParameters
          covariant T @4
            defaultType: dynamic
        returnType: T
''');
  }

  test_function_type_parameter_with_function_typed_parameter() async {
    var library = await buildLibrary('void f<T, U>(T x(U u)) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
          covariant U @10
            defaultType: dynamic
        parameters
          requiredPositional x @15
            type: T Function(U)
            parameters
              requiredPositional u @19
                type: U
        returnType: void
''');
  }

  test_function_typed_parameter_implicit() async {
    var library = await buildLibrary('f(g()) => null;');
    expect(
        library
            .definingCompilationUnit.functions[0].parameters[0].hasImplicitType,
        isFalse);
  }

  test_function_typeParameters_hasBound() async {
    var library = await buildLibrary('''
void f<T extends num>() {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            bound: num
            defaultType: num
        returnType: void
''');
  }

  test_function_typeParameters_noBound() async {
    var library = await buildLibrary('''
void f<T>() {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        returnType: void
''');
  }

  test_functions() async {
    var library = await buildLibrary('f() {} g() {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        returnType: dynamic
      g @7
        returnType: dynamic
''');
  }

  test_functionTypeAlias_enclosingElements() async {
    var library = await buildLibrary(r'''
typedef void F<T>(int a);
''');
    var unit = library.definingCompilationUnit;

    var F = unit.typeAliases[0];
    expect(F.name, 'F');

    var T = F.typeParameters[0];
    expect(T.name, 'T');
    expect(T.enclosingElement, same(F));

    var function = F.aliasedElement as GenericFunctionTypeElement;
    expect(function.enclosingElement, same(F));

    var a = function.parameters[0];
    expect(a.name, 'a');
    expect(a.enclosingElement, same(function));
  }

  test_functionTypeAlias_type_element() async {
    var library = await buildLibrary(r'''
typedef T F<T>();
F<int> a;
''');
    var unit = library.definingCompilationUnit;
    var type = unit.topLevelVariables[0].type as FunctionType;

    expect(type.alias!.element, same(unit.typeAliases[0]));
    _assertTypeStrings(type.alias!.typeArguments, ['int']);
  }

  test_functionTypeAlias_typeParameters_variance_contravariant() async {
    var library = await buildLibrary(r'''
typedef void F<T>(T a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @13
        typeParameters
          contravariant T @15
            defaultType: dynamic
        aliasedType: void Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @20
              type: T
          returnType: void
''');
  }

  test_functionTypeAlias_typeParameters_variance_contravariant2() async {
    var library = await buildLibrary(r'''
typedef void F1<T>(T a);
typedef F1<T> F2<T>();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F1 @13
        typeParameters
          contravariant T @16
            defaultType: dynamic
        aliasedType: void Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @21
              type: T
          returnType: void
      functionTypeAliasBased F2 @39
        typeParameters
          contravariant T @42
            defaultType: dynamic
        aliasedType: void Function(T) Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void Function(T)
            alias: self::@typeAlias::F1
              typeArguments
                T
''');
  }

  test_functionTypeAlias_typeParameters_variance_contravariant3() async {
    var library = await buildLibrary(r'''
typedef F1<T> F2<T>();
typedef void F1<T>(T a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F2 @14
        typeParameters
          contravariant T @17
            defaultType: dynamic
        aliasedType: void Function(T) Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void Function(T)
            alias: self::@typeAlias::F1
              typeArguments
                T
      functionTypeAliasBased F1 @36
        typeParameters
          contravariant T @39
            defaultType: dynamic
        aliasedType: void Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @44
              type: T
          returnType: void
''');
  }

  test_functionTypeAlias_typeParameters_variance_covariant() async {
    var library = await buildLibrary(r'''
typedef T F<T>();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @10
        typeParameters
          covariant T @12
            defaultType: dynamic
        aliasedType: T Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: T
''');
  }

  test_functionTypeAlias_typeParameters_variance_covariant2() async {
    var library = await buildLibrary(r'''
typedef List<T> F<T>();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @16
        typeParameters
          covariant T @18
            defaultType: dynamic
        aliasedType: List<T> Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: List<T>
''');
  }

  test_functionTypeAlias_typeParameters_variance_covariant3() async {
    var library = await buildLibrary(r'''
typedef T F1<T>();
typedef F1<T> F2<T>();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F1 @10
        typeParameters
          covariant T @13
            defaultType: dynamic
        aliasedType: T Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: T
      functionTypeAliasBased F2 @33
        typeParameters
          covariant T @36
            defaultType: dynamic
        aliasedType: T Function() Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: T Function()
            alias: self::@typeAlias::F1
              typeArguments
                T
''');
  }

  test_functionTypeAlias_typeParameters_variance_covariant4() async {
    var library = await buildLibrary(r'''
typedef void F1<T>(T a);
typedef void F2<T>(F1<T> a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F1 @13
        typeParameters
          contravariant T @16
            defaultType: dynamic
        aliasedType: void Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @21
              type: T
          returnType: void
      functionTypeAliasBased F2 @38
        typeParameters
          covariant T @41
            defaultType: dynamic
        aliasedType: void Function(void Function(T))
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @50
              type: void Function(T)
                alias: self::@typeAlias::F1
                  typeArguments
                    T
          returnType: void
''');
  }

  test_functionTypeAlias_typeParameters_variance_invariant() async {
    var library = await buildLibrary(r'''
typedef T F<T>(T a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @10
        typeParameters
          invariant T @12
            defaultType: dynamic
        aliasedType: T Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @17
              type: T
          returnType: T
''');
  }

  test_functionTypeAlias_typeParameters_variance_invariant2() async {
    var library = await buildLibrary(r'''
typedef T F1<T>();
typedef F1<T> F2<T>(T a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F1 @10
        typeParameters
          covariant T @13
            defaultType: dynamic
        aliasedType: T Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: T
      functionTypeAliasBased F2 @33
        typeParameters
          invariant T @36
            defaultType: dynamic
        aliasedType: T Function() Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @41
              type: T
          returnType: T Function()
            alias: self::@typeAlias::F1
              typeArguments
                T
''');
  }

  test_functionTypeAlias_typeParameters_variance_unrelated() async {
    var library = await buildLibrary(r'''
typedef void F<T>(int a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @13
        typeParameters
          unrelated T @15
            defaultType: dynamic
        aliasedType: void Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @22
              type: int
          returnType: void
''');
  }

  test_futureOr() async {
    var library = await buildLibrary('import "dart:async"; FutureOr<int> x;');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static x @35
        type: FutureOr<int>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get x @-1
        returnType: FutureOr<int>
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: FutureOr<int>
        returnType: void
''');
    var variables = library.definingCompilationUnit.topLevelVariables;
    expect(variables, hasLength(1));
    _assertTypeStr(variables[0].type, 'FutureOr<int>');
  }

  test_futureOr_const() async {
    var library =
        await buildLibrary('import "dart:async"; const x = FutureOr;');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static const x @27
        type: Type
        shouldUseTypeForInitializerInference: false
        constantInitializer
          SimpleIdentifier
            token: FutureOr @31
            staticElement: dart:async::@class::FutureOr
            staticType: Type
    accessors
      synthetic static get x @-1
        returnType: Type
''');
    var variables = library.definingCompilationUnit.topLevelVariables;
    expect(variables, hasLength(1));
    var x = variables[0] as ConstTopLevelVariableElementImpl;
    _assertTypeStr(x.type, 'Type');
    expect(x.constantInitializer.toString(), 'FutureOr');
  }

  test_futureOr_inferred() async {
    var library = await buildLibrary('''
import "dart:async";
FutureOr<int> f() => null;
var x = f();
var y = x.then((z) => z.asDouble());
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static x @52
        type: FutureOr<int>
        shouldUseTypeForInitializerInference: false
      static y @65
        type: InvalidType
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: FutureOr<int>
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: FutureOr<int>
        returnType: void
      synthetic static get y @-1
        returnType: InvalidType
      synthetic static set y @-1
        parameters
          requiredPositional _y @-1
            type: InvalidType
        returnType: void
    functions
      f @35
        returnType: FutureOr<int>
''');
    var variables = library.definingCompilationUnit.topLevelVariables;
    expect(variables, hasLength(2));
    var x = variables[0];
    expect(x.name, 'x');
    var y = variables[1];
    expect(y.name, 'y');
    _assertTypeStr(x.type, 'FutureOr<int>');
    _assertTypeStr(y.type, 'InvalidType');
  }

  test_generic_function_type_nullability_none() async {
    var library = await buildLibrary('''
void Function() f;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static f @16
        type: void Function()
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: void Function()
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: void Function()
        returnType: void
''');
  }

  test_generic_function_type_nullability_question() async {
    var library = await buildLibrary('''
void Function()? f;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static f @17
        type: void Function()?
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: void Function()?
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: void Function()?
        returnType: void
''');
  }

  test_generic_function_type_nullability_star() async {
    var library = await buildLibrary('''
// @dart = 2.9
void Function() f;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static f @31
        type: void Function()*
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: void Function()*
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: void Function()*
        returnType: void
''');
  }

  test_generic_gClass_gMethodStatic() async {
    var library = await buildLibrary('''
class C<T, U> {
  static void m<V, W>(V v, W w) {
    void f<X, Y>(V v, W w, X x, Y y) {
    }
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          static m @30
            typeParameters
              covariant V @32
                defaultType: dynamic
              covariant W @35
                defaultType: dynamic
            parameters
              requiredPositional v @40
                type: V
              requiredPositional w @45
                type: W
            returnType: void
''');
  }

  test_genericFunction_asFunctionReturnType() async {
    var library = await buildLibrary(r'''
int Function(int a, String b) f() => null;
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @30
        returnType: int Function(int, String)
''');
  }

  test_genericFunction_asFunctionTypedParameterReturnType() async {
    var library = await buildLibrary(r'''
void f(int Function(int a, String b) p(num c)) => null;
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional p @37
            type: int Function(int, String) Function(num)
            parameters
              requiredPositional c @43
                type: num
        returnType: void
''');
  }

  test_genericFunction_asGenericFunctionReturnType() async {
    var library = await buildLibrary(r'''
typedef F = void Function(String a) Function(int b);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @8
        aliasedType: void Function(String) Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional b @49
              type: int
          returnType: void Function(String)
''');
  }

  test_genericFunction_asMethodReturnType() async {
    var library = await buildLibrary(r'''
class C {
  int Function(int a, String b) m() => null;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          m @42
            returnType: int Function(int, String)
''');
  }

  test_genericFunction_asParameterType() async {
    var library = await buildLibrary(r'''
void f(int Function(int a, String b) p) => null;
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional p @37
            type: int Function(int, String)
        returnType: void
''');
  }

  test_genericFunction_asTopLevelVariableType() async {
    var library = await buildLibrary(r'''
int Function(int a, String b) v;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @30
        type: int Function(int, String)
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get v @-1
        returnType: int Function(int, String)
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: int Function(int, String)
        returnType: void
''');
  }

  test_genericFunction_asTypeArgument_ofAnnotation_class() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

@A<int Function(String a)>()
class B {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class B @64
        metadata
          Annotation
            atSign: @ @29
            name: SimpleIdentifier
              token: A @30
              staticElement: self::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @31
              arguments
                GenericFunctionType
                  returnType: NamedType
                    name: int @32
                    element: dart:core::@class::int
                    type: int
                  functionKeyword: Function @36
                  parameters: FormalParameterList
                    leftParenthesis: ( @44
                    parameter: SimpleFormalParameter
                      type: NamedType
                        name: String @45
                        element: dart:core::@class::String
                        type: String
                      name: a @52
                      declaredElement: a@52
                        type: String
                    rightParenthesis: ) @53
                  declaredElement: GenericFunctionTypeElement
                    parameters
                      a
                        kind: required positional
                        type: String
                    returnType: int
                    type: int Function(String)
                  type: int Function(String)
              rightBracket: > @54
            arguments: ArgumentList
              leftParenthesis: ( @55
              rightParenthesis: ) @56
            element: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int Function(String)}
        constructors
          synthetic @-1
''');
  }

  test_genericFunction_asTypeArgument_ofAnnotation_topLevelVariable() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

@A<int Function(String a)>()
var v = 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    topLevelVariables
      static v @62
        metadata
          Annotation
            atSign: @ @29
            name: SimpleIdentifier
              token: A @30
              staticElement: self::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @31
              arguments
                GenericFunctionType
                  returnType: NamedType
                    name: int @32
                    element: dart:core::@class::int
                    type: int
                  functionKeyword: Function @36
                  parameters: FormalParameterList
                    leftParenthesis: ( @44
                    parameter: SimpleFormalParameter
                      type: NamedType
                        name: String @45
                        element: dart:core::@class::String
                        type: String
                      name: a @52
                      declaredElement: a@52
                        type: String
                    rightParenthesis: ) @53
                  declaredElement: GenericFunctionTypeElement
                    parameters
                      a
                        kind: required positional
                        type: String
                    returnType: int
                    type: int Function(String)
                  type: int Function(String)
              rightBracket: > @54
            arguments: ArgumentList
              leftParenthesis: ( @55
              rightParenthesis: ) @56
            element: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int Function(String)}
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: int
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: int
        returnType: void
''');
  }

  test_genericFunction_asTypeArgument_parameters_optionalNamed() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

const v = A<String Function({int? a})>();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    topLevelVariables
      static const v @35
        type: A<String Function({int? a})>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            constructorName: ConstructorName
              type: NamedType
                name: A @39
                typeArguments: TypeArgumentList
                  leftBracket: < @40
                  arguments
                    GenericFunctionType
                      returnType: NamedType
                        name: String @41
                        element: dart:core::@class::String
                        type: String
                      functionKeyword: Function @48
                      parameters: FormalParameterList
                        leftParenthesis: ( @56
                        leftDelimiter: { @57
                        parameter: DefaultFormalParameter
                          parameter: SimpleFormalParameter
                            type: NamedType
                              name: int @58
                              question: ? @61
                              element: dart:core::@class::int
                              type: int?
                            name: a @63
                            declaredElement: a@63
                              type: int?
                          declaredElement: a@63
                            type: int?
                        rightDelimiter: } @64
                        rightParenthesis: ) @65
                      declaredElement: GenericFunctionTypeElement
                        parameters
                          a
                            kind: optional named
                            type: int?
                        returnType: String
                        type: String Function({int? a})
                      type: String Function({int? a})
                  rightBracket: > @66
                element: self::@class::A
                type: A<String Function({int? a})>
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::new
                substitution: {T: String Function({int? a})}
            argumentList: ArgumentList
              leftParenthesis: ( @67
              rightParenthesis: ) @68
            staticType: A<String Function({int? a})>
    accessors
      synthetic static get v @-1
        returnType: A<String Function({int? a})>
''');
  }

  test_genericFunction_asTypeArgument_parameters_optionalPositional() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

const v = A<String Function([int? a])>();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    topLevelVariables
      static const v @35
        type: A<String Function([int?])>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            constructorName: ConstructorName
              type: NamedType
                name: A @39
                typeArguments: TypeArgumentList
                  leftBracket: < @40
                  arguments
                    GenericFunctionType
                      returnType: NamedType
                        name: String @41
                        element: dart:core::@class::String
                        type: String
                      functionKeyword: Function @48
                      parameters: FormalParameterList
                        leftParenthesis: ( @56
                        leftDelimiter: [ @57
                        parameter: DefaultFormalParameter
                          parameter: SimpleFormalParameter
                            type: NamedType
                              name: int @58
                              question: ? @61
                              element: dart:core::@class::int
                              type: int?
                            name: a @63
                            declaredElement: a@63
                              type: int?
                          declaredElement: a@63
                            type: int?
                        rightDelimiter: ] @64
                        rightParenthesis: ) @65
                      declaredElement: GenericFunctionTypeElement
                        parameters
                          a
                            kind: optional positional
                            type: int?
                        returnType: String
                        type: String Function([int?])
                      type: String Function([int?])
                  rightBracket: > @66
                element: self::@class::A
                type: A<String Function([int?])>
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::new
                substitution: {T: String Function([int?])}
            argumentList: ArgumentList
              leftParenthesis: ( @67
              rightParenthesis: ) @68
            staticType: A<String Function([int?])>
    accessors
      synthetic static get v @-1
        returnType: A<String Function([int?])>
''');
  }

  test_genericFunction_asTypeArgument_parameters_requiredNamed() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

const v = A<String Function({required int a})>();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    topLevelVariables
      static const v @35
        type: A<String Function({required int a})>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            constructorName: ConstructorName
              type: NamedType
                name: A @39
                typeArguments: TypeArgumentList
                  leftBracket: < @40
                  arguments
                    GenericFunctionType
                      returnType: NamedType
                        name: String @41
                        element: dart:core::@class::String
                        type: String
                      functionKeyword: Function @48
                      parameters: FormalParameterList
                        leftParenthesis: ( @56
                        leftDelimiter: { @57
                        parameter: DefaultFormalParameter
                          parameter: SimpleFormalParameter
                            requiredKeyword: required @58
                            type: NamedType
                              name: int @67
                              element: dart:core::@class::int
                              type: int
                            name: a @71
                            declaredElement: a@71
                              type: int
                          declaredElement: a@71
                            type: int
                        rightDelimiter: } @72
                        rightParenthesis: ) @73
                      declaredElement: GenericFunctionTypeElement
                        parameters
                          a
                            kind: required named
                            type: int
                        returnType: String
                        type: String Function({required int a})
                      type: String Function({required int a})
                  rightBracket: > @74
                element: self::@class::A
                type: A<String Function({required int a})>
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::new
                substitution: {T: String Function({required int a})}
            argumentList: ArgumentList
              leftParenthesis: ( @75
              rightParenthesis: ) @76
            staticType: A<String Function({required int a})>
    accessors
      synthetic static get v @-1
        returnType: A<String Function({required int a})>
''');
  }

  test_genericFunction_asTypeArgument_parameters_requiredPositional() async {
    var library = await buildLibrary(r'''
class A<T> {
  const A();
}

const v = A<String Function(int a)>();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
    topLevelVariables
      static const v @35
        type: A<String Function(int)>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            constructorName: ConstructorName
              type: NamedType
                name: A @39
                typeArguments: TypeArgumentList
                  leftBracket: < @40
                  arguments
                    GenericFunctionType
                      returnType: NamedType
                        name: String @41
                        element: dart:core::@class::String
                        type: String
                      functionKeyword: Function @48
                      parameters: FormalParameterList
                        leftParenthesis: ( @56
                        parameter: SimpleFormalParameter
                          type: NamedType
                            name: int @57
                            element: dart:core::@class::int
                            type: int
                          name: a @61
                          declaredElement: a@61
                            type: int
                        rightParenthesis: ) @62
                      declaredElement: GenericFunctionTypeElement
                        parameters
                          a
                            kind: required positional
                            type: int
                        returnType: String
                        type: String Function(int)
                      type: String Function(int)
                  rightBracket: > @63
                element: self::@class::A
                type: A<String Function(int)>
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::new
                substitution: {T: String Function(int)}
            argumentList: ArgumentList
              leftParenthesis: ( @64
              rightParenthesis: ) @65
            staticType: A<String Function(int)>
    accessors
      synthetic static get v @-1
        returnType: A<String Function(int)>
''');
  }

  test_genericFunction_boundOf_typeParameter_ofMixin() async {
    var library = await buildLibrary(r'''
mixin B<X extends void Function()> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin B @6
        typeParameters
          covariant X @8
            bound: void Function()
            defaultType: void Function()
        superclassConstraints
          Object
''');
  }

  test_genericFunction_typeArgument_ofSuperclass_ofClassAlias() async {
    var library = await buildLibrary(r'''
class A<T> {}
mixin M {}
class B = A<void Function()> with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class alias B @31
        supertype: A<void Function()>
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: void Function()}
    mixins
      mixin M @20
        superclassConstraints
          Object
''');
  }

  test_genericFunction_typeParameter_asTypedefArgument() async {
    var library = await buildLibrary(r'''
typedef F1 = Function<V1>(F2<V1>);
typedef F2<V2> = V2 Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F1 @8
        aliasedType: dynamic Function<V1>(V1 Function())
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant V1 @22
          parameters
            requiredPositional @-1
              type: V1 Function()
                alias: self::@typeAlias::F2
                  typeArguments
                    V1
          returnType: dynamic
      F2 @43
        typeParameters
          covariant V2 @46
            defaultType: dynamic
        aliasedType: V2 Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: V2
''');
  }

  test_genericTypeAlias_enclosingElements() async {
    var library = await buildLibrary(r'''
typedef F<T> = void Function<U>(int a);
''');
    var unit = library.definingCompilationUnit;

    var F = unit.typeAliases[0];
    expect(F.name, 'F');

    var T = F.typeParameters[0];
    expect(T.name, 'T');
    expect(T.enclosingElement, same(F));

    var function = F.aliasedElement as GenericFunctionTypeElement;
    expect(function.enclosingElement, same(F));

    var U = function.typeParameters[0];
    expect(U.name, 'U');
    expect(U.enclosingElement, same(function));

    var a = function.parameters[0];
    expect(a.name, 'a');
    expect(a.enclosingElement, same(function));
  }

  test_genericTypeAlias_recursive() async {
    var library = await buildLibrary('''
typedef F<X extends F> = Function(F);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        typeParameters
          unrelated X @10
            bound: dynamic
            defaultType: dynamic
        aliasedType: dynamic Function(dynamic)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional @-1
              type: dynamic
          returnType: dynamic
''');
  }

  test_getter_async() async {
    var library = await buildLibrary(r'''
Future<int> get foo async => 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: Future<int>
    accessors
      static get foo @16 async
        returnType: Future<int>
''');
  }

  test_getter_asyncStar() async {
    var library = await buildLibrary(r'''
import 'dart:async';
Stream<int> get foo async* {}
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: Stream<int>
    accessors
      static get foo @37 async*
        returnType: Stream<int>
''');
  }

  test_getter_documented() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
get x => null;''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static x @-1
        type: dynamic
    accessors
      static get x @64
        documentationComment: /**\n * Docs\n */
        returnType: dynamic
''');
  }

  test_getter_external() async {
    var library = await buildLibrary('external int get x;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static x @-1
        type: int
    accessors
      static external get x @17
        returnType: int
''');
  }

  test_getter_inferred_type_nonStatic_implicit_return() async {
    var library = await buildLibrary(
        'class C extends D { get f => null; } abstract class D { int get f; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        fields
          synthetic f @-1
            type: int
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        accessors
          get f @24
            returnType: int
      abstract class D @52
        fields
          synthetic f @-1
            type: int
        constructors
          synthetic @-1
        accessors
          abstract get f @64
            returnType: int
''');
  }

  test_getter_syncStar() async {
    var library = await buildLibrary(r'''
Iterator<int> get foo sync* {}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: Iterator<int>
    accessors
      static get foo @18 sync*
        returnType: Iterator<int>
''');
  }

  test_getters() async {
    var library = await buildLibrary('int get x => null; get y => null;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static x @-1
        type: int
      synthetic static y @-1
        type: dynamic
    accessors
      static get x @8
        returnType: int
      static get y @23
        returnType: dynamic
''');
  }

  test_implicitCallTearoff() async {
    var library = await buildLibrary(r'''
class C {
  void call() {}
}

class D {
  const D(C c) : this.named(c);

  const D.named(void Function() f);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          call @17
            returnType: void
      class D @36
        constructors
          const @48
            parameters
              requiredPositional c @52
                type: C
            constantInitializers
              RedirectingConstructorInvocation
                thisKeyword: this @57
                period: . @61
                constructorName: SimpleIdentifier
                  token: named @62
                  staticElement: self::@class::D::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @67
                  arguments
                    ImplicitCallReference
                      expression: SimpleIdentifier
                        token: c @68
                        staticElement: self::@class::D::@constructor::new::@parameter::c
                        staticType: C
                      staticElement: self::@class::C::@method::call
                      staticType: void Function()
                  rightParenthesis: ) @69
                staticElement: self::@class::D::@constructor::named
            redirectedConstructor: self::@class::D::@constructor::named
          const named @83
            periodOffset: 82
            nameEnd: 88
            parameters
              requiredPositional f @105
                type: void Function()
''');
  }

  test_implicitConstructor_named_const() async {
    var library = await buildLibrary('''
class C {
  final Object x;
  const C.named(this.x);
}
const x = C.named(42);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          final x @25
            type: Object
            shouldUseTypeForInitializerInference: true
        constructors
          const named @38
            periodOffset: 37
            nameEnd: 43
            parameters
              requiredPositional final this.x @49
                type: Object
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: Object
    topLevelVariables
      static const x @61
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            constructorName: ConstructorName
              type: NamedType
                name: C @65
                element: self::@class::C
                type: C
              period: . @66
              name: SimpleIdentifier
                token: named @67
                staticElement: self::@class::C::@constructor::named
                staticType: null
              staticElement: self::@class::C::@constructor::named
            argumentList: ArgumentList
              leftParenthesis: ( @72
              arguments
                IntegerLiteral
                  literal: 42 @73
                  staticType: int
              rightParenthesis: ) @75
            staticType: C
    accessors
      synthetic static get x @-1
        returnType: C
''');
  }

  test_implicitTopLevelVariable_getterFirst() async {
    var library =
        await buildLibrary('int get x => 0; void set x(int value) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static x @-1
        type: int
    accessors
      static get x @8
        returnType: int
      static set x @25
        parameters
          requiredPositional value @31
            type: int
        returnType: void
''');
  }

  test_implicitTopLevelVariable_setterFirst() async {
    var library =
        await buildLibrary('void set x(int value) {} int get x => 0;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static x @-1
        type: int
    accessors
      static set x @9
        parameters
          requiredPositional value @15
            type: int
        returnType: void
      static get x @33
        returnType: int
''');
  }

  test_import_configurations_useDefault() async {
    declaredVariables = {
      'dart.library.io': 'false',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
import 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';

class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart
  definingUnit
    classes
      class B @104
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo.dart');
  }

  test_import_configurations_useFirst() async {
    declaredVariables = {
      'dart.library.io': 'true',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
import 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';

class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo_io.dart
  definingUnit
    classes
      class B @104
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_io.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_io.dart');
  }

  test_import_configurations_useFirst_eqTrue() async {
    declaredVariables = {
      'dart.library.io': 'true',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
import 'foo.dart'
  if (dart.library.io == 'true') 'foo_io.dart'
  if (dart.library.html == 'true') 'foo_html.dart';

class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo_io.dart
  definingUnit
    classes
      class B @124
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_io.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_io.dart');
  }

  test_import_configurations_useSecond() async {
    declaredVariables = {
      'dart.library.io': 'false',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
import 'foo.dart'
  if (dart.library.io) 'foo_io.dart'
  if (dart.library.html) 'foo_html.dart';

class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo_html.dart
  definingUnit
    classes
      class B @104
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_html.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_html.dart');
  }

  test_import_configurations_useSecond_eqTrue() async {
    declaredVariables = {
      'dart.library.io': 'false',
      'dart.library.html': 'true',
    };
    addSource('$testPackageLibPath/foo.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_io.dart', 'class A {}');
    addSource('$testPackageLibPath/foo_html.dart', 'class A {}');
    var library = await buildLibrary(r'''
import 'foo.dart'
  if (dart.library.io == 'true') 'foo_io.dart'
  if (dart.library.html == 'true') 'foo_html.dart';

class B extends A {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo_html.dart
  definingUnit
    classes
      class B @124
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/foo_html.dart::@class::A::@constructor::new
''');
    var typeA = library.definingCompilationUnit.getClass('B')!.supertype!;
    expect(typeA.element.source.shortName, 'foo_html.dart');
  }

  test_import_dartCore_explicit() async {
    var library = await buildLibrary('''
import 'dart:core';
import 'dart:math';
''');
    checkElementText(library, r'''
library
  imports
    dart:core
    dart:math
  definingUnit
''');
  }

  test_import_dartCore_implicit() async {
    var library = await buildLibrary('''
import 'dart:math';
''');
    checkElementText(library, r'''
library
  imports
    dart:math
  definingUnit
''');
  }

  test_import_deferred() async {
    addSource('$testPackageLibPath/a.dart', 'f() {}');
    var library = await buildLibrary('''
import 'a.dart' deferred as p;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart deferred as p @28
  definingUnit
''');
  }

  test_import_export() async {
    var library = await buildLibrary('''
import 'dart:async' as i1;
export 'dart:math';
import 'dart:async' as i2;
export 'dart:math';
import 'dart:async' as i3;
export 'dart:math';
''');
    checkElementText(library, r'''
library
  imports
    dart:async as i1 @23
    dart:async as i2 @70
    dart:async as i3 @117
  exports
    dart:math
    dart:math
    dart:math
  definingUnit
''');
  }

  test_import_hide() async {
    var library = await buildLibrary('''
import 'dart:async' hide Stream, Completer; Future f;
''');
    checkElementText(library, r'''
library
  imports
    dart:async
      combinators
        hide: Stream, Completer
  definingUnit
    topLevelVariables
      static f @51
        type: Future<dynamic>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: Future<dynamic>
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: Future<dynamic>
        returnType: void
''');
  }

  test_import_invalidUri_metadata() async {
    var library = await buildLibrary('''
@foo
import 'ht:';
''');
    checkElementText(library, r'''
library
  metadata
    Annotation
      atSign: @ @0
      name: SimpleIdentifier
        token: foo @1
        staticElement: <null>
        staticType: null
      element: <null>
  imports
    relativeUri 'ht:'
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: foo @1
            staticElement: <null>
            staticType: null
          element: <null>
  definingUnit
''');
  }

  test_import_multiple_combinators() async {
    var library = await buildLibrary('''
import "dart:async" hide Stream show Future;
Future f;
''');
    checkElementText(library, r'''
library
  imports
    dart:async
      combinators
        hide: Stream
        show: Future
  definingUnit
    topLevelVariables
      static f @52
        type: Future<dynamic>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: Future<dynamic>
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: Future<dynamic>
        returnType: void
''');
  }

  test_import_prefixed() async {
    addSource('$testPackageLibPath/a.dart', 'library a; class C {}');
    var library = await buildLibrary('import "a.dart" as a; a.C c;');

    final prefixElement = library.libraryImports[0].prefix!.element;
    expect(prefixElement.nameOffset, 19);

    checkElementText(library, r'''
library
  imports
    package:test/a.dart as a @19
  definingUnit
    topLevelVariables
      static c @26
        type: C
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C
        returnType: void
''');
  }

  test_import_self() async {
    var library = await buildLibrary('''
import 'test.dart' as p;
class C {}
class D extends p.C {} // Prevent "unused import" warning
''');
    expect(library.libraryImports, hasLength(2));
    expect(
        library.libraryImports[0].importedLibrary!.location, library.location);
    expect(library.libraryImports[1].importedLibrary!.isDartCore, true);
    checkElementText(library, r'''
library
  imports
    package:test/test.dart as p @22
  definingUnit
    classes
      class C @31
        constructors
          synthetic @-1
      class D @42
        supertype: C
        constructors
          synthetic @-1
            superConstructor: self::@class::C::@constructor::new
''');
  }

  test_import_show() async {
    var library = await buildLibrary('''
import "dart:async" show Future, Stream;
Future f;
Stream s;
''');
    checkElementText(library, r'''
library
  imports
    dart:async
      combinators
        show: Future, Stream
  definingUnit
    topLevelVariables
      static f @48
        type: Future<dynamic>
        shouldUseTypeForInitializerInference: true
      static s @58
        type: Stream<dynamic>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: Future<dynamic>
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: Future<dynamic>
        returnType: void
      synthetic static get s @-1
        returnType: Stream<dynamic>
      synthetic static set s @-1
        parameters
          requiredPositional _s @-1
            type: Stream<dynamic>
        returnType: void
''');
  }

  test_import_show_offsetEnd() async {
    var library = await buildLibrary('''
import "dart:math" show e, pi;
''');
    var import = library.libraryImports[0];
    var combinator = import.combinators[0] as ShowElementCombinator;
    expect(combinator.offset, 19);
    expect(combinator.end, 29);
  }

  test_import_uri() async {
    var library = await buildLibrary('''
import 'foo.dart';
''');

    final uri = library.libraryImports[0].uri as DirectiveUriWithLibrary;
    expect(uri.relativeUriString, 'foo.dart');
  }

  test_imports() async {
    addSource('$testPackageLibPath/a.dart', 'library a; class C {}');
    addSource('$testPackageLibPath/b.dart', 'library b; class D {}');
    var library =
        await buildLibrary('import "a.dart"; import "b.dart"; C c; D d;');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
    package:test/b.dart
  definingUnit
    topLevelVariables
      static c @36
        type: C
        shouldUseTypeForInitializerInference: true
      static d @41
        type: D
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C
        returnType: void
      synthetic static get d @-1
        returnType: D
      synthetic static set d @-1
        parameters
          requiredPositional _d @-1
            type: D
        returnType: void
''');
  }

  test_infer_generic_typedef_complex() async {
    var library = await buildLibrary('''
typedef F<T> = D<T,U> Function<U>();
class C<V> {
  const C(F<V> f);
}
class D<T,U> {}
D<int,U> f<U>() => null;
const x = const C(f);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @43
        typeParameters
          covariant V @45
            defaultType: dynamic
        constructors
          const @58
            parameters
              requiredPositional f @65
                type: D<V, U> Function<U>()
                  alias: self::@typeAlias::F
                    typeArguments
                      V
      class D @77
        typeParameters
          covariant T @79
            defaultType: dynamic
          covariant U @81
            defaultType: dynamic
        constructors
          synthetic @-1
    typeAliases
      F @8
        typeParameters
          covariant T @10
            defaultType: dynamic
        aliasedType: D<T, U> Function<U>()
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant U @31
          returnType: D<T, U>
    topLevelVariables
      static const x @118
        type: C<int>
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @122
            constructorName: ConstructorName
              type: NamedType
                name: C @128
                element: self::@class::C
                type: C<int>
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::new
                substitution: {V: int}
            argumentList: ArgumentList
              leftParenthesis: ( @129
              arguments
                SimpleIdentifier
                  token: f @130
                  staticElement: self::@function::f
                  staticType: D<int, U> Function<U>()
              rightParenthesis: ) @131
            staticType: C<int>
    accessors
      synthetic static get x @-1
        returnType: C<int>
    functions
      f @96
        typeParameters
          covariant U @98
            defaultType: dynamic
        returnType: D<int, U>
''');
  }

  test_infer_generic_typedef_simple() async {
    var library = await buildLibrary('''
typedef F = D<T> Function<T>();
class C {
  const C(F f);
}
class D<T> {}
D<T> f<T>() => null;
const x = const C(f);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @38
        constructors
          const @50
            parameters
              requiredPositional f @54
                type: D<T> Function<T>()
                  alias: self::@typeAlias::F
      class D @66
        typeParameters
          covariant T @68
            defaultType: dynamic
        constructors
          synthetic @-1
    typeAliases
      F @8
        aliasedType: D<T> Function<T>()
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant T @26
          returnType: D<T>
    topLevelVariables
      static const x @101
        type: C
        shouldUseTypeForInitializerInference: false
        constantInitializer
          InstanceCreationExpression
            keyword: const @105
            constructorName: ConstructorName
              type: NamedType
                name: C @111
                element: self::@class::C
                type: C
              staticElement: self::@class::C::@constructor::new
            argumentList: ArgumentList
              leftParenthesis: ( @112
              arguments
                SimpleIdentifier
                  token: f @113
                  staticElement: self::@function::f
                  staticType: D<T> Function<T>()
              rightParenthesis: ) @114
            staticType: C
    accessors
      synthetic static get x @-1
        returnType: C
    functions
      f @79
        typeParameters
          covariant T @81
            defaultType: dynamic
        returnType: D<T>
''');
  }

  test_infer_instanceCreation_fromArguments() async {
    var library = await buildLibrary('''
class A {}

class B extends A {}

class S<T extends A> {
  S(T _);
}

var s = new S(new B());
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @18
        supertype: A
        constructors
          synthetic @-1
            superConstructor: self::@class::A::@constructor::new
      class S @40
        typeParameters
          covariant T @42
            bound: A
            defaultType: A
        constructors
          @59
            parameters
              requiredPositional _ @63
                type: T
    topLevelVariables
      static s @74
        type: S<B>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get s @-1
        returnType: S<B>
      synthetic static set s @-1
        parameters
          requiredPositional _s @-1
            type: S<B>
        returnType: void
''');
  }

  test_infer_property_set() async {
    var library = await buildLibrary('''
class A {
  B b;
}
class B {
  C get c => null;
  void set c(C value) {}
}
class C {}
class D extends C {}
var a = new A();
var x = a.b.c ??= new D();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          b @14
            type: B
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get b @-1
            returnType: B
          synthetic set b @-1
            parameters
              requiredPositional _b @-1
                type: B
            returnType: void
      class B @25
        fields
          synthetic c @-1
            type: C
        constructors
          synthetic @-1
        accessors
          get c @37
            returnType: C
          set c @59
            parameters
              requiredPositional value @63
                type: C
            returnType: void
      class C @81
        constructors
          synthetic @-1
      class D @92
        supertype: C
        constructors
          synthetic @-1
            superConstructor: self::@class::C::@constructor::new
    topLevelVariables
      static a @111
        type: A
        shouldUseTypeForInitializerInference: false
      static x @128
        type: C
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get a @-1
        returnType: A
      synthetic static set a @-1
        parameters
          requiredPositional _a @-1
            type: A
        returnType: void
      synthetic static get x @-1
        returnType: C
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: C
        returnType: void
''');
  }

  test_inference_issue_32394() async {
    // Test the type inference involved in dartbug.com/32394
    var library = await buildLibrary('''
var x = y.map((a) => a.toString());
var y = [3];
var z = x.toList();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static x @4
        type: Iterable<String>
        shouldUseTypeForInitializerInference: false
      static y @40
        type: List<int>
        shouldUseTypeForInitializerInference: false
      static z @53
        type: List<String>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: Iterable<String>
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: Iterable<String>
        returnType: void
      synthetic static get y @-1
        returnType: List<int>
      synthetic static set y @-1
        parameters
          requiredPositional _y @-1
            type: List<int>
        returnType: void
      synthetic static get z @-1
        returnType: List<String>
      synthetic static set z @-1
        parameters
          requiredPositional _z @-1
            type: List<String>
        returnType: void
''');
  }

  test_inference_map() async {
    var library = await buildLibrary('''
class C {
  int p;
}
var x = <C>[];
var y = x.map((c) => c.p);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          p @16
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get p @-1
            returnType: int
          synthetic set p @-1
            parameters
              requiredPositional _p @-1
                type: int
            returnType: void
    topLevelVariables
      static x @25
        type: List<C>
        shouldUseTypeForInitializerInference: false
      static y @40
        type: Iterable<int>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: List<C>
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: List<C>
        returnType: void
      synthetic static get y @-1
        returnType: Iterable<int>
      synthetic static set y @-1
        parameters
          requiredPositional _y @-1
            type: Iterable<int>
        returnType: void
''');
  }

  test_inferred_function_type_for_variable_in_generic_function() async {
    // In the code below, `x` has an inferred type of `() => int`, with 2
    // (unused) type parameters from the enclosing top level function.
    var library = await buildLibrary('''
f<U, V>() {
  var x = () => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        typeParameters
          covariant U @2
            defaultType: dynamic
          covariant V @5
            defaultType: dynamic
        returnType: dynamic
''');
  }

  test_inferred_function_type_in_generic_class_constructor() async {
    // In the code below, `() => () => 0` has an inferred return type of
    // `() => int`, with 2 (unused) type parameters from the enclosing class.
    var library = await buildLibrary('''
class C<U, V> {
  final x;
  C() : x = (() => () => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant U @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        fields
          final x @24
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @29
        accessors
          synthetic get x @-1
            returnType: dynamic
''');
  }

  test_inferred_function_type_in_generic_class_getter() async {
    // In the code below, `() => () => 0` has an inferred return type of
    // `() => int`, with 2 (unused) type parameters from the enclosing class.
    var library = await buildLibrary('''
class C<U, V> {
  get x => () => () => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant U @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          get x @22
            returnType: dynamic
''');
  }

  test_inferred_function_type_in_generic_class_in_generic_method() async {
    // In the code below, `() => () => 0` has an inferred return type of
    // `() => int`, with 3 (unused) type parameters from the enclosing class
    // and method.
    var library = await buildLibrary('''
class C<T> {
  f<U, V>() {
    print(() => () => 0);
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          f @15
            typeParameters
              covariant U @17
                defaultType: dynamic
              covariant V @20
                defaultType: dynamic
            returnType: dynamic
''');
  }

  test_inferred_function_type_in_generic_class_setter() async {
    // In the code below, `() => () => 0` has an inferred return type of
    // `() => int`, with 2 (unused) type parameters from the enclosing class.
    var library = await buildLibrary('''
class C<U, V> {
  void set x(value) {
    print(() => () => 0);
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant U @8
            defaultType: dynamic
          covariant V @11
            defaultType: dynamic
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @27
            parameters
              requiredPositional value @29
                type: dynamic
            returnType: void
''');
  }

  test_inferred_function_type_in_generic_closure() async {
    // In the code below, `<U, V>() => () => 0` has an inferred return type of
    // `() => int`, with 3 (unused) type parameters.
    var library = await buildLibrary('''
f<T>() {
  print(/*<U, V>*/() => () => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        typeParameters
          covariant T @2
            defaultType: dynamic
        returnType: dynamic
''');
  }

  test_inferred_generic_function_type_in_generic_closure() async {
    // In the code below, `<U, V>() => <W, X, Y, Z>() => 0` has an inferred
    // return type of `() => int`, with 7 (unused) type parameters.
    var library = await buildLibrary('''
f<T>() {
  print(/*<U, V>*/() => /*<W, X, Y, Z>*/() => 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        typeParameters
          covariant T @2
            defaultType: dynamic
        returnType: dynamic
''');
  }

  test_inferred_type_could_not_infer() async {
    var library = await buildLibrary(r'''
class C<P extends num> {
  factory C(Iterable<P> p) => C._();
  C._();
}

var c = C([]);
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant P @8
            bound: num
            defaultType: num
        constructors
          factory @35
            parameters
              requiredPositional p @49
                type: Iterable<P>
          _ @66
            periodOffset: 65
            nameEnd: 67
    topLevelVariables
      static c @78
        type: C<dynamic>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get c @-1
        returnType: C<dynamic>
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<dynamic>
        returnType: void
''');
  }

  test_inferred_type_functionExpressionInvocation_oppositeOrder() async {
    var library = await buildLibrary('''
class A {
  static final foo = bar(1.2);
  static final bar = baz();

  static int Function(double) baz() => (throw 0);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          static final foo @25
            type: int
            shouldUseTypeForInitializerInference: false
          static final bar @56
            type: int Function(double)
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic static get foo @-1
            returnType: int
          synthetic static get bar @-1
            returnType: int Function(double)
        methods
          static baz @100
            returnType: int Function(double)
''');
  }

  test_inferred_type_inference_failure_on_function_invocation() async {
    writeTestPackageAnalysisOptionsFile(
      AnalysisOptionsFileConfig(
        strictInference: true,
      ),
    );
    var library = await buildLibrary(r'''
int m<T>() => 1;
var x = m();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static x @21
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: int
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int
        returnType: void
    functions
      m @4
        typeParameters
          covariant T @6
            defaultType: dynamic
        returnType: int
''');
  }

  test_inferred_type_inference_failure_on_generic_invocation() async {
    writeTestPackageAnalysisOptionsFile(
      AnalysisOptionsFileConfig(
        strictInference: true,
      ),
    );
    var library = await buildLibrary(r'''
int Function<T>()? m = <T>() => 1;
int Function<T>() n = <T>() => 2;
var x = (m ?? n)();
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static m @19
        type: int Function<T>()?
        shouldUseTypeForInitializerInference: true
      static n @53
        type: int Function<T>()
        shouldUseTypeForInitializerInference: true
      static x @73
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get m @-1
        returnType: int Function<T>()?
      synthetic static set m @-1
        parameters
          requiredPositional _m @-1
            type: int Function<T>()?
        returnType: void
      synthetic static get n @-1
        returnType: int Function<T>()
      synthetic static set n @-1
        parameters
          requiredPositional _n @-1
            type: int Function<T>()
        returnType: void
      synthetic static get x @-1
        returnType: int
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int
        returnType: void
''');
  }

  test_inferred_type_inference_failure_on_instance_creation() async {
    writeTestPackageAnalysisOptionsFile(
      AnalysisOptionsFileConfig(
        strictInference: true,
      ),
    );
    var library = await buildLibrary(r'''
import 'dart:collection';
var m = HashMap();
''');
    checkElementText(library, r'''
library
  imports
    dart:collection
  definingUnit
    topLevelVariables
      static m @30
        type: HashMap<dynamic, dynamic>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get m @-1
        returnType: HashMap<dynamic, dynamic>
      synthetic static set m @-1
        parameters
          requiredPositional _m @-1
            type: HashMap<dynamic, dynamic>
        returnType: void
''');
  }

  test_inferred_type_initializer_cycle() async {
    var library = await buildLibrary(r'''
var a = b + 1;
var b = c + 2;
var c = a + 3;
var d = 4;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static a @4
        typeInferenceError: dependencyCycle
          arguments: [a, b, c]
        type: dynamic
        shouldUseTypeForInitializerInference: false
      static b @19
        typeInferenceError: dependencyCycle
          arguments: [a, b, c]
        type: dynamic
        shouldUseTypeForInitializerInference: false
      static c @34
        typeInferenceError: dependencyCycle
          arguments: [a, b, c]
        type: dynamic
        shouldUseTypeForInitializerInference: false
      static d @49
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static set a @-1
        parameters
          requiredPositional _a @-1
            type: dynamic
        returnType: void
      synthetic static get b @-1
        returnType: dynamic
      synthetic static set b @-1
        parameters
          requiredPositional _b @-1
            type: dynamic
        returnType: void
      synthetic static get c @-1
        returnType: dynamic
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: dynamic
        returnType: void
      synthetic static get d @-1
        returnType: int
      synthetic static set d @-1
        parameters
          requiredPositional _d @-1
            type: int
        returnType: void
''');
  }

  test_inferred_type_is_typedef() async {
    var library = await buildLibrary('typedef int F(String s);'
        ' class C extends D { var v; }'
        ' abstract class D { F get v; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @31
        supertype: D
        fields
          v @49
            type: int Function(String)
              alias: self::@typeAlias::F
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        accessors
          synthetic get v @-1
            returnType: int Function(String)
              alias: self::@typeAlias::F
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: int Function(String)
                  alias: self::@typeAlias::F
            returnType: void
      abstract class D @69
        fields
          synthetic v @-1
            type: int Function(String)
              alias: self::@typeAlias::F
        constructors
          synthetic @-1
        accessors
          abstract get v @79
            returnType: int Function(String)
              alias: self::@typeAlias::F
    typeAliases
      functionTypeAliasBased F @12
        aliasedType: int Function(String)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional s @21
              type: String
          returnType: int
''');
  }

  test_inferred_type_nullability_class_ref_none() async {
    addSource('$testPackageLibPath/a.dart', 'int f() => 0;');
    var library = await buildLibrary('''
import 'a.dart';
var x = f();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static x @21
        type: int
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: int
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int
        returnType: void
''');
  }

  test_inferred_type_nullability_class_ref_question() async {
    addSource('$testPackageLibPath/a.dart', 'int? f() => 0;');
    var library = await buildLibrary('''
import 'a.dart';
var x = f();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static x @21
        type: int?
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: int?
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int?
        returnType: void
''');
  }

  test_inferred_type_nullability_function_type_none() async {
    addSource('$testPackageLibPath/a.dart', 'void Function() f() => () {};');
    var library = await buildLibrary('''
import 'a.dart';
var x = f();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static x @21
        type: void Function()
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: void Function()
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: void Function()
        returnType: void
''');
  }

  test_inferred_type_nullability_function_type_question() async {
    addSource('$testPackageLibPath/a.dart', 'void Function()? f() => () {};');
    var library = await buildLibrary('''
import 'a.dart';
var x = f();
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    topLevelVariables
      static x @21
        type: void Function()?
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get x @-1
        returnType: void Function()?
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: void Function()?
        returnType: void
''');
  }

  test_inferred_type_refers_to_bound_type_param() async {
    var library = await buildLibrary('''
class C<T> extends D<int, T> {
  var v;
}
abstract class D<U, V> {
  Map<V, U> get v;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        supertype: D<int, T>
        fields
          v @37
            type: Map<T, int>
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::D::@constructor::new
              substitution: {U: int, V: T}
        accessors
          synthetic get v @-1
            returnType: Map<T, int>
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: Map<T, int>
            returnType: void
      abstract class D @57
        typeParameters
          covariant U @59
            defaultType: dynamic
          covariant V @62
            defaultType: dynamic
        fields
          synthetic v @-1
            type: Map<V, U>
        constructors
          synthetic @-1
        accessors
          abstract get v @83
            returnType: Map<V, U>
''');
  }

  test_inferred_type_refers_to_function_typed_param_of_typedef() async {
    var library = await buildLibrary('''
typedef void F(int g(String s));
h(F f) => null;
var v = h((y) {});
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @13
        aliasedType: void Function(int Function(String))
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional g @19
              type: int Function(String)
              parameters
                requiredPositional s @28
                  type: String
          returnType: void
    topLevelVariables
      static v @53
        type: dynamic
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: dynamic
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: dynamic
        returnType: void
    functions
      h @33
        parameters
          requiredPositional f @37
            type: void Function(int Function(String))
              alias: self::@typeAlias::F
        returnType: dynamic
''');
  }

  test_inferred_type_refers_to_function_typed_parameter_type_generic_class() async {
    var library = await buildLibrary('''
class C<T, U> extends D<U, int> {
  void f(int x, g) {}
}
abstract class D<V, W> {
  void f(int x, W g(V s));
}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        supertype: D<U, int>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::D::@constructor::new
              substitution: {V: U, W: int}
        methods
          f @41
            parameters
              requiredPositional x @47
                type: int
              requiredPositional g @50
                type: int Function(U)
            returnType: void
      abstract class D @73
        typeParameters
          covariant V @75
            defaultType: dynamic
          covariant W @78
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          abstract f @90
            parameters
              requiredPositional x @96
                type: int
              requiredPositional g @101
                type: W Function(V)
                parameters
                  requiredPositional s @105
                    type: V
            returnType: void
''');
  }

  test_inferred_type_refers_to_function_typed_parameter_type_other_lib() async {
    addSource('$testPackageLibPath/a.dart', '''
import 'b.dart';
abstract class D extends E {}
''');
    addSource('$testPackageLibPath/b.dart', '''
abstract class E {
  void f(int x, int g(String s));
}
''');
    var library = await buildLibrary('''
import 'a.dart';
class C extends D {
  void f(int x, g) {}
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class C @23
        supertype: D
        constructors
          synthetic @-1
            superConstructor: package:test/a.dart::@class::D::@constructor::new
        methods
          f @44
            parameters
              requiredPositional x @50
                type: int
              requiredPositional g @53
                type: int Function(String)
            returnType: void
''');
  }

  test_inferred_type_refers_to_method_function_typed_parameter_type() async {
    var library = await buildLibrary('class C extends D { void f(int x, g) {} }'
        ' abstract class D { void f(int x, int g(String s)); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        methods
          f @25
            parameters
              requiredPositional x @31
                type: int
              requiredPositional g @34
                type: int Function(String)
            returnType: void
      abstract class D @57
        constructors
          synthetic @-1
        methods
          abstract f @66
            parameters
              requiredPositional x @72
                type: int
              requiredPositional g @79
                type: int Function(String)
                parameters
                  requiredPositional s @88
                    type: String
            returnType: void
''');
  }

  test_inferred_type_refers_to_nested_function_typed_param() async {
    var library = await buildLibrary('''
f(void g(int x, void h())) => null;
var v = f((x, y) {});
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @40
        type: dynamic
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: dynamic
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: dynamic
        returnType: void
    functions
      f @0
        parameters
          requiredPositional g @7
            type: void Function(int, void Function())
            parameters
              requiredPositional x @13
                type: int
              requiredPositional h @21
                type: void Function()
        returnType: dynamic
''');
  }

  test_inferred_type_refers_to_nested_function_typed_param_named() async {
    var library = await buildLibrary('''
f({void g(int x, void h())}) => null;
var v = f(g: (x, y) {});
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @42
        type: dynamic
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: dynamic
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: dynamic
        returnType: void
    functions
      f @0
        parameters
          optionalNamed default g @8
            type: void Function(int, void Function())
            parameters
              requiredPositional x @14
                type: int
              requiredPositional h @22
                type: void Function()
        returnType: dynamic
''');
  }

  test_inferred_type_refers_to_setter_function_typed_parameter_type() async {
    var library = await buildLibrary('class C extends D { void set f(g) {} }'
        ' abstract class D { void set f(int g(String s)); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        supertype: D
        fields
          synthetic f @-1
            type: int Function(String)
        constructors
          synthetic @-1
            superConstructor: self::@class::D::@constructor::new
        accessors
          set f @29
            parameters
              requiredPositional g @31
                type: int Function(String)
            returnType: void
      abstract class D @54
        fields
          synthetic f @-1
            type: int Function(String)
        constructors
          synthetic @-1
        accessors
          abstract set f @67
            parameters
              requiredPositional g @73
                type: int Function(String)
                parameters
                  requiredPositional s @82
                    type: String
            returnType: void
''');
  }

  test_inferredType_definedInSdkLibraryPart() async {
    addSource('$testPackageLibPath/a.dart', r'''
import 'dart:async';
class A {
  m(Stream p) {}
}
''');
    var library = await buildLibrary(r'''
import 'a.dart';
class B extends A {
  m(p) {}
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class B @23
        supertype: A
        constructors
          synthetic @-1
            superConstructor: package:test/a.dart::@class::A::@constructor::new
        methods
          m @39
            parameters
              requiredPositional p @41
                type: Stream<dynamic>
            returnType: dynamic
''');
    ClassElement b = library.definingCompilationUnit.classes[0];
    ParameterElement p = b.methods[0].parameters[0];
    // This test should verify that we correctly record inferred types,
    // when the type is defined in a part of an SDK library. So, test that
    // the type is actually in a part.
    final streamElement = (p.type as InterfaceType).element;
    expect(streamElement.source, isNot(streamElement.library.source));
  }

  test_inferredType_implicitCreation() async {
    var library = await buildLibrary(r'''
class A {
  A();
  A.named();
}
var a1 = A();
var a2 = A.named();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
          named @21
            periodOffset: 20
            nameEnd: 26
    topLevelVariables
      static a1 @36
        type: A
        shouldUseTypeForInitializerInference: false
      static a2 @50
        type: A
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get a1 @-1
        returnType: A
      synthetic static set a1 @-1
        parameters
          requiredPositional _a1 @-1
            type: A
        returnType: void
      synthetic static get a2 @-1
        returnType: A
      synthetic static set a2 @-1
        parameters
          requiredPositional _a2 @-1
            type: A
        returnType: void
''');
  }

  test_inferredType_implicitCreation_prefixed() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A {
  A();
  A.named();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
var a1 = foo.A();
var a2 = foo.A.named();
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    topLevelVariables
      static a1 @30
        type: A
        shouldUseTypeForInitializerInference: false
      static a2 @48
        type: A
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get a1 @-1
        returnType: A
      synthetic static set a1 @-1
        parameters
          requiredPositional _a1 @-1
            type: A
        returnType: void
      synthetic static get a2 @-1
        returnType: A
      synthetic static set a2 @-1
        parameters
          requiredPositional _a2 @-1
            type: A
        returnType: void
''');
  }

  test_inferredType_usesSyntheticFunctionType_functionTypedParam() async {
    // AnalysisContext does not set the enclosing element for the synthetic
    // FunctionElement created for the [f, g] type argument.
    var library = await buildLibrary('''
int f(int x(String y)) => null;
String g(int x(String y)) => null;
var v = [f, g];
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @71
        type: List<Object Function(int Function(String))>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: List<Object Function(int Function(String))>
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: List<Object Function(int Function(String))>
        returnType: void
    functions
      f @4
        parameters
          requiredPositional x @10
            type: int Function(String)
            parameters
              requiredPositional y @19
                type: String
        returnType: int
      g @39
        parameters
          requiredPositional x @45
            type: int Function(String)
            parameters
              requiredPositional y @54
                type: String
        returnType: String
''');
  }

  test_inheritance_errors() async {
    var library = await buildLibrary('''
abstract class A {
  int m();
}

abstract class B {
  String m();
}

abstract class C implements A, B {}

abstract class D extends C {
  var f;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        constructors
          synthetic @-1
        methods
          abstract m @25
            returnType: int
      abstract class B @48
        constructors
          synthetic @-1
        methods
          abstract m @61
            returnType: String
      abstract class C @84
        interfaces
          A
          B
        constructors
          synthetic @-1
      abstract class D @121
        supertype: C
        fields
          f @141
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
            superConstructor: self::@class::C::@constructor::new
        accessors
          synthetic get f @-1
            returnType: dynamic
          synthetic set f @-1
            parameters
              requiredPositional _f @-1
                type: dynamic
            returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure() async {
    var library = await buildLibrary('var v = () => 0;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @4
        type: int Function()
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: int Function()
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: int Function()
        returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_await_dynamic() async {
    var library = await buildLibrary('var v = (f) async => await f;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static v @4
        type: Future<dynamic> Function(dynamic)
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: Future<dynamic> Function(dynamic)
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: Future<dynamic> Function(dynamic)
        returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_await_future3_int() async {
    var library = await buildLibrary(r'''
import 'dart:async';
var v = (Future<Future<Future<int>>> f) async => await f;
''');
    // The analyzer type system over-flattens - see dartbug.com/31887
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static v @25
        type: Future<int> Function(Future<Future<Future<int>>>)
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: Future<int> Function(Future<Future<Future<int>>>)
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: Future<int> Function(Future<Future<Future<int>>>)
        returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_await_future_int() async {
    var library = await buildLibrary(r'''
import 'dart:async';
var v = (Future<int> f) async => await f;
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static v @25
        type: Future<int> Function(Future<int>)
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: Future<int> Function(Future<int>)
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: Future<int> Function(Future<int>)
        returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_await_future_noArg() async {
    var library = await buildLibrary(r'''
import 'dart:async';
var v = (Future f) async => await f;
''');
    checkElementText(library, r'''
library
  imports
    dart:async
  definingUnit
    topLevelVariables
      static v @25
        type: Future<dynamic> Function(Future<dynamic>)
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get v @-1
        returnType: Future<dynamic> Function(Future<dynamic>)
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: Future<dynamic> Function(Future<dynamic>)
        returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_field() async {
    var library = await buildLibrary('''
class C {
  var v = () => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          v @16
            type: int Function()
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get v @-1
            returnType: int Function()
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: int Function()
            returnType: void
''');
  }

  test_initializer_executable_with_return_type_from_closure_local() async {
    var library = await buildLibrary('''
void f() {
  int u = 0;
  var v = () => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        returnType: void
''');
  }

  test_instanceInference_operator_equal_legacy_from_legacy() async {
    addSource('$testPackageLibPath/legacy.dart', r'''
// @dart = 2.9
class LegacyDefault {
  bool operator==(other) => false;
}
class LegacyObject {
  bool operator==(Object other) => false;
}
class LegacyInt {
  bool operator==(int other) => false;
}
''');
    var library = await buildLibrary(r'''
// @dart = 2.9
import 'legacy.dart';
class X1 extends LegacyDefault  {
  bool operator==(other) => false;
}
class X2 extends LegacyObject {
  bool operator==(other) => false;
}
class X3 extends LegacyInt {
  bool operator==(other) => false;
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/legacy.dart
  definingUnit
    classes
      class X1 @43
        supertype: LegacyDefault*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyDefault::@constructor::new
        methods
          == @86
            parameters
              requiredPositional other @89
                type: dynamic
            returnType: bool*
      class X2 @114
        supertype: LegacyObject*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyObject::@constructor::new
        methods
          == @155
            parameters
              requiredPositional other @158
                type: Object*
            returnType: bool*
      class X3 @183
        supertype: LegacyInt*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyInt::@constructor::new
        methods
          == @221
            parameters
              requiredPositional other @224
                type: int*
            returnType: bool*
''');
  }

  test_instanceInference_operator_equal_legacy_from_legacy_nullSafe() async {
    addSource('$testPackageLibPath/legacy.dart', r'''
// @dart = 2.7
class LegacyDefault {
  bool operator==(other) => false;
}
class LegacyObject {
  bool operator==(Object other) => false;
}
class LegacyInt {
  bool operator==(int other) => false;
}
''');
    addSource('$testPackageLibPath/nullSafe.dart', r'''
class NullSafeDefault {
  bool operator==(other) => false;
}
class NullSafeObject {
  bool operator==(Object other) => false;
}
class NullSafeInt {
  bool operator==(int other) => false;
}
''');
    var library = await buildLibrary(r'''
// @dart = 2.7
import 'legacy.dart';
import 'nullSafe.dart';
class X1 extends LegacyDefault implements NullSafeDefault {
  bool operator==(other) => false;
}
class X2 extends LegacyObject implements NullSafeObject {
  bool operator==(other) => false;
}
class X3 extends LegacyInt implements NullSafeInt {
  bool operator==(other) => false;
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/legacy.dart
    package:test/nullSafe.dart
  definingUnit
    classes
      class X1 @67
        supertype: LegacyDefault*
        interfaces
          NullSafeDefault*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyDefault::@constructor::new
        methods
          == @136
            parameters
              requiredPositional other @139
                type: dynamic
            returnType: bool*
      class X2 @164
        supertype: LegacyObject*
        interfaces
          NullSafeObject*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyObject::@constructor::new
        methods
          == @231
            parameters
              requiredPositional other @234
                type: Object*
            returnType: bool*
      class X3 @259
        supertype: LegacyInt*
        interfaces
          NullSafeInt*
        constructors
          synthetic @-1
            superConstructor: package:test/legacy.dart::@class::LegacyInt::@constructor::new
        methods
          == @320
            parameters
              requiredPositional other @323
                type: int*
            returnType: bool*
''');
  }

  test_instanceInference_operator_equal_nullSafe_from_nullSafe() async {
    addSource('$testPackageLibPath/nullSafe.dart', r'''
class NullSafeDefault {
  bool operator==(other) => false;
}
class NullSafeObject {
  bool operator==(Object other) => false;
}
class NullSafeInt {
  bool operator==(int other) => false;
}
''');
    var library = await buildLibrary(r'''
import 'nullSafe.dart';
class X1 extends NullSafeDefault {
  bool operator==(other) => false;
}
class X2 extends NullSafeObject {
  bool operator==(other) => false;
}
class X3 extends NullSafeInt {
  bool operator==(other) => false;
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/nullSafe.dart
  definingUnit
    classes
      class X1 @30
        supertype: NullSafeDefault
        constructors
          synthetic @-1
            superConstructor: package:test/nullSafe.dart::@class::NullSafeDefault::@constructor::new
        methods
          == @74
            parameters
              requiredPositional other @77
                type: Object
            returnType: bool
      class X2 @102
        supertype: NullSafeObject
        constructors
          synthetic @-1
            superConstructor: package:test/nullSafe.dart::@class::NullSafeObject::@constructor::new
        methods
          == @145
            parameters
              requiredPositional other @148
                type: Object
            returnType: bool
      class X3 @173
        supertype: NullSafeInt
        constructors
          synthetic @-1
            superConstructor: package:test/nullSafe.dart::@class::NullSafeInt::@constructor::new
        methods
          == @213
            parameters
              requiredPositional other @216
                type: int
            returnType: bool
''');
  }

  test_instantiateToBounds_boundRefersToEarlierTypeArgument() async {
    var library = await buildLibrary('''
class C<S extends num, T extends C<S, T>> {}
C c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant S @8
            bound: num
            defaultType: num
          covariant T @23
            bound: C<S, T>
            defaultType: C<num, dynamic>
        constructors
          synthetic @-1
    topLevelVariables
      static c @47
        type: C<num, C<num, dynamic>>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C<num, C<num, dynamic>>
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<num, C<num, dynamic>>
        returnType: void
''');
  }

  test_instantiateToBounds_boundRefersToItself() async {
    var library = await buildLibrary('''
class C<T extends C<T>> {}
C c;
var c2 = new C();
class B {
  var c3 = new C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: C<T>
            defaultType: C<dynamic>
        constructors
          synthetic @-1
      class B @56
        fields
          c3 @66
            type: C<C<Object?>>
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get c3 @-1
            returnType: C<C<Object?>>
          synthetic set c3 @-1
            parameters
              requiredPositional _c3 @-1
                type: C<C<Object?>>
            returnType: void
    topLevelVariables
      static c @29
        type: C<C<dynamic>>
        shouldUseTypeForInitializerInference: true
      static c2 @36
        type: C<C<Object?>>
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get c @-1
        returnType: C<C<dynamic>>
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<C<dynamic>>
        returnType: void
      synthetic static get c2 @-1
        returnType: C<C<Object?>>
      synthetic static set c2 @-1
        parameters
          requiredPositional _c2 @-1
            type: C<C<Object?>>
        returnType: void
''');
  }

  test_instantiateToBounds_boundRefersToItself_legacy() async {
    var library = await buildLibrary('''
// @dart = 2.9
class C<T extends C<T>> {}
C c;
var c2 = new C();
class B {
  var c3 = new C();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @21
        typeParameters
          covariant T @23
            bound: C<T*>*
            defaultType: C<dynamic>*
        constructors
          synthetic @-1
      class B @71
        fields
          c3 @81
            type: C<C<dynamic>*>*
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get c3 @-1
            returnType: C<C<dynamic>*>*
          synthetic set c3 @-1
            parameters
              requiredPositional _c3 @-1
                type: C<C<dynamic>*>*
            returnType: void
    topLevelVariables
      static c @44
        type: C<C<dynamic>*>*
        shouldUseTypeForInitializerInference: true
      static c2 @51
        type: C<C<dynamic>*>*
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get c @-1
        returnType: C<C<dynamic>*>*
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<C<dynamic>*>*
        returnType: void
      synthetic static get c2 @-1
        returnType: C<C<dynamic>*>*
      synthetic static set c2 @-1
        parameters
          requiredPositional _c2 @-1
            type: C<C<dynamic>*>*
        returnType: void
''');
  }

  test_instantiateToBounds_boundRefersToLaterTypeArgument() async {
    var library = await buildLibrary('''
class C<T extends C<T, U>, U extends num> {}
C c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      notSimplyBounded class C @6
        typeParameters
          covariant T @8
            bound: C<T, U>
            defaultType: C<dynamic, num>
          covariant U @27
            bound: num
            defaultType: num
        constructors
          synthetic @-1
    topLevelVariables
      static c @47
        type: C<C<dynamic, num>, num>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C<C<dynamic, num>, num>
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<C<dynamic, num>, num>
        returnType: void
''');
  }

  test_instantiateToBounds_functionTypeAlias_reexported() async {
    addSource('$testPackageLibPath/a.dart', r'''
class O {}
typedef T F<T extends O>(T p);
''');
    addSource('$testPackageLibPath/b.dart', r'''
export 'a.dart' show F;
''');
    var library = await buildLibrary('''
import 'b.dart';
class C {
  F f() => null;
}
''');
    checkElementText(library, r'''
library
  imports
    package:test/b.dart
  definingUnit
    classes
      class C @23
        constructors
          synthetic @-1
        methods
          f @31
            returnType: O Function(O)
              alias: package:test/a.dart::@typeAlias::F
                typeArguments
                  O
''');
  }

  test_instantiateToBounds_functionTypeAlias_simple() async {
    var library = await buildLibrary('''
typedef F<T extends num>(T p);
F f;
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @8
        typeParameters
          contravariant T @10
            bound: num
            defaultType: num
        aliasedType: dynamic Function(T)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional p @27
              type: T
          returnType: dynamic
    topLevelVariables
      static f @33
        type: dynamic Function(num)
          alias: self::@typeAlias::F
            typeArguments
              num
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: dynamic Function(num)
          alias: self::@typeAlias::F
            typeArguments
              num
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: dynamic Function(num)
              alias: self::@typeAlias::F
                typeArguments
                  num
        returnType: void
''');
  }

  test_instantiateToBounds_genericFunctionAsBound() async {
    var library = await buildLibrary('''
class A<T> {}
class B<T extends int Function(), U extends A<T>> {}
B b;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      notSimplyBounded class B @20
        typeParameters
          covariant T @22
            bound: int Function()
            defaultType: int Function()
          covariant U @48
            bound: A<T>
            defaultType: A<int Function()>
        constructors
          synthetic @-1
    topLevelVariables
      static b @69
        type: B<int Function(), A<int Function()>>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get b @-1
        returnType: B<int Function(), A<int Function()>>
      synthetic static set b @-1
        parameters
          requiredPositional _b @-1
            type: B<int Function(), A<int Function()>>
        returnType: void
''');
  }

  test_instantiateToBounds_genericTypeAlias_simple() async {
    var library = await buildLibrary('''
typedef F<T extends num> = S Function<S>(T p);
F f;
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @8
        typeParameters
          contravariant T @10
            bound: num
            defaultType: num
        aliasedType: S Function<S>(T)
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant S @38
          parameters
            requiredPositional p @43
              type: T
          returnType: S
    topLevelVariables
      static f @49
        type: S Function<S>(num)
          alias: self::@typeAlias::F
            typeArguments
              num
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get f @-1
        returnType: S Function<S>(num)
          alias: self::@typeAlias::F
            typeArguments
              num
      synthetic static set f @-1
        parameters
          requiredPositional _f @-1
            type: S Function<S>(num)
              alias: self::@typeAlias::F
                typeArguments
                  num
        returnType: void
''');
  }

  test_instantiateToBounds_issue38498() async {
    var library = await buildLibrary('''
class A<R extends B> {
  final values = <B>[];
}
class B<T extends num> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant R @8
            bound: B<num>
            defaultType: B<num>
        fields
          final values @31
            type: List<B<num>>
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic get values @-1
            returnType: List<B<num>>
      class B @55
        typeParameters
          covariant T @57
            bound: num
            defaultType: num
        constructors
          synthetic @-1
''');
  }

  test_instantiateToBounds_simple() async {
    var library = await buildLibrary('''
class C<T extends num> {}
C c;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            bound: num
            defaultType: num
        constructors
          synthetic @-1
    topLevelVariables
      static c @28
        type: C<num>
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get c @-1
        returnType: C<num>
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: C<num>
        returnType: void
''');
  }

  test_invalid_annotation_prefixed_constructor() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {
  const A.named();
}
''');
    var library = await buildLibrary('''
import "a.dart" as a;
@a.A.named
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as a @19
  definingUnit
    classes
      class C @39
        metadata
          Annotation
            atSign: @ @22
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: a @23
                staticElement: self::@prefix::a
                staticType: null
              period: . @24
              identifier: SimpleIdentifier
                token: A @25
                staticElement: package:test/a.dart::@class::A
                staticType: null
              staticElement: package:test/a.dart::@class::A
              staticType: null
            period: . @26
            constructorName: SimpleIdentifier
              token: named @27
              staticElement: package:test/a.dart::@class::A::@constructor::named
              staticType: null
            element: package:test/a.dart::@class::A::@constructor::named
        constructors
          synthetic @-1
''');
  }

  test_invalid_annotation_unprefixed_constructor() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A {
  const A.named();
}
''');
    var library = await buildLibrary('''
import "a.dart";
@A.named
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class C @32
        metadata
          Annotation
            atSign: @ @17
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: A @18
                staticElement: package:test/a.dart::@class::A
                staticType: null
              period: . @19
              identifier: SimpleIdentifier
                token: named @20
                staticElement: package:test/a.dart::@class::A::@constructor::named
                staticType: null
              staticElement: package:test/a.dart::@class::A::@constructor::named
              staticType: null
            element: package:test/a.dart::@class::A::@constructor::named
        constructors
          synthetic @-1
''');
  }

  test_invalid_importPrefix_asTypeArgument() async {
    var library = await buildLibrary('''
import 'dart:async' as ppp;
class C {
  List<ppp> v;
}
''');
    checkElementText(library, r'''
library
  imports
    dart:async as ppp @23
  definingUnit
    classes
      class C @34
        fields
          v @50
            type: List<dynamic>
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get v @-1
            returnType: List<dynamic>
          synthetic set v @-1
            parameters
              requiredPositional _v @-1
                type: List<dynamic>
            returnType: void
''');
  }

  test_invalid_nameConflict_imported() async {
    addSource('$testPackageLibPath/a.dart', 'V() {}');
    addSource('$testPackageLibPath/b.dart', 'V() {}');
    var library = await buildLibrary('''
import 'a.dart';
import 'b.dart';
foo([p = V]) {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
    package:test/b.dart
  definingUnit
    functions
      foo @34
        parameters
          optionalPositional default p @39
            type: dynamic
            constantInitializer
              SimpleIdentifier
                token: V @43
                staticElement: <null>
                staticType: InvalidType
        returnType: dynamic
''');
  }

  test_invalid_nameConflict_imported_exported() async {
    addSource('$testPackageLibPath/a.dart', 'V() {}');
    addSource('$testPackageLibPath/b.dart', 'V() {}');
    addSource('$testPackageLibPath/c.dart', r'''
export 'a.dart';
export 'b.dart';
''');
    var library = await buildLibrary('''
import 'c.dart';
foo([p = V]) {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/c.dart
  definingUnit
    functions
      foo @17
        parameters
          optionalPositional default p @22
            type: dynamic
            constantInitializer
              SimpleIdentifier
                token: V @26
                staticElement: package:test/a.dart::@function::V
                staticType: dynamic Function()
        returnType: dynamic
''');
  }

  test_invalid_nameConflict_local() async {
    var library = await buildLibrary('''
foo([p = V]) {}
V() {}
var V;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static V @27
        type: dynamic
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get V @-1
        returnType: dynamic
      synthetic static set V @-1
        parameters
          requiredPositional _V @-1
            type: dynamic
        returnType: void
    functions
      foo @0
        parameters
          optionalPositional default p @5
            type: dynamic
            constantInitializer
              SimpleIdentifier
                token: V @9
                staticElement: self::@getter::V
                staticType: dynamic
        returnType: dynamic
      V @16
        returnType: dynamic
''');
  }

  test_invalid_setterParameter_fieldFormalParameter() async {
    var library = await buildLibrary('''
class C {
  int foo;
  void set bar(this.foo) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
          synthetic bar @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
          set bar @32
            parameters
              requiredPositional final this.foo @41
                type: dynamic
                field: <null>
            returnType: void
''');
  }

  test_invalid_setterParameter_fieldFormalParameter_self() async {
    var library = await buildLibrary('''
class C {
  set x(this.x) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic x @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set x @16
            parameters
              requiredPositional final this.x @23
                type: dynamic
                field: <null>
            returnType: void
''');
  }

  test_library() async {
    var library = await buildLibrary('');
    checkElementText(library, r'''
library
  definingUnit
''');
  }

  test_library_augmentationImports_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
class A {}
''');
    final library = await buildLibrary(r'''
import augment 'a.dart';
class B {}
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class A @35
            constructors
              synthetic @-1
  definingUnit
    classes
      class B @31
        constructors
          synthetic @-1
''');

    final import_0 = library.augmentationImports[0];
    final augmentation = import_0.importedAugmentation!;
    expect(augmentation.enclosingElement, same(library));
  }

  test_library_augmentationImports_noRelativeUriStr() async {
    final library = await buildLibrary(r'''
import augment '${'foo'}.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    noRelativeUriString
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUri_emptyUriSelf() async {
    final library = await buildLibrary(r'''
import augment '';
''');
    checkElementText(library, r'''
library
  augmentationImports
    source 'package:test/test.dart'
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUri_noSource() async {
    final library = await buildLibrary(r'''
import augment 'foo:bar';
''');
    checkElementText(library, r'''
library
  augmentationImports
    relativeUri 'foo:bar'
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUri_notAugmentation_library() async {
    newFile('$testPackageLibPath/a.dart', r'''
library my.lib;
''');
    final library = await buildLibrary(r'''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUri_notAugmentation_part() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of other.lib;
''');
    final library = await buildLibrary(r'''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUri_notExists() async {
    final library = await buildLibrary(r'''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_augmentationImports_withRelativeUriString() async {
    final library = await buildLibrary(r'''
import augment ':';
''');
    checkElementText(library, r'''
library
  augmentationImports
    relativeUriString ':'
  definingUnit
''');
  }

  test_library_documented_lines() async {
    var library = await buildLibrary('''
/// aaa
/// bbb
library test;
''');
    checkElementText(library, r'''
library
  name: test
  nameOffset: 24
  documentationComment: /// aaa\n/// bbb
  definingUnit
''');
  }

  test_library_documented_stars() async {
    var library = await buildLibrary('''
/**
 * aaa
 * bbb
 */
library test;''');
    checkElementText(library, r'''
library
  name: test
  nameOffset: 30
  documentationComment: /**\n * aaa\n * bbb\n */
  definingUnit
''');
  }

  test_library_exports_noRelativeUriStr() async {
    final library = await buildLibrary(r'''
export '${'foo'}.dart';
''');
    checkElementText(library, r'''
library
  exports
    noRelativeUriString
  definingUnit
''');
  }

  test_library_exports_withRelativeUri_emptyUriSelf() async {
    final library = await buildLibrary(r'''
export '';
''');
    checkElementText(library, r'''
library
  exports
    package:test/test.dart
  definingUnit
''');
  }

  test_library_exports_withRelativeUri_noSource() async {
    final library = await buildLibrary(r'''
export 'foo:bar';
''');
    checkElementText(library, r'''
library
  exports
    relativeUri 'foo:bar'
  definingUnit
''');
  }

  test_library_exports_withRelativeUri_notExists() async {
    final library = await buildLibrary(r'''
export 'a.dart';
''');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_library_exports_withRelativeUri_notLibrary_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
''');
    final library = await buildLibrary(r'''
export 'a.dart';
''');
    checkElementText(library, r'''
library
  exports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_exports_withRelativeUri_notLibrary_part() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of other.lib;
''');
    final library = await buildLibrary(r'''
export 'a.dart';
''');
    checkElementText(library, r'''
library
  exports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_exports_withRelativeUriString() async {
    final library = await buildLibrary(r'''
export ':';
''');
    checkElementText(library, r'''
library
  exports
    relativeUriString ':'
  definingUnit
''');
  }

  test_library_imports_noRelativeUriStr() async {
    final library = await buildLibrary(r'''
import '${'foo'}.dart';
''');
    checkElementText(library, r'''
library
  imports
    noRelativeUriString
  definingUnit
''');
  }

  test_library_imports_prefix_importedLibraries() async {
    final library = await buildLibrary(r'''
import 'dart:async' as p1;
import 'dart:collection' as p2;
import 'dart:math' as p1;
''');
    final p1 = library.prefixes.singleWhere((prefix) => prefix.name == 'p1');
    final import_async = library.libraryImports[0];
    final import_math = library.libraryImports[2];
    expect(p1.imports, unorderedEquals([import_async, import_math]));
  }

  test_library_imports_syntheticDartCore() async {
    final library = await buildLibrary('');
    configuration.withSyntheticDartCoreImport = true;
    checkElementText(library, r'''
library
  imports
    dart:core synthetic
  definingUnit
''');
  }

  test_library_imports_withRelativeUri_emptyUriSelf() async {
    final library = await buildLibrary(r'''
import '';
''');
    checkElementText(library, r'''
library
  imports
    package:test/test.dart
  definingUnit
''');
  }

  test_library_imports_withRelativeUri_noSource() async {
    final library = await buildLibrary(r'''
import 'foo:bar';
''');
    checkElementText(library, r'''
library
  imports
    relativeUri 'foo:bar'
  definingUnit
''');
  }

  test_library_imports_withRelativeUri_notExists() async {
    final library = await buildLibrary(r'''
import 'a.dart';
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
''');
  }

  test_library_imports_withRelativeUri_notLibrary_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
''');
    final library = await buildLibrary(r'''
import 'a.dart';
''');
    checkElementText(library, r'''
library
  imports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_imports_withRelativeUri_notLibrary_part() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of other.lib;
''');
    final library = await buildLibrary(r'''
import 'a.dart';
''');
    checkElementText(library, r'''
library
  imports
    source 'package:test/a.dart'
  definingUnit
''');
  }

  test_library_imports_withRelativeUriString() async {
    final library = await buildLibrary(r'''
import ':';
''');
    checkElementText(library, r'''
library
  imports
    relativeUriString ':'
  definingUnit
''');
  }

  test_library_name_with_spaces() async {
    var library = await buildLibrary('library foo . bar ;');
    checkElementText(library, r'''
library
  name: foo.bar
  nameOffset: 8
  definingUnit
''');
  }

  test_library_named() async {
    var library = await buildLibrary('library foo.bar;');
    checkElementText(library, r'''
library
  name: foo.bar
  nameOffset: 8
  definingUnit
''');
  }

  test_library_parts() async {
    addSource('$testPackageLibPath/a.dart', 'part of my.lib;');
    addSource('$testPackageLibPath/b.dart', 'part of my.lib;');
    var library =
        await buildLibrary('library my.lib; part "a.dart"; part "b.dart";');
    checkElementText(library, r'''
library
  name: my.lib
  nameOffset: 8
  definingUnit
  parts
    package:test/a.dart
    package:test/b.dart
''');
  }

  test_library_parts_noRelativeUriStr() async {
    final library = await buildLibrary(r'''
part '${'foo'}.dart';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    noRelativeUriString
''');
  }

  test_library_parts_withPart_partOfName() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of my.lib;
class B {}
''');
    final library = await buildLibrary(r'''
library my.lib;
part 'a.dart';
class A {}
''');
    checkElementText(library, r'''
library
  name: my.lib
  nameOffset: 8
  definingUnit
    classes
      class A @37
        constructors
          synthetic @-1
  parts
    package:test/a.dart
      classes
        class B @22
          constructors
            synthetic @-1
''');
  }

  test_library_parts_withPart_partOfUri() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
class B {}
''');
    final library = await buildLibrary(r'''
part 'a.dart';
class A {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @21
        constructors
          synthetic @-1
  parts
    package:test/a.dart
      classes
        class B @27
          constructors
            synthetic @-1
''');
  }

  test_library_parts_withRelativeUri_noSource() async {
    final library = await buildLibrary(r'''
part 'foo:bar';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    relativeUri 'foo:bar'
''');
  }

  test_library_parts_withRelativeUri_notPart_emptyUriSelf() async {
    final library = await buildLibrary(r'''
part '';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    source 'package:test/test.dart'
''');
  }

  test_library_parts_withRelativeUri_notPart_library() async {
    newFile('$testPackageLibPath/a.dart', '');
    final library = await buildLibrary(r'''
part 'a.dart';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    source 'package:test/a.dart'
''');
  }

  test_library_parts_withRelativeUri_notPart_notExists() async {
    final library = await buildLibrary(r'''
part 'a.dart';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    source 'package:test/a.dart'
''');
  }

  test_library_parts_withRelativeUriString() async {
    final library = await buildLibrary(r'''
part ':';
''');
    checkElementText(library, r'''
library
  definingUnit
  parts
    relativeUriString ':'
''');
  }

  test_library_prefixes() async {
    final library = await buildLibrary(r'''
import 'dart:async' as p1;
import 'dart:collection' as p2;
import 'dart:math' as p1;
''');
    final prefixNames = library.prefixes.map((e) => e.name).toList();
    expect(prefixNames, unorderedEquals(['p1', 'p2']));
  }

  test_localFunctions() async {
    var library = await buildLibrary(r'''
f() {
  f1() {}
  {
    f2() {}
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @0
        returnType: dynamic
''');
  }

  test_localFunctions_inConstructor() async {
    var library = await buildLibrary(r'''
class C {
  C() {
    f() {}
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @12
''');
  }

  test_localFunctions_inMethod() async {
    var library = await buildLibrary(r'''
class C {
  m() {
    f() {}
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          m @12
            returnType: dynamic
''');
  }

  test_localFunctions_inTopLevelGetter() async {
    var library = await buildLibrary(r'''
get g {
  f() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static g @-1
        type: dynamic
    accessors
      static get g @4
        returnType: dynamic
''');
  }

  test_localLabels_inConstructor() async {
    var library = await buildLibrary(r'''
class C {
  C() {
    aaa: while (true) {}
    bbb: switch (42) {
      ccc: case 0:
        break;
    }
  }
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          @12
''');
  }

  test_localLabels_inMethod() async {
    var library = await buildLibrary(r'''
class C {
  m() {
    aaa: while (true) {}
    bbb: switch (42) {
      ccc: case 0:
        break;
    }
  }
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        constructors
          synthetic @-1
        methods
          m @12
            returnType: dynamic
''');
  }

  test_localLabels_inTopLevelFunction() async {
    var library = await buildLibrary(r'''
main() {
  aaa: while (true) {}
  bbb: switch (42) {
    ccc: case 0:
      break;
  }
}
''', allowErrors: true);
    checkElementText(library, r'''
library
  definingUnit
    functions
      main @0
        returnType: dynamic
''');
  }

  test_main_class() async {
    var library = await buildLibrary('class main {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class main @6
        constructors
          synthetic @-1
''');
  }

  test_main_class_alias() async {
    var library =
        await buildLibrary('class main = C with D; class C {} class D {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias main @6
        supertype: C
        mixins
          D
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::C::@constructor::new
            superConstructor: self::@class::C::@constructor::new
      class C @29
        constructors
          synthetic @-1
      class D @40
        constructors
          synthetic @-1
''');
  }

  test_main_class_alias_via_export() async {
    addSource('$testPackageLibPath/a.dart',
        'class main = C with D; class C {} class D {}');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_main_class_via_export() async {
    addSource('$testPackageLibPath/a.dart', 'class main {}');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_main_getter() async {
    var library = await buildLibrary('get main => null;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static main @-1
        type: dynamic
    accessors
      static get main @4
        returnType: dynamic
''');
  }

  test_main_getter_via_export() async {
    addSource('$testPackageLibPath/a.dart', 'get main => null;');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_main_typedef() async {
    var library = await buildLibrary('typedef main();');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased main @8
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
''');
  }

  test_main_typedef_via_export() async {
    addSource('$testPackageLibPath/a.dart', 'typedef main();');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_main_variable() async {
    var library = await buildLibrary('var main;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static main @4
        type: dynamic
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get main @-1
        returnType: dynamic
      synthetic static set main @-1
        parameters
          requiredPositional _main @-1
            type: dynamic
        returnType: void
''');
  }

  test_main_variable_via_export() async {
    addSource('$testPackageLibPath/a.dart', 'var main;');
    var library = await buildLibrary('export "a.dart";');
    checkElementText(library, r'''
library
  exports
    package:test/a.dart
  definingUnit
''');
  }

  test_metadata_augmentation_class() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
@deprecated
class A {}
''');
    var library = await buildLibrary('''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      definingUnit
        classes
          class A @47
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: deprecated @30
                  staticElement: dart:core::@getter::deprecated
                  staticType: null
                element: dart:core::@getter::deprecated
            constructors
              synthetic @-1
  definingUnit
''');
  }

  test_metadata_augmentation_directive() async {
    newFile('$testPackageLibPath/a.dart', r'''
@deprecated
library augment 'test.dart';
''');
    var library = await buildLibrary(r'''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: deprecated @1
            staticElement: dart:core::@getter::deprecated
            staticType: null
          element: dart:core::@getter::deprecated
      definingUnit
  definingUnit
''');
  }

  test_metadata_augmentation_exportLibrary() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
@deprecated
export 'dart:math';
''');
    var library = await buildLibrary('''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      exports
        dart:math
          metadata
            Annotation
              atSign: @ @29
              name: SimpleIdentifier
                token: deprecated @30
                staticElement: dart:core::@getter::deprecated
                staticType: null
              element: dart:core::@getter::deprecated
      definingUnit
  definingUnit
''');
  }

  test_metadata_augmentation_importAugmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'b.dart';
''');
    newFile('$testPackageLibPath/b.dart', r'''
library augment 'test.dart';
@deprecated
import augment 'a.dart';
''');
    var library = await buildLibrary('''
import augment 'b.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/b.dart
      augmentationImports
        package:test/a.dart
          metadata
            Annotation
              atSign: @ @29
              name: SimpleIdentifier
                token: deprecated @30
                staticElement: dart:core::@getter::deprecated
                staticType: null
              element: dart:core::@getter::deprecated
          definingUnit
      definingUnit
  definingUnit
''');
  }

  test_metadata_augmentation_importLibrary() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
@deprecated
import 'dart:math';
''');
    var library = await buildLibrary('''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      imports
        dart:math
          metadata
            Annotation
              atSign: @ @29
              name: SimpleIdentifier
                token: deprecated @30
                staticElement: dart:core::@getter::deprecated
                staticType: null
              element: dart:core::@getter::deprecated
      definingUnit
  definingUnit
''');
  }

  test_metadata_augmentation_libraryAugmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
@deprecated
library augment 'test.dart';
''');
    final library = await buildLibrary('''
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: deprecated @1
            staticElement: dart:core::@getter::deprecated
            staticType: null
          element: dart:core::@getter::deprecated
      definingUnit
  definingUnit
''');
  }

  test_metadata_class_field_first() async {
    var library = await buildLibrary(r'''
const a = 0;
class C {
  @a
  int x = 0;
}
''');
    // Check metadata without asking any other properties.
    var x = _elementOfDefiningUnit(library, ['@class', 'C', '@field', 'x'])
        as FieldElement;
    expect(x.metadata, hasLength(1));
    // Check details.
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @19
        fields
          x @34
            metadata
              Annotation
                atSign: @ @25
                name: SimpleIdentifier
                  token: a @26
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_class_scope() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
class C<@foo T> {
  static const foo = 1;
  @foo
  void bar() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @27
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @34
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: foo @30
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        fields
          static const foo @54
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 1 @60
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get foo @-1
            returnType: int
        methods
          bar @77
            metadata
              Annotation
                atSign: @ @65
                name: SimpleIdentifier
                  token: foo @66
                  staticElement: self::@class::C::@getter::foo
                  staticType: null
                element: self::@class::C::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_classDeclaration() async {
    var library = await buildLibrary(r'''
const a = null;
const b = null;
@a
@b
class C {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @44
        metadata
          Annotation
            atSign: @ @32
            name: SimpleIdentifier
              token: a @33
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
          Annotation
            atSign: @ @35
            name: SimpleIdentifier
              token: b @36
              staticElement: self::@getter::b
              staticType: null
            element: self::@getter::b
        constructors
          synthetic @-1
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static const b @22
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @26
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get b @-1
        returnType: dynamic
''');
  }

  test_metadata_classTypeAlias() async {
    var library = await buildLibrary(
        'const a = null; @a class C = D with E; class D {} class E {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @25
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @45
        constructors
          synthetic @-1
      class E @56
        constructors
          synthetic @-1
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_constructor_call_named() async {
    var library = await buildLibrary('''
class A {
  const A.named(int _);
}
@A.named(0)
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const named @20
            periodOffset: 19
            nameEnd: 25
            parameters
              requiredPositional _ @30
                type: int
      class C @54
        metadata
          Annotation
            atSign: @ @36
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: A @37
                staticElement: self::@class::A
                staticType: null
              period: . @38
              identifier: SimpleIdentifier
                token: named @39
                staticElement: self::@class::A::@constructor::named
                staticType: null
              staticElement: self::@class::A::@constructor::named
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @44
              arguments
                IntegerLiteral
                  literal: 0 @45
                  staticType: int
              rightParenthesis: ) @46
            element: self::@class::A::@constructor::named
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_generic_inference() async {
    var library = await buildLibrary('''
class A<T> {
  const A.named(T _);
}

@A.named(0)
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const named @23
            periodOffset: 22
            nameEnd: 28
            parameters
              requiredPositional _ @31
                type: T
      class C @56
        metadata
          Annotation
            atSign: @ @38
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: A @39
                staticElement: self::@class::A
                staticType: null
              period: . @40
              identifier: SimpleIdentifier
                token: named @41
                staticElement: ConstructorMember
                  base: self::@class::A::@constructor::named
                  substitution: {T: int}
                staticType: null
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::named
                substitution: {T: int}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @46
              arguments
                IntegerLiteral
                  literal: 0 @47
                  staticType: int
              rightParenthesis: ) @48
            element: ConstructorMember
              base: self::@class::A::@constructor::named
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_generic_typeArguments() async {
    var library = await buildLibrary('''
class A<T> {
  const A.named();
}

@A<int>.named()
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const named @23
            periodOffset: 22
            nameEnd: 28
      class C @57
        metadata
          Annotation
            atSign: @ @35
            name: SimpleIdentifier
              token: A @36
              staticElement: self::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @37
              arguments
                NamedType
                  name: int @38
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @41
            period: . @42
            constructorName: SimpleIdentifier
              token: named @43
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::named
                substitution: {T: int}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @48
              rightParenthesis: ) @49
            element: ConstructorMember
              base: self::@class::A::@constructor::named
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_generic_typeArguments_disabledGenericMetadata() async {
    var library = await buildLibrary('''
class A<T> {
  const A.named();
}

@A<int>.named()
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const named @23
            periodOffset: 22
            nameEnd: 28
      class C @57
        metadata
          Annotation
            atSign: @ @35
            name: SimpleIdentifier
              token: A @36
              staticElement: self::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @37
              arguments
                NamedType
                  name: int @38
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @41
            period: . @42
            constructorName: SimpleIdentifier
              token: named @43
              staticElement: ConstructorMember
                base: self::@class::A::@constructor::named
                substitution: {T: int}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @48
              rightParenthesis: ) @49
            element: ConstructorMember
              base: self::@class::A::@constructor::named
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_prefixed() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A {
  const A.named(int _);
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
@foo.A.named(0)
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @48
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            period: . @32
            constructorName: SimpleIdentifier
              token: named @33
              staticElement: package:test/foo.dart::@class::A::@constructor::named
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @38
              arguments
                IntegerLiteral
                  literal: 0 @39
                  staticType: int
              rightParenthesis: ) @40
            element: package:test/foo.dart::@class::A::@constructor::named
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_prefixed_generic_inference() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A<T> {
  const A.named(T _);
}
''');
    var library = await buildLibrary('''
import "foo.dart" as foo;
@foo.A.named(0)
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @48
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            period: . @32
            constructorName: SimpleIdentifier
              token: named @33
              staticElement: ConstructorMember
                base: package:test/foo.dart::@class::A::@constructor::named
                substitution: {T: int}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @38
              arguments
                IntegerLiteral
                  literal: 0 @39
                  staticType: int
              rightParenthesis: ) @40
            element: ConstructorMember
              base: package:test/foo.dart::@class::A::@constructor::named
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_prefixed_generic_typeArguments() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A<T> {
  const A.named();
}
''');
    var library = await buildLibrary('''
import "foo.dart" as foo;
@foo.A<int>.named()
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @52
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @32
              arguments
                NamedType
                  name: int @33
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @36
            period: . @37
            constructorName: SimpleIdentifier
              token: named @38
              staticElement: ConstructorMember
                base: package:test/foo.dart::@class::A::@constructor::named
                substitution: {T: int}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @43
              rightParenthesis: ) @44
            element: ConstructorMember
              base: package:test/foo.dart::@class::A::@constructor::named
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_named_synthetic_ofClassAlias_generic() async {
    var library = await buildLibrary('''
class A {
  const A.named();
}

mixin B {}

class C<T> = A with B;

@C.named()
class D {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const named @20
            periodOffset: 19
            nameEnd: 25
      class alias C @50
        typeParameters
          covariant T @52
            defaultType: dynamic
        supertype: A
        mixins
          B
        constructors
          synthetic const named @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                period: . @0
                constructorName: SimpleIdentifier
                  token: named @-1
                  staticElement: self::@class::A::@constructor::named
                  staticType: null
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::named
            superConstructor: self::@class::A::@constructor::named
      class D @85
        metadata
          Annotation
            atSign: @ @68
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: C @69
                staticElement: self::@class::C
                staticType: null
              period: . @70
              identifier: SimpleIdentifier
                token: named @71
                staticElement: ConstructorMember
                  base: self::@class::C::@constructor::named
                  substitution: {T: dynamic}
                staticType: null
              staticElement: ConstructorMember
                base: self::@class::C::@constructor::named
                substitution: {T: dynamic}
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @76
              rightParenthesis: ) @77
            element: ConstructorMember
              base: self::@class::C::@constructor::named
              substitution: {T: dynamic}
        constructors
          synthetic @-1
    mixins
      mixin B @38
        superclassConstraints
          Object
''');
  }

  test_metadata_constructor_call_unnamed() async {
    var library = await buildLibrary('''
class A {
  const A(int _);
}
@A(0)
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
            parameters
              requiredPositional _ @24
                type: int
      class C @42
        metadata
          Annotation
            atSign: @ @30
            name: SimpleIdentifier
              token: A @31
              staticElement: self::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @32
              arguments
                IntegerLiteral
                  literal: 0 @33
                  staticType: int
              rightParenthesis: ) @34
            element: self::@class::A::@constructor::new
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_generic_inference() async {
    var library = await buildLibrary('''
class A<T> {
  const A(T _);
}

@A(0)
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
            parameters
              requiredPositional _ @25
                type: T
      class C @44
        metadata
          Annotation
            atSign: @ @32
            name: SimpleIdentifier
              token: A @33
              staticElement: self::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @34
              arguments
                IntegerLiteral
                  literal: 0 @35
                  staticType: int
              rightParenthesis: ) @36
            element: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_generic_typeArguments() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}

@A<int>()
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          const @21
      class C @45
        metadata
          Annotation
            atSign: @ @29
            name: SimpleIdentifier
              token: A @30
              staticElement: self::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @31
              arguments
                NamedType
                  name: int @32
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @35
            arguments: ArgumentList
              leftParenthesis: ( @36
              rightParenthesis: ) @37
            element: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_prefixed() async {
    addSource('$testPackageLibPath/foo.dart', 'class A { const A(_); }');
    var library =
        await buildLibrary('import "foo.dart" as foo; @foo.A(0) class C {}');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @42
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @32
              arguments
                IntegerLiteral
                  literal: 0 @33
                  staticType: int
              rightParenthesis: ) @34
            element: package:test/foo.dart::@class::A::@constructor::new
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_prefixed_generic_inference() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A<T> {
  const A(T _);
}
''');
    var library = await buildLibrary('''
import "foo.dart" as foo;
@foo.A(0)
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @42
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @32
              arguments
                IntegerLiteral
                  literal: 0 @33
                  staticType: int
              rightParenthesis: ) @34
            element: ConstructorMember
              base: package:test/foo.dart::@class::A::@constructor::new
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_prefixed_generic_typeArguments() async {
    addSource('$testPackageLibPath/foo.dart', '''
class A<T> {
  const A();
}
''');
    var library = await buildLibrary('''
import "foo.dart" as foo;
@foo.A<int>()
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @46
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: A @31
                staticElement: package:test/foo.dart::@class::A
                staticType: null
              staticElement: package:test/foo.dart::@class::A
              staticType: null
            typeArguments: TypeArgumentList
              leftBracket: < @32
              arguments
                NamedType
                  name: int @33
                  element: dart:core::@class::int
                  type: int
              rightBracket: > @36
            arguments: ArgumentList
              leftParenthesis: ( @37
              rightParenthesis: ) @38
            element: ConstructorMember
              base: package:test/foo.dart::@class::A::@constructor::new
              substitution: {T: int}
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructor_call_unnamed_synthetic_ofClassAlias_generic() async {
    var library = await buildLibrary('''
class A {
  const A();
}

mixin B {}

class C<T> = A with B;

@C()
class D {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @18
      class alias C @44
        typeParameters
          covariant T @46
            defaultType: dynamic
        supertype: A
        mixins
          B
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
      class D @73
        metadata
          Annotation
            atSign: @ @62
            name: SimpleIdentifier
              token: C @63
              staticElement: self::@class::C
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @64
              rightParenthesis: ) @65
            element: ConstructorMember
              base: self::@class::C::@constructor::new
              substitution: {T: dynamic}
        constructors
          synthetic @-1
    mixins
      mixin B @32
        superclassConstraints
          Object
''');
  }

  test_metadata_constructor_call_with_args() async {
    var library =
        await buildLibrary('class A { const A(x); } @A(null) class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          const @16
            parameters
              requiredPositional x @18
                type: dynamic
      class C @39
        metadata
          Annotation
            atSign: @ @24
            name: SimpleIdentifier
              token: A @25
              staticElement: self::@class::A
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @26
              arguments
                NullLiteral
                  literal: null @27
                  staticType: Null
              rightParenthesis: ) @31
            element: self::@class::A::@constructor::new
        constructors
          synthetic @-1
''');
  }

  test_metadata_constructorDeclaration_named() async {
    var library =
        await buildLibrary('const a = null; class C { @a C.named(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        constructors
          named @31
            metadata
              Annotation
                atSign: @ @26
                name: SimpleIdentifier
                  token: a @27
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            periodOffset: 30
            nameEnd: 36
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_constructorDeclaration_unnamed() async {
    var library = await buildLibrary('const a = null; class C { @a C(); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        constructors
          @29
            metadata
              Annotation
                atSign: @ @26
                name: SimpleIdentifier
                  token: a @27
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_enum_constant() async {
    var library = await buildLibrary('const a = 42; enum E { @a v }');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @19
        supertype: Enum
        fields
          static const enumConstant v @26
            metadata
              Annotation
                atSign: @ @23
                name: SimpleIdentifier
                  token: a @24
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_enum_constant_instanceCreation() async {
    var library = await buildLibrary('''
class A {
  final dynamic value;
  const A(this.value);
}

enum E {
  @A(100) a,
  b,
  @A(300) c,
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          final value @26
            type: dynamic
            shouldUseTypeForInitializerInference: true
        constructors
          const @41
            parameters
              requiredPositional final this.value @48
                type: dynamic
                field: self::@class::A::@field::value
        accessors
          synthetic get value @-1
            returnType: dynamic
    enums
      enum E @64
        supertype: Enum
        fields
          static const enumConstant a @78
            metadata
              Annotation
                atSign: @ @70
                name: SimpleIdentifier
                  token: A @71
                  staticElement: self::@class::A
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @72
                  arguments
                    IntegerLiteral
                      literal: 100 @73
                      staticType: int
                  rightParenthesis: ) @76
                element: self::@class::A::@constructor::new
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @83
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant c @96
            metadata
              Annotation
                atSign: @ @88
                name: SimpleIdentifier
                  token: A @89
                  staticElement: self::@class::A
                  staticType: null
                arguments: ArgumentList
                  leftParenthesis: ( @90
                  arguments
                    IntegerLiteral
                      literal: 300 @91
                      staticType: int
                  rightParenthesis: ) @94
                element: self::@class::A::@constructor::new
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get c @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_metadata_enum_constant_self() async {
    var library = await buildLibrary(r'''
enum E {
  @v
  v
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant v @16
            metadata
              Annotation
                atSign: @ @11
                name: SimpleIdentifier
                  token: v @12
                  staticElement: self::@enum::E::@getter::v
                  staticType: null
                element: self::@enum::E::@getter::v
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_metadata_enum_constructor() async {
    var library = await buildLibrary(r'''
const a = 42;
enum E {
  v;
  @a
  const E();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @19
        supertype: Enum
        fields
          static const enumConstant v @25
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          const @41
            metadata
              Annotation
                atSign: @ @30
                name: SimpleIdentifier
                  token: a @31
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_enum_method() async {
    var library = await buildLibrary(r'''
const a = 42;
enum E {
  v;
  @a
  void foo() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @19
        supertype: Enum
        fields
          static const enumConstant v @25
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
        methods
          foo @40
            metadata
              Annotation
                atSign: @ @30
                name: SimpleIdentifier
                  token: a @31
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            returnType: void
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_enum_scope() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
enum E<@foo T> {
  v;
  static const foo = 1;
  @foo
  void bar() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @26
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @33
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        supertype: Enum
        fields
          static const enumConstant v @40
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
          static const foo @58
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 1 @64
                staticType: int
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
          synthetic static get foo @-1
            returnType: int
        methods
          bar @81
            metadata
              Annotation
                atSign: @ @69
                name: SimpleIdentifier
                  token: foo @70
                  staticElement: self::@enum::E::@getter::foo
                  staticType: null
                element: self::@enum::E::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_enum_typeParameter() async {
    var library = await buildLibrary('''
const a = 42;
enum E<@a T> {
  v
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @19
        typeParameters
          covariant T @24
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @21
                name: SimpleIdentifier
                  token: a @22
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        supertype: Enum
        fields
          static const enumConstant v @31
            type: E<dynamic>
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E<dynamic>
                  staticElement: ConstructorMember
                    base: self::@enum::E::@constructor::new
                    substitution: {T: dynamic}
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E<dynamic>
          synthetic static const values @-1
            type: List<E<dynamic>>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E<dynamic>
                rightBracket: ] @0
                staticType: List<E<dynamic>>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E<dynamic>
          synthetic static get values @-1
            returnType: List<E<dynamic>>
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_enumDeclaration() async {
    var library = await buildLibrary('const a = 42; @a enum E { v }');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @22
        metadata
          Annotation
            atSign: @ @14
            name: SimpleIdentifier
              token: a @15
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        supertype: Enum
        fields
          static const enumConstant v @26
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: v @-1
                    staticElement: self::@enum::E::@getter::v
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get v @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_exportDirective() async {
    addSource('$testPackageLibPath/foo.dart', '');
    var library = await buildLibrary('@a export "foo.dart"; const a = null;');
    checkElementText(library, r'''
library
  metadata
    Annotation
      atSign: @ @0
      name: SimpleIdentifier
        token: a @1
        staticElement: self::@getter::a
        staticType: null
      element: self::@getter::a
  exports
    package:test/foo.dart
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: a @1
            staticElement: self::@getter::a
            staticType: null
          element: self::@getter::a
  definingUnit
    topLevelVariables
      static const a @28
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @32
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_extension_scope() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
extension E<@foo T> on int {
  static const foo = 1;
  @foo
  void bar() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @31
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @38
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @33
                name: SimpleIdentifier
                  token: foo @34
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        extendedType: int
        fields
          static const foo @65
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 1 @71
                staticType: int
        accessors
          synthetic static get foo @-1
            returnType: int
        methods
          bar @88
            metadata
              Annotation
                atSign: @ @76
                name: SimpleIdentifier
                  token: foo @77
                  staticElement: self::@extension::E::@getter::foo
                  staticType: null
                element: self::@extension::E::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_extensionDeclaration() async {
    var library = await buildLibrary(r'''
const a = null;
class A {}
@a
@Object()
extension E on A {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        constructors
          synthetic @-1
    extensions
      E @50
        metadata
          Annotation
            atSign: @ @27
            name: SimpleIdentifier
              token: a @28
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
          Annotation
            atSign: @ @30
            name: SimpleIdentifier
              token: Object @31
              staticElement: dart:core::@class::Object
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @37
              rightParenthesis: ) @38
            element: dart:core::@class::Object::@constructor::new
        extendedType: A
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_fieldDeclaration() async {
    var library = await buildLibrary('const a = null; class C { @a int x; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        fields
          x @33
            metadata
              Annotation
                atSign: @ @26
                name: SimpleIdentifier
                  token: a @27
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get x @-1
            returnType: int
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: int
            returnType: void
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_fieldFormalParameter() async {
    var library = await buildLibrary('''
const a = null;
class C {
  var x;
  C(@a this.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        fields
          x @32
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @37
            parameters
              requiredPositional final this.x @47
                type: dynamic
                metadata
                  Annotation
                    atSign: @ @39
                    name: SimpleIdentifier
                      token: a @40
                      staticElement: self::@getter::a
                      staticType: null
                    element: self::@getter::a
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_fieldFormalParameter_withDefault() async {
    var library = await buildLibrary(
        'const a = null; class C { var x; C([@a this.x = null]); }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        fields
          x @30
            type: dynamic
            shouldUseTypeForInitializerInference: false
        constructors
          @33
            parameters
              optionalPositional default final this.x @44
                type: dynamic
                metadata
                  Annotation
                    atSign: @ @36
                    name: SimpleIdentifier
                      token: a @37
                      staticElement: self::@getter::a
                      staticType: null
                    element: self::@getter::a
                constantInitializer
                  NullLiteral
                    literal: null @48
                    staticType: Null
                field: self::@class::C::@field::x
        accessors
          synthetic get x @-1
            returnType: dynamic
          synthetic set x @-1
            parameters
              requiredPositional _x @-1
                type: dynamic
            returnType: void
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_functionDeclaration_function() async {
    var library = await buildLibrary('''
const a = null;
@a
f() {}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @19
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        returnType: dynamic
''');
  }

  test_metadata_functionDeclaration_getter() async {
    var library = await buildLibrary('const a = null; @a get f => null;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      synthetic static f @-1
        type: dynamic
    accessors
      synthetic static get a @-1
        returnType: dynamic
      static get f @23
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        returnType: dynamic
''');
  }

  test_metadata_functionDeclaration_setter() async {
    var library = await buildLibrary('const a = null; @a set f(value) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      synthetic static f @-1
        type: dynamic
    accessors
      synthetic static get a @-1
        returnType: dynamic
      static set f @23
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        parameters
          requiredPositional value @25
            type: dynamic
        returnType: void
''');
  }

  test_metadata_functionTypeAlias() async {
    var library = await buildLibrary('const a = null; @a typedef F();');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @27
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_functionTypedFormalParameter() async {
    var library = await buildLibrary('const a = null; f(@a g()) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @16
        parameters
          requiredPositional g @21
            type: dynamic Function()
            metadata
              Annotation
                atSign: @ @18
                name: SimpleIdentifier
                  token: a @19
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        returnType: dynamic
''');
  }

  test_metadata_functionTypedFormalParameter_withDefault() async {
    var library = await buildLibrary('const a = null; f([@a g() = null]) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @16
        parameters
          optionalPositional default g @22
            type: dynamic Function()
            metadata
              Annotation
                atSign: @ @19
                name: SimpleIdentifier
                  token: a @20
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            constantInitializer
              NullLiteral
                literal: null @28
                staticType: null
        returnType: dynamic
''');
  }

  test_metadata_genericTypeAlias() async {
    var library = await buildLibrary(r'''
const a = null;
const b = null;
@a
@b
typedef F = void Function();''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @46
        metadata
          Annotation
            atSign: @ @32
            name: SimpleIdentifier
              token: a @33
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
          Annotation
            atSign: @ @35
            name: SimpleIdentifier
              token: b @36
              staticElement: self::@getter::b
              staticType: null
            element: self::@getter::b
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static const b @22
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @26
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get b @-1
        returnType: dynamic
''');
  }

  test_metadata_importDirective() async {
    var library = await buildLibrary('''
@a
import "dart:math";
const a = 0;
''');
    checkElementText(library, r'''
library
  metadata
    Annotation
      atSign: @ @0
      name: SimpleIdentifier
        token: a @1
        staticElement: self::@getter::a
        staticType: null
      element: self::@getter::a
  imports
    dart:math
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: a @1
            staticElement: self::@getter::a
            staticType: null
          element: self::@getter::a
  definingUnit
    topLevelVariables
      static const a @29
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @33
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_importDirective_hasShow() async {
    var library = await buildLibrary(r'''
@a
import "dart:math" show Random;

const a = 0;
''');
    checkElementText(library, r'''
library
  metadata
    Annotation
      atSign: @ @0
      name: SimpleIdentifier
        token: a @1
        staticElement: self::@getter::a
        staticType: null
      element: self::@getter::a
  imports
    dart:math
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: a @1
            staticElement: self::@getter::a
            staticType: null
          element: self::@getter::a
      combinators
        show: Random
  definingUnit
    topLevelVariables
      static const a @42
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @46
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_inAliasedElement_formalParameter() async {
    var library = await buildLibrary('''
const a = 42;
typedef F = void Function(@a int first)
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @22
        aliasedType: void Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional first @47
              type: int
              metadata
                Annotation
                  atSign: @ @40
                  name: SimpleIdentifier
                    token: a @41
                    staticElement: self::@getter::a
                    staticType: null
                  element: self::@getter::a
          returnType: void
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_inAliasedElement_formalParameter2() async {
    var library = await buildLibrary('''
const a = 42;
typedef F = void Function(int foo(@a int bar))
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @22
        aliasedType: void Function(int Function(int))
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional foo @44
              type: int Function(int)
              parameters
                requiredPositional bar @55
                  type: int
                  metadata
                    Annotation
                      atSign: @ @48
                      name: SimpleIdentifier
                        token: a @49
                        staticElement: self::@getter::a
                        staticType: null
                      element: self::@getter::a
          returnType: void
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_inAliasedElement_typeParameter() async {
    var library = await buildLibrary('''
const a = 42;
typedef F = void Function<@a T>(int first)
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @22
        aliasedType: void Function<T>(int)
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant T @43
              metadata
                Annotation
                  atSign: @ @40
                  name: SimpleIdentifier
                    token: a @41
                    staticElement: self::@getter::a
                    staticType: null
                  element: self::@getter::a
          parameters
            requiredPositional first @50
              type: int
          returnType: void
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 42 @10
            staticType: int
    accessors
      synthetic static get a @-1
        returnType: int
''');
  }

  test_metadata_invalid_classDeclaration() async {
    var library = await buildLibrary('f(_) {} @f(42) class C {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @21
        metadata
          Annotation
            atSign: @ @8
            name: SimpleIdentifier
              token: f @9
              staticElement: self::@function::f
              staticType: null
            arguments: ArgumentList
              leftParenthesis: ( @10
              arguments
                IntegerLiteral
                  literal: 42 @11
                  staticType: int
              rightParenthesis: ) @13
            element: self::@function::f
        constructors
          synthetic @-1
    functions
      f @0
        parameters
          requiredPositional _ @2
            type: dynamic
        returnType: dynamic
''');
  }

  test_metadata_library_importAugmentation_augmentation() async {
    newFile('$testPackageLibPath/a.dart', r'''
library augment 'test.dart';
''');
    var library = await buildLibrary('''
@deprecated
import augment 'a.dart';
''');
    checkElementText(library, r'''
library
  augmentationImports
    package:test/a.dart
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: deprecated @1
            staticElement: dart:core::@getter::deprecated
            staticType: null
          element: dart:core::@getter::deprecated
      definingUnit
  definingUnit
''');
  }

  /// Even though the target is not an augmentation, metadata is available.
  test_metadata_library_importAugmentation_notAugmentation_library() async {
    var library = await buildLibrary('''
@deprecated
import augment 'dart:math';
''');
    checkElementText(library, r'''
library
  augmentationImports
    source 'dart:math'
      metadata
        Annotation
          atSign: @ @0
          name: SimpleIdentifier
            token: deprecated @1
            staticElement: dart:core::@getter::deprecated
            staticType: null
          element: dart:core::@getter::deprecated
  definingUnit
''');
  }

  test_metadata_libraryDirective() async {
    var library = await buildLibrary('@a library L; const a = null;');
    checkElementText(library, r'''
library
  name: L
  nameOffset: 11
  metadata
    Annotation
      atSign: @ @0
      name: SimpleIdentifier
        token: a @1
        staticElement: self::@getter::a
        staticType: null
      element: self::@getter::a
  definingUnit
    topLevelVariables
      static const a @20
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @24
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_methodDeclaration_getter() async {
    var library =
        await buildLibrary('const a = null; class C { @a get m => null; }');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        fields
          synthetic m @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          get m @33
            metadata
              Annotation
                atSign: @ @26
                name: SimpleIdentifier
                  token: a @27
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_methodDeclaration_method() async {
    var library = await buildLibrary(r'''
const a = null;
const b = null;
class C {
  @a
  @b
  m() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @38
        constructors
          synthetic @-1
        methods
          m @54
            metadata
              Annotation
                atSign: @ @44
                name: SimpleIdentifier
                  token: a @45
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
              Annotation
                atSign: @ @49
                name: SimpleIdentifier
                  token: b @50
                  staticElement: self::@getter::b
                  staticType: null
                element: self::@getter::b
            returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static const b @22
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @26
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get b @-1
        returnType: dynamic
''');
  }

  test_metadata_methodDeclaration_method_mixin() async {
    var library = await buildLibrary(r'''
const a = null;
const b = null;
mixin M {
  @a
  @b
  m() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @38
        superclassConstraints
          Object
        methods
          m @54
            metadata
              Annotation
                atSign: @ @44
                name: SimpleIdentifier
                  token: a @45
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
              Annotation
                atSign: @ @49
                name: SimpleIdentifier
                  token: b @50
                  staticElement: self::@getter::b
                  staticType: null
                element: self::@getter::b
            returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static const b @22
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @26
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get b @-1
        returnType: dynamic
''');
  }

  test_metadata_methodDeclaration_setter() async {
    var library = await buildLibrary('''
const a = null;
class C {
  @a
  set m(value) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        fields
          synthetic m @-1
            type: dynamic
        constructors
          synthetic @-1
        accessors
          set m @37
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: a @29
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            parameters
              requiredPositional value @39
                type: dynamic
            returnType: void
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_mixin_scope() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
mixin M<@foo T> {
  static const foo = 1;
  @foo
  void bar() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @27
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @34
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: foo @30
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        superclassConstraints
          Object
        fields
          static const foo @54
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 1 @60
                staticType: int
        accessors
          synthetic static get foo @-1
            returnType: int
        methods
          bar @77
            metadata
              Annotation
                atSign: @ @65
                name: SimpleIdentifier
                  token: foo @66
                  staticElement: self::@mixin::M::@getter::foo
                  staticType: null
                element: self::@mixin::M::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_mixinDeclaration() async {
    var library = await buildLibrary(r'''
const a = null;
const b = null;
@a
@b
mixin M {}''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @44
        metadata
          Annotation
            atSign: @ @32
            name: SimpleIdentifier
              token: a @33
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
          Annotation
            atSign: @ @35
            name: SimpleIdentifier
              token: b @36
              staticElement: self::@getter::b
              staticType: null
            element: self::@getter::b
        superclassConstraints
          Object
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static const b @22
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @26
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get b @-1
        returnType: dynamic
''');
  }

  test_metadata_offsets_onClass() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
class A<@foo T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @27
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @34
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: foo @30
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        constructors
          synthetic @-1
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onClassConstructor() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {
  @foo
  A(@foo int a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        constructors
          @35
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            parameters
              requiredPositional a @46
                type: int
                metadata
                  Annotation
                    atSign: @ @37
                    name: SimpleIdentifier
                      token: foo @38
                      staticElement: self::@getter::foo
                      staticType: null
                    element: self::@getter::foo
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onClassGetter() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {
  @foo
  int get getter => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        fields
          synthetic getter @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get getter @43
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            returnType: int
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onClassMethod() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {
  @foo
  void method<@foo T>(@foo int a) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        constructors
          synthetic @-1
        methods
          method @40
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            typeParameters
              covariant T @52
                defaultType: dynamic
                metadata
                  Annotation
                    atSign: @ @47
                    name: SimpleIdentifier
                      token: foo @48
                      staticElement: self::@getter::foo
                      staticType: null
                    element: self::@getter::foo
            parameters
              requiredPositional a @64
                type: int
                metadata
                  Annotation
                    atSign: @ @55
                    name: SimpleIdentifier
                      token: foo @56
                      staticElement: self::@getter::foo
                      staticType: null
                    element: self::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onClassSetter() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {
  @foo
  set setter(@foo int a) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        fields
          synthetic setter @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set setter @39
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            parameters
              requiredPositional a @55
                type: int
                metadata
                  Annotation
                    atSign: @ @46
                    name: SimpleIdentifier
                      token: foo @47
                      staticElement: self::@getter::foo
                      staticType: null
                    element: self::@getter::foo
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onClassTypeAlias() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {}
mixin M {}

@foo
class B<@foo T> = A with M;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        constructors
          synthetic @-1
      class alias B @50
        metadata
          Annotation
            atSign: @ @39
            name: SimpleIdentifier
              token: foo @40
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @57
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @52
                name: SimpleIdentifier
                  token: foo @53
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        supertype: A
        mixins
          M
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: self::@class::A::@constructor::new
    mixins
      mixin M @33
        superclassConstraints
          Object
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onEnum() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
enum E {
  @foo e1,
  e2,
  @foo e3,
}
''');
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @26
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        supertype: Enum
        fields
          static const enumConstant e1 @37
            metadata
              Annotation
                atSign: @ @32
                name: SimpleIdentifier
                  token: foo @33
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant e2 @43
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant e3 @54
            metadata
              Annotation
                atSign: @ @49
                name: SimpleIdentifier
                  token: foo @50
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: e1 @-1
                    staticElement: self::@enum::E::@getter::e1
                    staticType: E
                  SimpleIdentifier
                    token: e2 @-1
                    staticElement: self::@enum::E::@getter::e2
                    staticType: E
                  SimpleIdentifier
                    token: e3 @-1
                    staticElement: self::@enum::E::@getter::e3
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get e1 @-1
            returnType: E
          synthetic static get e2 @-1
            returnType: E
          synthetic static get e3 @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onExtension() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
extension E<@foo T> on List<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @31
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @38
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @33
                name: SimpleIdentifier
                  token: foo @34
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        extendedType: List<T>
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onFieldDeclaration() async {
    var library = await buildLibrary(r'''
const foo = 0;

class A {
  @foo
  static isStatic = 1;

  @foo
  static const isStaticConst = 2;

  @foo
  var isInstance = 3;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @22
        fields
          static isStatic @42
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            type: int
            shouldUseTypeForInitializerInference: false
          static const isStaticConst @79
            metadata
              Annotation
                atSign: @ @59
                name: SimpleIdentifier
                  token: foo @60
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 2 @95
                staticType: int
          isInstance @112
            metadata
              Annotation
                atSign: @ @101
                name: SimpleIdentifier
                  token: foo @102
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            type: int
            shouldUseTypeForInitializerInference: false
        constructors
          synthetic @-1
        accessors
          synthetic static get isStatic @-1
            returnType: int
          synthetic static set isStatic @-1
            parameters
              requiredPositional _isStatic @-1
                type: int
            returnType: void
          synthetic static get isStaticConst @-1
            returnType: int
          synthetic get isInstance @-1
            returnType: int
          synthetic set isInstance @-1
            parameters
              requiredPositional _isInstance @-1
                type: int
            returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onLibrary() async {
    var library = await buildLibrary('''
/// Some documentation.
@foo
library my.lib;

const foo = 0;
''');
    checkElementText(library, r'''
library
  name: my.lib
  nameOffset: 37
  documentationComment: /// Some documentation.
  metadata
    Annotation
      atSign: @ @24
      name: SimpleIdentifier
        token: foo @25
        staticElement: self::@getter::foo
        staticType: null
      element: self::@getter::foo
  definingUnit
    topLevelVariables
      static const foo @52
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @58
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onMixin() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
mixin A<@foo T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin A @27
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @34
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @29
                name: SimpleIdentifier
                  token: foo @30
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        superclassConstraints
          Object
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onTypeAlias_classic() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
typedef void F<@foo T>(@foo int a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @34
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          unrelated T @41
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @36
                name: SimpleIdentifier
                  token: foo @37
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        aliasedType: void Function(int)
        aliasedElement: GenericFunctionTypeElement
          parameters
            requiredPositional a @53
              type: int
              metadata
                Annotation
                  atSign: @ @44
                  name: SimpleIdentifier
                    token: foo @45
                    staticElement: self::@getter::foo
                    staticType: null
                  element: self::@getter::foo
          returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onTypeAlias_genericFunctionType() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
typedef A<@foo T> = void Function<@foo U>(@foo int a);
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      A @29
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          unrelated T @36
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @31
                name: SimpleIdentifier
                  token: foo @32
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        aliasedType: void Function<U>(int)
        aliasedElement: GenericFunctionTypeElement
          typeParameters
            covariant U @60
              metadata
                Annotation
                  atSign: @ @55
                  name: SimpleIdentifier
                    token: foo @56
                    staticElement: self::@getter::foo
                    staticType: null
                  element: self::@getter::foo
          parameters
            requiredPositional a @72
              type: int
              metadata
                Annotation
                  atSign: @ @63
                  name: SimpleIdentifier
                    token: foo @64
                    staticElement: self::@getter::foo
                    staticType: null
                  element: self::@getter::foo
          returnType: void
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
''');
  }

  test_metadata_offsets_onUnit() async {
    addSource('$testPackageLibPath/a.dart', '''
part of my.lib;
''');

    addSource('$testPackageLibPath/b.dart', '''
part of my.lib;
''');

    var library = await buildLibrary('''
library my.lib;

@foo
part 'a.dart';

@foo
part 'b.dart';

const foo = 0;
''');
    checkElementText(library, r'''
library
  name: my.lib
  nameOffset: 8
  definingUnit
    topLevelVariables
      static const foo @65
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @71
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
  parts
    package:test/a.dart
      metadata
        Annotation
          atSign: @ @17
          name: SimpleIdentifier
            token: foo @18
            staticElement: self::@getter::foo
            staticType: null
          element: self::@getter::foo
    package:test/b.dart
      metadata
        Annotation
          atSign: @ @38
          name: SimpleIdentifier
            token: foo @39
            staticElement: self::@getter::foo
            staticType: null
          element: self::@getter::foo
''');
  }

  test_metadata_offsets_onUnitFunction() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
void f<@foo T>({@foo int? a = 42}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
    functions
      f @26
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        typeParameters
          covariant T @33
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @28
                name: SimpleIdentifier
                  token: foo @29
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        parameters
          optionalNamed default a @47
            type: int?
            metadata
              Annotation
                atSign: @ @37
                name: SimpleIdentifier
                  token: foo @38
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
            constantInitializer
              IntegerLiteral
                literal: 42 @51
                staticType: int
        returnType: void
''');
  }

  test_metadata_offsets_onUnitGetter() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
int get getter => 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
      synthetic static getter @-1
        type: int
    accessors
      synthetic static get foo @-1
        returnType: int
      static get getter @29
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        returnType: int
''');
  }

  test_metadata_offsets_onUnitSetter() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
set setter(@foo int a) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
      synthetic static setter @-1
        type: int
    accessors
      synthetic static get foo @-1
        returnType: int
      static set setter @25
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        parameters
          requiredPositional a @41
            type: int
            metadata
              Annotation
                atSign: @ @32
                name: SimpleIdentifier
                  token: foo @33
                  staticElement: self::@getter::foo
                  staticType: null
                element: self::@getter::foo
        returnType: void
''');
  }

  test_metadata_offsets_onUnitVariable() async {
    var library = await buildLibrary(r'''
const foo = 0;

@foo
var isNotConst = 1;

@foo
const isConst = 2;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const foo @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @12
            staticType: int
      static isNotConst @25
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: foo @17
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        type: int
        shouldUseTypeForInitializerInference: false
      static const isConst @53
        metadata
          Annotation
            atSign: @ @42
            name: SimpleIdentifier
              token: foo @43
              staticElement: self::@getter::foo
              staticType: null
            element: self::@getter::foo
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 2 @63
            staticType: int
    accessors
      synthetic static get foo @-1
        returnType: int
      synthetic static get isNotConst @-1
        returnType: int
      synthetic static set isNotConst @-1
        parameters
          requiredPositional _isNotConst @-1
            type: int
        returnType: void
      synthetic static get isConst @-1
        returnType: int
''');
  }

  test_metadata_partDirective() async {
    addSource('$testPackageLibPath/foo.dart', 'part of L;');
    var library = await buildLibrary('''
library L;
@a
part 'foo.dart';
const a = null;''');
    checkElementText(library, r'''
library
  name: L
  nameOffset: 8
  definingUnit
    topLevelVariables
      static const a @37
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @41
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
  parts
    package:test/foo.dart
      metadata
        Annotation
          atSign: @ @11
          name: SimpleIdentifier
            token: a @12
            staticElement: self::@getter::a
            staticType: null
          element: self::@getter::a
''');
  }

  test_metadata_partDirective2() async {
    addSource('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
''');
    addSource('$testPackageLibPath/b.dart', r'''
part of 'test.dart';
''');
    var library = await buildLibrary('''
part 'a.dart';
part 'b.dart';
''');

    // The difference with the test above is that we ask the part first.
    // There was a bug that we were not loading library directives.
    expect(library.parts[0].metadata, isEmpty);
  }

  test_metadata_prefixed_variable() async {
    addSource('$testPackageLibPath/a.dart', 'const b = null;');
    var library = await buildLibrary('import "a.dart" as a; @a.b class C {}');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart as a @19
  definingUnit
    classes
      class C @33
        metadata
          Annotation
            atSign: @ @22
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: a @23
                staticElement: self::@prefix::a
                staticType: null
              period: . @24
              identifier: SimpleIdentifier
                token: b @25
                staticElement: package:test/a.dart::@getter::b
                staticType: null
              staticElement: package:test/a.dart::@getter::b
              staticType: null
            element: package:test/a.dart::@getter::b
        constructors
          synthetic @-1
''');
  }

  test_metadata_simpleFormalParameter() async {
    var library = await buildLibrary('const a = null; f(@a x) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @16
        parameters
          requiredPositional x @21
            type: dynamic
            metadata
              Annotation
                atSign: @ @18
                name: SimpleIdentifier
                  token: a @19
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        returnType: dynamic
''');
  }

  test_metadata_simpleFormalParameter_method() async {
    var library = await buildLibrary('''
const a = null;

class C {
  m(@a x) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @23
        constructors
          synthetic @-1
        methods
          m @29
            parameters
              requiredPositional x @34
                type: dynamic
                metadata
                  Annotation
                    atSign: @ @31
                    name: SimpleIdentifier
                      token: a @32
                      staticElement: self::@getter::a
                      staticType: null
                    element: self::@getter::a
            returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_simpleFormalParameter_unit_setter() async {
    var library = await buildLibrary('''
const a = null;

set foo(@a int x) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      synthetic static foo @-1
        type: int
    accessors
      synthetic static get a @-1
        returnType: dynamic
      static set foo @21
        parameters
          requiredPositional x @32
            type: int
            metadata
              Annotation
                atSign: @ @25
                name: SimpleIdentifier
                  token: a @26
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        returnType: void
''');
  }

  test_metadata_simpleFormalParameter_withDefault() async {
    var library = await buildLibrary('const a = null; f([@a x = null]) {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @16
        parameters
          optionalPositional default x @22
            type: dynamic
            metadata
              Annotation
                atSign: @ @19
                name: SimpleIdentifier
                  token: a @20
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
            constantInitializer
              NullLiteral
                literal: null @26
                staticType: Null
        returnType: dynamic
''');
  }

  test_metadata_superFormalParameter() async {
    var library = await buildLibrary('''
const a = null;

class A {
  A(int x);
}

class B extends A {
  B(@a super.x);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @23
        constructors
          @29
            parameters
              requiredPositional x @35
                type: int
      class B @48
        supertype: A
        constructors
          @64
            parameters
              requiredPositional final super.x @75
                type: int
                metadata
                  Annotation
                    atSign: @ @66
                    name: SimpleIdentifier
                      token: a @67
                      staticElement: self::@getter::a
                      staticType: null
                    element: self::@getter::a
                superConstructorParameter: self::@class::A::@constructor::new::@parameter::x
            superConstructor: self::@class::A::@constructor::new
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_topLevelVariableDeclaration() async {
    var library = await buildLibrary('const a = null; @a int v;');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
      static v @23
        metadata
          Annotation
            atSign: @ @16
            name: SimpleIdentifier
              token: a @17
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        type: int
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get a @-1
        returnType: dynamic
      synthetic static get v @-1
        returnType: int
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: int
        returnType: void
''');
  }

  test_metadata_typeParameter_ofClass() async {
    var library = await buildLibrary('const a = null; class C<@a T> {}');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @22
        typeParameters
          covariant T @27
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @24
                name: SimpleIdentifier
                  token: a @25
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        constructors
          synthetic @-1
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_typeParameter_ofClassTypeAlias() async {
    var library = await buildLibrary('''
const a = null;
class C<@a T> = D with E;
class D {}
class E {}''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class alias C @22
        typeParameters
          covariant T @27
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @24
                name: SimpleIdentifier
                  token: a @25
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        supertype: D
        mixins
          E
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::D::@constructor::new
            superConstructor: self::@class::D::@constructor::new
      class D @48
        constructors
          synthetic @-1
      class E @59
        constructors
          synthetic @-1
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_typeParameter_ofFunction() async {
    var library = await buildLibrary('const a = null; f<@a T>() {}');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
    functions
      f @16
        typeParameters
          covariant T @21
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @18
                name: SimpleIdentifier
                  token: a @19
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        returnType: dynamic
''');
  }

  test_metadata_typeParameter_ofTypedef() async {
    var library = await buildLibrary('const a = null; typedef F<@a T>();');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @24
        typeParameters
          unrelated T @29
            defaultType: dynamic
            metadata
              Annotation
                atSign: @ @26
                name: SimpleIdentifier
                  token: a @27
                  staticElement: self::@getter::a
                  staticType: null
                element: self::@getter::a
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
    topLevelVariables
      static const a @6
        type: dynamic
        shouldUseTypeForInitializerInference: false
        constantInitializer
          NullLiteral
            literal: null @10
            staticType: Null
    accessors
      synthetic static get a @-1
        returnType: dynamic
''');
  }

  test_metadata_unit_topLevelVariable_first() async {
    var library = await buildLibrary(r'''
const a = 0;
@a
int x = 0;
''');
    // Check metadata without asking any other properties.
    var x = _elementOfDefiningUnit(library, ['@variable', 'x'])
        as TopLevelVariableElement;
    expect(x.metadata, hasLength(1));
    // Check details.
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static const a @6
        type: int
        shouldUseTypeForInitializerInference: false
        constantInitializer
          IntegerLiteral
            literal: 0 @10
            staticType: int
      static x @20
        metadata
          Annotation
            atSign: @ @13
            name: SimpleIdentifier
              token: a @14
              staticElement: self::@getter::a
              staticType: null
            element: self::@getter::a
        type: int
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get a @-1
        returnType: int
      synthetic static get x @-1
        returnType: int
      synthetic static set x @-1
        parameters
          requiredPositional _x @-1
            type: int
        returnType: void
''');
  }

  test_metadata_value_class_staticField() async {
    var library = await buildLibrary('''
class A {
  static const x = 0;
}
@A.x
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          static const x @25
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 0 @29
                staticType: int
        constructors
          synthetic @-1
        accessors
          synthetic static get x @-1
            returnType: int
      class C @45
        metadata
          Annotation
            atSign: @ @34
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: A @35
                staticElement: self::@class::A
                staticType: null
              period: . @36
              identifier: SimpleIdentifier
                token: x @37
                staticElement: self::@class::A::@getter::x
                staticType: null
              staticElement: self::@class::A::@getter::x
              staticType: null
            element: self::@class::A::@getter::x
        constructors
          synthetic @-1
''');
  }

  test_metadata_value_enum_constant() async {
    var library = await buildLibrary('''
enum E {a, b, c}
@E.b
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @28
        metadata
          Annotation
            atSign: @ @17
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: E @18
                staticElement: self::@enum::E
                staticType: null
              period: . @19
              identifier: SimpleIdentifier
                token: b @20
                staticElement: self::@enum::E::@getter::b
                staticType: null
              staticElement: self::@enum::E::@getter::b
              staticType: null
            element: self::@enum::E::@getter::b
        constructors
          synthetic @-1
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @8
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant b @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          static const enumConstant c @14
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                  SimpleIdentifier
                    token: c @-1
                    staticElement: self::@enum::E::@getter::c
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
        constructors
          synthetic const @-1
        accessors
          synthetic static get a @-1
            returnType: E
          synthetic static get b @-1
            returnType: E
          synthetic static get c @-1
            returnType: E
          synthetic static get values @-1
            returnType: List<E>
''');
  }

  test_metadata_value_extension_staticField() async {
    var library = await buildLibrary('''
extension E on int {
  static const x = 0;
}
@E.x
class C {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @56
        metadata
          Annotation
            atSign: @ @45
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: E @46
                staticElement: self::@extension::E
                staticType: null
              period: . @47
              identifier: SimpleIdentifier
                token: x @48
                staticElement: self::@extension::E::@getter::x
                staticType: null
              staticElement: self::@extension::E::@getter::x
              staticType: null
            element: self::@extension::E::@getter::x
        constructors
          synthetic @-1
    extensions
      E @10
        extendedType: int
        fields
          static const x @36
            type: int
            shouldUseTypeForInitializerInference: false
            constantInitializer
              IntegerLiteral
                literal: 0 @40
                staticType: int
        accessors
          synthetic static get x @-1
            returnType: int
''');
  }

  test_metadata_value_prefix_extension_staticField() async {
    addSource('$testPackageLibPath/foo.dart', '''
extension E on int {
  static const x = 0;
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
@foo.E.x
class C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/foo.dart as foo @21
  definingUnit
    classes
      class C @41
        metadata
          Annotation
            atSign: @ @26
            name: PrefixedIdentifier
              prefix: SimpleIdentifier
                token: foo @27
                staticElement: self::@prefix::foo
                staticType: null
              period: . @30
              identifier: SimpleIdentifier
                token: E @31
                staticElement: package:test/foo.dart::@extension::E
                staticType: null
              staticElement: package:test/foo.dart::@extension::E
              staticType: null
            period: . @32
            constructorName: SimpleIdentifier
              token: x @33
              staticElement: package:test/foo.dart::@extension::E::@getter::x
              staticType: null
            element: package:test/foo.dart::@extension::E::@getter::x
        constructors
          synthetic @-1
''');
  }

  test_methodInvocation_implicitCall() async {
    var library = await buildLibrary(r'''
class A {
  double call() => 0.0;
}
class B {
  A a;
}
var c = new B().a();
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          call @19
            returnType: double
      class B @42
        fields
          a @50
            type: A
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get a @-1
            returnType: A
          synthetic set a @-1
            parameters
              requiredPositional _a @-1
                type: A
            returnType: void
    topLevelVariables
      static c @59
        type: double
        shouldUseTypeForInitializerInference: false
    accessors
      synthetic static get c @-1
        returnType: double
      synthetic static set c @-1
        parameters
          requiredPositional _c @-1
            type: double
        returnType: void
''');
  }

  test_mixin() async {
    var library = await buildLibrary(r'''
class A {}
class B {}
class C {}
class D {}

mixin M<T extends num, U> on A, B implements C, D {
  T f;
  U get g => 0;
  set s(int v) {}
  int m(double v) => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
      class B @17
        constructors
          synthetic @-1
      class C @28
        constructors
          synthetic @-1
      class D @39
        constructors
          synthetic @-1
    mixins
      mixin M @51
        typeParameters
          covariant T @53
            bound: num
            defaultType: num
          covariant U @68
            defaultType: dynamic
        superclassConstraints
          A
          B
        interfaces
          C
          D
        fields
          f @101
            type: T
            shouldUseTypeForInitializerInference: true
          synthetic g @-1
            type: U
          synthetic s @-1
            type: int
        accessors
          synthetic get f @-1
            returnType: T
          synthetic set f @-1
            parameters
              requiredPositional _f @-1
                type: T
            returnType: void
          get g @112
            returnType: U
          set s @126
            parameters
              requiredPositional v @132
                type: int
            returnType: void
        methods
          m @144
            parameters
              requiredPositional v @153
                type: double
            returnType: int
''');
  }

  test_mixin_base() async {
    var library = await buildLibrary(r'''
base mixin M on A {}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      base mixin M @11
        superclassConstraints
          Object
''');
  }

  test_mixin_field_inferredType_final() async {
    var library = await buildLibrary('''
mixin M {
  final x = 0;
}''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          final x @18
            type: int
            shouldUseTypeForInitializerInference: false
        accessors
          synthetic get x @-1
            returnType: int
''');
  }

  test_mixin_first() async {
    var library = await buildLibrary(r'''
mixin M {}
''');

    // We intentionally ask `mixins` directly, to check that we can ask them
    // separately, without asking classes.
    var mixins = library.definingCompilationUnit.mixins;
    expect(mixins, hasLength(1));
    expect(mixins[0].name, 'M');
  }

  test_mixin_getter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
mixin M on A {
  int get foo {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
        accessors
          get foo @25 invokesSuperSelf
            returnType: int
''');
  }

  test_mixin_getter_invokesSuperSelf_getter_nestedInAssignment() async {
    var library = await buildLibrary(r'''
mixin M on A {
  int get foo {
    (super.foo).foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
        accessors
          get foo @25 invokesSuperSelf
            returnType: int
''');
  }

  test_mixin_getter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
mixin M on A {
  int get foo {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
        accessors
          get foo @25
            returnType: int
''');
  }

  test_mixin_implicitObjectSuperclassConstraint() async {
    var library = await buildLibrary(r'''
mixin M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
''');
  }

  test_mixin_inference_classAlias_oneMixin() async {
    // In the code below, B's superclass constraints don't include A, because
    // superclass constraints are determined from the mixin's superclass, and
    // B's superclass is Object.  So no mixin type inference is attempted, and
    // "with B" is interpreted as "with B<dynamic>".
    var library = await buildLibrary(r'''
class A<T> {}
class B<T> = Object with A<T>;
class C = A<int> with B;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class alias B @20
        typeParameters
          covariant T @22
            defaultType: dynamic
        supertype: Object
        mixins
          A<T>
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class alias C @51
        supertype: A<int>
        mixins
          B<dynamic>
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::A::@constructor::new
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int}
''');
  }

  test_mixin_inference_classAlias_twoMixins() async {
    // In the code below, `B` has a single superclass constraint, A1, because
    // superclass constraints are determined from the mixin's superclass, and
    // B's superclass is "Object with A1<T>".  So mixin type inference succeeds
    // (since C's base class implements A1<int>), and "with B" is interpreted as
    // "with B<int>".
    var library = await buildLibrary(r'''
class A1<T> {}
class A2<T> {}
class B<T> = Object with A1<T>, A2<T>;
class Base implements A1<int> {}
class C = Base with B;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A1 @6
        typeParameters
          covariant T @9
            defaultType: dynamic
        constructors
          synthetic @-1
      class A2 @21
        typeParameters
          covariant T @24
            defaultType: dynamic
        constructors
          synthetic @-1
      class alias B @36
        typeParameters
          covariant T @38
            defaultType: dynamic
        supertype: Object
        mixins
          A1<T>
          A2<T>
        constructors
          synthetic const @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: dart:core::@class::Object::@constructor::new
      class Base @75
        interfaces
          A1<int>
        constructors
          synthetic @-1
      class alias C @108
        supertype: Base
        mixins
          B<int>
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::Base::@constructor::new
            superConstructor: self::@class::Base::@constructor::new
''');
  }

  test_mixin_inference_legacy() async {
    var library = await buildLibrary(r'''
// @dart = 2.9
class A<T> {}
mixin M<U> on A<U> {}
class B extends A<int> with M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @21
        typeParameters
          covariant T @23
            defaultType: dynamic
        constructors
          synthetic @-1
      class B @57
        supertype: A<int*>*
        mixins
          M<int*>*
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int*}
    mixins
      mixin M @35
        typeParameters
          covariant U @37
            defaultType: dynamic
        superclassConstraints
          A<U*>*
''');
  }

  test_mixin_inference_nested_functionType() async {
    var library = await buildLibrary(r'''
class A<T> {}
mixin M<T, U> on A<T Function(U)> {}
class C extends A<int Function(String)> with M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class C @57
        supertype: A<int Function(String)>
        mixins
          M<int, String>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int Function(String)}
    mixins
      mixin M @20
        typeParameters
          covariant T @22
            defaultType: dynamic
          covariant U @25
            defaultType: dynamic
        superclassConstraints
          A<T Function(U)>
''');
  }

  test_mixin_inference_nested_interfaceType() async {
    var library = await buildLibrary(r'''
abstract class A<T> {}
mixin M<T> on A<List<T>> {}
class C extends A<List<int>> with M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      abstract class A @15
        typeParameters
          covariant T @17
            defaultType: dynamic
        constructors
          synthetic @-1
      class C @57
        supertype: A<List<int>>
        mixins
          M<int>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: List<int>}
    mixins
      mixin M @29
        typeParameters
          covariant T @31
            defaultType: dynamic
        superclassConstraints
          A<List<T>>
''');
  }

  test_mixin_inference_nullSafety() async {
    var library = await buildLibrary(r'''
class A<T> {}
mixin M<U> on A<U> {}
class B extends A<int> with M {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class B @42
        supertype: A<int>
        mixins
          M<int>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: int}
    mixins
      mixin M @20
        typeParameters
          covariant U @22
            defaultType: dynamic
        superclassConstraints
          A<U>
''');
  }

  test_mixin_inference_nullSafety2() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A<T> {}

mixin B<T> on A<T> {}
mixin C<T> on A<T> {}
''');
    var library = await buildLibrary(r'''
// @dart=2.8
import 'a.dart';

class D extends A<int> with B<int>, C {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class D @37
        supertype: A<int*>*
        mixins
          B<int*>*
          C<int*>*
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: package:test/a.dart::@class::A::@constructor::new
              substitution: {T: int*}
''');
  }

  test_mixin_inference_nullSafety_mixed_inOrder() async {
    addSource('$testPackageLibPath/a.dart', r'''
class A<T> {}
mixin M<U> on A<U> {}
''');
    var library = await buildLibrary(r'''
// @dart = 2.8
import 'a.dart';
class B extends A<int> with M {}
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
  definingUnit
    classes
      class B @38
        supertype: A<int*>*
        mixins
          M<int*>*
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: package:test/a.dart::@class::A::@constructor::new
              substitution: {T: int*}
''');
  }

  @FailingTest(reason: 'Out-of-order inference is not specified yet')
  test_mixin_inference_nullSafety_mixed_outOfOrder() async {
    addSource('$testPackageLibPath/a.dart', r'''
// @dart = 2.8
class A<T> {}
mixin M<U> on A<U> {}
''');
    var library = await buildLibrary(r'''
import 'a.dart';

class B extends A<int> with M {}
''');
    checkElementText(library, r'''
import 'a.dart';
class B extends A<int> with M<int> {
  synthetic B();
}
''');
  }

  test_mixin_inference_twoMixins() async {
    // Both `M1` and `M2` have their type arguments inferred.
    var library = await buildLibrary(r'''
class I<X> {}
mixin M1<T> on I<T> {}
mixin M2<T> on I<T> {}
class A = I<int> with M1, M2;
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class I @6
        typeParameters
          covariant X @8
            defaultType: dynamic
        constructors
          synthetic @-1
      class alias A @66
        supertype: I<int>
        mixins
          M1<int>
          M2<int>
        constructors
          synthetic @-1
            constantInitializers
              SuperConstructorInvocation
                superKeyword: super @0
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticElement: self::@class::I::@constructor::new
            superConstructor: ConstructorMember
              base: self::@class::I::@constructor::new
              substitution: {X: int}
    mixins
      mixin M1 @20
        typeParameters
          covariant T @23
            defaultType: dynamic
        superclassConstraints
          I<T>
      mixin M2 @43
        typeParameters
          covariant T @46
            defaultType: dynamic
        superclassConstraints
          I<T>
''');
  }

  test_mixin_inference_viaTypeAlias() async {
    var library = await buildLibrary(r'''
mixin M<T, U> on S<T> {}

typedef M2<T2> = M<T2, int>;

class S<T3> {}

class X extends S<String> with M2 {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class S @62
        typeParameters
          covariant T3 @64
            defaultType: dynamic
        constructors
          synthetic @-1
      class X @78
        supertype: S<String>
        mixins
          M<String, int>
            alias: self::@typeAlias::M2
              typeArguments
                String
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::S::@constructor::new
              substitution: {T3: String}
    mixins
      mixin M @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        superclassConstraints
          S<T>
    typeAliases
      M2 @34
        typeParameters
          covariant T2 @37
            defaultType: dynamic
        aliasedType: M<T2, int>
''');
  }

  test_mixin_inference_viaTypeAlias2() async {
    var library = await buildLibrary(r'''
mixin M<T, U> on S<T> {}

typedef M2<T2> = M<T2, int>;

typedef M3<T3> = M2<T3>;

class S<T4> {}

class X extends S<String> with M3 {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class S @88
        typeParameters
          covariant T4 @90
            defaultType: dynamic
        constructors
          synthetic @-1
      class X @104
        supertype: S<String>
        mixins
          M<String, int>
            alias: self::@typeAlias::M3
              typeArguments
                String
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::S::@constructor::new
              substitution: {T4: String}
    mixins
      mixin M @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        superclassConstraints
          S<T>
    typeAliases
      M2 @34
        typeParameters
          covariant T2 @37
            defaultType: dynamic
        aliasedType: M<T2, int>
      M3 @64
        typeParameters
          covariant T3 @67
            defaultType: dynamic
        aliasedType: M<T3, int>
          alias: self::@typeAlias::M2
            typeArguments
              T3
''');
  }

  test_mixin_method_invokesSuperSelf() async {
    var library = await buildLibrary(r'''
mixin M on A {
  void foo() {
    super.foo();
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        methods
          foo @22 invokesSuperSelf
            returnType: void
''');
  }

  test_mixin_method_namedAsConstraint() async {
    var library = await buildLibrary(r'''
class A {}
mixin B on A {
  void A() {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
    mixins
      mixin B @17
        superclassConstraints
          A
        methods
          A @33
            returnType: void
''');
  }

  test_mixin_setter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
mixin M on A {
  set foo(int _) {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
        accessors
          set foo @21
            parameters
              requiredPositional _ @29
                type: int
            returnType: void
''');
  }

  test_mixin_setter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
mixin M on A {
  set foo(int _) {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
        accessors
          set foo @21 invokesSuperSelf
            parameters
              requiredPositional _ @29
                type: int
            returnType: void
''');
  }

  test_mixin_typeParameters_variance_contravariant() async {
    var library = await buildLibrary('mixin M<in T> {}');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        typeParameters
          contravariant T @11
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_mixin_typeParameters_variance_covariant() async {
    var library = await buildLibrary('mixin M<out T> {}');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        typeParameters
          covariant T @12
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_mixin_typeParameters_variance_invariant() async {
    var library = await buildLibrary('mixin M<inout T> {}');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        typeParameters
          invariant T @14
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_mixin_typeParameters_variance_multiple() async {
    var library = await buildLibrary('mixin M<inout T, in U, out V> {}');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        typeParameters
          invariant T @14
            defaultType: dynamic
          contravariant U @20
            defaultType: dynamic
          covariant V @27
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_nameConflict_exportedAndLocal() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    addSource('$testPackageLibPath/c.dart', '''
export 'a.dart';
class C {}
''');
    var library = await buildLibrary('''
import 'c.dart';
C v = null;
''');
    checkElementText(library, r'''
library
  imports
    package:test/c.dart
  definingUnit
    topLevelVariables
      static v @19
        type: C
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get v @-1
        returnType: C
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: C
        returnType: void
''');
  }

  test_nameConflict_exportedAndLocal_exported() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    addSource('$testPackageLibPath/c.dart', '''
export 'a.dart';
class C {}
''');
    addSource('$testPackageLibPath/d.dart', 'export "c.dart";');
    var library = await buildLibrary('''
import 'd.dart';
C v = null;
''');
    checkElementText(library, r'''
library
  imports
    package:test/d.dart
  definingUnit
    topLevelVariables
      static v @19
        type: C
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get v @-1
        returnType: C
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: C
        returnType: void
''');
  }

  test_nameConflict_exportedAndParted() async {
    addSource('$testPackageLibPath/a.dart', 'class C {}');
    addSource('$testPackageLibPath/b.dart', '''
part of lib;
class C {}
''');
    addSource('$testPackageLibPath/c.dart', '''
library lib;
export 'a.dart';
part 'b.dart';
''');
    var library = await buildLibrary('''
import 'c.dart';
C v = null;
''');
    checkElementText(library, r'''
library
  imports
    package:test/c.dart
  definingUnit
    topLevelVariables
      static v @19
        type: C
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get v @-1
        returnType: C
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: C
        returnType: void
''');
  }

  test_nameConflict_importWithRelativeUri_exportWithAbsolute() async {
    if (resourceProvider.pathContext.separator != '/') {
      return;
    }

    addSource('$testPackageLibPath/a.dart', 'class A {}');
    addSource('$testPackageLibPath/b.dart', 'export "/a.dart";');
    var library = await buildLibrary('''
import 'a.dart';
import 'b.dart';
A v = null;
''');
    checkElementText(library, r'''
library
  imports
    package:test/a.dart
    package:test/b.dart
  definingUnit
    topLevelVariables
      static v @36
        type: A
        shouldUseTypeForInitializerInference: true
    accessors
      synthetic static get v @-1
        returnType: A
      synthetic static set v @-1
        parameters
          requiredPositional _v @-1
            type: A
        returnType: void
''');
  }

  test_nameOffset_class_constructor() async {
    var library = await buildLibrary(r'''
class A {
  A();
  A.named();
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
          named @21
            periodOffset: 20
            nameEnd: 26
''');
  }

  test_nameOffset_class_constructor_parameter() async {
    var library = await buildLibrary(r'''
class A {
  A(int a);
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          @12
            parameters
              requiredPositional a @18
                type: int
''');
  }

  test_nameOffset_class_field() async {
    var library = await buildLibrary(r'''
class A {
  int foo = 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
        constructors
          synthetic @-1
        accessors
          synthetic get foo @-1
            returnType: int
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
            returnType: void
''');
  }

  test_nameOffset_class_getter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo => 0;
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          get foo @20
            returnType: int
''');
  }

  test_nameOffset_class_method() async {
    var library = await buildLibrary(r'''
class A {
  void foo<T>(int a) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          foo @17
            typeParameters
              covariant T @21
                defaultType: dynamic
            parameters
              requiredPositional a @28
                type: int
            returnType: void
''');
  }

  test_nameOffset_class_setter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int x) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        fields
          synthetic foo @-1
            type: int
        constructors
          synthetic @-1
        accessors
          set foo @16
            parameters
              requiredPositional x @24
                type: int
            returnType: void
''');
  }

  test_nameOffset_class_typeParameter() async {
    var library = await buildLibrary(r'''
class A<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
''');
  }

  test_nameOffset_extension_typeParameter() async {
    var library = await buildLibrary(r'''
extension E<T> on int {}
''');
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        typeParameters
          covariant T @12
            defaultType: dynamic
        extendedType: int
''');
  }

  test_nameOffset_function_functionTypedFormal_parameter() async {
    var library = await buildLibrary(r'''
void f(void f<U>(int a)) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredPositional f @12
            type: void Function<U>(int)
            typeParameters
              covariant U @14
            parameters
              requiredPositional a @21
                type: int
        returnType: void
''');
  }

  test_nameOffset_function_functionTypedFormal_parameter2() async {
    var library = await buildLibrary(r'''
void f({required void f<U>(int a)}) {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        parameters
          requiredNamed default f @22
            type: void Function<U>(int)
            typeParameters
              covariant U @24
            parameters
              requiredPositional a @31
                type: int
        returnType: void
''');
  }

  test_nameOffset_function_typeParameter() async {
    var library = await buildLibrary(r'''
void f<T>() {}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
        returnType: void
''');
  }

  test_nameOffset_functionTypeAlias_typeParameter() async {
    var library = await buildLibrary(r'''
typedef void F<T>();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      functionTypeAliasBased F @13
        typeParameters
          unrelated T @15
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_nameOffset_genericTypeAlias_typeParameter() async {
    var library = await buildLibrary(r'''
typedef F<T> = void Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @8
        typeParameters
          unrelated T @10
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_nameOffset_mixin_typeParameter() async {
    var library = await buildLibrary(r'''
mixin M<T> {}
''');
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        superclassConstraints
          Object
''');
  }

  test_nameOffset_unit_getter() async {
    var library = await buildLibrary(r'''
int get foo => 0;
''');
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: int
    accessors
      static get foo @8
        returnType: int
''');
  }

  test_nested_generic_functions_in_generic_class_with_function_typed_params() async {
    var library = await buildLibrary('''
class C<T, U> {
  void g<V, W>() {
    void h<X, Y>(void p(T t, U u, V v, W w, X x, Y y)) {
    }
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          g @23
            typeParameters
              covariant V @25
                defaultType: dynamic
              covariant W @28
                defaultType: dynamic
            returnType: void
''');
  }

  test_nested_generic_functions_in_generic_class_with_local_variables() async {
    var library = await buildLibrary('''
class C<T, U> {
  void g<V, W>() {
    void h<X, Y>() {
      T t;
      U u;
      V v;
      W w;
      X x;
      Y y;
    }
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        typeParameters
          covariant T @8
            defaultType: dynamic
          covariant U @11
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          g @23
            typeParameters
              covariant V @25
                defaultType: dynamic
              covariant W @28
                defaultType: dynamic
            returnType: void
''');
  }

  test_nested_generic_functions_with_function_typed_param() async {
    var library = await buildLibrary('''
void f<T, U>() {
  void g<V, W>() {
    void h<X, Y>(void p(T t, U u, V v, W w, X x, Y y)) {
    }
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
          covariant U @10
            defaultType: dynamic
        returnType: void
''');
  }

  test_nested_generic_functions_with_local_variables() async {
    var library = await buildLibrary('''
void f<T, U>() {
  void g<V, W>() {
    void h<X, Y>() {
      T t;
      U u;
      V v;
      W w;
      X x;
      Y y;
    }
  }
}
''');
    checkElementText(library, r'''
library
  definingUnit
    functions
      f @5
        typeParameters
          covariant T @7
            defaultType: dynamic
          covariant U @10
            defaultType: dynamic
        returnType: void
''');
  }

  test_new_typedef_function_notSimplyBounded_functionType_returnType() async {
    var library = await buildLibrary('''
typedef F = G Function();
typedef G = F Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
      notSimplyBounded G @34
        aliasedType: dynamic Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: dynamic
''');
  }

  test_new_typedef_function_notSimplyBounded_functionType_returnType_viaInterfaceType() async {
    var library = await buildLibrary('''
typedef F = List<F> Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        aliasedType: List<dynamic> Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: List<dynamic>
''');
  }

  test_new_typedef_function_notSimplyBounded_self() async {
    var library = await buildLibrary('''
typedef F<T extends F> = void Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        typeParameters
          unrelated T @10
            bound: dynamic
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_new_typedef_function_notSimplyBounded_simple_no_bounds() async {
    var library = await buildLibrary('''
typedef F<T> = void Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @8
        typeParameters
          unrelated T @10
            defaultType: dynamic
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_new_typedef_function_notSimplyBounded_simple_non_generic() async {
    var library = await buildLibrary('''
typedef F = void Function();
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      F @8
        aliasedType: void Function()
        aliasedElement: GenericFunctionTypeElement
          returnType: void
''');
  }

  test_new_typedef_nonFunction_notSimplyBounded_self() async {
    var library = await buildLibrary('''
typedef F<T extends F> = List<int>;
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        typeParameters
          unrelated T @10
            bound: dynamic
            defaultType: dynamic
        aliasedType: List<int>
''');
  }

  test_new_typedef_nonFunction_notSimplyBounded_viaInterfaceType() async {
    var library = await buildLibrary('''
typedef F = List<F>;
''');
    checkElementText(library, r'''
library
  definingUnit
    typeAliases
      notSimplyBounded F @8
        aliasedType: List<dynamic>
''');
  }

  test_nonSynthetic_class_field() async {
    var library = await buildLibrary(r'''
class C {
  int foo = 0;
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
            nonSynthetic: self::@class::C::@field::foo
        constructors
          synthetic @-1
            nonSynthetic: self::@class::C
        accessors
          synthetic get foo @-1
            returnType: int
            nonSynthetic: self::@class::C::@field::foo
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
                nonSynthetic: self::@class::C::@field::foo
            returnType: void
            nonSynthetic: self::@class::C::@field::foo
''');
  }

  test_nonSynthetic_class_getter() async {
    var library = await buildLibrary(r'''
class C {
  int get foo => 0;
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@class::C::@getter::foo
        constructors
          synthetic @-1
            nonSynthetic: self::@class::C
        accessors
          get foo @20
            returnType: int
            nonSynthetic: self::@class::C::@getter::foo
''');
  }

  test_nonSynthetic_class_setter() async {
    var library = await buildLibrary(r'''
class C {
  set foo(int value) {}
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    classes
      class C @6
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@class::C::@setter::foo
        constructors
          synthetic @-1
            nonSynthetic: self::@class::C
        accessors
          set foo @16
            parameters
              requiredPositional value @24
                type: int
                nonSynthetic: self::@class::C::@setter::foo::@parameter::value
            returnType: void
            nonSynthetic: self::@class::C::@setter::foo
''');
  }

  test_nonSynthetic_enum() async {
    var library = await buildLibrary(r'''
enum E {
  a, b
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    enums
      enum E @5
        supertype: Enum
        fields
          static const enumConstant a @11
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
            nonSynthetic: self::@enum::E::@field::a
          static const enumConstant b @14
            type: E
            shouldUseTypeForInitializerInference: false
            constantInitializer
              InstanceCreationExpression
                constructorName: ConstructorName
                  type: NamedType
                    name: E @-1
                    element: self::@enum::E
                    type: E
                  staticElement: self::@enum::E::@constructor::new
                argumentList: ArgumentList
                  leftParenthesis: ( @0
                  rightParenthesis: ) @0
                staticType: E
            nonSynthetic: self::@enum::E::@field::b
          synthetic static const values @-1
            type: List<E>
            constantInitializer
              ListLiteral
                leftBracket: [ @0
                elements
                  SimpleIdentifier
                    token: a @-1
                    staticElement: self::@enum::E::@getter::a
                    staticType: E
                  SimpleIdentifier
                    token: b @-1
                    staticElement: self::@enum::E::@getter::b
                    staticType: E
                rightBracket: ] @0
                staticType: List<E>
            nonSynthetic: self::@enum::E
        constructors
          synthetic const @-1
            nonSynthetic: self::@enum::E
        accessors
          synthetic static get a @-1
            returnType: E
            nonSynthetic: self::@enum::E::@field::a
          synthetic static get b @-1
            returnType: E
            nonSynthetic: self::@enum::E::@field::b
          synthetic static get values @-1
            returnType: List<E>
            nonSynthetic: self::@enum::E
''');
  }

  test_nonSynthetic_extension_getter() async {
    var library = await buildLibrary(r'''
extension E on int {
  int get foo => 0;
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        extendedType: int
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@extension::E::@getter::foo
        accessors
          get foo @31
            returnType: int
            nonSynthetic: self::@extension::E::@getter::foo
''');
  }

  test_nonSynthetic_extension_setter() async {
    var library = await buildLibrary(r'''
extension E on int {
  set foo(int value) {}
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    extensions
      E @10
        extendedType: int
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@extension::E::@setter::foo
        accessors
          set foo @27
            parameters
              requiredPositional value @35
                type: int
                nonSynthetic: self::@extension::E::@setter::foo::@parameter::value
            returnType: void
            nonSynthetic: self::@extension::E::@setter::foo
''');
  }

  test_nonSynthetic_mixin_field() async {
    var library = await buildLibrary(r'''
mixin M {
  int foo = 0;
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          foo @16
            type: int
            shouldUseTypeForInitializerInference: true
            nonSynthetic: self::@mixin::M::@field::foo
        accessors
          synthetic get foo @-1
            returnType: int
            nonSynthetic: self::@mixin::M::@field::foo
          synthetic set foo @-1
            parameters
              requiredPositional _foo @-1
                type: int
                nonSynthetic: self::@mixin::M::@field::foo
            returnType: void
            nonSynthetic: self::@mixin::M::@field::foo
''');
  }

  test_nonSynthetic_mixin_getter() async {
    var library = await buildLibrary(r'''
mixin M {
  int get foo => 0;
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@mixin::M::@getter::foo
        accessors
          get foo @20
            returnType: int
            nonSynthetic: self::@mixin::M::@getter::foo
''');
  }

  test_nonSynthetic_mixin_setter() async {
    var library = await buildLibrary(r'''
mixin M {
  set foo(int value) {}
}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    mixins
      mixin M @6
        superclassConstraints
          Object
        fields
          synthetic foo @-1
            type: int
            nonSynthetic: self::@mixin::M::@setter::foo
        accessors
          set foo @16
            parameters
              requiredPositional value @24
                type: int
                nonSynthetic: self::@mixin::M::@setter::foo::@parameter::value
            returnType: void
            nonSynthetic: self::@mixin::M::@setter::foo
''');
  }

  test_nonSynthetic_unit_getter() async {
    var library = await buildLibrary(r'''
int get foo => 0;
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: int
        nonSynthetic: self::@getter::foo
    accessors
      static get foo @8
        returnType: int
        nonSynthetic: self::@getter::foo
''');
  }

  test_nonSynthetic_unit_getterSetter() async {
    var library = await buildLibrary(r'''
int get foo => 0;
set foo(int value) {}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: int
        nonSynthetic: self::@getter::foo
    accessors
      static get foo @8
        returnType: int
        nonSynthetic: self::@getter::foo
      static set foo @22
        parameters
          requiredPositional value @30
            type: int
            nonSynthetic: self::@setter::foo::@parameter::value
        returnType: void
        nonSynthetic: self::@setter::foo
''');
  }

  test_nonSynthetic_unit_setter() async {
    var library = await buildLibrary(r'''
set foo(int value) {}
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      synthetic static foo @-1
        type: int
        nonSynthetic: self::@setter::foo
    accessors
      static set foo @4
        parameters
          requiredPositional value @12
            type: int
            nonSynthetic: self::@setter::foo::@parameter::value
        returnType: void
        nonSynthetic: self::@setter::foo
''');
  }

  test_nonSynthetic_unit_variable() async {
    var library = await buildLibrary(r'''
int foo = 0;
''');
    configuration.withNonSynthetic = true;
    checkElementText(library, r'''
library
  definingUnit
    topLevelVariables
      static foo @4
        type: int
        shouldUseTypeForInitializerInference: true
        nonSynthetic: self::@variable::foo
    accessors
      synthetic static get foo @-1
        returnType: int
        nonSynthetic: self::@variable::foo
      synthetic static set foo @-1
        parameters
          requiredPositional _foo @-1
            type: int
            nonSynthetic: self::@variable::foo
        returnType: void
        nonSynthetic: self::@variable::foo
''');
  }

  test_parameter() async {
    var library = await buildLibrary('void main(int p) {}');
    checkElementText(library, r'''
library
  definingUnit
    functions
      main @5
        parameters
          requiredPositional p @14
            type: int
        returnType: void
''');
  }

  test_parameter_covariant_explicit_named() async {
    var library = await buildLibrary('''
class A {
  void m({covariant A a}) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          m @17
            parameters
              optionalNamed default covariant a @32
                type: A
            returnType: void
''');
  }

  test_parameter_covariant_explicit_positional() async {
    var library = await buildLibrary('''
class A {
  void m([covariant A a]) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          m @17
            parameters
              optionalPositional default covariant a @32
                type: A
            returnType: void
''');
  }

  test_parameter_covariant_explicit_required() async {
    var library = await buildLibrary('''
class A {
  void m(covariant A a) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        constructors
          synthetic @-1
        methods
          m @17
            parameters
              requiredPositional covariant a @31
                type: A
            returnType: void
''');
  }

  test_parameter_covariant_inherited() async {
    var library = await buildLibrary(r'''
class A<T> {
  void f(covariant T t) {}
}
class B<T> extends A<T> {
  void f(T t) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
        typeParameters
          covariant T @8
            defaultType: dynamic
        constructors
          synthetic @-1
        methods
          f @20
            parameters
              requiredPositional covariant t @34
                type: T
            returnType: void
      class B @48
        typeParameters
          covariant T @50
            defaultType: dynamic
        supertype: A<T>
        constructors
          synthetic @-1
            superConstructor: ConstructorMember
              base: self::@class::A::@constructor::new
              substitution: {T: T}
        methods
          f @75
            parameters
              requiredPositional covariant t @79
                type: T
            returnType: void
''');
  }

  test_parameter_covariant_inherited_named() async {
    var library = await buildLibrary('''
class A {
  void m({covariant A a}) {}
}
class B extends A {
  void m({B a}) {}
}
''');
    checkElementText(library, r'''
library
  definingUnit
    classes
      class A @6
       