// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/// @docImport 'package:flutter_driver/flutter_driver.dart';
library;

import 'dart:typed_data';

/// Drives a native device or emulator that is running a Flutter application.
///
/// Unlike [FlutterDriver], a [NativeDriver] is backed by a platform specific
/// implementation that might interact with out-of-process services, such as
/// `adb` for Android or `ios-deploy` for iOS, and might require additional
/// setup (e.g., adding test-only plugins to the application under test) for
/// full functionality.
///
/// API that is available directly on [NativeDriver] is considered _lowest
/// common denominator_ and is guaranteed to work on all platforms supported by
/// Flutter Driver unless otherwise noted. Platform-specific functionality that
/// _cannot_ be exposed through this interface is available through
/// platform-specific extensions.
abstract interface class NativeDriver {
  /// Closes the native driver and releases any resources associated with it.
  ///
  /// After calling this method, the driver is no longer usable.
  Future<void> close();

  /// Take a screenshot using a platform-specific mechanism.
  ///
  /// The image is returned as an opaque handle that can be used to retrieve
  /// the screenshot data or to compare it with another screenshot, and may
  /// include platform-specific system UI elements, such as the status bar or
  /// navigation bar.
  Future<NativeScreenshot> screenshot();
}

/// An opaque handle to a screenshot taken on a native device.
///
/// Unlike [FlutterDriver.screenshot], the screenshot represented by this handle
/// is generated by a platform-specific mechanism and is often already stored
/// on disk. The handle can be used to retrieve the screenshot data or to
/// compare it with another screenshot.
abstract interface class NativeScreenshot {
  /// Saves the screenshot to a file at the specified [path].
  ///
  /// If [path] is not provided, a temporary file will be created.
  ///
  /// Returns the path to the saved file.
  Future<String> saveAs([String? path]);

  /// Reads the screenshot as a PNG-formatted list of bytes.
  Future<Uint8List> readAsBytes();
}
