// Copyright (c) 2023, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

/// A library for JS interop. Includes a JS type hierarchy to facilitate sound
/// interop with JS. The JS type hierarchy is modeled after the actual type
/// hierarchy in JS, and not the Dart type hierarchy.
///
/// Note: The exact semantics of JS types currently depend on their backend
/// implementation. JS backends currently support conflating Dart types and JS
/// types, whereas Wasm backends do not. Over time we will try and unify backend
/// semantics as much as possible.
import 'dart:_js_types' as js_types;
import 'dart:typed_data';

/// Export the `dart:_js_annotations` version of the `@JS` annotation. This is
/// mostly identical to the `package:js` version, except this is meant to be used
/// for sound top-level external members and inline classes instead of the
/// `package:js` classes.
export 'dart:_js_annotations' show JS;

/// The overall top type in the JS types hierarchy.
typedef JSAny = js_types.JSAny;

/// The representation type of all JavaScript objects for inline classes,
/// [JSObject] <: [JSAny].
///
/// This is the supertype of all JS objects, but not other JS types, like
/// primitives. This is the only allowed `on` type for inline classes written by
/// users to model JS interop objects. See https://dart.dev/web/js-interop for
/// more details on how to use JS interop.
/// TODO(srujzs): This class _must_ be sealed before we can make this library
/// public. Either use the CFE mechanisms that exist today, or use the Dart 3
/// sealed classes feature.
/// TODO(joshualitt): Do we need to seal any other JS types on JS backends? We
/// probably want to seal all JS types on Wasm backends.
typedef JSObject = js_types.JSObject;

/// The type of all JS functions, [JSFunction] <: [JSObject].
typedef JSFunction = js_types.JSFunction;

/// The type of all Dart functions adapted to be callable from JS. We only allow
/// a subset of Dart functions to be callable from JS, [JSExportedDartFunction]
/// <: [JSFunction].
/// TODO(joshualitt): Detail exactly what are the requirements.
typedef JSExportedDartFunction = js_types.JSExportedDartFunction;

/// The type of all JS arrays, [JSArray] <: [JSObject].
typedef JSArray = js_types.JSArray;

/// The type of all Dart objects exported to JS. If a Dart type is not
/// explicitly marked with `@JSExport`, then no guarantees are made about its
/// representation in JS. [JSExportedDartObject] <: [JSObject].
typedef JSExportedDartObject = js_types.JSExportedDartObject;

/// The type of JS array buffers, [JSArrayBuffer] <: [JSObject].
typedef JSArrayBuffer = js_types.JSArrayBuffer;

/// The type of JS byte data, [JSDataView] <: [JSObject].
typedef JSDataView = js_types.JSDataView;

/// The abstract supertype of all JS typed arrays, [JSTypedArray] <: [JSObject].
typedef JSTypedArray = js_types.JSTypedArray;

/// The typed arrays themselves, `*Array` <: [JSTypedArray].
typedef JSInt8Array = js_types.JSInt8Array;
typedef JSUint8Array = js_types.JSUint8Array;
typedef JSUint8ClampedArray = js_types.JSUint8ClampedArray;
typedef JSInt16Array = js_types.JSInt16Array;
typedef JSUint16Array = js_types.JSUint16Array;
typedef JSInt32Array = js_types.JSInt32Array;
typedef JSUint32Array = js_types.JSUint32Array;
typedef JSFloat32Array = js_types.JSFloat32Array;
typedef JSFloat64Array = js_types.JSFloat64Array;

/// The various JS primitive types. Crucially, unlike the Dart type hierarchy,
/// none of these are subtypes of [JSObject], but rather they are logically
/// subtypes of [JSAny].

/// The type of JS numbers, [JSNumber] <: [JSAny].
typedef JSNumber = js_types.JSNumber;

/// The type of JS booleans, [JSBoolean] <: [JSAny].
typedef JSBoolean = js_types.JSBoolean;

/// The type of JS strings, [JSString] <: [JSAny].
typedef JSString = js_types.JSString;

/// TODO(joshualitt): Figure out how we want to handle JSUndefined and JSNull.

/// Extension members to support conversions between Dart types and JS types.
/// Not all Dart types can be converted to JS types and vice versa.
/// TODO(joshualitt): We might want to investigate using inline classes instead
/// of extension methods for these methods.

/// [JSExportedDartFunction] <-> [Function]
extension JSExportedDartFunctionToFunction on JSExportedDartFunction {
  external T toDart<T extends Function>();
}

extension FunctionToJSExportedDartFunction on Function {
  external JSExportedDartFunction toJS<T extends Function>();
}

/// [JSExportedDartObject] <-> [Object]
extension JSExportedDartObjectToObject on JSExportedDartObject {
  external T toDart<T>();
}

extension ObjectToJSExportedDartObject on Object {
  external JSExportedDartObject toJS();
}

/// TODO(joshualitt): On Wasm backends List / Array conversion methods will
/// copy, and on JS backends they will not. We should find a path towards
/// consistent semantics.
/// [JSArrayBuffer] <-> [ByteBuffer]
extension JSArrayBufferToByteBuffer on JSArrayBuffer {
  external ByteBuffer toDart();
}

extension ByteBufferToJSArrayBuffer on ByteBuffer {
  external JSArrayBuffer toJS();
}

/// [JSDataView] <-> [ByteData]
extension JSDataViewToByteData on JSDataView {
  external ByteData toDart();
}

extension ByteDataToJSDataView on ByteData {
  external JSDataView toJS();
}

/// [JSInt8Array] <-> [Int8List]
extension JSInt8ArrayToInt8List on JSInt8Array {
  external Int8List toDart();
}

extension Int8ListToJSInt8Array on Int8List {
  external JSInt8Array toJS();
}

/// [JSUint8Array] <-> [Uint8List]
extension JSUint8ArrayToUint8List on JSUint8Array {
  external Uint8List toDart();
}

extension Uint8ListToJSUint8Array on Uint8List {
  external JSUint8Array toJS();
}

/// [JSUint8ClampedArray] <-> [Uint8ClampedList]
extension JSUint8ClampedArrayToUint8ClampedList on JSUint8ClampedArray {
  external Uint8ClampedList toDart();
}

extension Uint8ClampedListToJSUint8ClampedArray on Uint8ClampedList {
  external JSUint8ClampedArray toJS();
}

/// [JSInt16Array] <-> [Int16List]
extension JSInt16ArrayToInt16List on JSInt16Array {
  external Int16List toDart();
}

extension Int16ListToJSInt16Array on Int16List {
  external JSInt16Array toJS();
}

/// [JSUint16Array] <-> [Uint16List]
extension JSUint16ArrayToInt16List on JSUint16Array {
  external Uint16List toDart();
}

extension Uint16ListToJSInt16Array on Uint16List {
  external JSUint16Array toJS();
}

/// [JSInt32Array] <-> [Int32List]
extension JSInt32ArrayToInt32List on JSInt32Array {
  external Int32List toDart();
}

extension Int32ListToJSInt32Array on Int32List {
  external JSInt32Array toJS();
}

/// [JSUint32Array] <-> [Uint32List]
extension JSUint32ArrayToUint32List on JSUint32Array {
  external Uint32List toDart();
}

extension Uint32ListToJSUint32Array on Uint32List {
  external JSUint32Array toJS();
}

/// [JSFloat32Array] <-> [Float32List]
extension JSFloat32ArrayToFloat32List on JSFloat32Array {
  external Float32List toDart();
}

extension Float32ListToJSFloat32Array on Float32List {
  external JSFloat32Array toJS();
}

/// [JSFloat64Array] <-> [Float64List]
extension JSFloat64ArrayToFloat64List on JSFloat64Array {
  external Float64List toDart();
}

extension Float64ListToJSFloat64Array on Float64List {
  external JSFloat64Array toJS();
}

/// [JSArray] <-> [List]
extension JSArrayToList on JSArray {
  external List<JSAny?> toDart();
}

extension ListToJSArray on List<JSAny?> {
  external JSArray toJS();
}

/// [JSNumber] <-> [double].
extension JSNumberToDouble on JSNumber {
  external double toDart();
}

extension DoubleToJSNumber on double {
  external JSNumber toJS();
}

/// [JSBoolean] <-> [bool]
extension JSBooleanToBool on JSBoolean {
  external bool toDart();
}

extension BoolToJSBoolean on bool {
  external JSBoolean toJS();
}

/// [JSString] <-> [String]
extension JSStringToString on JSString {
  external String toDart();
}

extension StringToJSString on String {
  external JSString toJS();
}
