library petitparser.core.actions.trimming;

import 'package:petitparser/src/core/combinators/delegate.dart';
import 'package:petitparser/src/core/contexts/context.dart';
import 'package:petitparser/src/core/contexts/result.dart';
import 'package:petitparser/src/core/parser.dart';

/// A parser that silently consumes input of another parser around
/// its delegate.
class TrimmingParser<T> extends DelegateParser<T> {
  Parser left;
  Parser right;

  TrimmingParser(Parser<T> delegate, this.left, this.right)
      : assert(left != null),
        assert(right != null),
        super(delegate);

  @override
  Result<T> parseOn(Context context) {
    final buffer = context.buffer;
    final before = trim_(left, buffer, context.position);
    final result = delegate.parseOn(
        before == context.position ? context : Context(buffer, before));
    if (result.isFailure) {
      return result;
    }
    final after = trim_(right, buffer, result.position);
    return result.success(result.value, after);
  }

  @override
  int fastParseOn(String buffer, int position) {
    final result = delegate.fastParseOn(buffer, trim_(left, buffer, position));
    return result < 0 ? result : trim_(right, buffer, result);
  }

  int trim_(Parser parser, String buffer, int position) {
    for (;;) {
      final result = parser.fastParseOn(buffer, position);
      if (result < 0) {
        return position;
      }
      position = result;
    }
  }

  @override
  TrimmingParser<T> copy() => TrimmingParser<T>(delegate, left, right);

  @override
  List<Parser> get children => [delegate, left, right];

  @override
  void replace(Parser source, Parser target) {
    super.replace(source, target);
    if (left == source) {
      left = target;
    }
    if (right == source) {
      right = target;
    }
  }
}
