// Copyright (c) 2019, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:analyzer/src/context/packages.dart';
import 'package:analyzer/src/test_utilities/resource_provider_mixin.dart';
import 'package:meta/meta.dart';
import 'package:pub_semver/pub_semver.dart';
import 'package:test/test.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(PackagesTest);
  });
}

@reflectiveTest
class PackagesTest with ResourceProviderMixin {
  void setUp() {
    newFile('/test/lib/test.dart', content: '');
  }

  test_parseDotPackagesFile() {
    var file = newFile('/test/.packages', content: '''
# Generated by pub
aaa:${toUriStr('/packages/aaa/lib/')}
test:lib/
''');
    var packages = parseDotPackagesFile(resourceProvider, file);

    _assertPackage(
      packages,
      name: 'test',
      expectedLibPath: '/test/lib',
      expectedVersion: null,
    );

    _assertPackage(
      packages,
      name: 'aaa',
      expectedLibPath: '/packages/aaa/lib',
      expectedVersion: null,
    );
  }

  test_parsePackageConfigJsonFile() {
    var file = newFile('/test/.dart_tool/package_config.json', content: '''
{
  "configVersion": 2,
  "packages": [
    {
      "name": "test",
      "rootUri": "../",
      "packageUri": "lib/",
      "languageVersion": "2.6"
    },
    {
      "name": "aaa",
      "rootUri": "${toUriStr('/packages/aaa')}",
      "packageUri": "lib/",
      "languageVersion": "2.3"
    },
    {
      "name": "bbb",
      "rootUri": "${toUriStr('/packages/bbb')}",
      "packageUri": "lib/"
    }
  ]
}
''');
    var packages = parsePackageConfigJsonFile(resourceProvider, file);

    _assertPackage(
      packages,
      name: 'test',
      expectedLibPath: '/test/lib',
      expectedVersion: Version(2, 6, 0),
    );

    _assertPackage(
      packages,
      name: 'aaa',
      expectedLibPath: '/packages/aaa/lib',
      expectedVersion: Version(2, 3, 0),
    );

    _assertPackage(
      packages,
      name: 'bbb',
      expectedLibPath: '/packages/bbb/lib',
      expectedVersion: null,
    );
  }

  /// New features were added in `2.2.2` over `2.2.0`.
  /// But `2.2.2` is not representable, so we special case it.
  test_parsePackageConfigJsonFile_version222() {
    var file = newFile('/test/.dart_tool/package_config.json', content: '''
{
  "configVersion": 2,
  "packages": [
    {
      "name": "test",
      "rootUri": "../",
      "packageUri": "lib/",
      "languageVersion": "2.2"
    }
  ]
}
''');
    var packages = parsePackageConfigJsonFile(resourceProvider, file);

    _assertPackage(
      packages,
      name: 'test',
      expectedLibPath: '/test/lib',
      expectedVersion: Version(2, 2, 2),
    );
  }

  void _assertPackage(
    Packages packages, {
    @required String name,
    @required String expectedLibPath,
    @required Version expectedVersion,
  }) {
    var package = packages[name];
    expect(package.name, name);
    expect(package.libFolder.path, convertPath(expectedLibPath));
    expect(package.languageVersion, expectedVersion);
  }
}
