---
title: Diagnostic messages
description: Details for diagnostics produced by the Dart analyzer.
---
{%- comment %}
WARNING: Do NOT EDIT this file directly. It is autogenerated by the script in
`pkg/analyzer/tool/diagnostics/generate.dart` in the sdk repository.
Update instructions: https://github.com/dart-lang/site-www/issues/1949
{% endcomment -%}

This page lists diagnostic messages produced by the Dart analyzer,
with details about what those messages mean and how you can fix your code.
For more information about the analyzer, see
[Customizing static analysis](/guides/language/analysis-options).

## Glossary

This page uses the following terms.

### Constant context

A _constant context_ is a region of code in which it isn't necessary to include
the `const` keyword because it's implied by the fact that everything in that
region is required to be a constant. The following locations are constant
contexts:

* Everything inside a list, map or set literal that's prefixed by the
  `const` keyword. Example:

  ```dart
  var l = const [/*constant context*/];
  ```

* The arguments inside an invocation of a constant constructor. Example:

  ```dart
  var p = const Point(/*constant context*/);
  ```

* The initializer for a variable that's prefixed by the `const` keyword.
  Example:

  ```dart
  const v = /*constant context*/;
  ```

* Annotations

* The expression in a case clause. Example:

  ```dart
  void f(int e) {
    switch (e) {
      case /*constant context*/:
        break;
    }
  }
  ```

## Diagnostics

The analyzer produces the following diagnostics for code that
doesn't conform to the language specification or
that might work in unexpected ways.

### abstract_super_member_reference

_The {0} '{1}' is always abstract in the supertype._

#### Description

The analyzer produces this diagnostic when an inherited member is
referenced using `super`, but there is no concrete implementation of the
member in the superclass chain. Abstract members can't be invoked.

#### Example

The following code produces this diagnostic because `B` doesn't inherit a
concrete implementation of `a`:

{% prettify dart %}
abstract class A {
  int get a;
}
class B extends A {
  int get a => super.[!a!];
}
{% endprettify %}

#### Common fixes

Remove the invocation of the abstract member, possibly replacing it with an
invocation of a concrete member.

### ambiguous_extension_member_access

_A member named '{0}' is defined in extensions '{1}' and '{2}' and neither is
more specific._

#### Description

When code refers to a member of an object (for example, `o.m()` or `o.m` or
`o[i]`) where the static type of `o` doesn't declare the member (`m` or
`[]`, for example), then the analyzer tries to find the member in an
extension. For example, if the member is `m`, then the analyzer looks for
extensions that declare a member named `m` and have an extended type that
the static type of `o` can be assigned to. When there's more than one such
extension in scope, the extension whose extended type is most specific is
selected.

The analyzer produces this diagnostic when none of the extensions has an
extended type that's more specific than the extended types of all of the
other extensions, making the reference to the member ambiguous.

#### Example

The following code produces this diagnostic because there's no way to
choose between the member in `E1` and the member in `E2`:

{% prettify dart %}
extension E1 on String {
  int get charCount => 1;
}

extension E2 on String {
  int get charCount => 2;
}

void f(String s) {
  print(s.[!charCount!]);
}
{% endprettify %}

#### Common fixes

If you don't need both extensions, then you can delete or hide one of them.

If you need both, then explicitly select the one you want to use by using
an extension override:

{% prettify dart %}
extension E1 on String {
  int get charCount => length;
}

extension E2 on String {
  int get charCount => length;
}

void f(String s) {
  print(E2(s).charCount);
}
{% endprettify %}

### ambiguous_import

_The name '{0}' is defined in the libraries {1}._

#### Description

The analyzer produces this diagnostic when a name is referenced that is
declared in two or more imported libraries.

#### Example

Given a library (`a.dart`) that defines a class (`C` in this example):

{% prettify dart %}
class A {}
class C {}
{% endprettify %}

And a library (`b.dart`) that defines a different class with the same name:

{% prettify dart %}
class B {}
class C {}
{% endprettify %}

The following code produces this diagnostic:

{% prettify dart %}
import 'a.dart';
import 'b.dart';

void f([!C!] c1, [!C!] c2) {}
{% endprettify %}

#### Common fixes

If any of the libraries aren't needed, then remove the import directives
for them:

{% prettify dart %}
import 'a.dart';

void f(C c1, C c2) {}
{% endprettify %}

If the name is still defined by more than one library, then add a `hide`
clause to the import directives for all except one library:

{% prettify dart %}
import 'a.dart' hide C;
import 'b.dart';

void f(C c1, C c2) {}
{% endprettify %}

If you must be able to reference more than one of these types, then add a
prefix to each of the import directives, and qualify the references with
the appropriate prefix:

{% prettify dart %}
import 'a.dart' as a;
import 'b.dart' as b;

void f(a.C c1, b.C c2) {}
{% endprettify %}

### ambiguous_set_or_map_literal_both

_This literal contains both 'Map' and 'Iterable' spreads, which makes it
impossible to determine whether the literal is a map or a set._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments and all of the
elements are spread elements (which are allowed in both kinds of literals),
then the analyzer uses the types of the expressions that are being spread.
If all of the expressions have the type `Iterable`, then it's a set
literal; if they all have the type `Map`, then it's a map literal.

The analyzer produces this diagnostic when some of the expressions being
spread have the type `Iterable` and others have the type `Map`, making it
impossible for the analyzer to determine whether you are writing a map
literal or a set literal.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    [!{...a, ...b, ...c}!];
{% endprettify %}

The list `b` can only be spread into a set, and the maps `a` and `c` can
only be spread into a map, and the literal can't be both.

#### Common fixes

There are two common ways to fix this problem. The first is to remove all
of the spread elements of one kind or another, so that the elements are
consistent. In this case, that likely means removing the list and deciding
what to do about the now unused parameter:

{% prettify dart %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, ...c};
{% endprettify %}

The second fix is to change the elements of one kind into elements that are
consistent with the other elements. For example, you can add the elements
of the list as keys that map to themselves:

{% prettify dart %}
union(Map<String, String> a, List<String> b, Map<String, String> c) =>
    {...a, for (String s in b) s: s, ...c};
{% endprettify %}

### ambiguous_set_or_map_literal_either

_This literal must be either a map or a set, but the elements don't have enough
information for type inference to work._

#### Description

Because map and set literals use the same delimiters (`{` and `}`), the
analyzer looks at the type arguments and the elements to determine which
kind of literal you meant. When there are no type arguments and all of the
elements are spread elements (which are allowed in both kinds of literals)
then the analyzer uses the types of the expressions that are being spread.
If all of the expressions have the type `Iterable`, then it's a set
literal; if they all have the type `Map`, then it's a map literal.

This diagnostic is produced when none of the expressions being spread have
a type that allows the analyzer to decide whether you were writing a map
literal or a set literal.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
union(a, b) => [!{...a, ...b}!];
{% endprettify %}

The problem occurs because there are no type arguments, and there is no
information about the type of either `a` or `b`.

#### Common fixes

There are three common ways to fix this problem. The first is to add type
arguments to the literal. For example, if the literal is intended to be a
map literal, you might write something like this:

{% prettify dart %}
union(a, b) => <String, String>{...a, ...b};
{% endprettify %}

The second fix is to add type information so that the expressions have
either the type `Iterable` or the type `Map`. You can add an explicit cast
or, in this case, add types to the declarations of the two parameters:

{% prettify dart %}
union(List<int> a, List<int> b) => {...a, ...b};
{% endprettify %}

The third fix is to add context information. In this case, that means
adding a return type to the function:

{% prettify dart %}
Set<String> union(a, b) => {...a, ...b};
{% endprettify %}

In other cases, you might add a type somewhere else. For example, say the
original code looks like this:

{% prettify dart %}
union(a, b) {
  var x = [!{...a, ...b}!];
  return x;
}
{% endprettify %}

You might add a type annotation on `x`, like this:

{% prettify dart %}
union(a, b) {
  Map<String, String> x = {...a, ...b};
  return x;
}
{% endprettify %}

### argument_type_not_assignable

_The argument type '{0}' can't be assigned to the parameter type '{1}'._

#### Description

The analyzer produces this diagnostic when the static type of an argument
can't be assigned to the static type of the corresponding parameter.

#### Example

The following code produces this diagnostic because a `num` can't be
assigned to a `String`:

{% prettify dart %}
String f(String x) => x;
String g(num y) => f([!y!]);
{% endprettify %}

#### Common fixes

If possible, rewrite the code so that the static type is assignable. In the
example above you might be able to change the type of the parameter `y`:

{% prettify dart %}
String f(String x) => x;
String g(String y) => f(y);
{% endprettify %}

If that fix isn't possible, then add code to handle the case where the
argument value isn't the required type. One approach is to coerce other
types to the required type:

{% prettify dart %}
String f(String x) => x;
String g(num y) => f(y.toString());
{% endprettify %}

Another approach is to add explicit type tests and fallback code:

{% prettify dart %}
String f(String x) => x;
String g(num y) => f(y is String ? y : '');
{% endprettify %}

If you believe that the runtime type of the argument will always be the
same as the static type of the parameter, and you're willing to risk having
an exception thrown at runtime if you're wrong, then add an explicit cast:

{% prettify dart %}
String f(String x) => x;
String g(num y) => f(y as String);
{% endprettify %}

### assignment_to_final

_'{0}' can't be used as a setter because it's final._

#### Description

The analyzer produces this diagnostic when it finds an invocation of a
setter, but there's no setter because the field with the same name was
declared to be `final` or `const`.

#### Example

The following code produces this diagnostic because `v` is final:

{% prettify dart %}
class C {
  final v = 0;
}

f(C c) {
  c.[!v!] = 1;
}
{% endprettify %}

#### Common fixes

If you need to be able to set the value of the field, then remove the
modifier `final` from the field:

{% prettify dart %}
class C {
  int v = 0;
}

f(C c) {
  c.v = 1;
}
{% endprettify %}

### assignment_to_final_no_setter

_There isn’t a setter named '{0}' in class '{1}'._

#### Description

The analyzer produces this diagnostic when a reference to a setter is
found; there is no setter defined for the type; but there is a getter
defined with the same name.

#### Example

The following code produces this diagnostic because there is no setter
named `x` in `C`, but there is a getter named `x`:

{% prettify dart %}
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.[!x!] = 1;
}
{% endprettify %}

#### Common fixes

If you want to invoke an existing setter, then correct the name:

{% prettify dart %}
class C {
  int get x => 0;
  set y(int p) {}
}

void f(C c) {
  c.y = 1;
}
{% endprettify %}

If you want to invoke the setter but it just doesn't exist yet, then
declare it:

{% prettify dart %}
class C {
  int get x => 0;
  set x(int p) {}
  set y(int p) {}
}

void f(C c) {
  c.x = 1;
}
{% endprettify %}

### built_in_identifier_as_extension_name

_The built-in identifier '{0}' can't be used as an extension name._

#### Description

The analyzer produces this diagnostic when the name of an extension is a
built-in identifier. Built-in identifiers can’t be used as extension names.

#### Example

The following code produces this diagnostic because `mixin` is a built-in
identifier:

{% prettify dart %}
extension [!mixin!] on int {}
{% endprettify %}

#### Common fixes

Choose a different name for the extension.

### cast_to_non_type

_The name '{0}' isn't a type, so it can't be used in an 'as' expression._

#### Description

The analyzer produces this diagnostic when the name following the `as` in a
cast expression is defined to be something other than a type.

#### Example

The following code produces this diagnostic because `x` is a variable, not
a type:

{% prettify dart %}
num x = 0;
int y = x as [!x!];
{% endprettify %}

#### Common fixes

Replace the name with the name of a type:

{% prettify dart %}
num x = 0;
int y = x as int;
{% endprettify %}

### concrete_class_with_abstract_member

_'{0}' must have a method body because '{1}' isn't abstract._

#### Description

The analyzer produces this diagnostic when a member of a concrete class is
found that doesn't have a concrete implementation. Concrete classes aren't
allowed to contain abstract members.

#### Example

The following code produces this diagnostic because `m` is an abstract
method but `C` isn't an abstract class:

{% prettify dart %}
class C {
  [!void m();!]
}
{% endprettify %}

#### Common fixes

If it's valid to create instances of the class, provide an implementation
for the member:

{% prettify dart %}
class C {
  void m() {}
}
{% endprettify %}

If it isn't valid to create instances of the class, mark the class as being
abstract:

{% prettify dart %}
abstract class C {
  void m();
}
{% endprettify %}

### const_initialized_with_non_constant_value

_Const variables must be initialized with a constant value._

#### Description

The analyzer produces this diagnostic when a value that isn't statically
known to be a constant is assigned to a variable that's declared to be a
'const' variable.

#### Example

The following code produces this diagnostic because `x` isn't declared to
be `const`:

{% prettify dart %}
var x = 0;
const y = [!x!];
{% endprettify %}

#### Common fixes

If the value being assigned can be declared to be `const`, then change the
declaration:

{% prettify dart %}
const x = 0;
const y = x;
{% endprettify %}

If the value can't be declared to be `const`, then remove the `const`
modifier from the variable, possibly using `final` in its place:

{% prettify dart %}
var x = 0;
final y = x;
{% endprettify %}

### const_instance_field

_Only static fields can be declared as const._

#### Description

The analyzer produces this diagnostic when an instance field is marked as
being const.

#### Example

The following code produces this diagnostic because `f` is an instance
field:

{% prettify dart %}
class C {
  [!const!] int f = 3;
}
{% endprettify %}

#### Common fixes

If the field needs to be an instance field, then remove the keyword
`const`, or replace it with `final`:

{% prettify dart %}
class C {
  final int f = 3;
}
{% endprettify %}

If the field really should be a const field, then make it a static field:

{% prettify dart %}
class C {
  static const int f = 3;
}
{% endprettify %}

### const_not_initialized

_The constant '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a variable that is declared to
be a constant doesn't have an initializer.

#### Example

The following code produces this diagnostic because `c` isn't initialized:

{% prettify dart %}
const [!c!];
{% endprettify %}

#### Common fixes

Add an initializer:

{% prettify dart %}
const c = 'c';
{% endprettify %}

### const_spread_expected_list_or_set

_A list or a set is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant list or set evaluates to something other than a list
or a set.

#### Example

The following code produces this diagnostic because the value of `list1` is
`null`, which is neither a list nor a set:

{% prettify dart %}
const List<int> list1 = null;
const List<int> list2 = [...[!list1!]];
{% endprettify %}

#### Common fixes

Change the expression to something that evaluates to either a constant list
or a constant set:

{% prettify dart %}
const List<int> list1 = [];
const List<int> list2 = [...list1];
{% endprettify %}

### const_spread_expected_map

_A map is expected in this spread._

#### Description

The analyzer produces this diagnostic when the expression of a spread
operator in a constant map evaluates to something other than a map.

#### Example

The following code produces this diagnostic because the value of `map1` is
`null`, which isn't a map:

{% prettify dart %}
const Map<String, int> map1 = null;
const Map<String, int> map2 = {...[!map1!]};
{% endprettify %}

#### Common fixes

Change the expression to something that evaluates to a constant map:

{% prettify dart %}
const Map<String, int> map1 = {};
const Map<String, int> map2 = {...map1};
{% endprettify %}

### const_with_non_const

_The constructor being called isn't a const constructor._

#### Description

The analyzer produces this diagnostic when the keyword `const` is used to
invoke a constructor that isn't marked with `const`.

#### Example

The following code produces this diagnostic because the constructor in `A`
isn't a const constructor:

{% prettify dart %}
class A {
  A();
}

A f() => [!const!] A();
{% endprettify %}

#### Common fixes

If it's desirable and possible to make the class a constant class (by
making all of the fields of the class, including inherited fields, final),
then add the keyword `const` to the constructor:

{% prettify dart %}
class A {
  const A();
}

A f() => const A();
{% endprettify %}

Otherwise, remove the keyword `const`:

{% prettify dart %}
class A {
  A();
}

A f() => A();
{% endprettify %}

### const_with_non_constant_argument

_Arguments of a constant creation must be constant expressions._

#### Description

The analyzer produces this diagnostic when a const constructor is invoked
with an argument that isn't a constant expression.

#### Example

The following code produces this diagnostic because `i` isn't a constant:

{% prettify dart %}
class C {
  final int i;
  const C(this.i);
}
C f(int i) => const C([!i!]);
{% endprettify %}

#### Common fixes

Either make all of the arguments constant expressions, or remove the
`const` keyword to use the non-constant form of the constructor:

{% prettify dart %}
class C {
  final int i;
  const C(this.i);
}
C f(int i) => C(i);
{% endprettify %}

### dead_code

_Dead code._

#### Description

The analyzer produces this diagnostic when code is found that won't be
executed because execution will never reach the code.

#### Example

The following code produces this diagnostic because the invocation of
`print` occurs after the function has returned:

{% prettify dart %}
void f() {
  return;
  [!print('here');!]
}
{% endprettify %}

#### Common fixes

If the code isn't needed, then remove it:

{% prettify dart %}
void f() {
  return;
}
{% endprettify %}

If the code needs to be executed, then either move the code to a place
where it will be executed:

{% prettify dart %}
void f() {
  print('here');
  return;
}
{% endprettify %}

Or, rewrite the code before it, so that it can be reached:

{% prettify dart %}
void f({bool skipPrinting = true}) {
  if (skipPrinting) {
    return;
  }
  print('here');
}
{% endprettify %}

### dead_code_catch_following_catch

_Dead code: Catch clauses after a 'catch (e)' or an 'on Object catch (e)' are
never reached._

#### Description

The analyzer produces this diagnostic when a catch clause is found that
can't be executed because it’s after a catch clause of the form `catch (e)`
or `on Object catch (e)`. The first catch clause that matches the thrown
object is selected, and both of those forms will match any object, so no
catch clauses that follow them will be selected.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
void f() {
  try {
  } catch (e) {
  } [!on String {
  }!]
}
{% endprettify %}

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

{% prettify dart %}
void f() {
  try {
  } on String {
  } catch (e) {
  }
}
{% endprettify %}

If the clause doesn't need to be selectable, then remove it:

{% prettify dart %}
void f() {
  try {
  } catch (e) {
  }
}
{% endprettify %}

### dead_code_on_catch_subtype

_Dead code: This on-catch block won’t be executed because '{0}' is a subtype of
'{1}' and hence will have been caught already._

#### Description

The analyzer produces this diagnostic when a catch clause is found that
can't be executed because it is after a catch clause that catches either
the same type or a supertype of the clause's type. The first catch clause
that matches the thrown object is selected, and the earlier clause l always
matches anything matchable by the highlighted clause, so the highlighted
clause will never be selected.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
void f() {
  try {
  } on num {
  } [!on int {
  }!]
}
{% endprettify %}

#### Common fixes

If the clause should be selectable, then move the clause before the general
clause:

{% prettify dart %}
void f() {
  try {
  } on int {
  } on num {
  }
}
{% endprettify %}

If the clause doesn't need to be selectable, then remove it:

{% prettify dart %}
void f() {
  try {
  } on num {
  }
}
{% endprettify %}

### deprecated_member_use

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}._

#### Description

The analyzer produces this diagnostic when a deprecated library or class
member is used in a different package.

#### Example

If the method `m` in the class `C` is annotated with `@deprecated`, then
the following code produces this diagnostic:

{% prettify dart %}
void f(C c) {
  c.[!m!]();
}
{% endprettify %}

#### Common fixes

The documentation for declarations that are annotated with `@deprecated`
should indicate what code to use in place of the deprecated code.

### deprecated_member_use_from_same_package

_'{0}' is deprecated and shouldn't be used._

_'{0}' is deprecated and shouldn't be used. {1}._

#### Description

The analyzer produces this diagnostic when a deprecated library member or
class member is used in the same package in which it's declared.

#### Example

The following code produces this diagnostic because `x` is deprecated:

{% prettify dart %}
@deprecated
var x = 0;
var y = [!x!];
{% endprettify %}

#### Common fixes

The fix depends on what's been deprecated and what the replacement is. The
documentation for deprecated declarations should indicate what code to use
in place of the deprecated code.

### duplicate_definition

_The name '{0}' is already defined._

#### Description

The analyzer produces this diagnostic when a name is declared, and there is
a previous declaration with the same name in the same scope.

#### Example

The following code produces this diagnostic because the name `x` is
declared twice:

{% prettify dart %}
int x = 0;
int [!x!] = 1;
{% endprettify %}

#### Common fixes

Choose a different name for one of the declarations.

{% prettify dart %}
int x = 0;
int y = 1;
{% endprettify %}

### duplicate_import

_Duplicate import._

#### Description

The analyzer produces this diagnostic when an import directive is found
that is the same as an import before it in the file. The second import
doesn’t add value and should be removed.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
import 'package:meta/meta.dart';
import [!'package:meta/meta.dart'!];

@sealed class C {}
{% endprettify %}

#### Common fixes

Remove the unnecessary import:

{% prettify dart %}
import 'package:meta/meta.dart';

@sealed class C {}
{% endprettify %}

### equal_elements_in_const_set

_Two values in a constant set can't be equal._

#### Description

The analyzer produces this diagnostic when two elements in a constant set
literal have the same value. The set can only contain each value once,
which means that one of the values is unnecessary.

#### Example

The following code produces this diagnostic because the string `'a'` is
specified twice:

{% prettify dart %}
const Set<String> set = {'a', [!'a'!]};
{% endprettify %}

#### Common fixes

Remove one of the duplicate values:

{% prettify dart %}
const Set<String> set = {'a'};
{% endprettify %}

Note that literal sets preserve the order of their elements, so the choice
of which element to remove might affect the order in which elements are
returned by an iterator.

### equal_keys_in_const_map

_Two keys in a constant map literal can't be equal._

#### Description

The analyzer produces this diagnostic when a key in a constant map is the
same as a previous key in the same map. If two keys are the same, then the
second value would overwrite the first value, which makes having both pairs
pointless.

#### Example

The following code produces this diagnostic because the key `1` is used
twice:

{% prettify dart %}
const map = <int, String>{1: 'a', 2: 'b', [!1!]: 'c', 4: 'd'};
{% endprettify %}

#### Common fixes

If both entries should be included in the map, then change one of the keys
to be different:

{% prettify dart %}
const map = <int, String>{1: 'a', 2: 'b', 3: 'c', 4: 'd'};
{% endprettify %}

If only one of the entries is needed, then remove the one that isn't
needed:

{% prettify dart %}
const map = <int, String>{1: 'a', 2: 'b', 4: 'd'};
{% endprettify %}

Note that literal maps preserve the order of their entries, so the choice
of which entry to remove might affect the order in which keys and values
are returned by an iterator.

### expression_in_map

_Expressions can't be used in a map literal._

#### Description

The analyzer produces this diagnostic when the analyzer finds an
expression, rather than a map entry, in what appears to be a map literal.

#### Example

The following code generates this diagnostic:

{% prettify dart %}
var map = <String, int>{'a': 0, 'b': 1, [!'c'!]};
{% endprettify %}

#### Common fixes

If the expression is intended to compute either a key or a value in an
entry, fix the issue by replacing the expression with the key or the value.
For example:

{% prettify dart %}
var map = <String, int>{'a': 0, 'b': 1, 'c': 2};
{% endprettify %}

### extends_non_class

_Classes can only extend other classes._

#### Description

The analyzer produces this diagnostic when an extends clause contains a
name that is declared to be something other than a class.

#### Example

The following code produces this diagnostic because `f` is declared to be a
function:

{% prettify dart %}
void f() {}

class C extends [!f!] {}
{% endprettify %}

#### Common fixes

If you want the class to extend a class other than `Object`, then replace
the name in the extends clause with the name of that class:

{% prettify dart %}
void f() {}

class C extends B {}

class B {}
{% endprettify %}

If you want the class to extend `Object`, then remove the extends clause:

{% prettify dart %}
void f() {}

class C {}
{% endprettify %}

### extension_as_expression

_Extension '{0}' can't be used as an expression._

#### Description

The analyzer produces this diagnostic when the name of an extension is used
in an expression other than in an extension override or to qualify an
access to a static member of the extension. Because classes define a type,
the name of a class can be used to refer to the instance of `Type`
representing the type of the class. Extensions, on the other hand, don't
define a type and can't be used as a type literal.

#### Example

The following code produces this diagnostic because `E` is an extension:

{% prettify dart %}
extension E on int {
  static String m() => '';
}

var x = [!E!];
{% endprettify %}

#### Common fixes

Replace the name of the extension with a name that can be referenced, such
as a static member defined on the extension:

{% prettify dart %}
extension E on int {
  static String m() => '';
}

var x = E.m();
{% endprettify %}

### extension_conflicting_static_and_instance

_Extension '{0}' can't define static member '{1}' and an instance member with
the same name._

#### Description

The analyzer produces this diagnostic when an extension declaration
contains both an instance member and a static member that have the same
name. The instance member and the static member can't have the same name
because it's unclear which member is being referenced by an unqualified use
of the name within the body of the extension.

#### Example

The following code produces this diagnostic because the name `a` is being
used for two different members:

{% prettify dart %}
extension E on Object {
  int get a => 0;
  static int [!a!]() => 0;
}
{% endprettify %}

#### Common fixes

Rename or remove one of the members:

{% prettify dart %}
extension E on Object {
  int get a => 0;
  static int b() => 0;
}
{% endprettify %}

### extension_declares_abstract_member

_Extensions can't declare abstract members._

#### Description

The analyzer produces this diagnostic when an abstract declaration is
declared in an extension. Extensions can declare only concrete members.

#### Example

The following code produces this diagnostic because the method `a` doesn't
have a body:

{% prettify dart %}
extension E on String {
  int [!a!]();
}
{% endprettify %}

#### Common fixes

Either provide an implementation for the member or remove it.

### extension_declares_constructor

_Extensions can't declare constructors._

#### Description

The analyzer produces this diagnostic when a constructor declaration is
found in an extension. It isn't valid to define a constructor because
extensions aren't classes, and it isn't possible to create an instance of
an extension.

#### Example

The following code produces this diagnostic because there is a constructor
declaration in `E`:

{% prettify dart %}
extension E on String {
  [!E!]() : super();
}
{% endprettify %}

#### Common fixes

Remove the constructor or replace it with a static method.

### extension_declares_instance_field

_Extensions can't declare instance fields_

#### Description

The analyzer produces this diagnostic when an instance field declaration is
found in an extension. It isn't valid to define an instance field because
extensions can only add behavior, not state.

#### Example

The following code produces this diagnostic because `s` is an instance
field:

{% prettify dart %}
extension E on String {
  String [!s!];
}
{% endprettify %}

#### Common fixes

Remove the field, make it a static field, or convert it to be a getter,
setter, or method.

### extension_declares_member_of_object

_Extensions can't declare members with the same name as a member declared by
'Object'._

#### Description

The analyzer produces this diagnostic when an extension declaration
declares a member with the same name as a member declared in the class
`Object`. Such a member can never be used because the member in `Object` is
always found first.

#### Example

The following code produces this diagnostic because `toString` is defined
by `Object`:

{% prettify dart %}
extension E on String {
  String [!toString!]() => this;
}
{% endprettify %}

#### Common fixes

Remove the member or rename it so that the name doesn't conflict with the
member in `Object`:

{% prettify dart %}
extension E on String {
  String displayString() => this;
}
{% endprettify %}

### extension_override_access_to_static_member

_An extension override can't be used to access a static member from an
extension._

#### Description

The analyzer produces this diagnostic when an extension override is the
target of the invocation of a static member. Similar to static members in
classes, the static members of an extension should be accessed using the
name of the extension, not an extension override.

#### Example

The following code produces this diagnostic because `m` is static:

{% prettify dart %}
extension E on String {
  static void m() {}
}

void f() {
  E('').[!m!]();
}
{% endprettify %}

#### Common fixes

Replace the extension override with the name of the extension:

{% prettify dart %}
extension E on String {
  static void m() {}
}

void f() {
  E.m();
}
{% endprettify %}

### extension_override_argument_not_assignable

_The type of the argument to the extension override '{0}' isn't assignable to
the extended type '{1}'._

#### Description

The analyzer produces this diagnostic when the argument to an extension
override isn't assignable to the type being extended by the extension.

#### Example

The following code produces this diagnostic because `3` isn't a `String`:

{% prettify dart %}
extension E on String {
  void method() {}
}

void f() {
  E([!3!]).method();
}
{% endprettify %}

#### Common fixes

If you're using the correct extension, then update the argument to have the
correct type:

{% prettify dart %}
extension E on String {
  void method() {}
}

void f() {
  E(3.toString()).method();
}
{% endprettify %}

If there's a different extension that's valid for the type of the argument,
then either replace the name of the extension or unwrap the target so that
the correct extension is found.

### extension_override_without_access

_An extension override can only be used to access instance members._

#### Description

The analyzer produces this diagnostic when an extension override is found
that isn't being used to access one of the members of the extension. The
extension override syntax doesn't have any runtime semantics; it only
controls which member is selected at compile time.

#### Example

The following code produces this diagnostic because `E(i)` isn't an
expression:

{% prettify dart %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print([!E(i)!]);
}
{% endprettify %}

#### Common fixes

If you want to invoke one of the members of the extension, then add the
invocation:

{% prettify dart %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(E(i).a);
}
{% endprettify %}

If you don't want to invoke a member, then unwrap the target:

{% prettify dart %}
extension E on int {
  int get a => 0;
}

void f(int i) {
  print(i);
}
{% endprettify %}

### extension_override_with_cascade

_Extension overrides have no value so they can't be used as the target of a
cascade expression._

#### Description

The analyzer produces this diagnostic when an extension override is used as
the target of a cascade expression. The value of a cascade expression
`e..m` is the value of the target `e`, but extension overrides are not
expressions and don't have a value.

#### Example

The following code produces this diagnostic because `E(3)` isn't an
expression:

{% prettify dart %}
extension E on int {
  void m() {}
}
f() {
  E(3)[!..!]m();
}
{% endprettify %}

#### Common fixes

Use '.' rather than '..':

{% prettify dart %}
extension E on int {
  void m() {}
}
f() {
  E(3).m();
}
{% endprettify %}

If there are multiple cascaded accesses, you'll need to duplicate the
extension override for each one.

### extra_positional_arguments

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows.

#### Example

The following code produces this diagnostic because `f` defines 2
parameters but is invoked with 3 arguments:

{% prettify dart %}
void f(int a, int b) {}
void g() {
  f[!(1, 2, 3)!];
}
{% endprettify %}

#### Common fixes

Remove the arguments that don't correspond to parameters:

{% prettify dart %}
void f(int a, int b) {}
void g() {
  f(1, 2);
}
{% endprettify %}

### extra_positional_arguments_could_be_named

_Too many positional arguments: {0} expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has more positional arguments than the method or function allows, but the
method or function defines named parameters.

#### Example

The following code produces this diagnostic because `f` defines 2
positional parameters but has a named parameter that could be used for the
third argument:

{% prettify dart %}
void f(int a, int b, {int c}) {}
void g() {
  f[!(1, 2, 3)!];
}
{% endprettify %}

#### Common fixes

If some of the arguments should be values for named parameters, then add
the names before the arguments:

{% prettify dart %}
void f(int a, int b, {int c}) {}
void g() {
  f(1, 2, c: 3);
}
{% endprettify %}

Otherwise, remove the arguments that don't correspond to positional
parameters:

{% prettify dart %}
void f(int a, int b, {int c}) {}
void g() {
  f(1, 2);
}
{% endprettify %}

### final_not_initialized

_The final variable '{0}' must be initialized._

#### Description

The analyzer produces this diagnostic when a final field or variable isn't
initialized.

#### Example

The following code produces this diagnostic because `x` doesn't have an
initializer:

{% prettify dart %}
final [!x!];
{% endprettify %}

#### Common fixes

For variables and static fields, you can add an initializer:

{% prettify dart %}
final x = 0;
{% endprettify %}

For instance fields, you can add an initializer as shown in the previous
example, or you can initialize the field in every constructor. You can
initialize the field by using a field formal parameter:

{% prettify dart %}
class C {
  final int x;
  C(this.x);
}
{% endprettify %}

You can also initialize the field by using an initializer in the
constructor:

{% prettify dart %}
class C {
  final int x;
  C(int y) : x = y * 2;
}
{% endprettify %}

### final_not_initialized_constructor

_All final variables must be initialized, but '{0}' and '{1}' are not._

_All final variables must be initialized, but '{0}' is not._

_All final variables must be initialized, but '{0}', '{1}', and {2} others are
not._

#### Description

The analyzer produces this diagnostic when a class defines one or more
final instance fields without initializers and has at least one constructor
that doesn't initialize those fields. All final instance fields must be
initialized when the instance is created, either by the field's initializer
or by the constructor.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
class C {
  final String value;

  [!C!]();
}
{% endprettify %}

#### Common fixes

If the value should be passed in to the constructor directly, then use a
field formal parameter to initialize the field `value`:

{% prettify dart %}
class C {
  final String value;

  C(this.value);
}
{% endprettify %}

If the value should be computed indirectly from a value provided by the
caller, then add a parameter and include an initializer:

{% prettify dart %}
class C {
  final String value;

  C(Object o) : value = o.toString();
}
{% endprettify %}

If the value of the field doesn't depend on values that can be passed to
the constructor, then add an initializer for the field as part of the field
declaration:

{% prettify dart %}
class C {
  final String value = '';

  C();
}
{% endprettify %}

If the value of the field doesn't depend on values that can be passed to
the constructor but different constructors need to initialize it to
different values, then add an initializer for the field in the initializer
list:

{% prettify dart %}
class C {
  final String value;

  C() : value = '';

  C.named() : value = 'c';
}
{% endprettify %}

However, if the value is the same for all instances, then consider using a
static field instead of an instance field:

{% prettify dart %}
class C {
  static const String value = '';

  C();
}
{% endprettify %}

### for_in_of_invalid_type

_The type '{0}' used in the 'for' loop must implement {1}._

#### Description

The analyzer produces this diagnostic when the expression following `in` in
a for-in loop has a type that isn't a subclass of `Iterable`.

#### Example

The following code produces this diagnostic because `m` is a `Map`, and
`Map` isn't a subclass of `Iterable`:

{% prettify dart %}
void f(Map<String, String> m) {
  for (String s in [!m!]) {
    print(s);
  }
}
{% endprettify %}

#### Common fixes

Replace the expression with one that produces an iterable value:

{% prettify dart %}
void f(Map<String, String> m) {
  for (String s in m.values) {
    print(s);
  }
}
{% endprettify %}

### implements_non_class

_Classes and mixins can only implement other classes and mixins._

#### Description

The analyzer produces this diagnostic when a name used in the implements
clause of a class or mixin declaration is defined to be something other
than a class or mixin.

#### Example

The following code produces this diagnostic because `x` is a variable
rather than a class or mixin:

{% prettify dart %}
var x;
class C implements [!x!] {}
{% endprettify %}

#### Common fixes

If the name is the name of an existing class or mixin that's already being
imported, then add a prefix to the import so that the local definition of
the name doesn't shadow the imported name.

If the name is the name of an existing class or mixin that isn't being
imported, then add an import, with a prefix, for the library in which it’s
declared.

Otherwise, either replace the name in the implements clause with the name
of an existing class or mixin, or remove the name from the implements
clause.

### implements_repeated

_'{0}' can only be implemented once._

#### Description

The analyzer produces this diagnostic when a single class is specified more
than once in an implements clause.

#### Example

The following code produces this diagnostic because `A` is in the list
twice:

{% prettify dart %}
class A {}
class B implements A, [!A!] {}
{% endprettify %}

#### Common fixes

Remove all except one occurrence of the class name:

{% prettify dart %}
class A {}
class B implements A {}
{% endprettify %}

### implicit_this_reference_in_initializer

_Only static members can be accessed in initializers._

#### Description

The analyzer produces this diagnostic when it finds a reference to an
instance member in a constructor's initializer list.

#### Example

The following code produces this diagnostic because `defaultX` is an
instance member:

{% prettify dart %}
class C {
  int x;

  C() : x = [!defaultX!];

  int get defaultX => 0;
}
{% endprettify %}

#### Common fixes

If the member can be made static, then do so:

{% prettify dart %}
class C {
  int x;

  C() : x = defaultX;

  static int get defaultX => 0;
}
{% endprettify %}

If not, then replace the reference in the initializer with a different
expression that doesn't use an instance member:

{% prettify dart %}
class C {
  int x;

  C() : x = 0;

  int get defaultX => 0;
}
{% endprettify %}

### initializer_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when a constructor initializes a
field that isn't declared in the class containing the constructor.
Constructors can't initialize fields that aren't declared and fields that
are inherited from superclasses.

#### Example

The following code produces this diagnostic because the initializer is
initializing `x`, but `x` isn't a field in the class:

{% prettify dart %}
class C {
  int y;

  C() : [!x = 0!];
}
{% endprettify %}

#### Common fixes

If a different field should be initialized, then change the name to the
name of the field:

{% prettify dart %}
class C {
  int y;

  C() : y = 0;
}
{% endprettify %}

If the field must be declared, then add a declaration:

{% prettify dart %}
class C {
  int x;
  int y;

  C() : x = 0;
}
{% endprettify %}

### initializing_formal_for_non_existent_field

_'{0}' isn't a field in the enclosing class._

#### Description

The analyzer produces this diagnostic when a field formal parameter is
found in a constructor in a class that doesn't declare the field being
initialized. Constructors can't initialize fields that aren't declared and
fields that are inherited from superclasses.

#### Example

The following code produces this diagnostic because the field `x` isn't
defined:

{% prettify dart %}
class C {
  int y;

  C([!this.x!]);
}
{% endprettify %}

#### Common fixes

If the field name was wrong, then change it to the name of an existing
field:

{% prettify dart %}
class C {
  int y;

  C(this.y);
}
{% endprettify %}

If the field name is correct but hasn't yet been defined, then declare the
field:

{% prettify dart %}
class C {
  int x;
  int y;

  C(this.x);
}
{% endprettify %}

If the parameter is needed but shouldn't initialize a field, then convert
it to a normal parameter and use it:

{% prettify dart %}
class C {
  int y;

  C(int x) : y = x * 2;
}
{% endprettify %}

If the parameter isn't needed, then remove it:

{% prettify dart %}
class C {
  int y;

  C();
}
{% endprettify %}

### instance_member_access_from_factory

_Instance members can't be accessed from a factory constructor._

#### Description

The analyzer produces this diagnostic when a factory constructor contains
an unqualified reference to an instance member. In a generative
constructor, the instance of the class is created and initialized before
the body of the constructor is executed, so the instance can be bound to
`this` and accessed just like it would be in an instance method. But, in a
factory constructor, the instance isn't created before executing the body,
so `this` can't be used to reference it.

#### Example

The following code produces this diagnostic because `x` isn't in scope in
the factory constructor:

{% prettify dart %}
class C {
  int x;
  factory C() {
    return C._([!x!]);
  }
  C._(this.x);
}
{% endprettify %}

#### Common fixes

Rewrite the code so that it doesn't reference the instance member:

{% prettify dart %}
class C {
  int x;
  factory C() {
    return C._(0);
  }
  C._(this.x);
}
{% endprettify %}

### instance_member_access_from_static

_Instance members can't be accessed from a static method._

#### Description

The analyzer produces this diagnostic when a static method contains an
unqualified reference to an instance member.

#### Example

The following code produces this diagnostic because the instance field `x`
is being referenced in a static method:

{% prettify dart %}
class C {
  int x;

  static int m() {
    return [!x!];
  }
}
{% endprettify %}

#### Common fixes

If the method must reference the instance member, then it can't be static,
so remove the keyword:

{% prettify dart %}
class C {
  int x;

  int m() {
    return x;
  }
}
{% endprettify %}

If the method can't be made an instance method, then add a parameter so
that an instance of the class can be passed in:

{% prettify dart %}
class C {
  int x;

  static int m(C c) {
    return c.x;
  }
}
{% endprettify %}

### instantiate_abstract_class

_Abstract classes can't be instantiated._

#### Description

The analyzer produces this diagnostic when it finds a constructor
invocation and the constructor is declared in an abstract class. Even
though you can't create an instance of an abstract class, abstract classes
can declare constructors that can be invoked by subclasses.

#### Example

The following code produces this diagnostic because `C` is an abstract
class:

{% prettify dart %}
abstract class C {}

var c = new [!C!]();
{% endprettify %}

#### Common fixes

If there's a concrete subclass of the abstract class that can be used, then
create an instance of the concrete subclass.

### invalid_assignment

_A value of type '{0}' can't be assigned to a variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the static type of an expression
that is assigned to a variable isn't assignable to the type of the
variable.

#### Example

The following code produces this diagnostic because the type of the
initializer (`int`) isn't assignable to the type of the variable
(`String`):

{% prettify dart %}
int i = 0;
String s = [!i!];
{% endprettify %}

#### Common fixes

If the value being assigned is always assignable at runtime, even though
the static types don't reflect that, then add an explicit cast.

Otherwise, change the value being assigned so that it has the expected
type. In the previous example, this might look like:

{% prettify dart %}
int i = 0;
String s = i.toString();
{% endprettify %}

If you can’t change the value, then change the type of the variable to be
compatible with the type of the value being assigned:

{% prettify dart %}
int i = 0;
int s = i;
{% endprettify %}

### invalid_extension_argument_count

_Extension overrides must have exactly one argument: the value of 'this' in the
extension method._

#### Description

The analyzer produces this diagnostic when an extension override doesn't
have exactly one argument. The argument is the expression used to compute
the value of `this` within the extension method, so there must be one
argument.

#### Example

The following code produces this diagnostic because there are no arguments:

{% prettify dart %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!()!].join('b');
}
{% endprettify %}

And, the following code produces this diagnostic because there's more than
one argument:

{% prettify dart %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E[!('a', 'b')!].join('c');
}
{% endprettify %}

#### Common fixes

Provide one argument for the extension override:

{% prettify dart %}
extension E on String {
  String join(String other) => '$this $other';
}

void f() {
  E('a').join('b');
}
{% endprettify %}

### invalid_factory_name_not_a_class

_The name of a factory constructor must be the same as the name of the
immediately enclosing class._

#### Description

The analyzer produces this diagnostic when the name of a factory
constructor isn't the same as the name of the surrounding class.

#### Example

The following code produces this diagnostic because the name of the factory
constructor (`A`) isn't the same as the surrounding class (`C`):

{% prettify dart %}
class A {}

class C {
  factory [!A!]() => null;
}
{% endprettify %}

#### Common fixes

If the factory returns an instance of the surrounding class, then rename
the factory:

{% prettify dart %}
class A {}

class C {
  factory C() => null;
}
{% endprettify %}

If the factory returns an instance of a different class, then move the
factory to that class:

{% prettify dart %}
class A {
  factory A() => null;
}

class C {}
{% endprettify %}

If the factory returns an instance of a different class, but you can't
modify that class or don't want to move the factory, then convert it to be
a static method:

{% prettify dart %}
class A {}

class C {
  static A a() => null;
}
{% endprettify %}

### invalid_literal_annotation

_Only const constructors can have the `@literal` annotation._

#### Description

The analyzer produces this diagnostic when the `@literal` annotation is
applied to anything other than a const constructor.

#### Example

The following code produces this diagnostic because the constructor is not
a `const` constructor:

{% prettify dart %}
import 'package:meta/meta.dart';

class C {
  [!@literal!]
  C();
}
{% endprettify %}

The following code produces this diagnostic because `x` isn't a
constructor:

{% prettify dart %}
import 'package:meta/meta.dart';

[!@literal!]
var x;
{% endprettify %}

#### Common fixes

If the annotation is on a constructor and the constructor should always be
invoked with `const`, when possible, then mark the constructor with the
`const` keyword:

{% prettify dart %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}
{% endprettify %}

If the constructor can't be marked as `const`, then remove the annotation.

If the annotation is on anything other than a constructor, then remove the
annotation:

{% prettify dart %}
var x;
{% endprettify %}

### invalid_override

_'{1}.{0}' ('{2}') isn't a valid override of '{3}.{0}' ('{4}')._

#### Description

The analyzer produces this diagnostic when a member of a class is found
that overrides a member from a supertype and the override isn't valid. An
override is valid if all of these are true:
* It allows all of the arguments allowed by the overridden member.
* It doesn't require any arguments that aren't required by the overridden
  member.
* The type of every parameter of the overridden member is assignable to the
  corresponding parameter of the override.
* The return type of the override is assignable to the return type of the
  overridden member.

#### Example

The following code produces this diagnostic because the type of the
parameter `s` (`String`) isn't assignable to the type of the parameter `i`
(`int`):

{% prettify dart %}
class A {
  void m(int i) {}
}

class B extends A {
  void [!m!](String s) {}
}
{% endprettify %}

#### Common fixes

If the invalid method is intended to override the method from the
superclass, then change it to conform:

{% prettify dart %}
class A {
  void m(int i) {}
}

class B extends A {
  void m(int i) {}
}
{% endprettify %}

If it isn't intended to override the method from the superclass, then
rename it:

{% prettify dart %}
class A {
  void m(int i) {}
}

class B extends A {
  void m2(String s) {}
}
{% endprettify %}

### invalid_reference_to_this

_Invalid reference to 'this' expression._

#### Description

The analyzer produces this diagnostic when `this` is used outside of an
instance method or a generative constructor. The reserved word `this` is
only defined in the context of an instance method or a generative
constructor.

#### Example

The following code produces this diagnostic because `v` is a top-level
variable:

{% prettify dart %}
C f() => [!this!];

class C {}
{% endprettify %}

#### Common fixes

Use a variable of the appropriate type in place of `this`, declaring it if
necessary:

{% prettify dart %}
C f(C c) => c;

class C {}
{% endprettify %}

### invalid_use_of_covariant_in_extension

_Can't have modifier '#lexeme' in an extension._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the keyword `covariant` in the declaration of a parameter.
Extensions aren't classes and don't have subclasses, so the keyword serves
no purpose.

#### Example

The following code produces this diagnostic because `i` is marked as being
covariant:

{% prettify dart %}
extension E on String {
  void a([!covariant!] int i) {}
}
{% endprettify %}

#### Common fixes

Remove the 'covariant' keyword:

{% prettify dart %}
extension E on String {
  void a(int i) {}
}
{% endprettify %}

### invalid_visibility_annotation

_The member '{0}' is annotated with '{1}', but this annotation is only
meaningful on declarations of public members._

#### Description

The analyzer produces this diagnostic when either the `@visibleForTemplate`
or `@visibleForTesting` annotation is applied to a non-public declaration.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
import 'package:meta/meta.dart';

[!@visibleForTesting!]
void _someFunction() {}

void f() => _someFunction();
{% endprettify %}

#### Common fixes

If the declaration doesn't need to be used by test code, then remove the
annotation:

{% prettify dart %}
void _someFunction() {}

void f() => _someFunction();
{% endprettify %}

If it does, then make it public:

{% prettify dart %}
import 'package:meta/meta.dart';

@visibleForTesting
void someFunction() {}

void f() => someFunction();
{% endprettify %}

### invocation_of_extension_without_call

_The extension '{0}' doesn't define a 'call' method so the override can't be
used in an invocation._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a function but the extension doesn't declare a `call` method.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't define a `call` method:

{% prettify dart %}
extension E on String {}

void f() {
  [!E('')!]();
}
{% endprettify %}

#### Common fixes

If the extension is intended to define a `call` method, then declare it:

{% prettify dart %}
extension E on String {
  int call() => 0;
}

void f() {
  E('')();
}
{% endprettify %}

If the extended type defines a `call` method, then remove the extension
override.

If the `call` method isn't defined, then rewrite the code so that it
doesn't invoke the `call` method.

### invocation_of_non_function

_'{0}' isn't a function._

#### Description

The analyzer produces this diagnostic when it finds a function invocation,
but the name of the function being invoked is defined to be something other
than a function.

#### Example

The following code produces this diagnostic because `Binary` is the name of
a function type, not a function:

{% prettify dart %}
typedef Binary = int Function(int, int);

int f() {
  return [!Binary!](1, 2);
}
{% endprettify %}

#### Common fixes

Replace the name with the name of a function.

### invocation_of_non_function_expression

_The expression doesn't evaluate to a function, so it can't be invoked._

#### Description

The analyzer produces this diagnostic when a function invocation is found,
but the name being referenced isn't the name of a function, or when the
expression computing the function doesn't compute a function.

#### Example

The following code produces this diagnostic because `x` isn't a function:

{% prettify dart %}
int x = 0;

int f() => x;

var y = [!x!]();
{% endprettify %}

The following code produces this diagnostic because `f()` doesn't return a
function:

{% prettify dart %}
int x = 0;

int f() => x;

var y = [!f()!]();
{% endprettify %}

#### Common fixes

If you need to invoke a function, then replace the code before the argument
list with the name of a function or with an expression that computes a
function:

{% prettify dart %}
int x = 0;

int f() => x;

var y = f();
{% endprettify %}

### list_element_type_not_assignable

_The element type '{0}' can't be assigned to the list type '{1}'._

#### Description

The analyzer produces this diagnostic when the type of an element in a list
literal isn't assignable to the element type of the list.

#### Example

The following code produces this diagnostic because `2.5` is a double, and
the list can hold only integers:

{% prettify dart %}
List<int> x = [1, [!2.5!], 3];
{% endprettify %}

#### Common fixes

If you intended to add a different object to the list, then replace the
element with an expression that computes the intended object:

{% prettify dart %}
List<int> x = [1, 2, 3];
{% endprettify %}

If the object shouldn't be in the list, then remove the element:

{% prettify dart %}
List<int> x = [1, 3];
{% endprettify %}

If the object being computed is correct, then widen the element type of the
list to allow all of the different types of objects it needs to contain:

{% prettify dart %}
List<num> x = [1, 2.5, 3];
{% endprettify %}

### map_entry_not_in_map

_Map entries can only be used in a map literal._

#### Description

The analyzer produces this diagnostic when a map entry (a key/value pair)
is found in a set literal.

#### Example

The following code produces this diagnostic because the literal has a map
entry even though it's a set literal:

{% prettify dart %}
const collection = <String>{[!'a' : 'b'!]};
{% endprettify %}

#### Common fixes

If you intended for the collection to be a map, then change the code so
that it is a map. In the previous example, you could do this by adding
another type argument:

{% prettify dart %}
const collection = <String, String>{'a' : 'b'};
{% endprettify %}

In other cases, you might need to change the explicit type from `Set` to
`Map`.

If you intended for the collection to be a set, then remove the map entry,
possibly by replacing the colon with a comma if both values should be
included in the set:

{% prettify dart %}
const collection = <String>{'a', 'b'};
{% endprettify %}

### map_key_type_not_assignable

_The element type '{0}' can't be assigned to the map key type '{1}'._

#### Description

The analyzer produces this diagnostic when a key of a key-value pair in a
map literal has a type that isn't assignable to the key type of the map.

#### Example

The following code produces this diagnostic because `2` is an `int`, but
the keys of the map are required to be `String`s:

{% prettify dart %}
var m = <String, String>{[!2!] : 'a'};
{% endprettify %}

#### Common fixes

If the type of the map is correct, then change the key to have the correct
type:

{% prettify dart %}
var m = <String, String>{'2' : 'a'};
{% endprettify %}

If the type of the key is correct, then change the key type of the map:

{% prettify dart %}
var m = <int, String>{2 : 'a'};
{% endprettify %}

### map_value_type_not_assignable

_The element type '{0}' can't be assigned to the map value type '{1}'._

#### Description

The analyzer produces this diagnostic when a value of a key-value pair in a
map literal has a type that isn't assignable to the the value type of the
map.

#### Example

The following code produces this diagnostic because `2` is an `int`, but/
the values of the map are required to be `String`s:

{% prettify dart %}
var m = <String, String>{'a' : [!2!]};
{% endprettify %}

#### Common fixes

If the type of the map is correct, then change the value to have the
correct type:

{% prettify dart %}
var m = <String, String>{'a' : '2'};
{% endprettify %}

If the type of the value is correct, then change the value type of the map:

{% prettify dart %}
var m = <String, int>{'a' : 2};
{% endprettify %}

### missing_enum_constant_in_switch

_Missing case clause for '{0}'._

#### Description

The analyzer produces this diagnostic when a switch statement for an enum
doesn't include an option for one of the values in the enumeration.

Note that `null` is always a possible value for an enum and therefore also
must be handled.

#### Example

The following code produces this diagnostic because the enum constant `e2`
isn't handled:

{% prettify dart %}
enum E { e1, e2 }

void f(E e) {
  [!switch (e)!] {
    case E.e1:
      break;
  }
}
{% endprettify %}

#### Common fixes

If there's special handling for the missing values, then add a case clause
for each of the missing values:

{% prettify dart %}
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    case E.e2:
      break;
  }
}
{% endprettify %}

If the missing values should be handled the same way, then add a default
clause:

{% prettify dart %}
enum E { e1, e2 }

void f(E e) {
  switch (e) {
    case E.e1:
      break;
    default:
      break;
  }
}
{% endprettify %}

### missing_required_param

_The parameter '{0}' is required._

_The parameter '{0}' is required. {1}._

#### Description

The analyzer produces this diagnostic when a method or function with a
named parameter that is annotated as being required is invoked without
providing a value for the parameter.

#### Example

The following code produces this diagnostic because the named parameter `x`
is required:

{% prettify dart %}
import 'package:meta/meta.dart';

void f({@required int x}) {}

void g() {
  [!f!]();
}
{% endprettify %}

#### Common fixes

Provide the required value:

{% prettify dart %}
import 'package:meta/meta.dart';

void f({@required int x}) {}

void g() {
  f(x: 2);
}
{% endprettify %}

### missing_return

_This function has a return type of '{0}', but doesn't end with a return
statement._

#### Description

Any function or method that doesn't end with either an explicit return or a
throw implicitly returns `null`. This is rarely the desired behavior. The
analyzer produces this diagnostic when it finds an implicit return.

#### Example

The following code produces this diagnostic because `f` doesn't end with a
return:

{% prettify dart %}
int [!f!](int x) {
  if (x < 0) {
    return 0;
  }
}
{% endprettify %}

#### Common fixes

Add a return statement that makes the return value explicit, even if `null`
is the appropriate value.

### mixin_of_non_class

_Classes can only mix in mixins and classes._

#### Description

The analyzer produces this diagnostic when a name in a mixin clause is
defined to be something other than a mixin or a class.

#### Example

The following code produces this diagnostic because `F` is defined to be a
function type:

{% prettify dart %}
typedef F = int Function(String);

class C with [!F!] {}
{% endprettify %}

#### Common fixes

Remove the invalid name from the list, possibly replacing it with the name of the intended mixin or class:

{% prettify dart %}
typedef F = int Function(String);

class C {}
{% endprettify %}

### mixin_on_sealed_class

_The class '{0}' shouldn't be used as a mixin constraint because it is sealed,
and any class mixing in this mixin must have '{0}' as a superclass._

#### Description

The analyzer produces this diagnostic when the superclass constraint of a
mixin is a class from a different package that was marked as `@sealed`.
Classes that are sealed can't be extended, implemented, mixed in, or used
as a superclass constraint.

#### Example

If the package 'p' defines a sealed class:

{% prettify dart %}
import 'package:meta/meta.dart';

@sealed
class C {}
{% endprettify %}

Then, the following code, when in a package other than 'p', produces this
diagnostic:

{% prettify dart %}
import 'package:p/p.dart';

[!mixin M on C {}!]
{% endprettify %}

#### Common fixes

If the classes that use the mixin don't need to be subclasses of the sealed
class, then consider adding a field and delegating to the wrapped instance
of the sealed class.

### mixin_super_class_constraint_non_interface

_Only classes and mixins can be used as superclass constraints._

#### Description

The analyzer produces this diagnostic when a type following the `on`
keyword in a mixin declaration is neither a class nor a mixin.

#### Example

The following code produces this diagnostic because `F` is neither a class
nor a mixin:

{% prettify dart %}
typedef F = void Function();

mixin M on [!F!] {}
{% endprettify %}

#### Common fixes

If the type was intended to be a class but was mistyped, then replace the
name.

Otherwise, remove the type from the on clause.

### must_be_immutable

_This class (or a class that this class inherits from) is marked as
'@immutable', but one or more of its instance fields aren't final: {0}_

#### Description

The analyzer produces this diagnostic when an immutable class defines one
or more instance fields that aren't final. A class is immutable if it's
marked as being immutable using the annotation `@immutable` or if it's a
subclass of an immutable class.

#### Example

The following code produces this diagnostic because the field `x` isn't
final:

{% prettify dart %}
import 'package:meta/meta.dart';

@immutable
class [!C!] {
  int x;

  C(this.x);
}
{% endprettify %}

#### Common fixes

If instances of the class should be immutable, then add the keyword `final`
to all non-final field declarations:

{% prettify dart %}
import 'package:meta/meta.dart';

@immutable
class C {
  final int x;

  C(this.x);
}
{% endprettify %}

If the instances of the class should be mutable, then remove the

{% prettify dart %}
class C {
  int x;

  C(this.x);
}
{% endprettify %}

### new_with_undefined_constructor_default

_The class '{0}' doesn't have a default constructor._

#### Description

The analyzer produces this diagnostic when an unnamed constructor is
invoked on a class that defines named constructors but the class doesn’t
have an unnamed constructor.

#### Example

The following code produces this diagnostic because `A` doesn't define an
unnamed constructor:

{% prettify dart %}
class A {
  A.a();
}

A f() => [!A!]();
{% endprettify %}

#### Common fixes

If one of the named constructors does what you need, then use it:

{% prettify dart %}
class A {
  A.a();
}

A f() => A.a();
{% endprettify %}

If none of the named constructors does what you need, and you're able to
add an unnamed constructor, then add the constructor:

{% prettify dart %}
class A {
  A();
  A.a();
}

A f() => A();
{% endprettify %}

### non_abstract_class_inherits_abstract_member

_Missing concrete implementation of '{0}'._

_Missing concrete implementations of '{0}' and '{1}'._

_Missing concrete implementations of '{0}', '{1}', '{2}', '{3}', and {4} more._

_Missing concrete implementations of '{0}', '{1}', '{2}', and '{3}'._

_Missing concrete implementations of '{0}', '{1}', and '{2}'._

#### Description

The analyzer produces this diagnostic when a concrete class inherits one or
more abstract members, and doesn't provide or inherit an implementation for
at least one of those abstract members.

#### Example

The following code produces this diagnostic because the class `B` doesn't
have a concrete implementation of `m`:

{% prettify dart %}
abstract class A {
  void m();
}

class [!B!] extends A {}
{% endprettify %}

#### Common fixes

If the subclass can provide a concrete implementation for some or all of
the abstract inherited members, then add the concrete implementations:

{% prettify dart %}
abstract class A {
  void m();
}

class B extends A {
  void m() {}
}
{% endprettify %}

If there is a mixin that provides an implementation of the inherited
methods, then apply the mixin to the subclass:

{% prettify dart %}
abstract class A {
  void m();
}

class B extends A with M {}

mixin M {
  void m() {}
}
{% endprettify %}

If the subclass can't provide a concrete implementation for all of the
abstract inherited members, then mark the subclass as being abstract:

{% prettify dart %}
abstract class A {
  void m();
}

abstract class B extends A {}
{% endprettify %}

### non_bool_condition

_Conditions must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when a condition, such as an `if` or
`while` loop, doesn't have the static type `bool`.

#### Example

The following code produces this diagnostic because `x` has the static type
`int`:

{% prettify dart %}
void f(int x) {
  if ([!x!]) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

Change the condition so that it produces a Boolean value:

{% prettify dart %}
void f(int x) {
  if (x == 0) {
    // ...
  }
}
{% endprettify %}

### non_bool_expression

_The expression in an assert must be of type 'bool'._

#### Description

The analyzer produces this diagnostic when the first expression in an
assert has a type other than `bool`.

#### Example

The following code produces this diagnostic because the type of `p` is
`int`, but a `bool` is required:

{% prettify dart %}
void f(int p) {
  assert([!p!]);
}
{% endprettify %}

#### Common fixes

Change the expression so that it has the type `bool`:

{% prettify dart %}
void f(int p) {
  assert(p > 0);
}
{% endprettify %}

### non_bool_negation_expression

_A negation operand must have a static type of 'bool'._

#### Description

The analyzer produces this diagnostic when the operand of the unary
negation operator (`!`) doesn't have the type `bool`.

#### Example

The following code produces this diagnostic because `x` is an `int` when it
must be a `bool`:

{% prettify dart %}
int x = 0;
bool y = ![!x!];
{% endprettify %}

#### Common fixes

Replace the operand with an expression that has the type `bool`:

{% prettify dart %}
int x = 0;
bool y = !(x > 0);
{% endprettify %}

### non_bool_operand

_The operands of the operator '{0}' must be assignable to 'bool'._

#### Description

The analyzer produces this diagnostic when one of the operands of either
the `&&` or `||` operator doesn't have the type `bool`.

#### Example

The following code produces this diagnostic because `a` isn't a Boolean
value:

{% prettify dart %}
int a = 3;
bool b = [!a!] || a > 1;
{% endprettify %}

#### Common fixes

Change the operand to a Boolean value:

{% prettify dart %}
int a = 3;
bool b = a == 0 || a > 1;
{% endprettify %}

### non_constant_case_expression

_Case expressions must be constant._

#### Description

The analyzer produces this diagnostic when the expression in a case clause
isn't a constant expression.

#### Example

The following code produces this diagnostic because `j` isn't a constant:

{% prettify dart %}
void f(int i, int j) {
  switch (i) {
    case [!j!]:
      // ...
      break;
  }
}
{% endprettify %}

#### Common fixes

Either make the expression a constant expression, or rewrite the switch
statement as a sequence of if statements:

{% prettify dart %}
void f(int i, int j) {
  if (i == j) {
    // ...
  }
}
{% endprettify %}

### non_constant_default_value

_The default value of an optional parameter must be constant._

#### Description

The analyzer produces this diagnostic when an optional parameter, either
named or positional, has a default value that isn't a compile-time
constant.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
var defaultValue = 3;

void f([int value = [!defaultValue!]]) {}
{% endprettify %}

#### Common fixes

If the default value can be converted to be a constant, then convert it:

{% prettify dart %}
const defaultValue = 3;

void f([int value = defaultValue]) {}
{% endprettify %}

If the default value needs to change over time, then apply the default
value inside the function:

{% prettify dart %}
var defaultValue = 3;

void f([int value]) {
  value ??= defaultValue;
}
{% endprettify %}

### non_constant_list_element

_The values in a const list literal must be constants._

#### Description

The analyzer produces this diagnostic when an element in a constant list
literal isn't a constant value. The list literal can be constant either
explicitly (because it's prefixed by the `const` keyword) or implicitly
(because it appears in a [constant context](#constant-context)).

#### Example

The following code produces this diagnostic because `x` isn't a constant,
even though it appears in an implicitly constant list literal:

{% prettify dart %}
var x = 2;
var y = const <int>[0, 1, [!x!]];
{% endprettify %}

#### Common fixes

If the list needs to be a constant list, then convert the element to be a
constant. In the example above, you might add the `const` keyword to the
declaration of `x`:

{% prettify dart %}
const x = 2;
var y = const <int>[0, 1, x];
{% endprettify %}

If the expression can't be made a constant, then the list can't be a
constant either, so you must change the code so that the list isn't a
constant. In the example above this means removing the `const` keyword
before the list literal:

{% prettify dart %}
var x = 2;
var y = <int>[0, 1, x];
{% endprettify %}

### non_constant_map_element

_The elements in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when an if element or a spread
element in a constant map isn't a constant element.

#### Example

The following code produces this diagnostic because it is attempting to
spread a non-constant map:

{% prettify dart %}
var notConst = <int, int>{};
var map = const <int, int>{...[!notConst!]};
{% endprettify %}

Similarly, the following code produces this diagnostic because the
condition in the if element isn't a constant expression:

{% prettify dart %}
bool notConst = true;
var map = const <int, int>{if ([!notConst!]) 1 : 2};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the elements  constants.
In the spread example, you might do that by making the collection being
spread a constant:

{% prettify dart %}
const notConst = <int, int>{};
var map = const <int, int>{...notConst};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart %}
bool notConst = true;
var map = <int, int>{if (notConst) 1 : 2};
{% endprettify %}

### non_constant_map_key

_The keys in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a key in a constant map literal
isn't a constant value.

#### Example

The following code produces this diagnostic beause `a` isn't a constant:

{% prettify dart %}
var a = 'a';
var m = const {[!a!]: 0};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

{% prettify dart %}
const a = 'a';
var m = const {a: 0};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart %}
var a = 'a';
var m = {a: 0};
{% endprettify %}

### non_constant_map_value

_The values in a const map literal must be constant._

#### Description

The analyzer produces this diagnostic when a value in a constant map
literal isn't a constant value.

#### Example

The following code produces this diagnostic because `a` isn't a constant:

{% prettify dart %}
var a = 'a';
var m = const {0: [!a!]};
{% endprettify %}

#### Common fixes

If the map needs to be a constant map, then make the key a constant:

{% prettify dart %}
const a = 'a';
var m = const {0: a};
{% endprettify %}

If the map doesn't need to be a constant map, then remove the `const`
keyword:

{% prettify dart %}
var a = 'a';
var m = {0: a};
{% endprettify %}

### non_constant_set_element

_The values in a const set literal must be constants._

#### Description

The analyzer produces this diagnostic when a constant set literal contains
an element that isn't a compile-time constant.

#### Example

The following code produces this diagnostic because `i` isn't a constant:

{% prettify dart %}
var i = 0;

var s = const {[!i!]};
{% endprettify %}

#### Common fixes

If the element can be changed to be a constant, then change it:

{% prettify dart %}
const i = 0;

var s = const {i};
{% endprettify %}

If the element can't be a constant, then remove the keyword `const`:

{% prettify dart %}
var i = 0;

var s = {i};
{% endprettify %}

### non_const_call_to_literal_constructor

_This instance creation must be 'const', because the {0} constructor is marked
as '@literal'._

_This instance creation must be 'const', because the {0} constructor is marked
as '@literal'._

#### Description

The analyzer produces this diagnostic when a constructor that has the
`@literal` annotation is invoked without using the `const` keyword, but all
of the arguments to the constructor are constants. The annotation indicates
that the constructor should be used to create a constant value whenever
possible.

#### Example

The following code produces this diagnostic:

{% prettify dart %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

C f() => [!C()!];
{% endprettify %}

#### Common fixes

Add the keyword `const` before the constructor invocation:

{% prettify dart %}
import 'package:meta/meta.dart';

class C {
  @literal
  const C();
}

void f() => const C();
{% endprettify %}

### non_type_as_type_argument

_The name '{0}' isn't a type so it can't be used as a type argument._

#### Description

The analyzer produces this diagnostic when an identifier that isn't a type
is used as a type argument.

#### Example

The following code produces this diagnostic because `x` is a variable, not
a type:

{% prettify dart %}
var x = 0;
List<[!x!]> xList = [];
{% endprettify %}

#### Common fixes

Change the type argument to be a type:

{% prettify dart %}
var x = 0;
List<int> xList = [];
{% endprettify %}

### not_a_type

_{0} isn't a type._

#### Description

The analyzer produces this diagnostic when a name is used as a type but
declared to be something other than a type.

#### Example

The following code produces this diagnostic because `f` is a function:

{% prettify dart %}
f() {}
g([!f!] v) {}
{% endprettify %}

#### Common fixes

Replace the name with the name of a type.

### not_enough_positional_arguments

_{0} positional argument(s) expected, but {1} found._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has fewer positional arguments than the number of required positional
parameters.

#### Example

The following code produces this diagnostic because `f` declares two
required parameters, but only one argument is provided:

{% prettify dart %}
void f(int a, int b) {}
void g() {
  f[!(0)!];
}
{% endprettify %}

#### Common fixes

Add arguments corresponding to the remaining parameters:

{% prettify dart %}
void f(int a, int b) {}
void g() {
  f(0, 1);
}
{% endprettify %}

### not_iterable_spread

_Spread elements in list or set literals must implement 'Iterable'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in either a list literal or a
set literal doesn't implement the type `Iterable`.

#### Example

The following code generates this diagnostic:

{% prettify dart %}
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...[!m!]};
{% endprettify %}

#### Common fixes

The most common fix is to replace the expression with one that produces an
iterable object:

{% prettify dart %}
var m = <String, int>{'a': 0, 'b': 1};
var s = <String>{...m.keys};
{% endprettify %}

### not_map_spread

_Spread elements in map literals must implement 'Map'._

#### Description

The analyzer produces this diagnostic when the static type of the
expression of a spread element that appears in a map literal doesn't
implement the type `Map`.

#### Example

The following code produces this diagnostic because `l` isn't a `Map`:

{% prettify dart %}
var l =  <String>['a', 'b'];
var m = <int, String>{...[!l!]};
{% endprettify %}

#### Common fixes

The most common fix is to replace the expression with one that produces a
map:

{% prettify dart %}
var l =  <String>['a', 'b'];
var m = <int, String>{...l.asMap()};
{% endprettify %}

### override_on_non_overriding_member

_The field doesn't override an inherited getter or setter._

_The getter doesn't override an inherited getter._

_The method doesn't override an inherited method._

_The setter doesn't override an inherited setter._

#### Description

The analyzer produces this diagnostic when a class member is annotated with
the `@override` annotation, but the member isn’t declared in any of the
supertypes of the class.

#### Example

The following code produces this diagnostic because `m` isn't declared in
any of the supertypes of `C`:

{% prettify dart %}
class C {
  @override
  String [!m!]() => '';
}
{% endprettify %}

#### Common fixes

If the member is intended to override a member with a different name, then
update the member to have the same name:

{% prettify dart %}
class C {
  @override
  String toString() => '';
}
{% endprettify %}

If the member is intended to override a member that was removed from the
superclass, then consider removing the member from the subclass.

If the member can't be removed, then remove the annotation.

### part_of_non_part

_The included part '{0}' must have a part-of directive._

#### Description

The analyzer produces this diagnostic when a part directive is found and
the referenced file doesn't have a part-of directive.

#### Example

Given a file (`a.dart`) containing:

{% prettify dart %}
class A {}
{% endprettify %}

The following code produces this diagnostic because `a.dart` doesn't
contain a part-of directive:

{% prettify dart %}
part [!'a.dart'!];
{% endprettify %}

#### Common fixes

If the referenced file is intended to be a part of another library, then
add a part-of directive to the file:

{% prettify dart %}
part of 'test.dart';

class A {}
{% endprettify %}

If the referenced file is intended to be a library, then replace the part
directive with an import directive:

{% prettify dart %}
import 'a.dart';
{% endprettify %}

### redirect_to_non_class

_The name '{0}' isn't a type and can't be used in a redirected constructor._

#### Description

One way to implement a factory constructor is to redirect to another
constructor by referencing the name of the constructor. The analyzer
produces this diagnostic when the redirect is to something other than a
constructor.

#### Example

The following code produces this diagnostic because `f` is a function:

{% prettify dart %}
C f() => null;

class C {
  factory C() = [!f!];
}
{% endprettify %}

#### Common fixes

If the constructor isn't defined, then either define it or replace it with
a constructor that is defined.

If the constructor is defined but the class that defines it isn't visible,
then you probably need to add an import.

If you're trying to return the value returned by a function, then rewrite
the constructor to return the value from the constructor's body:

{% prettify dart %}
C f() => null;

class C {
  factory C() => f();
}
{% endprettify %}

### referenced_before_declaration

_Local variable '{0}' can't be referenced before it is declared._

#### Description

The analyzer produces this diagnostic when a variable is referenced before
it’s declared. In Dart, variables are visible everywhere in the block in
which they are declared, but can only be referenced after they are
declared.

The analyzer also produces a context message that indicates where the
declaration is located.

#### Example

The following code produces this diagnostic because `i` is used before it
is declared:

{% prettify dart %}
void f() {
  print([!i!]);
  int i = 5;
}
{% endprettify %}

#### Common fixes

If you intended to reference the local variable, move the declaration
before the first reference:

{% prettify dart %}
void f() {
  int i = 5;
  print(i);
}
{% endprettify %}

If you intended to reference a name from an outer scope, such as a
parameter, instance field or top-level variable, then rename the local
declaration so that it doesn't hide the outer variable.

{% prettify dart %}
void f(int i) {
  print(i);
  int x = 5;
  print(x);
}
{% endprettify %}

### return_of_invalid_type

_A value of type '{0}' can't be returned from function '{2}' because it has a
return type of '{1}'._

_A value of type '{0}' can't be returned from method '{2}' because it has a
return type of '{1}'._

#### Description

The analyzer produces this diagnostic when a method or function returns a
value whose type isn't assignable to the declared return type.

#### Example

The following code produces this diagnostic because `f` has a return type
of `String` but is returning an `int`:

{% prettify dart %}
String f() => [!3!];
{% endprettify %}

#### Common fixes

If the return type is correct, then replace the value being returned with a
value of the correct type, possibly by converting the existing value:

{% prettify dart %}
String f() => 3.toString();
{% endprettify %}

If the value is correct, then change the return type to match:

{% prettify dart %}
int f() => 3;
{% endprettify %}

### return_of_invalid_type_from_closure

_The return type '{0}' isn't a '{1}', as required by the closure's context._

#### Description

The analyzer produces this diagnostic when the static type of a returned
expression isn't assignable to the return type that the closure is required
to have.

#### Example

The following code produces this diagnostic because `f` is defined to be a
function that returns a `String`, but the closure assigned to it returns an
`int`:

{% prettify dart %}
String Function(String) f = (s) => [!3!];
{% endprettify %}

#### Common fixes

If the return type is correct, then replace the returned value with a value
of the correct type, possibly by converting the existing value:

{% prettify dart %}
String Function(String) f = (s) => 3.toString();
{% endprettify %}

### sdk_version_async_exported_from_core

_The class '{0}' wasn't exported from 'dart:core' until version 2.1, but this
code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when either the class `Future` or
`Stream` is referenced in a library that doesn't import `dart:async` in
code that has an SDK constraint whose lower bound is less than 2.1.0. In
earlier versions, these classes weren't defined in `dart:core`, so the
import was necessary.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.1.0:

```yaml
environment:
  sdk: '>=2.0.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart %}
void f([!Future!] f) {}
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the classes to be referenced:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

If you need to support older versions of the SDK, then import the
`dart:async` library.

{% prettify dart %}
import 'dart:async';

void f(Future f) {}
{% endprettify %}

### sdk_version_as_expression_in_const_context

_The use of an as expression in a constant expression wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an as expression inside a
[constant context](#constant-context) is found in code that has an SDK
constraint whose lower bound is less than 2.3.2. Using an as expression in
a [constant context](#constant-context) wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following generates
this diagnostic:

{% prettify dart %}
const num n = 3;
const int i = [!n as int!];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use an as expression, or change the code so that the as
expression is not in a [constant context](#constant-context).:

{% prettify dart %}
num x = 3;
int y = x as int;
{% endprettify %}

### sdk_version_bool_operator_in_const_context

_The use of the operator '{0}' for 'bool' operands in a constant context wasn't
supported until version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when any use of the `&`, `|` or `^`
operators on the class `bool` inside a
[constant context](#constant-context) is found in code that has an SDK
constraint whose lower bound is less than 2.3.2. Using these operators in a
[constant context](#constant-context) wasn't supported in earlier versions,
so this code won't be able to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart %}
const bool a = true;
const bool b = false;
const bool c = a [!&!] b;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operators to be used:

```yaml
environment:
 sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use these operators, or change the code so that the expression
is not in a [constant context](#constant-context).:

{% prettify dart %}
const bool a = true;
const bool b = false;
bool c = a & b;
{% endprettify %}

### sdk_version_eq_eq_operator_in_const_context

_Using the operator '==' for non-primitive types wasn't supported until version
2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when the operator `==` is used on a
non-primitive type inside a [constant context](#constant-context) is found
in code that has an SDK constraint whose lower bound is less than 2.3.2.
Using this operator in a [constant context](#constant-context) wasn't
supported in earlier versions, so this code won't be able to run against
earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart %}
class C {}
const C a = null;
const C b = null;
const bool same = a [!==!] b;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the operator to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the `==` operator, or change the code so that the
expression is not in a [constant context](#constant-context).:

{% prettify dart %}
class C {}
const C a = null;
const C b = null;
bool same = a == b;
{% endprettify %}

### sdk_version_extension_methods

_Extension methods weren't supported until version 2.6.0, but this code is
required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an extension declaration or an
extension override is found in code that has an SDK constraint whose lower
bound is less than 2.6.0. Using extensions wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.6.0:

```yaml
environment:
 sdk: '>=2.4.0 <2.7.0'
```

In the package that has that pubspec, code like the following generates
this diagnostic:

{% prettify dart %}
[!extension!] E on String {
  void sayHello() {
    print('Hello $this');
  }
}
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.6.0 <2.7.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of extensions. The most common way to do this is to rewrite
the members of the extension as top-level functions (or methods) that take
the value that would have been bound to `this` as a parameter:

{% prettify dart %}
void sayHello(String s) {
  print('Hello $s');
}
{% endprettify %}

### sdk_version_is_expression_in_const_context

_The use of an is expression in a constant context wasn't supported until
version 2.3.2, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an is expression inside a
[constant context](#constant-context) is found in code that has an SDK
constraint whose lower bound is less than 2.3.2. Using an is expression in
a [constant context](#constant-context) wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.2:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following generates
this diagnostic:

{% prettify dart %}
const x = 4;
const y = [!x is int!] ? 0 : 1;
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the expression to be used:

```yaml
environment:
  sdk: '>=2.3.2 <2.4.0'
```

If you need to support older versions of the SDK, then either rewrite the
code to not use the is operator, or, if that's not possible, change the
code so that the is expression is not in a
[constant context](#constant-context).:

{% prettify dart %}
const x = 4;
var y = x is int ? 0 : 1;
{% endprettify %}

### sdk_version_set_literal

_Set literals weren't supported until version 2.2, but this code is required to
be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a set literal is found in code
that has an SDK constraint whose lower bound is less than 2.2.0. Set
literals weren't supported in earlier versions, so this code won't be able
to run against earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.2.0:

```yaml
environment:
  sdk: '>=2.1.0 <2.4.0'
```

In the package that has that pubspec, code like the following produces this
diagnostic:

{% prettify dart %}
var s = [!<int>{}!];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

If you do need to support older versions of the SDK, then replace the set
literal with code that creates the set without the use of a literal:

{% prettify dart %}
var s = new Set<int>();
{% endprettify %}

### sdk_version_ui_as_code

_The for, if, and spread elements weren't supported until version 2.3.0, but
this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when a for, if, or spread element is
found in code that has an SDK constraint whose lower bound is less than
2.3.0. Using a for, if, or spread element wasn't supported in earlier
versions, so this code won't be able to run against earlier versions of the
SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.3.0:

```yaml
environment:
  sdk: '>=2.2.0 <2.4.0'
```

In the package that has that pubspec, code like the following generates
this diagnostic:

{% prettify dart %}
var digits = [[!for (int i = 0; i < 10; i++) i!]];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.3.0 <2.4.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

{% prettify dart %}
var digits = _initializeDigits();

List<int> _initializeDigits() {
  var digits = <int>[];
  for (int i = 0; i < 10; i++) {
    digits.add(i);
  }
  return digits;
}
{% endprettify %}

### sdk_version_ui_as_code_in_const_context

_The if and spread elements weren't supported in constant expressions until
version 2.5.0, but this code is required to be able to run on earlier versions._

#### Description

The analyzer produces this diagnostic when an if or spread element inside
a [constant context](#constant-context) is found in code that has an
SDK constraint whose lower bound is less than 2.5.0. Using an if or
spread element inside a [constant context](#constant-context) wasn't
supported in earlier versions, so this code won't be able to run against
earlier versions of the SDK.

#### Example

Here's an example of a pubspec that defines an SDK constraint with a lower
bound of less than 2.5.0:

```yaml
environment:
  sdk: '>=2.4.0 <2.6.0'
```

In the package that has that pubspec, code like the following generates
this diagnostic:

{% prettify dart %}
const a = [1, 2];
const b = [[!...a!]];
{% endprettify %}

#### Common fixes

If you don't need to support older versions of the SDK, then you can
increase the SDK constraint to allow the syntax to be used:

```yaml
environment:
  sdk: '>=2.5.0 <2.6.0'
```

If you need to support older versions of the SDK, then rewrite the code to
not make use of those elements:

{% prettify dart %}
const a = [1, 2];
const b = [1, 2];
{% endprettify %}

If that's not possible, change the code so that the element is not in a
[constant context](#constant-context).:

{% prettify dart %}
const a = [1, 2];
var b = [...a];
{% endprettify %}

### static_access_to_instance_member

_Instance member '{0}' can't be accessed using static access._

#### Description

The analyzer produces this diagnostic when a class name is used to access
an instance field. Instance fields don't exist on a class; they exist only
on an instance of the class.

#### Example

The following code produces this diagnostic because `x` is an instance
field:

{% prettify dart %}
class C {
  static int a;

  int b;
}

int f() => C.[!b!];
{% endprettify %}

#### Common fixes

If you intend to access a static field, then change the name of the field
to an existing static field:

{% prettify dart %}
class C {
  static int a;

  int b;
}

int f() => C.a;
{% endprettify %}

If you intend to access the instance field, then use an instance of the
class to access the field:

{% prettify dart %}
class C {
  static int a;

  int b;
}

int f(C c) => c.b;
{% endprettify %}

### super_in_extension

_The 'super' keyword can't be used in an extension because an extension doesn't
have a superclass._

#### Description

The analyzer produces this diagnostic when a member declared inside an
extension uses the `super` keyword . Extensions aren't classes and don't
have superclasses, so the `super` keyword serves no purpose.

#### Example

The following code produces this diagnostic because `super` can't be used
in an extension:

{% prettify dart %}
extension E on Object {
  String get displayString => [!super!].toString();
}
{% endprettify %}

#### Common fixes

Remove the `super` keyword :

{% prettify dart %}
extension E on Object {
  String get displayString => toString();
}
{% endprettify %}

### type_argument_not_matching_bounds

_'{0}' doesn't extend '{1}'._

#### Description

The analyzer produces this diagnostic when a type argument isn't the same
as or a subclass of the bounds of the corresponding type parameter.

#### Example

The following code produces this diagnostic because `String` isn't a
subclass of `num`:

{% prettify dart %}
class A<E extends num> {}

var a = A<[!String!]>();
{% endprettify %}

#### Common fixes

Change the type argument to be a subclass of the bounds:

{% prettify dart %}
class A<E extends num> {}

var a = A<int>();
{% endprettify %}

### type_test_with_undefined_name

_The name '{0}' isn't defined, so it can't be used in an 'is' expression._

#### Description

The analyzer produces this diagnostic when the name following the `is` in a
type test expression isn't defined.

#### Example

The following code produces this diagnostic because the name `Srting` isn't
defined:

{% prettify dart %}
void f(Object o) {
  if (o is [!Srting!]) {
    // ...
  }
}
{% endprettify %}

#### Common fixes

Replace the name with the name of a type:

{% prettify dart %}
void f(Object o) {
  if (o is String) {
    // ...
  }
}
{% endprettify %}

### undefined_annotation

_Undefined name '{0}' used as an annotation._

#### Description

The analyzer produces this diagnostic when a name that isn't defined is
used as an annotation.

#### Example

The following code produces this diagnostic because the name `undefined`
isn't defined:

{% prettify dart %}
[!@undefined!]
void f() {}
{% endprettify %}

#### Common fixes

If the name is correct, but it isn’t declared yet, then declare the name as
a constant value:

{% prettify dart %}
const undefined = 'undefined';

@undefined
void f() {}
{% endprettify %}

If the name is wrong, replace the name with the name of a valid constant:

{% prettify dart %}
@deprecated
void f() {}
{% endprettify %}

Otherwise, remove the annotation.

### undefined_class

_Undefined class '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a class but either isn't defined or isn't visible
in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because `Piont` isn't defined:

{% prettify dart %}
class Point {}

void f([!Piont!] p) {}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a class that is defined. The example above can be corrected by
fixing the spelling of the class:

{% prettify dart %}
class Point {}

void f(Point p) {}
{% endprettify %}

If the class is defined but isn't visible, then you probably need to add an
import.

### undefined_extension_getter

_The getter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a getter, but the getter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static getter is
referenced but isn't defined by the specified extension.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't declare an instance getter named `b`:

{% prettify dart %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').[!b!];
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static getter named `a`:

{% prettify dart %}
extension E on String {}

var x = E.[!a!];
{% endprettify %}

#### Common fixes

If the name of the getter is incorrect, then change it to the name of an
existing getter:

{% prettify dart %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').a;
}
{% endprettify %}

If the name of the getter is correct but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart %}
extension E on String {
  String get a => 'a';
}

extension F on String {
  String get b => 'b';
}

void f() {
  F('c').b;
}
{% endprettify %}

If the name of the getter and extension are both correct, but the getter
isn't defined, then define the getter:

{% prettify dart %}
extension E on String {
  String get a => 'a';
  String get b => 'z';
}

extension F on String {
  String get b => 'b';
}

void f() {
  E('c').b;
}
{% endprettify %}

### undefined_extension_method

_The method '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a method, but the method isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static method is
referenced but isn't defined by the specified extension.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't declare an instance method named `b`:

{% prettify dart %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').[!b!]();
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static method named `a`:

{% prettify dart %}
extension E on String {}

var x = E.[!a!]();
{% endprettify %}

#### Common fixes

If the name of the method is incorrect, then change it to the name of an
existing method:

{% prettify dart %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').a();
}
{% endprettify %}

If the name of the method is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart %}
extension E on String {
  String a() => 'a';
}

extension F on String {
  String b() => 'b';
}

void f() {
  F('c').b();
}
{% endprettify %}

If the name of the method and extension are both correct, but the method
isn't defined, then define the method:

{% prettify dart %}
extension E on String {
  String a() => 'a';
  String b() => 'z';
}

extension F on String {
  String b() => 'b';
}

void f() {
  E('c').b();
}
{% endprettify %}

### undefined_extension_setter

_The setter '{0}' isn't defined for the extension '{1}'._

#### Description

The analyzer produces this diagnostic when an extension override is used to
invoke a setter, but the setter isn't defined by the specified extension.
The analyzer also produces this diagnostic when a static setter is
referenced but isn't defined by the specified extension.

#### Example

The following code produces this diagnostic because the extension `E`
doesn't declare an instance setter named `b`:

{% prettify dart %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').[!b!] = 'd';
}
{% endprettify %}

The following code produces this diagnostic because the extension `E`
doesn't declare a static setter named `a`:

{% prettify dart %}
extension E on String {}

void f() {
  E.[!a!] = 3;
}
{% endprettify %}

#### Common fixes

If the name of the setter is incorrect, then change it to the name of an
existing setter:

{% prettify dart %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').a = 'd';
}
{% endprettify %}

If the name of the setter is correct, but the name of the extension is
wrong, then change the name of the extension to the correct name:

{% prettify dart %}
extension E on String {
  set a(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  F('c').b = 'd';
}
{% endprettify %}

If the name of the setter and extension are both correct, but the setter
isn't defined, then define the setter:

{% prettify dart %}
extension E on String {
  set a(String v) {}
  set b(String v) {}
}

extension F on String {
  set b(String v) {}
}

void f() {
  E('c').b = 'd';
}
{% endprettify %}

### undefined_function

_The function '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a function but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because the name `emty` isn't
defined:

{% prettify dart %}
List<int> empty() => [];

void main() {
  print([!emty!]());
}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a function that is defined. The example above can be corrected
by fixing the spelling of the function:

{% prettify dart %}
List<int> empty() => [];

void main() {
  print(empty());
}
{% endprettify %}

If the function is defined but isn't visible, then you probably need to add
an import or re-arrange your code to make the function visible.

### undefined_getter

_The getter '{0}' isn't defined for the class '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a getter but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because `String` has no member
named `len`:

{% prettify dart %}
int f(String s) => s.[!len!];
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a getter that is defined. The example above can be corrected by
fixing the spelling of the getter:

{% prettify dart %}
int f(String s) => s.length;
{% endprettify %}

### undefined_identifier

_Undefined name '{0}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
either isn't defined or isn't visible in the scope in which it's being
referenced.

#### Example

The following code produces this diagnostic because the name `rihgt` isn't
defined:

{% prettify dart %}
int min(int left, int right) => left <= [!rihgt!] ? left : right;
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
an identifier that is defined. The example above can be corrected by
fixing the spelling of the variable:

{% prettify dart %}
int min(int left, int right) => left <= right ? left : right;
{% endprettify %}

If the identifier is defined but isn't visible, then you probably need to
add an import or re-arrange your code to make the identifier visible.

### undefined_method

_The method '{0}' isn't defined for the class '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a method but either isn't defined or isn't
visible in the scope in which it's being referenced.

#### Example

The following code produces this diagnostic because the identifier
`removeMiddle` isn't defined:

{% prettify dart %}
int f(List<int> l) => l.[!removeMiddle!]();
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a method that is defined. The example above can be corrected by
fixing the spelling of the method:

{% prettify dart %}
int f(List<int> l) => l.removeLast();
{% endprettify %}

### undefined_named_parameter

_The named parameter '{0}' isn't defined._

#### Description

The analyzer produces this diagnostic when a method or function invocation
has a named argument, but the method or function being invoked doesn't
define a parameter with the same name.

#### Example

The following code produces this diagnostic because `m` doesn't declare a
named parameter named `a`:

{% prettify dart %}
class C {
  m({int b}) {}
}

void f(C c) {
  c.m([!a!]: 1);
}
{% endprettify %}

#### Common fixes

If the argument name is mistyped, then replace it with the correct name.
The example above can be fixed by changing `a` to `b`:

{% prettify dart %}
class C {
  m({int b}) {}
}

void f(C c) {
  c.m(b: 1);
}
{% endprettify %}

If a subclass adds a parameter with the name in question, then cast the
target to the subclass:

{% prettify dart %}
class C {
  m({int b}) {}
}

class D extends C {
  m({int a, int b}) {}
}

void f(C c) {
  (c as D).m(a: 1);
}
{% endprettify %}

If the parameter should be added to the function, then add it:

{% prettify dart %}
class C {
  m({int a, int b}) {}
}

void f(C c) {
  c.m(a: 1);
}
{% endprettify %}

### undefined_operator

_The operator '{0}' isn't defined for the class '{1}'._

#### Description

The analyzer produces this diagnostic when a user-definable operator is
invoked on an object for which the operator isn't defined.

#### Example

The following code produces this diagnostic because the class `C` doesn't
define the operator `+`:

{% prettify dart %}
class C {}

C f(C c) => c [!+!] 2;
{% endprettify %}

#### Common fixes

If the operator should be defined for the class, then define it:

{% prettify dart %}
class C {
  C operator +(int i) => this;
}

C f(C c) => c + 2;
{% endprettify %}

### undefined_prefixed_name

_The name '{0}' is being referenced through the prefix '{1}', but it isn't
defined in any of the libraries imported using that prefix._

#### Description

The analyzer produces this diagnostic when a prefixed identifier is found
where the prefix is valid, but the identifier isn't declared in any of the
libraries imported using that prefix.

#### Example

The following code produces this diagnostic because `dart:core` doesn't
define anything named `a`:

{% prettify dart %}
import 'dart:core' as p;

void f() {
  p.[!a!];
}
{% endprettify %}

#### Common fixes

If the library in which the name is declared isn't imported yet, add an
import for the library.

If the name is wrong, then change it to one of the names that's declared in
the imported libraries.

### undefined_setter

_The setter '{0}' isn't defined for the class '{1}'._

#### Description

The analyzer produces this diagnostic when it encounters an identifier that
appears to be the name of a setter but either isn't defined or isn't
visible in the scope in which the identifier is being referenced.

#### Example

The following code produces this diagnostic because there isn't a setter
named `z`:

{% prettify dart %}
class C {
  int x = 0;
  void m(int y) {
    this.[!z!] = y;
  }
}
{% endprettify %}

#### Common fixes

If the identifier isn't defined, then either define it or replace it with
the name of a setter that is defined. The example above can be corrected by
fixing the spelling of the setter:

{% prettify dart %}
class C {
  int x = 0;
  void m(int y) {
    this.x = y;
  }
}
{% endprettify %}

### undefined_super_method

_The method '{0}' isn't defined in a superclass of '{1}'._

#### Description

The analyzer produces this diagnostic when an inherited method is
referenced using `super`, but there’s no method with that name in the
superclass chain.

#### Example

The following code produces this diagnostic because `Object` doesn't define
a member named `n`:

{% prettify dart %}
class C {
  void m() {
    super.[!n!]();
  }
}
{% endprettify %}

#### Common fixes

If the inherited method you intend to invoke has a different name, then
make the name of the invoked method  match the inherited method.

If the method you intend to invoke is defined in the same class, then
remove the `super.`.

If not, then either add the method to one of the superclasses or remove the
invocation.

### unqualified_reference_to_static_member_of_extended_type

_Static members from the extended type or one of its superclasses must be
qualified by the name of the defining type._

#### Description

The analyzer produces this diagnostic when an undefined name is found, and
the name is the same as a static member of the extended type or one of its
superclasses.

#### Example

The following code produces this diagnostic because `m` is a static member
of the extended type `C`:

{% prettify dart %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    [!m!]();
  }
}
{% endprettify %}

#### Common fixes

If you're trying to reference a static member that's declared outside the
extension, then add the name of the class or extension before the reference
to the member:

{% prettify dart %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    C.m();
  }
}
{% endprettify %}

If you're referencing a member that isn't declared yet, add a declaration:

{% prettify dart %}
class C {
  static void m() {}
}

extension E on C {
  void f() {
    m();
  }

  void m() {}
}
{% endprettify %}

### unused_element

_The declaration '{0}' isn't referenced._

#### Description

The analyzer produces this diagnostic when a private class, enum, mixin,
typedef, top level variable, top level function, or method is declared but
never referenced.

#### Example

Assuming that no code in the library references `_C`, the following code
produces this diagnostic:

{% prettify dart %}
class [!_C!] {}
{% endprettify %}

#### Common fixes

If the declaration isn't needed, then remove it.

If the declaration was intended to be used, then add the missing code.

### unused_field

_The value of the field '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a private field is declared but
never read, even if it's written in one or more places.

#### Example

The following code produces this diagnostic because `_x` isn't referenced
anywhere in the library:

{% prettify dart %}
class Point {
  int [!_x!];
}
{% endprettify %}

#### Common fixes

If the field isn't needed, then remove it.

If the field was intended to be used, then add the missing code.

### unused_import

_Unused import: '{0}'._

#### Description

The analyzer produces this diagnostic when an import isn't needed because
none of the names that are imported are referenced within the importing
library.

#### Example

The following code produces this diagnostic because nothing defined in
`dart:async` is referenced in the library:

{% prettify dart %}
import [!'dart:async'!];

void main() {}
{% endprettify %}

#### Common fixes

If the import isn't needed, then remove it.

If some of the imported names are intended to be used, then add the missing
code.

### unused_local_variable

_The value of the local variable '{0}' isn't used._

#### Description

The analyzer produces this diagnostic when a local variable is declared but
never read, even if it's written in one or more places.

#### Example

The following code produces this diagnostic because the value of `count` is
never read:

{% prettify dart %}
void main() {
  int [!count!] = 0;
}
{% endprettify %}

#### Common fixes

If the variable isn't needed, then remove it.

If the variable was intended to be used, then add the missing code.

### uri_does_not_exist

_Target of URI doesn't exist: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist.

#### Example

If the file `lib.dart` doesn't exist, the following code produces this
diagnostic:

{% prettify dart %}
import [!'lib.dart'!];
{% endprettify %}

#### Common fixes

If the URI was mistyped or invalid, then correct the URI.

If the URI is correct, then create the file.

### uri_has_not_been_generated

_Target of URI hasn't been generated: '{0}'._

#### Description

The analyzer produces this diagnostic when an import, export, or part
directive is found where the URI refers to a file that doesn't exist and
the name of the file ends with a pattern that's commonly produced by code
generators, such as one of the following:
- `.g.dart`
- `.pb.dart`
- `.pbenum.dart`
- `.pbserver.dart`
- `.pbjson.dart`
- `.template.dart`

#### Example

If the file `lib.g.dart` doesn't exist, the following code produces this
diagnostic:

{% prettify dart %}
import [!'lib.g.dart'!];
{% endprettify %}

#### Common fixes

If the file is a generated file, then run the generator that generates the
file.

If the file isn't a generated file, then check the spelling of the URI or
create the file.

### use_of_void_result

_This expression has a type of 'void' so its value can't be used._

#### Description

The analyzer produces this diagnostic when it finds an expression whose
type is `void`, and the expression is used in a place where a value is
expected, such as before a member access or on the right-hand side of an
assignment.

#### Example

The following code produces this diagnostic because `f` doesn't produce an
object on which `toString` can be invoked:

{% prettify dart %}
void f() {}

void g() {
  [!f()!].toString();
}
{% endprettify %}

#### Common fixes

Either rewrite the code so that the expression has a value or rewrite the
code so that it doesn't depend on the value.

### variable_type_mismatch

_A value of type '{0}' can't be assigned to a variable of type '{1}'._

#### Description

The analyzer produces this diagnostic when the evaluation of a constant
expression would result in a `CastException`.

#### Example

The following code produces this diagnostic because the value of `x` is an
`int`, which can't be assigned to `y` because an `int` isn't a `String`:

{% prettify dart %}
const Object x = 0;
const String y = [!x!];
{% endprettify %}

#### Common fixes

If the declaration of the constant is correct, then change the value being
assigned to be of the correct type:

{% prettify dart %}
const Object x = 0;
const String y = '$x';
{% endprettify %}

If the assigned value is correct, then change the declaration to have the
correct type:

{% prettify dart %}
const Object x = 0;
const int y = x;
{% endprettify %}

### wrong_number_of_parameters_for_operator

_Operator '{0}' should declare exactly {1} parameters, but {2} found._

#### Description

The analyzer produces this diagnostic when a declaration of an operator has
the wrong number of parameters.

#### Example

The following code produces this diagnostic because the operator `+` must
have a single parameter corresponding to the right operand:

{% prettify dart %}
class C {
  int operator [!+!](a, b) => 0;
}
{% endprettify %}

#### Common fixes

Add or remove parameters to match the required number:

{% prettify dart %}
class C {
  int operator +(a) => 0;
}
{% endprettify %}

### wrong_number_of_type_arguments

_The type '{0}' is declared with {1} type parameters, but {2} type arguments
were given._

#### Description

The analyzer produces this diagnostic when a type that has type parameters
is used and type arguments are provided, but the number of type arguments
isn't the same as the number of type parameters.

#### Example

The following code produces this diagnostic because `C` has one type
parameter but two type arguments are provided:

{% prettify dart %}
class C<E> {}

void f([!C<int, int>!] x) {}
{% endprettify %}

#### Common fixes

Add or remove type arguments, as necessary, to match the number of type
parameters defined for the type:

{% prettify dart %}
class C<E> {}

void f(C<int> x) {}
{% endprettify %}
