// Copyright (c) 2020, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// Generates a file to test that all the Win32 API typedefs can be loaded (i.e.
// that lookupFunction works for all the APIs generated.)

import 'dart:io';

import 'package:winmd/winmd.dart';

import '../manual_gen/win32api.dart';
import '../projection/struct.dart';
import '../projection/utils.dart';

const structFileHeader = '''
// Copyright (c) 2020, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

// Dart representations of common structs used in the Win32 API.

// THIS FILE IS GENERATED AUTOMATICALLY AND SHOULD NOT BE EDITED DIRECTLY.

// -----------------------------------------------------------------------------
// Linter exceptions
// -----------------------------------------------------------------------------
// ignore_for_file: camel_case_types
// ignore_for_file: camel_case_extensions
//
// Why? The linter defaults to throw a warning for types not named as camel
// case. We deliberately break this convention to match the Win32 underlying
// types.
//
//
// ignore_for_file: unused_field
//
// Why? The linter complains about unused fields (e.g. a class that contains
// underscore-prefixed members that are not used in the library. In this class,
// we use this feature to ensure that sizeOf<STRUCT_NAME> returns a size at
// least as large as the underlying native struct. See, for example,
// ENUMLOGFONTEX.
//
//
// ignore_for_file: unnecessary_getters_setters
//
// Why? In structs like VARIANT, we're using getters and setters to project the
// same underlying data property to various union types. The trivial overhead is
// outweighed by readability.
//
//
// ignore_for_file: unused_import
//
// Why? We speculatively include some imports that we might generate a
// requirement for.
// -----------------------------------------------------------------------------

import 'dart:ffi';
import 'dart:typed_data';

import 'package:ffi/ffi.dart';

import 'callbacks.dart';
import 'com/IDispatch.dart';
import 'com/IUnknown.dart';
import 'combase.dart';
import 'guid.dart';
import 'oleaut32.dart';
import 'structs.dart';
''';

bool supports64Bit(TypeDef type) {
  final supportedArchAttribute = type.customAttributes.where((attr) =>
      attr.name == 'Windows.Win32.Interop.SupportedArchitectureAttribute');

  // [0x01, 00x00] prolog, then first digit is arch attribute.
  final arch = Architecture(supportedArchAttribute.first.signatureBlob[2]);
  return arch.x64;
}

int generateStructs(Win32API win32) {
  final scope = MetadataStore.getWin32Scope();

  var structsGenerated = 0;
  final writer =
      File('lib/src/structs.g.dart').openSync(mode: FileMode.writeOnly);

  writer.writeStringSync(structFileHeader);

  for (final struct in win32.structs.keys) {
    final win32struct = win32.structs[struct]!;
    final typeDefs = scope.typeDefs
        .where((typeDef) => typeDef.name == win32struct.namespace)
        .toList();
    if (typeDefs.isEmpty) {
      throw Exception('$struct missing');
    }
    if (typeDefs.length > 1) {
      typeDefs.retainWhere(supports64Bit);
    }
    final typeDef = typeDefs.first;

    writer.writeStringSync(wrapCommentText(win32struct.comment));
    writer.writeStringSync('\n///\n');

    final projectedStruct = StructProjection(typeDef, struct);
    writer.writeStringSync(projectedStruct.toString());
    structsGenerated++;
  }

  return structsGenerated;
}
